// Licensed under the Apache License, Version 2.0
// <LICENSE-APACHE or http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your option.
// All files in the project carrying such notice may not be copied, modified, or distributed
// except according to those terms.
//! Definitions and data structures for wlan auto config client side API.
use shared::devpropdef::DEVPROPKEY;
use shared::guiddef::{GUID, LPGUID};
use shared::minwindef::{BOOL, BYTE, DWORD, LPBYTE, PBOOL, PBYTE, PDWORD, PUCHAR, ULONG, USHORT};
use shared::windef::HWND;
use shared::windot11::{
    DOT11_COUNTRY_OR_REGION_STRING, DOT11_HESSID, DOT11_MAC_ADDRESS, DOT11_PHY_TYPE,
    DOT11_RATE_SET_MAX_LENGTH, PDOT11_BSSID_LIST, PDOT11_HESSID, PDOT11_MAC_ADDRESS,
};
use shared::wlantypes::{
    DOT11_ACCESSNETWORKOPTIONS, DOT11_AUTH_ALGORITHM, DOT11_AUTH_CIPHER_PAIR, DOT11_BSS_TYPE,
    DOT11_CIPHER_ALGORITHM, DOT11_SSID, DOT11_VENUEINFO, PDOT11_ACCESSNETWORKOPTIONS, PDOT11_SSID,
};
use um::eaptypes::EAP_METHOD_TYPE;
use um::l2cmn::{
    L2_NOTIFICATION_CODE_PUBLIC_BEGIN, L2_NOTIFICATION_CODE_V2_BEGIN, L2_NOTIFICATION_DATA,
    L2_NOTIFICATION_SOURCE_ALL, L2_NOTIFICATION_SOURCE_NONE, L2_NOTIFICATION_SOURCE_ONEX,
    L2_NOTIFICATION_SOURCE_WLAN_ACM, L2_NOTIFICATION_SOURCE_WLAN_HNWK,
    L2_NOTIFICATION_SOURCE_WLAN_IHV, L2_NOTIFICATION_SOURCE_WLAN_MSM,
    L2_NOTIFICATION_SOURCE_WLAN_SECURITY, L2_PROFILE_MAX_NAME_LENGTH,
    L2_REASON_CODE_DOT11_AC_BASE, L2_REASON_CODE_DOT11_MSM_BASE,
    L2_REASON_CODE_DOT11_SECURITY_BASE, L2_REASON_CODE_GROUP_SIZE, L2_REASON_CODE_PROFILE_BASE,
    L2_REASON_CODE_SUCCESS, L2_REASON_CODE_UNKNOWN,
};
use um::winnt::{
    BOOLEAN, DELETE, FILE_EXECUTE, FILE_READ_DATA, FILE_WRITE_DATA, HANDLE, LONG, LPCWSTR, LPWSTR,
    PHANDLE, PVOID, PWCHAR, STANDARD_RIGHTS_EXECUTE, STANDARD_RIGHTS_READ, STANDARD_RIGHTS_WRITE,
    ULONGLONG, WCHAR, WRITE_DAC,
};
pub const WLAN_API_VERSION_1_0: DWORD = 0x00000001;
pub const WLAN_API_VERSION_2_0: DWORD = 0x00000002;
#[inline]
pub fn WLAN_API_VERSION_MAJOR(v: u32) -> u32 {
    v & 0xffff
}
#[inline]
pub fn WLAN_API_VERSION_MINOR(v: u32) -> u32 {
    v >> 16
}
#[inline]
pub fn WLAN_API_MAKE_VERSION(major: u32, minor: u32) -> u32 {
    minor << 16 | major
}
pub const WLAN_API_VERSION: u32 = WLAN_API_VERSION_2_0;
pub const WLAN_MAX_NAME_LENGTH: usize = L2_PROFILE_MAX_NAME_LENGTH;
pub const WLAN_PROFILE_GROUP_POLICY: DWORD = 0x00000001;
pub const WLAN_PROFILE_USER: DWORD = 0x00000002;
pub const WLAN_PROFILE_GET_PLAINTEXT_KEY: DWORD = 0x00000004;
pub const WLAN_PROFILE_CONNECTION_MODE_SET_BY_CLIENT: DWORD = 0x00010000;
pub const WLAN_PROFILE_CONNECTION_MODE_AUTO: DWORD = 0x00020000;
pub const WLAN_SET_EAPHOST_DATA_ALL_USERS: DWORD = 0x00000001;
STRUCT!{struct WLAN_PROFILE_INFO {
    strProfileName: [WCHAR; WLAN_MAX_NAME_LENGTH],
    dwFlags: DWORD,
}}
pub type PWLAN_PROFILE_INFO = *mut WLAN_PROFILE_INFO;
STRUCT!{struct DOT11_NETWORK {
    dot11Ssid: DOT11_SSID,
    dot11BssType: DOT11_BSS_TYPE,
}}
pub type PDOT11_NETWORK = *mut DOT11_NETWORK;
pub const DOT11_PSD_IE_MAX_DATA_SIZE: DWORD = 240;
pub const DOT11_PSD_IE_MAX_ENTRY_NUMBER: DWORD = 5;
STRUCT!{struct WLAN_RAW_DATA {
    dwDataSize: DWORD,
    DataBlob: [BYTE; 1],
}}
pub type PWLAN_RAW_DATA = *mut WLAN_RAW_DATA;
STRUCT!{struct WLAN_RAW_DATA_LIST_DataList {
    dwDataOffset: DWORD,
    dwDataSize: DWORD,
}}
STRUCT!{struct WLAN_RAW_DATA_LIST {
    dwTotalSize: DWORD,
    dwNumberOfItems: DWORD,
    DataList: [WLAN_RAW_DATA_LIST_DataList; 1],
}}
pub type PWLAN_RAW_DATA_LIST = *mut WLAN_RAW_DATA_LIST;
ENUM!{enum WLAN_CONNECTION_MODE {
    wlan_connection_mode_profile = 0,
    wlan_connection_mode_temporary_profile = 1,
    wlan_connection_mode_discovery_secure = 2,
    wlan_connection_mode_discovery_unsecure = 3,
    wlan_connection_mode_auto = 4,
    wlan_connection_mode_invalid = 5,
}}
pub type PWLAN_CONNECTION_MODE = *mut WLAN_CONNECTION_MODE;
pub type WLAN_REASON_CODE = DWORD;
pub type PWLAN_REASON_CODE = *mut DWORD;
pub const WLAN_REASON_CODE_SUCCESS: WLAN_REASON_CODE = L2_REASON_CODE_SUCCESS;
pub const WLAN_REASON_CODE_UNKNOWN: WLAN_REASON_CODE = L2_REASON_CODE_UNKNOWN;
pub const WLAN_REASON_CODE_RANGE_SIZE: WLAN_REASON_CODE = L2_REASON_CODE_GROUP_SIZE;
pub const WLAN_REASON_CODE_BASE: WLAN_REASON_CODE = L2_REASON_CODE_DOT11_AC_BASE;
pub const WLAN_REASON_CODE_AC_BASE: WLAN_REASON_CODE = L2_REASON_CODE_DOT11_AC_BASE;
pub const WLAN_REASON_CODE_AC_CONNECT_BASE: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_BASE
    + WLAN_REASON_CODE_RANGE_SIZE / 2;
pub const WLAN_REASON_CODE_AC_END: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_BASE
    + WLAN_REASON_CODE_RANGE_SIZE - 1;
pub const WLAN_REASON_CODE_PROFILE_BASE: WLAN_REASON_CODE = L2_REASON_CODE_PROFILE_BASE;
pub const WLAN_REASON_CODE_PROFILE_CONNECT_BASE: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE
    + WLAN_REASON_CODE_RANGE_SIZE / 2;
pub const WLAN_REASON_CODE_PROFILE_END: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE
    + WLAN_REASON_CODE_RANGE_SIZE - 1;
pub const WLAN_REASON_CODE_MSM_BASE: WLAN_REASON_CODE = L2_REASON_CODE_DOT11_MSM_BASE;
pub const WLAN_REASON_CODE_MSM_CONNECT_BASE: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_BASE
    + WLAN_REASON_CODE_RANGE_SIZE / 2;
pub const WLAN_REASON_CODE_MSM_END: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_BASE
    + WLAN_REASON_CODE_RANGE_SIZE - 1;
pub const WLAN_REASON_CODE_MSMSEC_BASE: WLAN_REASON_CODE = L2_REASON_CODE_DOT11_SECURITY_BASE;
pub const WLAN_REASON_CODE_MSMSEC_CONNECT_BASE: WLAN_REASON_CODE = WLAN_REASON_CODE_MSMSEC_BASE
    + WLAN_REASON_CODE_RANGE_SIZE / 2;
pub const WLAN_REASON_CODE_MSMSEC_END: WLAN_REASON_CODE = WLAN_REASON_CODE_MSMSEC_BASE
    + WLAN_REASON_CODE_RANGE_SIZE - 1;
pub const WLAN_REASON_CODE_NETWORK_NOT_COMPATIBLE: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_BASE + 1;
pub const WLAN_REASON_CODE_PROFILE_NOT_COMPATIBLE: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_BASE + 2;
pub const WLAN_REASON_CODE_NO_AUTO_CONNECTION: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_CONNECT_BASE
    + 1;
pub const WLAN_REASON_CODE_NOT_VISIBLE: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_CONNECT_BASE + 2;
pub const WLAN_REASON_CODE_GP_DENIED: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_CONNECT_BASE + 3;
pub const WLAN_REASON_CODE_USER_DENIED: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_CONNECT_BASE + 4;
pub const WLAN_REASON_CODE_BSS_TYPE_NOT_ALLOWED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_AC_CONNECT_BASE + 5;
pub const WLAN_REASON_CODE_IN_FAILED_LIST: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_CONNECT_BASE + 6;
pub const WLAN_REASON_CODE_IN_BLOCKED_LIST: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_CONNECT_BASE
    + 7;
pub const WLAN_REASON_CODE_SSID_LIST_TOO_LONG: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_CONNECT_BASE
    + 8;
pub const WLAN_REASON_CODE_CONNECT_CALL_FAIL: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_CONNECT_BASE
    + 9;
pub const WLAN_REASON_CODE_SCAN_CALL_FAIL: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_CONNECT_BASE
    + 10;
pub const WLAN_REASON_CODE_NETWORK_NOT_AVAILABLE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_AC_CONNECT_BASE + 11;
pub const WLAN_REASON_CODE_PROFILE_CHANGED_OR_DELETED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_AC_CONNECT_BASE + 12;
pub const WLAN_REASON_CODE_KEY_MISMATCH: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_CONNECT_BASE + 13;
pub const WLAN_REASON_CODE_USER_NOT_RESPOND: WLAN_REASON_CODE = WLAN_REASON_CODE_AC_CONNECT_BASE
    + 14;
pub const WLAN_REASON_CODE_AP_PROFILE_NOT_ALLOWED_FOR_CLIENT: WLAN_REASON_CODE =
    WLAN_REASON_CODE_AC_CONNECT_BASE + 15;
pub const WLAN_REASON_CODE_AP_PROFILE_NOT_ALLOWED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_AC_CONNECT_BASE + 16;
pub const WLAN_REASON_CODE_HOTSPOT2_PROFILE_DENIED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_AC_CONNECT_BASE + 17;
pub const WLAN_REASON_CODE_INVALID_PROFILE_SCHEMA: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE
    + 1;
pub const WLAN_REASON_CODE_PROFILE_MISSING: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE + 2;
pub const WLAN_REASON_CODE_INVALID_PROFILE_NAME: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE
    + 3;
pub const WLAN_REASON_CODE_INVALID_PROFILE_TYPE: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE
    + 4;
pub const WLAN_REASON_CODE_INVALID_PHY_TYPE: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE + 5;
pub const WLAN_REASON_CODE_MSM_SECURITY_MISSING: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE
    + 6;
pub const WLAN_REASON_CODE_IHV_SECURITY_NOT_SUPPORTED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_PROFILE_BASE + 7;
pub const WLAN_REASON_CODE_IHV_OUI_MISMATCH: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE + 8;
pub const WLAN_REASON_CODE_IHV_OUI_MISSING: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE + 9;
pub const WLAN_REASON_CODE_IHV_SETTINGS_MISSING: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE
    + 10;
pub const WLAN_REASON_CODE_CONFLICT_SECURITY: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE
    + 11;
pub const WLAN_REASON_CODE_SECURITY_MISSING: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE + 12;
pub const WLAN_REASON_CODE_INVALID_BSS_TYPE: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE + 13;
pub const WLAN_REASON_CODE_INVALID_ADHOC_CONNECTION_MODE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_PROFILE_BASE + 14;
pub const WLAN_REASON_CODE_NON_BROADCAST_SET_FOR_ADHOC: WLAN_REASON_CODE =
    WLAN_REASON_CODE_PROFILE_BASE + 15;
pub const WLAN_REASON_CODE_AUTO_SWITCH_SET_FOR_ADHOC: WLAN_REASON_CODE =
    WLAN_REASON_CODE_PROFILE_BASE + 16;
pub const WLAN_REASON_CODE_AUTO_SWITCH_SET_FOR_MANUAL_CONNECTION: WLAN_REASON_CODE =
    WLAN_REASON_CODE_PROFILE_BASE + 17;
pub const WLAN_REASON_CODE_IHV_SECURITY_ONEX_MISSING: WLAN_REASON_CODE =
    WLAN_REASON_CODE_PROFILE_BASE + 18;
pub const WLAN_REASON_CODE_PROFILE_SSID_INVALID: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE
    + 19;
pub const WLAN_REASON_CODE_TOO_MANY_SSID: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE + 20;
pub const WLAN_REASON_CODE_IHV_CONNECTIVITY_NOT_SUPPORTED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_PROFILE_BASE + 21;
pub const WLAN_REASON_CODE_BAD_MAX_NUMBER_OF_CLIENTS_FOR_AP: WLAN_REASON_CODE =
    WLAN_REASON_CODE_PROFILE_BASE + 22;
pub const WLAN_REASON_CODE_INVALID_CHANNEL: WLAN_REASON_CODE = WLAN_REASON_CODE_PROFILE_BASE + 23;
pub const WLAN_REASON_CODE_OPERATION_MODE_NOT_SUPPORTED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_PROFILE_BASE + 24;
pub const WLAN_REASON_CODE_AUTO_AP_PROFILE_NOT_ALLOWED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_PROFILE_BASE + 25;
pub const WLAN_REASON_CODE_AUTO_CONNECTION_NOT_ALLOWED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_PROFILE_BASE + 26;
pub const WLAN_REASON_CODE_HOTSPOT2_PROFILE_NOT_ALLOWED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_PROFILE_BASE + 27;
pub const WLAN_REASON_CODE_UNSUPPORTED_SECURITY_SET_BY_OS: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSM_BASE + 1;
pub const WLAN_REASON_CODE_UNSUPPORTED_SECURITY_SET: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_BASE
    + 2;
pub const WLAN_REASON_CODE_BSS_TYPE_UNMATCH: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_BASE + 3;
pub const WLAN_REASON_CODE_PHY_TYPE_UNMATCH: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_BASE + 4;
pub const WLAN_REASON_CODE_DATARATE_UNMATCH: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_BASE + 5;
pub const WLAN_REASON_CODE_USER_CANCELLED: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_CONNECT_BASE
    + 1;
pub const WLAN_REASON_CODE_ASSOCIATION_FAILURE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSM_CONNECT_BASE + 2;
pub const WLAN_REASON_CODE_ASSOCIATION_TIMEOUT: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSM_CONNECT_BASE + 3;
pub const WLAN_REASON_CODE_PRE_SECURITY_FAILURE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSM_CONNECT_BASE + 4;
pub const WLAN_REASON_CODE_START_SECURITY_FAILURE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSM_CONNECT_BASE + 5;
pub const WLAN_REASON_CODE_SECURITY_FAILURE: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_CONNECT_BASE
    + 6;
pub const WLAN_REASON_CODE_SECURITY_TIMEOUT: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_CONNECT_BASE
    + 7;
pub const WLAN_REASON_CODE_ROAMING_FAILURE: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_CONNECT_BASE
    + 8;
pub const WLAN_REASON_CODE_ROAMING_SECURITY_FAILURE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSM_CONNECT_BASE + 9;
pub const WLAN_REASON_CODE_ADHOC_SECURITY_FAILURE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSM_CONNECT_BASE + 10;
pub const WLAN_REASON_CODE_DRIVER_DISCONNECTED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSM_CONNECT_BASE + 11;
pub const WLAN_REASON_CODE_DRIVER_OPERATION_FAILURE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSM_CONNECT_BASE + 12;
pub const WLAN_REASON_CODE_IHV_NOT_AVAILABLE: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_CONNECT_BASE
    + 13;
pub const WLAN_REASON_CODE_IHV_NOT_RESPONDING: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_CONNECT_BASE
    + 14;
pub const WLAN_REASON_CODE_DISCONNECT_TIMEOUT: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_CONNECT_BASE
    + 15;
pub const WLAN_REASON_CODE_INTERNAL_FAILURE: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_CONNECT_BASE
    + 16;
pub const WLAN_REASON_CODE_UI_REQUEST_TIMEOUT: WLAN_REASON_CODE = WLAN_REASON_CODE_MSM_CONNECT_BASE
    + 17;
pub const WLAN_REASON_CODE_TOO_MANY_SECURITY_ATTEMPTS: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSM_CONNECT_BASE + 18;
pub const WLAN_REASON_CODE_AP_STARTING_FAILURE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSM_CONNECT_BASE + 19;
pub const WLAN_REASON_CODE_MSMSEC_MIN: WLAN_REASON_CODE = WLAN_REASON_CODE_MSMSEC_BASE;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_KEY_INDEX: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 1;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_PSK_PRESENT: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 2;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_KEY_LENGTH: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 3;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_PSK_LENGTH: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 4;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_NO_AUTH_CIPHER_SPECIFIED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 5;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_TOO_MANY_AUTH_CIPHER_SPECIFIED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 6;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_DUPLICATE_AUTH_CIPHER: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 7;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_RAWDATA_INVALID: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 8;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_AUTH_CIPHER: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 9;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_ONEX_DISABLED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 10;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_ONEX_ENABLED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 11;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_PMKCACHE_MODE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 12;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_PMKCACHE_SIZE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 13;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_PMKCACHE_TTL: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 14;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_PREAUTH_MODE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 15;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_PREAUTH_THROTTLE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 16;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_PREAUTH_ONLY_ENABLED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 17;
pub const WLAN_REASON_CODE_MSMSEC_CAPABILITY_NETWORK: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 18;
pub const WLAN_REASON_CODE_MSMSEC_CAPABILITY_NIC: WLAN_REASON_CODE = WLAN_REASON_CODE_MSMSEC_BASE
    + 19;
pub const WLAN_REASON_CODE_MSMSEC_CAPABILITY_PROFILE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 20;
pub const WLAN_REASON_CODE_MSMSEC_CAPABILITY_DISCOVERY: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 21;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_PASSPHRASE_CHAR: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 22;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_KEYMATERIAL_CHAR: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 23;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_WRONG_KEYTYPE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 24;
pub const WLAN_REASON_CODE_MSMSEC_MIXED_CELL: WLAN_REASON_CODE = WLAN_REASON_CODE_MSMSEC_BASE + 25;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_AUTH_TIMERS_INVALID: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 26;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_INVALID_GKEY_INTV: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 27;
pub const WLAN_REASON_CODE_MSMSEC_TRANSITION_NETWORK: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 28;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_KEY_UNMAPPED_CHAR: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 29;
pub const WLAN_REASON_CODE_MSMSEC_CAPABILITY_PROFILE_AUTH: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 30;
pub const WLAN_REASON_CODE_MSMSEC_CAPABILITY_PROFILE_CIPHER: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 31;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_SAFE_MODE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 32;
pub const WLAN_REASON_CODE_MSMSEC_CAPABILITY_PROFILE_SAFE_MODE_NIC: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 33;
pub const WLAN_REASON_CODE_MSMSEC_CAPABILITY_PROFILE_SAFE_MODE_NW: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 34;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_UNSUPPORTED_AUTH: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 35;
pub const WLAN_REASON_CODE_MSMSEC_PROFILE_UNSUPPORTED_CIPHER: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 36;
pub const WLAN_REASON_CODE_MSMSEC_CAPABILITY_MFP_NW_NIC: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_BASE + 37;
pub const WLAN_REASON_CODE_MSMSEC_UI_REQUEST_FAILURE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 1;
pub const WLAN_REASON_CODE_MSMSEC_AUTH_START_TIMEOUT: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 2;
pub const WLAN_REASON_CODE_MSMSEC_AUTH_SUCCESS_TIMEOUT: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 3;
pub const WLAN_REASON_CODE_MSMSEC_KEY_START_TIMEOUT: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 4;
pub const WLAN_REASON_CODE_MSMSEC_KEY_SUCCESS_TIMEOUT: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 5;
pub const WLAN_REASON_CODE_MSMSEC_M3_MISSING_KEY_DATA: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 6;
pub const WLAN_REASON_CODE_MSMSEC_M3_MISSING_IE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 7;
pub const WLAN_REASON_CODE_MSMSEC_M3_MISSING_GRP_KEY: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 8;
pub const WLAN_REASON_CODE_MSMSEC_PR_IE_MATCHING: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 9;
pub const WLAN_REASON_CODE_MSMSEC_SEC_IE_MATCHING: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 10;
pub const WLAN_REASON_CODE_MSMSEC_NO_PAIRWISE_KEY: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 11;
pub const WLAN_REASON_CODE_MSMSEC_G1_MISSING_KEY_DATA: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 12;
pub const WLAN_REASON_CODE_MSMSEC_G1_MISSING_GRP_KEY: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 13;
pub const WLAN_REASON_CODE_MSMSEC_PEER_INDICATED_INSECURE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 14;
pub const WLAN_REASON_CODE_MSMSEC_NO_AUTHENTICATOR: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 15;
pub const WLAN_REASON_CODE_MSMSEC_NIC_FAILURE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 16;
pub const WLAN_REASON_CODE_MSMSEC_CANCELLED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 17;
pub const WLAN_REASON_CODE_MSMSEC_KEY_FORMAT: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 18;
pub const WLAN_REASON_CODE_MSMSEC_DOWNGRADE_DETECTED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 19;
pub const WLAN_REASON_CODE_MSMSEC_PSK_MISMATCH_SUSPECTED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 20;
pub const WLAN_REASON_CODE_MSMSEC_FORCED_FAILURE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 21;
pub const WLAN_REASON_CODE_MSMSEC_M3_TOO_MANY_RSNIE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 22;
pub const WLAN_REASON_CODE_MSMSEC_M2_MISSING_KEY_DATA: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 23;
pub const WLAN_REASON_CODE_MSMSEC_M2_MISSING_IE: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 24;
pub const WLAN_REASON_CODE_MSMSEC_AUTH_WCN_COMPLETED: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 25;
pub const WLAN_REASON_CODE_MSMSEC_M3_MISSING_MGMT_GRP_KEY: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 26;
pub const WLAN_REASON_CODE_MSMSEC_G1_MISSING_MGMT_GRP_KEY: WLAN_REASON_CODE =
    WLAN_REASON_CODE_MSMSEC_CONNECT_BASE + 27;
pub const WLAN_REASON_CODE_MSMSEC_MAX: WLAN_REASON_CODE = WLAN_REASON_CODE_MSMSEC_END;
pub type WLAN_SIGNAL_QUALITY = ULONG;
pub type PWLAN_SIGNAL_QUALITY = *mut ULONG;
pub const WLAN_AVAILABLE_NETWORK_CONNECTED: DWORD = 0x00000001;
pub const WLAN_AVAILABLE_NETWORK_HAS_PROFILE: DWORD = 0x00000002;
pub const WLAN_AVAILABLE_NETWORK_CONSOLE_USER_PROFILE: DWORD = 0x00000004;
pub const WLAN_AVAILABLE_NETWORK_INTERWORKING_SUPPORTED: DWORD = 0x00000008;
pub const WLAN_AVAILABLE_NETWORK_HOTSPOT2_ENABLED: DWORD = 0x00000010;
pub const WLAN_AVAILABLE_NETWORK_ANQP_SUPPORTED: DWORD = 0x00000020;
pub const WLAN_AVAILABLE_NETWORK_HOTSPOT2_DOMAIN: DWORD = 0x00000040;
pub const WLAN_AVAILABLE_NETWORK_HOTSPOT2_ROAMING: DWORD = 0x00000080;
pub const WLAN_AVAILABLE_NETWORK_AUTO_CONNECT_FAILED: DWORD = 0x00000100;
pub const WLAN_AVAILABLE_NETWORK_INCLUDE_ALL_ADHOC_PROFILES: DWORD = 0x00000001;
pub const WLAN_AVAILABLE_NETWORK_INCLUDE_ALL_MANUAL_HIDDEN_PROFILES: DWORD = 0x00000002;
STRUCT!{struct WLAN_RATE_SET {
    uRateSetLength: ULONG,
    usRateSet: [USHORT; DOT11_RATE_SET_MAX_LENGTH],
}}
pub type PWLAN_RATE_SET = *mut WLAN_RATE_SET;
pub const WLAN_MAX_PHY_TYPE_NUMBER: usize = 8;
STRUCT!{struct WLAN_AVAILABLE_NETWORK {
    strProfileName: [WCHAR; WLAN_MAX_NAME_LENGTH],
    dot11Ssid: DOT11_SSID,
    dot11BssType: DOT11_BSS_TYPE,
    uNumberOfBssids: ULONG,
    bNetworkConnectable: BOOL,
    wlanNotConnectableReason: WLAN_REASON_CODE,
    uNumberOfPhyTypes: ULONG,
    dot11PhyTypes: [DOT11_PHY_TYPE; WLAN_MAX_PHY_TYPE_NUMBER],
    bMorePhyTypes: BOOL,
    wlanSignalQuality: WLAN_SIGNAL_QUALITY,
    bSecurityEnabled: BOOL,
    dot11DefaultAuthAlgorithm: DOT11_AUTH_ALGORITHM,
    dot11DefaultCipherAlgorithm: DOT11_CIPHER_ALGORITHM,
    dwFlags: DWORD,
    dwReserved: DWORD,
}}
pub type PWLAN_AVAILABLE_NETWORK = *mut WLAN_AVAILABLE_NETWORK;
STRUCT!{struct WLAN_AVAILABLE_NETWORK_V2 {
    strProfileName: [WCHAR; WLAN_MAX_NAME_LENGTH],
    dot11Ssid: DOT11_SSID,
    dot11BssType: DOT11_BSS_TYPE,
    uNumberOfBssids: ULONG,
    bNetworkConnectable: BOOL,
    wlanNotConnectableReason: WLAN_REASON_CODE,
    uNumberOfPhyTypes: ULONG,
    dot11PhyTypes: [DOT11_PHY_TYPE; WLAN_MAX_PHY_TYPE_NUMBER],
    bMorePhyTypes: BOOL,
    wlanSignalQuality: WLAN_SIGNAL_QUALITY,
    bSecurityEnabled: BOOL,
    dot11DefaultAuthAlgorithm: DOT11_AUTH_ALGORITHM,
    dot11DefaultCipherAlgorithm: DOT11_CIPHER_ALGORITHM,
    dwFlags: DWORD,
    AccessNetworkOptions: DOT11_ACCESSNETWORKOPTIONS,
    dot11HESSID: DOT11_HESSID,
    VenueInfo: DOT11_VENUEINFO,
    dwReserved: DWORD,
}}
pub type PWLAN_AVAILABLE_NETWORK_V2 = *mut WLAN_AVAILABLE_NETWORK_V2;
STRUCT!{struct WLAN_BSS_ENTRY {
    dot11Ssid: DOT11_SSID,
    uPhyId: ULONG,
    dot11Bssid: DOT11_MAC_ADDRESS,
    dot11BssType: DOT11_BSS_TYPE,
    dot11BssPhyType: DOT11_PHY_TYPE,
    lRssi: LONG,
    uLinkQuality: ULONG,
    bInRegDomain: BOOLEAN,
    usBeaconPeriod: USHORT,
    ullTimestamp: ULONGLONG,
    ullHostTimestamp: ULONGLONG,
    usCapabilityInformation: USHORT,
    ulChCenterFrequency: ULONG,
    wlanRateSet: WLAN_RATE_SET,
    ulIeOffset: ULONG,
    ulIeSize: ULONG,
}}
pub type PWLAN_BSS_ENTRY = *mut WLAN_BSS_ENTRY;
STRUCT!{struct WLAN_BSS_LIST {
    dwTotalSize: DWORD,
    dwNumberOfItems: DWORD,
    wlanBssEntries: [WLAN_BSS_ENTRY; 1],
}}
pub type PWLAN_BSS_LIST = *mut WLAN_BSS_LIST;
ENUM!{enum WLAN_INTERFACE_STATE {
    wlan_interface_state_not_ready = 0,
    wlan_interface_state_connected = 1,
    wlan_interface_state_ad_hoc_network_formed = 2,
    wlan_interface_state_disconnecting = 3,
    wlan_interface_state_disconnected = 4,
    wlan_interface_state_associating = 5,
    wlan_interface_state_discovering = 6,
    wlan_interface_state_authenticating = 7,
}}
pub type PWLAN_INTERFACE_STATE = *mut WLAN_INTERFACE_STATE;
ENUM!{enum WLAN_ADHOC_NETWORK_STATE {
    wlan_adhoc_network_state_formed = 0,
    wlan_adhoc_network_state_connected = 1,
}}
pub type PWLAN_ADHOC_NETWORK_STATE = *mut WLAN_ADHOC_NETWORK_STATE;
STRUCT!{struct WLAN_INTERFACE_INFO {
    InterfaceGuid: GUID,
    strInterfaceDescription: [WCHAR; WLAN_MAX_NAME_LENGTH],
    isState: WLAN_INTERFACE_STATE,
}}
pub type PWLAN_INTERFACE_INFO = *mut WLAN_INTERFACE_INFO;
STRUCT!{struct WLAN_ASSOCIATION_ATTRIBUTES {
    dot11Ssid: DOT11_SSID,
    dot11BssType: DOT11_BSS_TYPE,
    dot11Bssid: DOT11_MAC_ADDRESS,
    dot11PhyType: DOT11_PHY_TYPE,
    uDot11PhyIndex: ULONG,
    wlanSignalQuality: WLAN_SIGNAL_QUALITY,
    ulRxRate: ULONG,
    ulTxRate: ULONG,
}}
pub type PWLAN_ASSOCIATION_ATTRIBUTES = *mut WLAN_ASSOCIATION_ATTRIBUTES;
STRUCT!{struct WLAN_SECURITY_ATTRIBUTES {
    bSecurityEnabled: BOOL,
    bOneXEnabled: BOOL,
    dot11AuthAlgorithm: DOT11_AUTH_ALGORITHM,
    dot11CipherAlgorithm: DOT11_CIPHER_ALGORITHM,
}}
pub type PWLAN_SECURITY_ATTRIBUTES = *mut WLAN_SECURITY_ATTRIBUTES;
STRUCT!{struct WLAN_CONNECTION_ATTRIBUTES {
    isState: WLAN_INTERFACE_STATE,
    wlanConnectionMode: WLAN_CONNECTION_MODE,
    strProfileName: [WCHAR; WLAN_MAX_NAME_LENGTH],
    wlanAssociationAttributes: WLAN_ASSOCIATION_ATTRIBUTES,
    wlanSecurityAttributes: WLAN_SECURITY_ATTRIBUTES,
}}
pub type PWLAN_CONNECTION_ATTRIBUTES = *mut WLAN_CONNECTION_ATTRIBUTES;
ENUM!{enum DOT11_RADIO_STATE {
    dot11_radio_state_unknown = 0,
    dot11_radio_state_on = 1,
    dot11_radio_state_off = 2,
}}
pub type PDOT11_RADIO_STATE = *mut DOT11_RADIO_STATE;
pub const WLAN_MAX_PHY_INDEX: usize = 64;
STRUCT!{struct WLAN_PHY_RADIO_STATE {
    dwPhyIndex: DWORD,
    dot11SoftwareRadioState: DOT11_RADIO_STATE,
    dot11HardwareRadioState: DOT11_RADIO_STATE,
}}
pub type PWLAN_PHY_RADIO_STATE = *mut WLAN_PHY_RADIO_STATE;
STRUCT!{struct WLAN_RADIO_STATE {
    dwNumberOfPhys: DWORD,
    PhyRadioState: [WLAN_PHY_RADIO_STATE; WLAN_MAX_PHY_INDEX],
}}
pub type PWLAN_RADIO_STATE = *mut WLAN_RADIO_STATE;
ENUM!{enum WLAN_OPERATIONAL_STATE {
    wlan_operational_state_unknown = 0,
    wlan_operational_state_off = 1,
    wlan_operational_state_on = 2,
    wlan_operational_state_going_off = 3,
    wlan_operational_state_going_on = 4,
}}
pub type PWLAN_OPERATIONAL_STATE = *mut WLAN_OPERATIONAL_STATE;
ENUM!{enum WLAN_INTERFACE_TYPE {
    wlan_interface_type_emulated_802_11 = 0,
    wlan_interface_type_native_802_11 = 1,
    wlan_interface_type_invalid = 2,
}}
pub type PWLAN_INTERFACE_TYPE = *mut WLAN_INTERFACE_TYPE;
STRUCT!{struct WLAN_INTERFACE_CAPABILITY {
    interfaceType: WLAN_INTERFACE_TYPE,
    bDot11DSupported: BOOL,
    dwMaxDesiredSsidListSize: DWORD,
    dwMaxDesiredBssidListSize: DWORD,
    dwNumberOfSupportedPhys: DWORD,
    dot11PhyTypes: [DOT11_PHY_TYPE; WLAN_MAX_PHY_INDEX],
}}
pub type PWLAN_INTERFACE_CAPABILITY = *mut WLAN_INTERFACE_CAPABILITY;
STRUCT!{struct WLAN_AUTH_CIPHER_PAIR_LIST {
    dwNumberOfItems: DWORD,
    pAuthCipherPairList: [DOT11_AUTH_CIPHER_PAIR; 1],
}}
pub type PWLAN_AUTH_CIPHER_PAIR_LIST = *mut WLAN_AUTH_CIPHER_PAIR_LIST;
STRUCT!{struct WLAN_COUNTRY_OR_REGION_STRING_LIST {
    dwNumberOfItems: DWORD,
    pCountryOrRegionStringList: [DOT11_COUNTRY_OR_REGION_STRING; 1],
}}
pub type PWLAN_COUNTRY_OR_REGION_STRING_LIST = *mut WLAN_COUNTRY_OR_REGION_STRING_LIST;
STRUCT!{struct WLAN_PROFILE_INFO_LIST {
    dwNumberOfItems: DWORD,
    dwIndex: DWORD,
    ProfileInfo: [WLAN_PROFILE_INFO; 1],
}}
pub type PWLAN_PROFILE_INFO_LIST = *mut WLAN_PROFILE_INFO_LIST;
STRUCT!{struct WLAN_AVAILABLE_NETWORK_LIST {
    dwNumberOfItems: DWORD,
    dwIndex: DWORD,
    Network: [WLAN_AVAILABLE_NETWORK; 1],
}}
pub type PWLAN_AVAILABLE_NETWORK_LIST = *mut WLAN_AVAILABLE_NETWORK_LIST;
STRUCT!{struct WLAN_AVAILABLE_NETWORK_LIST_V2 {
    dwNumberOfItems: DWORD,
    dwIndex: DWORD,
    Network: [WLAN_AVAILABLE_NETWORK_V2; 1],
}}
pub type PWLAN_AVAILABLE_NETWORK_LIST_V2 = *mut WLAN_AVAILABLE_NETWORK_LIST_V2;
STRUCT!{struct WLAN_INTERFACE_INFO_LIST {
    dwNumberOfItems: DWORD,
    dwIndex: DWORD,
    InterfaceInfo: [WLAN_INTERFACE_INFO; 1],
}}
pub type PWLAN_INTERFACE_INFO_LIST = *mut WLAN_INTERFACE_INFO_LIST;
STRUCT!{struct DOT11_NETWORK_LIST {
    dwNumberOfItems: DWORD,
    dwIndex: DWORD,
    Network: [DOT11_NETWORK; 1],
}}
pub type PDOT11_NETWORK_LIST = *mut DOT11_NETWORK_LIST;
ENUM!{enum WLAN_POWER_SETTING {
    wlan_power_setting_no_saving = 0,
    wlan_power_setting_low_saving = 1,
    wlan_power_setting_medium_saving = 2,
    wlan_power_setting_maximum_saving = 3,
    wlan_power_setting_invalid = 4,
}}
pub type PWLAN_POWER_SETTING = *mut WLAN_POWER_SETTING;
pub const WLAN_CONNECTION_HIDDEN_NETWORK: DWORD = 0x00000001;
pub const WLAN_CONNECTION_ADHOC_JOIN_ONLY: DWORD = 0x00000002;
pub const WLAN_CONNECTION_IGNORE_PRIVACY_BIT: DWORD = 0x00000004;
pub const WLAN_CONNECTION_EAPOL_PASSTHROUGH: DWORD = 0x00000008;
pub const WLAN_CONNECTION_PERSIST_DISCOVERY_PROFILE: DWORD = 0x00000010;
pub const WLAN_CONNECTION_PERSIST_DISCOVERY_PROFILE_CONNECTION_MODE_AUTO: DWORD = 0x00000020;
pub const WLAN_CONNECTION_PERSIST_DISCOVERY_PROFILE_OVERWRITE_EXISTING: DWORD = 0x00000040;
STRUCT!{struct WLAN_CONNECTION_PARAMETERS {
    wlanConnectionMode: WLAN_CONNECTION_MODE,
    strProfile: LPCWSTR,
    pDot11Ssid: PDOT11_SSID,
    pDesiredBssidList: PDOT11_BSSID_LIST,
    dot11BssType: DOT11_BSS_TYPE,
    dwFlags: DWORD,
}}
pub type PWLAN_CONNECTION_PARAMETERS = *mut WLAN_CONNECTION_PARAMETERS;
STRUCT!{struct WLAN_CONNECTION_PARAMETERS_V2 {
    wlanConnectionMode: WLAN_CONNECTION_MODE,
    strProfile: LPCWSTR,
    pDot11Ssid: PDOT11_SSID,
    pDot11Hessid: PDOT11_HESSID,
    pDesiredBssidList: PDOT11_BSSID_LIST,
    dot11BssType: DOT11_BSS_TYPE,
    dwFlags: DWORD,
    pDot11AccessNetworkOptions: PDOT11_ACCESSNETWORKOPTIONS,
}}
pub type PWLAN_CONNECTION_PARAMETERS_V2 = *mut WLAN_CONNECTION_PARAMETERS_V2;
STRUCT!{struct WLAN_MSM_NOTIFICATION_DATA {
    wlanConnectionMode: WLAN_CONNECTION_MODE,
    strProfileName: [WCHAR; WLAN_MAX_NAME_LENGTH],
    dot11Ssid: DOT11_SSID,
    dot11BssType: DOT11_BSS_TYPE,
    dot11MacAddr: DOT11_MAC_ADDRESS,
    bSecurityEnabled: BOOL,
    bFirstPeer: BOOL,
    bLastPeer: BOOL,
    wlanReasonCode: WLAN_REASON_CODE,
}}
pub type PWLAN_MSM_NOTIFICATION_DATA = *mut WLAN_MSM_NOTIFICATION_DATA;
pub const WLAN_CONNECTION_NOTIFICATION_ADHOC_NETWORK_FORMED: DWORD = 0x00000001;
pub const WLAN_CONNECTION_NOTIFICATION_CONSOLE_USER_PROFILE: DWORD = 0x00000004;
STRUCT!{struct WLAN_CONNECTION_NOTIFICATION_DATA {
    wlanConnectionMode: WLAN_CONNECTION_MODE,
    strProfileName: [WCHAR; WLAN_MAX_NAME_LENGTH],
    dot11Ssid: DOT11_SSID,
    dot11BssType: DOT11_BSS_TYPE,
    bSecurityEnabled: BOOL,
    wlanReasonCode: WLAN_REASON_CODE,
    dwFlags: DWORD,
    strProfileXml: [WCHAR; 1],
}}
pub type PWLAN_CONNECTION_NOTIFICATION_DATA = *mut WLAN_CONNECTION_NOTIFICATION_DATA;
pub const WLAN_NOTIFICATION_SOURCE_NONE: DWORD = L2_NOTIFICATION_SOURCE_NONE;
pub const WLAN_NOTIFICATION_SOURCE_ALL: DWORD = L2_NOTIFICATION_SOURCE_ALL;
pub const WLAN_NOTIFICATION_SOURCE_ACM: DWORD = L2_NOTIFICATION_SOURCE_WLAN_ACM;
pub const WLAN_NOTIFICATION_SOURCE_MSM: DWORD = L2_NOTIFICATION_SOURCE_WLAN_MSM;
pub const WLAN_NOTIFICATION_SOURCE_SECURITY: DWORD = L2_NOTIFICATION_SOURCE_WLAN_SECURITY;
pub const WLAN_NOTIFICATION_SOURCE_IHV: DWORD = L2_NOTIFICATION_SOURCE_WLAN_IHV;
pub const WLAN_NOTIFICATION_SOURCE_HNWK: DWORD = L2_NOTIFICATION_SOURCE_WLAN_HNWK;
pub const WLAN_NOTIFICATION_SOURCE_ONEX: DWORD = L2_NOTIFICATION_SOURCE_ONEX;
ENUM!{enum WLAN_NOTIFICATION_ACM {
    wlan_notification_acm_start = L2_NOTIFICATION_CODE_PUBLIC_BEGIN,
    wlan_notification_acm_autoconf_enabled = 1,
    wlan_notification_acm_autoconf_disabled = 2,
    wlan_notification_acm_background_scan_enabled = 3,
    wlan_notification_acm_background_scan_disabled = 4,
    wlan_notification_acm_bss_type_change = 5,
    wlan_notification_acm_power_setting_change = 6,
    wlan_notification_acm_scan_complete = 7,
    wlan_notification_acm_scan_fail = 8,
    wlan_notification_acm_connection_start = 9,
    wlan_notification_acm_connection_complete = 10,
    wlan_notification_acm_connection_attempt_fail = 11,
    wlan_notification_acm_filter_list_change = 12,
    wlan_notification_acm_interface_arrival = 13,
    wlan_notification_acm_interface_removal = 14,
    wlan_notification_acm_profile_change = 15,
    wlan_notification_acm_profile_name_change = 16,
    wlan_notification_acm_profiles_exhausted = 17,
    wlan_notification_acm_network_not_available = 18,
    wlan_notification_acm_network_available = 19,
    wlan_notification_acm_disconnecting = 20,
    wlan_notification_acm_disconnected = 21,
    wlan_notification_acm_adhoc_network_state_change = 22,
    wlan_notification_acm_profile_unblocked = 23,
    wlan_notification_acm_screen_power_change = 24,
    wlan_notification_acm_profile_blocked = 25,
    wlan_notification_acm_scan_list_refresh = 26,
    wlan_notification_acm_operational_state_change = 27,
    wlan_notification_acm_end = 28,
}}
pub type PWLAN_NOTIFICATION_ACM = *mut WLAN_NOTIFICATION_ACM;
ENUM!{enum WLAN_NOTIFICATION_MSM {
    wlan_notification_msm_start = L2_NOTIFICATION_CODE_PUBLIC_BEGIN,
    wlan_notification_msm_associating = 1,
    wlan_notification_msm_associated = 2,
    wlan_notification_msm_authenticating = 3,
    wlan_notification_msm_connected = 4,
    wlan_notification_msm_roaming_start = 5,
    wlan_notification_msm_roaming_end = 6,
    wlan_notification_msm_radio_state_change = 7,
    wlan_notification_msm_signal_quality_change = 8,
    wlan_notification_msm_disassociating = 9,
    wlan_notification_msm_disconnected = 10,
    wlan_notification_msm_peer_join = 11,
    wlan_notification_msm_peer_leave = 12,
    wlan_notification_msm_adapter_removal = 13,
    wlan_notification_msm_adapter_operation_mode_change = 14,
    wlan_notification_msm_link_degraded = 15,
    wlan_notification_msm_link_improved = 16,
    wlan_notification_msm_end = 17,
}}
pub type PWLAN_NOTIFICATION_MSM = *mut WLAN_NOTIFICATION_MSM;
ENUM!{enum WLAN_NOTIFICATION_SECURITY {
    wlan_notification_security_start = L2_NOTIFICATION_CODE_PUBLIC_BEGIN,
    wlan_notification_security_end = 1,
}}
pub type PWLAN_NOTIFICATION_SECURITY = *mut WLAN_NOTIFICATION_SECURITY;
pub type WLAN_NOTIFICATION_DATA = L2_NOTIFICATION_DATA;
pub type PWLAN_NOTIFICATION_DATA = *mut L2_NOTIFICATION_DATA;
FN!{stdcall WLAN_NOTIFICATION_CALLBACK(
    PWLAN_NOTIFICATION_DATA,
    PVOID,
) -> ()}
ENUM!{enum WLAN_OPCODE_VALUE_TYPE {
    wlan_opcode_value_type_query_only = 0,
    wlan_opcode_value_type_set_by_group_policy = 1,
    wlan_opcode_value_type_set_by_user = 2,
    wlan_opcode_value_type_invalid = 3,
}}
pub type PWLAN_OPCODE_VALUE_TYPE = *mut WLAN_OPCODE_VALUE_TYPE;
ENUM!{enum WLAN_INTF_OPCODE {
    wlan_intf_opcode_autoconf_start = 0x000000000,
    wlan_intf_opcode_autoconf_enabled = 1,
    wlan_intf_opcode_background_scan_enabled = 2,
    wlan_intf_opcode_media_streaming_mode = 3,
    wlan_intf_opcode_radio_state = 4,
    wlan_intf_opcode_bss_type = 5,
    wlan_intf_opcode_interface_state = 6,
    wlan_intf_opcode_current_connection = 7,
    wlan_intf_opcode_channel_number = 8,
    wlan_intf_opcode_supported_infrastructure_auth_cipher_pairs = 9,
    wlan_intf_opcode_supported_adhoc_auth_cipher_pairs = 10,
    wlan_intf_opcode_supported_country_or_region_string_list = 11,
    wlan_intf_opcode_current_operation_mode = 12,
    wlan_intf_opcode_supported_safe_mode = 13,
    wlan_intf_opcode_certified_safe_mode = 14,
    wlan_intf_opcode_hosted_network_capable = 15,
    wlan_intf_opcode_management_frame_protection_capable = 16,
    wlan_intf_opcode_autoconf_end = 0x0fffffff,
    wlan_intf_opcode_msm_start = 0x10000100,
    wlan_intf_opcode_statistics = 268435713,
    wlan_intf_opcode_rssi = 268435714,
    wlan_intf_opcode_msm_end = 0x1fffffff,
    wlan_intf_opcode_security_start = 0x20010000,
    wlan_intf_opcode_security_end = 0x2fffffff,
    wlan_intf_opcode_ihv_start = 0x30000000,
    wlan_intf_opcode_ihv_end = 0x3fffffff,
}}
pub type PWLAN_INTF_OPCODE = *mut WLAN_INTF_OPCODE;
ENUM!{enum WLAN_AUTOCONF_OPCODE {
    wlan_autoconf_opcode_start = 0,
    wlan_autoconf_opcode_show_denied_networks = 1,
    wlan_autoconf_opcode_power_setting = 2,
    wlan_autoconf_opcode_only_use_gp_profiles_for_allowed_networks = 3,
    wlan_autoconf_opcode_allow_explicit_creds = 4,
    wlan_autoconf_opcode_block_period = 5,
    wlan_autoconf_opcode_allow_virtual_station_extensibility = 6,
    wlan_autoconf_opcode_end = 7,
}}
pub type PWLAN_AUTOCONF_OPCODE = *mut WLAN_AUTOCONF_OPCODE;
ENUM!{enum WLAN_IHV_CONTROL_TYPE {
    wlan_ihv_control_type_service = 0,
    wlan_ihv_control_type_driver = 1,
}}
pub type PWLAN_IHV_CONTROL_TYPE = *mut WLAN_IHV_CONTROL_TYPE;
ENUM!{enum WLAN_FILTER_LIST_TYPE {
    wlan_filter_list_type_gp_permit = 0,
    wlan_filter_list_type_gp_deny = 1,
    wlan_filter_list_type_user_permit = 2,
    wlan_filter_list_type_user_deny = 3,
}}
pub type PWLAN_FILTER_LIST_TYPE = *mut WLAN_FILTER_LIST_TYPE;
STRUCT!{struct WLAN_PHY_FRAME_STATISTICS {
    ullTransmittedFrameCount: ULONGLONG,
    ullMulticastTransmittedFrameCount: ULONGLONG,
    ullFailedCount: ULONGLONG,
    ullRetryCount: ULONGLONG,
    ullMultipleRetryCount: ULONGLONG,
    ullMaxTXLifetimeExceededCount: ULONGLONG,
    ullTransmittedFragmentCount: ULONGLONG,
    ullRTSSuccessCount: ULONGLONG,
    ullRTSFailureCount: ULONGLONG,
    ullACKFailureCount: ULONGLONG,
    ullReceivedFrameCount: ULONGLONG,
    ullMulticastReceivedFrameCount: ULONGLONG,
    ullPromiscuousReceivedFrameCount: ULONGLONG,
    ullMaxRXLifetimeExceededCount: ULONGLONG,
    ullFrameDuplicateCount: ULONGLONG,
    ullReceivedFragmentCount: ULONGLONG,
    ullPromiscuousReceivedFragmentCount: ULONGLONG,
    ullFCSErrorCount: ULONGLONG,
}}
pub type PWLAN_PHY_FRAME_STATISTICS = *mut WLAN_PHY_FRAME_STATISTICS;
STRUCT!{struct WLAN_MAC_FRAME_STATISTICS {
    ullTransmittedFrameCount: ULONGLONG,
    ullReceivedFrameCount: ULONGLONG,
    ullWEPExcludedCount: ULONGLONG,
    ullTKIPLocalMICFailures: ULONGLONG,
    ullTKIPReplays: ULONGLONG,
    ullTKIPICVErrorCount: ULONGLONG,
    ullCCMPReplays: ULONGLONG,
    ullCCMPDecryptErrors: ULONGLONG,
    ullWEPUndecryptableCount: ULONGLONG,
    ullWEPICVErrorCount: ULONGLONG,
    ullDecryptSuccessCount: ULONGLONG,
    ullDecryptFailureCount: ULONGLONG,
}}
pub type PWLAN_MAC_FRAME_STATISTICS = *mut WLAN_MAC_FRAME_STATISTICS;
STRUCT!{struct WLAN_STATISTICS {
    ullFourWayHandshakeFailures: ULONGLONG,
    ullTKIPCounterMeasuresInvoked: ULONGLONG,
    ullReserved: ULONGLONG,
    MacUcastCounters: WLAN_MAC_FRAME_STATISTICS,
    MacMcastCounters: WLAN_MAC_FRAME_STATISTICS,
    dwNumberOfPhys: DWORD,
    PhyCounters: [WLAN_PHY_FRAME_STATISTICS; 1],
}}
pub type PWLAN_STATISTICS = *mut WLAN_STATISTICS;
pub const WLAN_READ_ACCESS: DWORD = STANDARD_RIGHTS_READ | FILE_READ_DATA;
pub const WLAN_EXECUTE_ACCESS: DWORD = WLAN_READ_ACCESS | STANDARD_RIGHTS_EXECUTE | FILE_EXECUTE;
pub const WLAN_WRITE_ACCESS: DWORD = WLAN_READ_ACCESS | WLAN_EXECUTE_ACCESS | STANDARD_RIGHTS_WRITE
    | FILE_WRITE_DATA | DELETE | WRITE_DAC;
ENUM!{enum WLAN_SECURABLE_OBJECT {
    wlan_secure_permit_list = 0,
    wlan_secure_deny_list = 1,
    wlan_secure_ac_enabled = 2,
    wlan_secure_bc_scan_enabled = 3,
    wlan_secure_bss_type = 4,
    wlan_secure_show_denied = 5,
    wlan_secure_interface_properties = 6,
    wlan_secure_ihv_control = 7,
    wlan_secure_all_user_profiles_order = 8,
    wlan_secure_add_new_all_user_profiles = 9,
    wlan_secure_add_new_per_user_profiles = 10,
    wlan_secure_media_streaming_mode_enabled = 11,
    wlan_secure_current_operation_mode = 12,
    wlan_secure_get_plaintext_key = 13,
    wlan_secure_hosted_network_elevated_access = 14,
    wlan_secure_virtual_station_extensibility = 15,
    wlan_secure_wfd_elevated_access = 16,
    WLAN_SECURABLE_OBJECT_COUNT = 17,
}}
pub type PWLAN_SECURABLE_OBJECT = *mut WLAN_SECURABLE_OBJECT;
STRUCT!{struct WLAN_DEVICE_SERVICE_GUID_LIST {
    dwNumberOfItems: DWORD,
    dwIndex: DWORD,
    DeviceService: [GUID; 1],
}}
pub type PWLAN_DEVICE_SERVICE_GUID_LIST = *mut WLAN_DEVICE_SERVICE_GUID_LIST;
pub const WFD_API_VERSION_1_0: DWORD = 0x00000001;
pub const WFD_API_VERSION: DWORD = WFD_API_VERSION_1_0;
ENUM!{enum WFD_ROLE_TYPE {
    WFD_ROLE_TYPE_NONE = 0x00,
    WFD_ROLE_TYPE_DEVICE = 0x01,
    WFD_ROLE_TYPE_GROUP_OWNER = 0x02,
    WFD_ROLE_TYPE_CLIENT = 0x04,
    WFD_ROLE_TYPE_MAX = 0x05,
}}
pub type PWFD_ROLE_TYPE = *mut WFD_ROLE_TYPE;
STRUCT!{struct WFD_GROUP_ID {
    DeviceAddress: DOT11_MAC_ADDRESS,
    GroupSSID: DOT11_SSID,
}}
pub type PWFD_GROUP_ID = *mut WFD_GROUP_ID;
extern "system" {
    pub fn WlanOpenHandle(
        dwClientVersion: DWORD,
        pReserved: PVOID,
        pdwNegotiatedVersion: PDWORD,
        phClientHandle: PHANDLE,
    ) -> DWORD;
    pub fn WlanCloseHandle(
        hClientHandle: HANDLE,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanEnumInterfaces(
        hClientHandle: HANDLE,
        pReserved: PVOID,
        ppInterfaceList: *mut PWLAN_INTERFACE_INFO_LIST,
    ) -> DWORD;
    pub fn WlanSetAutoConfigParameter(
        hClientHandle: HANDLE,
        OpCode: WLAN_AUTOCONF_OPCODE,
        dwDataSize: DWORD,
        pData: PVOID,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanQueryAutoConfigParameter(
        hClientHandle: HANDLE,
        OpCode: WLAN_AUTOCONF_OPCODE,
        pReserved: PVOID,
        pdwDataSize: PDWORD,
        ppData: *mut PVOID,
        pWlanOpcodeValueType: PWLAN_OPCODE_VALUE_TYPE,
    ) -> DWORD;
    pub fn WlanGetInterfaceCapability(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        pReserved: PVOID,
        ppCapability: *mut PWLAN_INTERFACE_CAPABILITY,
    ) -> DWORD;
    pub fn WlanSetInterface(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        OpCode: WLAN_INTF_OPCODE,
        dwDataSize: DWORD,
        pData: PVOID,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanQueryInterface(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        OpCode: WLAN_INTF_OPCODE,
        pReserved: PVOID,
        pdwDataSize: PDWORD,
        ppData: *mut PVOID,
        pWlanOpcodeValueType: PWLAN_OPCODE_VALUE_TYPE,
    ) -> DWORD;
    pub fn WlanIhvControl(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        Type: WLAN_IHV_CONTROL_TYPE,
        dwInBufferSize: DWORD,
        pInBuffer: PVOID,
        dwOutBufferSize: DWORD,
        pOutBuffer: PVOID,
        pdwBytesReturned: PDWORD,
    ) -> DWORD;
    pub fn WlanScan(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        pDot11Ssid: PDOT11_SSID,
        pIeData: PWLAN_RAW_DATA,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanGetAvailableNetworkList(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        dwFlags: DWORD,
        pReserved: PVOID,
        ppAvailableNetworkList: *mut PWLAN_AVAILABLE_NETWORK_LIST,
    ) -> DWORD;
    pub fn WlanGetAvailableNetworkList2(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        dwFlags: DWORD,
        pReserved: PVOID,
        ppAvailableNetworkList: *mut PWLAN_AVAILABLE_NETWORK_LIST_V2,
    ) -> DWORD;
    pub fn WlanGetNetworkBssList(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        pDot11Ssid: PDOT11_SSID,
        dot11BssType: DOT11_BSS_TYPE,
        bSecurityEnabled: BOOL,
        pReserved: PVOID,
        ppWlanBssList: *mut PWLAN_BSS_LIST,
    ) -> DWORD;
    pub fn WlanConnect(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        pConnectionParameters: PWLAN_CONNECTION_PARAMETERS,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanConnect2(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        pConnectionParameters: PWLAN_CONNECTION_PARAMETERS_V2,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanDisconnect(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanRegisterNotification(
        hClientHandle: HANDLE,
        dwNotifSource: DWORD,
        bIgnoreDuplicate: BOOL,
        funcCallback: WLAN_NOTIFICATION_CALLBACK,
        pCallbackContext: PVOID,
        pReserved: PVOID,
        pdwPrevNotifSource: PDWORD,
    ) -> DWORD;
    pub fn WlanGetProfile(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        strProfileName: LPCWSTR,
        pReserved: PVOID,
        pstrProfileXml: *mut LPWSTR,
        pdwFlags: *mut DWORD,
        pdwGrantedAccess: *mut DWORD,
    ) -> DWORD;
    pub fn WlanSetProfileEapUserData(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        strProfileName: LPCWSTR,
        eapType: EAP_METHOD_TYPE,
        dwFlags: DWORD,
        dwEapUserDataSize: DWORD,
        pbEapUserData: LPBYTE,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanSetProfileEapXmlUserData(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        strProfileName: LPCWSTR,
        dwFlags: DWORD,
        strEapXmlUserData: LPCWSTR,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanSetProfile(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        dwFlags: DWORD,
        strProfileXml: LPCWSTR,
        strAllUserProfileSecurity: LPCWSTR,
        bOverwrite: BOOL,
        pReserved: PVOID,
        pdwReasonCode: *mut DWORD,
    ) -> DWORD;
    pub fn WlanDeleteProfile(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        strProfileName: LPCWSTR,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanRenameProfile(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        strOldProfileName: LPCWSTR,
        strNewProfileName: LPCWSTR,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanGetProfileList(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        pReserved: PVOID,
        ppProfileList: *mut PWLAN_PROFILE_INFO_LIST,
    ) -> DWORD;
    pub fn WlanSetProfileList(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        dwItems: DWORD,
        strProfileNames: *mut LPCWSTR,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanSetProfilePosition(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        strProfileName: LPCWSTR,
        dwPosition: DWORD,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanSetProfileCustomUserData(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        strProfileName: LPCWSTR,
        dwDataSize: DWORD,
        pData: PBYTE,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanGetProfileCustomUserData(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        strProfileName: LPCWSTR,
        pReserved: PVOID,
        pdwDataSize: *mut DWORD,
        ppData: *mut PBYTE,
    ) -> DWORD;
    pub fn WlanSetFilterList(
        hClientHandle: HANDLE,
        wlanFilterListType: WLAN_FILTER_LIST_TYPE,
        pNetworkList: PDOT11_NETWORK_LIST,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanGetFilterList(
        hClientHandle: HANDLE,
        wlanFilterListType: WLAN_FILTER_LIST_TYPE,
        pReserved: PVOID,
        ppNetworkList: *mut PDOT11_NETWORK_LIST,
    ) -> DWORD;
    pub fn WlanSetPsdIEDataList(
        hClientHandle: HANDLE,
        strFormat: LPCWSTR,
        pPsdIEDataList: PWLAN_RAW_DATA_LIST,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanSaveTemporaryProfile(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        strProfileName: LPCWSTR,
        strAllUserProfileSecurity: LPCWSTR,
        dwFlags: DWORD,
        bOverWrite: BOOL,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanDeviceServiceCommand(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        pDeviceServiceGuid: LPGUID,
        dwOpCode: DWORD,
        dwInBufferSize: DWORD,
        pInBuffer: PVOID,
        dwOutBufferSize: DWORD,
        pOutBuffer: PVOID,
        pdwBytesReturned: PDWORD,
    ) -> DWORD;
    pub fn WlanGetSupportedDeviceServices(
        hClientHandle: HANDLE,
        pInterfaceGuid: *const GUID,
        ppDevSvcGuidList: *mut PWLAN_DEVICE_SERVICE_GUID_LIST,
    ) -> DWORD;
    pub fn WlanExtractPsdIEDataList(
        hClientHandle: HANDLE,
        dwIeDataSize: DWORD,
        pRawIeData: PBYTE,
        strFormat: LPCWSTR,
        pReserved: PVOID,
        ppPsdIEDataList: *mut PWLAN_RAW_DATA_LIST,
    ) -> DWORD;
    pub fn WlanReasonCodeToString(
        dwReasonCode: DWORD,
        dwBufferSize: DWORD,
        pStringBuffer: PWCHAR,
        pReserved: PVOID,
    ) -> DWORD;
    pub fn WlanAllocateMemory(
        dwMemorySize: DWORD,
    ) -> PVOID;
    pub fn WlanFreeMemory(
        pMemory: PVOID,
    );
    pub fn WlanSetSecuritySettings(
        hClientHandle: HANDLE,
        SecurableObject: WLAN_SECURABLE_OBJECT,
        strModifiedSDDL: LPCWSTR,
    ) -> DWORD;
    pub fn WlanGetSecuritySettings(
        hClientHandle: HANDLE,
        SecurableObject: WLAN_SECURABLE_OBJECT,
        pValueType: PWLAN_OPCODE_VALUE_TYPE,
        pstrCurrentSDDL: *mut LPWSTR,
        pdwGrantedAccess: PDWORD,
    ) -> DWORD;
}
pub const WLAN_UI_API_VERSION: DWORD = 1;
pub const WLAN_UI_API_INITIAL_VERSION: DWORD = 1;
ENUM!{enum WL_DISPLAY_PAGES {
    WLConnectionPage = 0,
    WLSecurityPage = 1,
    WLAdvPage = 2,
}}
pub type PWL_DISPLAY_PAGES = *mut WL_DISPLAY_PAGES;
extern "system" {
    pub fn WlanUIEditProfile(
        dwClientVersion: DWORD,
        wstrProfileName: LPCWSTR,
        pInterfaceGuid: *mut GUID,
        hWnd: HWND,
        wlStartPage: WL_DISPLAY_PAGES,
        pReserved: PVOID,
        pWlanReasonCode: PWLAN_REASON_CODE,
    ) -> DWORD;
}
ENUM!{enum WLAN_HOSTED_NETWORK_STATE {
    wlan_hosted_network_unavailable = 0,
    wlan_hosted_network_idle = 1,
    wlan_hosted_network_active = 2,
}}
pub type PWLAN_HOSTED_NETWORK_STATE = *mut WLAN_HOSTED_NETWORK_STATE;
ENUM!{enum WLAN_HOSTED_NETWORK_REASON {
    wlan_hosted_network_reason_success = 0,
    wlan_hosted_network_reason_unspecified = 1,
    wlan_hosted_network_reason_bad_parameters = 2,
    wlan_hosted_network_reason_service_shutting_down = 3,
    wlan_hosted_network_reason_insufficient_resources = 4,
    wlan_hosted_network_reason_elevation_required = 5,
    wlan_hosted_network_reason_read_only = 6,
    wlan_hosted_network_reason_persistence_failed = 7,
    wlan_hosted_network_reason_crypt_error = 8,
    wlan_hosted_network_reason_impersonation = 9,
    wlan_hosted_network_reason_stop_before_start = 10,
    wlan_hosted_network_reason_interface_available = 11,
    wlan_hosted_network_reason_interface_unavailable = 12,
    wlan_hosted_network_reason_miniport_stopped = 13,
    wlan_hosted_network_reason_miniport_started = 14,
    wlan_hosted_network_reason_incompatible_connection_started = 15,
    wlan_hosted_network_reason_incompatible_connection_stopped = 16,
    wlan_hosted_network_reason_user_action = 17,
    wlan_hosted_network_reason_client_abort = 18,
    wlan_hosted_network_reason_ap_start_failed = 19,
    wlan_hosted_network_reason_peer_arrived = 20,
    wlan_hosted_network_reason_peer_departed = 21,
    wlan_hosted_network_reason_peer_timeout = 22,
    wlan_hosted_network_reason_gp_denied = 23,
    wlan_hosted_network_reason_service_unavailable = 24,
    wlan_hosted_network_reason_device_change = 25,
    wlan_hosted_network_reason_properties_change = 26,
    wlan_hosted_network_reason_virtual_station_blocking_use = 27,
    wlan_hosted_network_reason_service_available_on_virtual_station = 28,
}}
pub type PWLAN_HOSTED_NETWORK_REASON = *mut WLAN_HOSTED_NETWORK_REASON;
ENUM!{enum WLAN_HOSTED_NETWORK_PEER_AUTH_STATE {
    wlan_hosted_network_peer_state_invalid = 0,
    wlan_hosted_network_peer_state_authenticated = 1,
}}
pub type PWLAN_HOSTED_NETWORK_PEER_AUTH_STATE = *mut WLAN_HOSTED_NETWORK_PEER_AUTH_STATE;
extern "system" {
    pub fn WlanHostedNetworkStartUsing(
        hClientHandle: HANDLE,
        pFailReason: PWLAN_HOSTED_NETWORK_REASON,
        pvReserved: PVOID,
    ) -> DWORD;
    pub fn WlanHostedNetworkStopUsing(
        hClientHandle: HANDLE,
        pFailReason: PWLAN_HOSTED_NETWORK_REASON,
        pvReserved: PVOID,
    ) -> DWORD;
    pub fn WlanHostedNetworkForceStart(
        hClientHandle: HANDLE,
        pFailReason: PWLAN_HOSTED_NETWORK_REASON,
        pvReserved: PVOID,
    ) -> DWORD;
    pub fn WlanHostedNetworkForceStop(
        hClientHandle: HANDLE,
        pFailReason: PWLAN_HOSTED_NETWORK_REASON,
        pvReserved: PVOID,
    ) -> DWORD;
}
STRUCT!{struct WLAN_HOSTED_NETWORK_PEER_STATE {
    PeerMacAddress: DOT11_MAC_ADDRESS,
    PeerAuthState: WLAN_HOSTED_NETWORK_PEER_AUTH_STATE,
}}
pub type PWLAN_HOSTED_NETWORK_PEER_STATE = *mut WLAN_HOSTED_NETWORK_PEER_STATE;
STRUCT!{struct WLAN_HOSTED_NETWORK_RADIO_STATE {
    dot11SoftwareRadioState: DOT11_RADIO_STATE,
    dot11HardwareRadioState: DOT11_RADIO_STATE,
}}
pub type PWLAN_HOSTED_NETWORK_RADIO_STATE = *mut WLAN_HOSTED_NETWORK_RADIO_STATE;
ENUM!{enum WLAN_HOSTED_NETWORK_NOTIFICATION_CODE {
    wlan_hosted_network_state_change = L2_NOTIFICATION_CODE_V2_BEGIN,
    wlan_hosted_network_peer_state_change = 4097,
    wlan_hosted_network_radio_state_change = 4098,
}}
pub type PWLAN_HOSTED_NETWORK_NOTIFICATION_CODE = *mut WLAN_HOSTED_NETWORK_NOTIFICATION_CODE;
STRUCT!{struct WLAN_HOSTED_NETWORK_STATE_CHANGE {
    OldState: WLAN_HOSTED_NETWORK_STATE,
    NewState: WLAN_HOSTED_NETWORK_STATE,
    StateChangeReason: WLAN_HOSTED_NETWORK_REASON,
}}
pub type PWLAN_HOSTED_NETWORK_STATE_CHANGE = *mut WLAN_HOSTED_NETWORK_STATE_CHANGE;
STRUCT!{struct WLAN_HOSTED_NETWORK_DATA_PEER_STATE_CHANGE {
    OldState: WLAN_HOSTED_NETWORK_PEER_STATE,
    NewState: WLAN_HOSTED_NETWORK_PEER_STATE,
    PeerStateChangeReason: WLAN_HOSTED_NETWORK_REASON,
}}
pub type PWLAN_HOSTED_NETWORK_DATA_PEER_STATE_CHANGE =
    *mut WLAN_HOSTED_NETWORK_DATA_PEER_STATE_CHANGE;
ENUM!{enum WLAN_HOSTED_NETWORK_OPCODE {
    wlan_hosted_network_opcode_connection_settings = 0,
    wlan_hosted_network_opcode_security_settings = 1,
    wlan_hosted_network_opcode_station_profile = 2,
    wlan_hosted_network_opcode_enable = 3,
}}
pub type PWLAN_HOSTED_NETWORK_OPCODE = *mut WLAN_HOSTED_NETWORK_OPCODE;
STRUCT!{struct WLAN_HOSTED_NETWORK_CONNECTION_SETTINGS {
    hostedNetworkSSID: DOT11_SSID,
    dwMaxNumberOfPeers: DWORD,
}}
pub type PWLAN_HOSTED_NETWORK_CONNECTION_SETTINGS = *mut WLAN_HOSTED_NETWORK_CONNECTION_SETTINGS;
STRUCT!{struct WLAN_HOSTED_NETWORK_SECURITY_SETTINGS {
    dot11AuthAlgo: DOT11_AUTH_ALGORITHM,
    dot11CipherAlgo: DOT11_CIPHER_ALGORITHM,
}}
pub type PWLAN_HOSTED_NETWORK_SECURITY_SETTINGS = *mut WLAN_HOSTED_NETWORK_SECURITY_SETTINGS;
extern "system" {
    pub fn WlanHostedNetworkQueryProperty(
        hClientHandle: HANDLE,
        OpCode: WLAN_HOSTED_NETWORK_OPCODE,
        pdwDataSize: PDWORD,
        ppvData: *mut PVOID,
        pWlanOpcodeValueType: PWLAN_OPCODE_VALUE_TYPE,
        pvReserved: PVOID,
    ) -> DWORD;
    pub fn WlanHostedNetworkSetProperty(
        hClientHandle: HANDLE,
        OpCode: WLAN_HOSTED_NETWORK_OPCODE,
        dwDataSize: DWORD,
        pvData: PVOID,
        pFailReason: PWLAN_HOSTED_NETWORK_REASON,
        pvReserved: PVOID,
    ) -> DWORD;
    pub fn WlanHostedNetworkInitSettings(
        hClientHandle: HANDLE,
        pFailReason: PWLAN_HOSTED_NETWORK_REASON,
        pvReserved: PVOID,
    ) -> DWORD;
    pub fn WlanHostedNetworkRefreshSecuritySettings(
        hClientHandle: HANDLE,
        pFailReason: PWLAN_HOSTED_NETWORK_REASON,
        pvReserved: PVOID,
    ) -> DWORD;
}
STRUCT!{struct WLAN_HOSTED_NETWORK_STATUS {
    HostedNetworkState: WLAN_HOSTED_NETWORK_STATE,
    IPDeviceID: GUID,
    wlanHostedNetworkBSSID: DOT11_MAC_ADDRESS,
    dot11PhyType: DOT11_PHY_TYPE,
    ulChannelFrequency: ULONG,
    dwNumberOfPeers: DWORD,
    PeerList: [WLAN_HOSTED_NETWORK_PEER_STATE; 1],
}}
pub type PWLAN_HOSTED_NETWORK_STATUS = *mut WLAN_HOSTED_NETWORK_STATUS;
extern "system" {
    pub fn WlanHostedNetworkQueryStatus(
        hClientHandle: HANDLE,
        ppWlanHostedNetworkStatus: *mut PWLAN_HOSTED_NETWORK_STATUS,
        pvReserved: PVOID,
    ) -> DWORD;
    pub fn WlanHostedNetworkSetSecondaryKey(
        hClientHandle: HANDLE,
        dwKeyLength: DWORD,
        pucKeyData: PUCHAR,
        bIsPassPhrase: BOOL,
        bPersistent: BOOL,
        pFailReason: PWLAN_HOSTED_NETWORK_REASON,
        pvReserved: PVOID,
    ) -> DWORD;
    pub fn WlanHostedNetworkQuerySecondaryKey(
        hClientHandle: HANDLE,
        pdwKeyLength: PDWORD,
        ppucKeyData: *mut PUCHAR,
        pbIsPassPhrase: PBOOL,
        pbPersistent: PBOOL,
        pFailReason: PWLAN_HOSTED_NETWORK_REASON,
        pvReserved: PVOID,
    ) -> DWORD;
    pub fn WlanRegisterVirtualStationNotification(
        hClientHandle: HANDLE,
        bRegister: BOOL,
        pReserved: PVOID,
    ) -> DWORD;
}
DEFINE_GUID!{GUID_DEVINTERFACE_WIFIDIRECT_DEVICE,
    0x439b20af, 0x8955, 0x405b, 0x99, 0xf0, 0xa6, 0x2a, 0xf0, 0xc6, 0x8d, 0x43}
DEFINE_GUID!{GUID_AEPSERVICE_WIFIDIRECT_DEVICE,
    0xcc29827c, 0x9caf, 0x4928, 0x99, 0xa9, 0x18, 0xf7, 0xc2, 0x38, 0x13, 0x89}
DEFINE_GUID!{GUID_DEVINTERFACE_ASP_INFRA_DEVICE,
    0xff823995, 0x7a72, 0x4c80, 0x87, 0x57, 0xc6, 0x7e, 0xe1, 0x3d, 0x1a, 0x49}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_DeviceAddress,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x01}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_InterfaceAddress,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x02}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_InterfaceGuid,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x03}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_GroupId,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x04}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_IsConnected,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x05}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_IsVisible,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x06}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_IsLegacyDevice,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x07}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_MiracastVersion,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x08}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_IsMiracastLCPSupported,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x09}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_Services,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x0a}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_SupportedChannelList,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x0b}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_InformationElements,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x0c}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_DeviceAddressCopy,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x0d}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_IsRecentlyAssociated,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x0e}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_Service_Aeps,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x0f}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_NoMiracastAutoProject,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x10}
DEFINE_DEVPROPKEY!{DEVPKEY_InfraCast_Supported,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x11}
DEFINE_DEVPROPKEY!{DEVPKEY_InfraCast_StreamSecuritySupported,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x12}
DEFINE_DEVPROPKEY!{DEVPKEY_InfraCast_AccessPointBssid,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x13}
DEFINE_DEVPROPKEY!{DEVPKEY_InfraCast_SinkHostName,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x14}
DEFINE_DEVPROPKEY!{DEVPKEY_InfraCast_ChallengeAep,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x15}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_IsDMGCapable,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x16}
DEFINE_DEVPROPKEY!{DEVPKEY_InfraCast_DevnodeAep,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x17}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_FoundWsbService,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x18}
DEFINE_DEVPROPKEY!{DEVPKEY_InfraCast_HostName_ResolutionMode,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x19}
DEFINE_DEVPROPKEY!{DEVPKEY_InfraCast_SinkIpAddress,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x1a}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirect_TransientAssociation,
    0x1506935d, 0xe3e7, 0x450f, 0x86, 0x37, 0x82, 0x23, 0x3e, 0xbe, 0x5f, 0x6e, 0x1b}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirectServices_ServiceAddress,
    0x31b37743, 0x7c5e, 0x4005, 0x93, 0xe6, 0xe9, 0x53, 0xf9, 0x2b, 0x82, 0xe9, 0x02}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirectServices_ServiceName,
    0x31b37743, 0x7c5e, 0x4005, 0x93, 0xe6, 0xe9, 0x53, 0xf9, 0x2b, 0x82, 0xe9, 0x03}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirectServices_ServiceInformation,
    0x31b37743, 0x7c5e, 0x4005, 0x93, 0xe6, 0xe9, 0x53, 0xf9, 0x2b, 0x82, 0xe9, 0x04}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirectServices_AdvertisementId,
    0x31b37743, 0x7c5e, 0x4005, 0x93, 0xe6, 0xe9, 0x53, 0xf9, 0x2b, 0x82, 0xe9, 0x05}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirectServices_ServiceConfigMethods,
    0x31b37743, 0x7c5e, 0x4005, 0x93, 0xe6, 0xe9, 0x53, 0xf9, 0x2b, 0x82, 0xe9, 0x06}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFiDirectServices_RequestServiceInformation,
    0x31b37743, 0x7c5e, 0x4005, 0x93, 0xe6, 0xe9, 0x53, 0xf9, 0x2b, 0x82, 0xe9, 0x07}
extern "system" {
    pub fn WFDOpenHandle(
        dwClientVersion: DWORD,
        pdwNegotiatedVersion: PDWORD,
        phClientHandle: PHANDLE,
    ) -> DWORD;
    pub fn WFDCloseHandle(
        hClientHandle: HANDLE,
    ) -> DWORD;
}
FN!{stdcall WFD_OPEN_SESSION_COMPLETE_CALLBACK(
    hSessionHandle: HANDLE,
    pvContext: PVOID,
    guidSessionInterface: GUID,
    dwError: DWORD,
    dwReasonCode: DWORD,
) -> ()}
extern "system" {
    pub fn WFDStartOpenSession(
        hClientHandle: HANDLE,
        pDeviceAddress: PDOT11_MAC_ADDRESS,
        pvContext: PVOID,
        pfnCallback: WFD_OPEN_SESSION_COMPLETE_CALLBACK,
        phSessionHandle: PHANDLE,
    ) -> DWORD;
    pub fn WFDCancelOpenSession(
        hSessionHandle: HANDLE,
    ) -> DWORD;
    pub fn WFDOpenLegacySession(
        hClientHandle: HANDLE,
        pLegacyMacAddress: PDOT11_MAC_ADDRESS,
        phSessionHandle: *mut HANDLE,
        pGuidSessionInterface: *mut GUID,
    ) -> DWORD;
    pub fn WFDCloseSession(
        hSessionHandle: HANDLE,
    ) -> DWORD;
    pub fn WFDUpdateDeviceVisibility(
        pDeviceAddress: PDOT11_MAC_ADDRESS,
    ) -> DWORD;
}
DEFINE_DEVPROPKEY!{DEVPKEY_WiFi_InterfaceGuid,
    0xef1167eb, 0xcbfc, 0x4341, 0xa5, 0x68, 0xa7, 0xc9, 0x1a, 0x68, 0x98, 0x2c, 0x02}
