/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.util.Observer;
import java.util.Observable;
import java.util.ResourceBundle;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

import com.redhat.rhdb.vise.resources.ActionResources;

/*
 * need to set things so that the dialog will take a model, clone it, work on
 * the clone, and set the original values to clone values if and only if the
 * user clicks ok. ???
 */

/**
 * Preferences dialog for Explain. This dialog is modal and
 * associated with a specific ConfigurationModel and Frame (parent).
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 *
 * @see com.redhat.rhdb.vise.ConfigurationModel
 */
public class PreferencesDialog extends JDialog implements Observer, ActionListener {
    private ResourceBundle res;
	
	private int result;
	private Frame owner;
	private JTabbedPane tabpane;
	private JButton cancelButton, okButton;
	private Component genprefs, dispprefs;
	private CheckBoxListener cbl;

	private JCheckBox antialias_button;
	private JCheckBox largeicons_button, tooltips_button, reload_button;
	private JRadioButton explain_button;
	private JRadioButton analyze_button;

	private ConfigurationModel cm;

	/** User has closed window with intention of accepting changes. */
	public static final int OK_OPTION = 0;

	/** User has closed window with intention of discarding changes. */
	public static final int CANCEL_OPTION = 1;

	/** User has closed window with intention of discarding changes. */
	public static final int CLOSED_OPTION = 2;

	/**
	 * Creates a new <code>PreferencesDialog</code> instance.
	 *
	 * @param cm a <code>ConfigurationModel</code> value
	 * @param owner a <code>Frame</code> value
	 */
	public PreferencesDialog(ConfigurationModel cm, Frame owner)
	{
		this(cm, owner, ExplainResources.getString(ExplainResources.PREFERENCES_TITLE));
	}

	/**
	 * Creates a new <code>PreferencesDialog</code> instance.
	 *
	 * @param cm a <code>ConfigurationModel</code> value
	 * @param owner a <code>Frame</code> value
	 * @param title a <code>String</code> value
	 */
	public PreferencesDialog(ConfigurationModel cm, Frame owner, String title)
	{
		// we are a modal dialog window, owned by owner
		super(owner, true);
		setTitle(title);

		this.owner = owner;
		this.result = CLOSED_OPTION;
		
		this.cm = cm;
		cm.addObserver(this);
		
		// Listeners
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e)
			{
				close();
			}
		});

		cbl = new CheckBoxListener();

        res = ResourceBundle.getBundle("com.redhat.rhdb.vise.resources.ActionResources");

		initComponents();

		update();
	}

	/**
	 * Get the ConfigurationModel used.
	 *
	 * @return a <code>ConfigurationModel</code> value
	 */
	public ConfigurationModel getConfigurationModel()
	{
		return cm;
	}

	/**
	 * Set the ConfigurationModel to use.
	 *
	 * @param cm a <code>ConfigurationModel</code> value
	 */
	public void setConfigurationModel(ConfigurationModel cm)
	{
		this.cm = cm;
		cm.addObserver(this);
		
		update();
	}

	/**
	 * Show the dialog.
	 */
	public void show()
	{
		tabpane.setSelectedIndex(0);
		tabpane.requestFocus();
		pack();
		setSize(415, getHeight());
		setLocationRelativeTo(owner);

		super.show();
	}

	/**
	 * Get how/why dialog was closed.
	 *
	 * @return an <code>int</code> value
	 */
	public int getResult()
	{
		return result;
	}

	//
	// Observer stuff
	//

	// inherits doc comment
	public void update(Observable o, Object arg)
	{
		ConfigurationModel m = (ConfigurationModel) o;

		if (m != cm)
			return;

		update();
	}

	//
	// ActionListener
	//
	
	/**
	 * Action listener.
	 *
	 * @param e an <code>ActionEvent</code> value
	 */
	public void actionPerformed(ActionEvent e)
	{
		String command = e.getActionCommand();

		if (command.equals(res.getString(ActionResources.OK_COMMAND)))
		{
			ok();
		}
		else if (command.equals(res.getString(ActionResources.CANCEL_COMMAND)))
		{
			cancel();
		}
		else
		{
			System.err.println("PreferencesDialog: unknown action command" + command);
		}
	}

	//
	// private methods
	//

	private void cancel()
	{
		result = CANCEL_OPTION;
		hide();
	}

	private void close()
	{
		result = CLOSED_OPTION;
		hide();
	}

	private void ok()
	{
		result = OK_OPTION;
		hide();
	}
	
	private void update()
	{
		tooltips_button.setSelected(cm.isShowToolTips());
		reload_button.setSelected(cm.isReload());
		if (cm.isAnalyze()) {
			analyze_button.setSelected(true);
		} else {
			explain_button.setSelected(true);
		}
		antialias_button.setSelected(cm.isAntiAlias());
		largeicons_button.setSelected(cm.isUseLargeIcons());
	}

	private void initComponents()
	{
		JPanel content;
		JPanel buttonPanel;

		content = new JPanel();
		content.setLayout(new BorderLayout());
		content.setBorder(BorderFactory.createEmptyBorder(0, 0, 0, 0));
		setContentPane(content);

		getRootPane().setDefaultButton(okButton);

		buttonPanel = new ButtonPanel(this);
		getContentPane().add(buttonPanel, BorderLayout.SOUTH);

		tabpane = new JTabbedPane();
		
		genprefs = createGeneralPanel();
		tabpane.addTab(ExplainResources.getString(ExplainResources.PREFS_TAB_GENERAL),
					   genprefs);
		tabpane.setSelectedIndex(0);

		dispprefs = createDisplayPanel();
		tabpane.addTab(ExplainResources.getString(ExplainResources.PREFS_TAB_DISPLAY),
					   dispprefs);

		tabpane.setToolTipTextAt(0, ExplainResources.getString(ExplainResources.PREFS_TAB_GENERAL_TOOLTIP));
		tabpane.setToolTipTextAt(1, ExplainResources.getString(ExplainResources.PREFS_TAB_DISPLAY_TOOLTIP));

		getContentPane().add(tabpane, BorderLayout.CENTER);
	}

	private JPanel createGeneralPanel()
	{
		JPanel panel;

		panel = new JPanel();
		panel.setBorder(BorderFactory.createEmptyBorder(12, 12, 11, 11));

		GridBagLayout gbl = new GridBagLayout();
		GridBagConstraints gbc = new GridBagConstraints();

		gbc.gridx = 0;
		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.weighty = 0.0;
		gbc.weightx = 1.0;
		gbc.ipadx = 0;
		gbc.ipady = 0;

		panel.setLayout(gbl);

		gbc.gridy = 0;
		gbc.gridheight = 3;
		gbc.anchor = GridBagConstraints.NORTHWEST;
		
		JPanel buttonPanel = new JPanel();
		Border etched = BorderFactory.createEtchedBorder();
		Border titled = BorderFactory.createTitledBorder(etched,
										ExplainResources.getString(ExplainResources.PREFS_LABEL_TOOLBAR_BUTTON));
		buttonPanel.setBorder(titled);
		buttonPanel.setLayout(new BorderLayout());
		
		explain_button = new JRadioButton(ExplainResources.getString(ExplainResources.PREFS_LABEL_EXPLAIN));
		explain_button.addItemListener(cbl);
		explain_button.setMnemonic(((Integer)ExplainResources.getObject(ExplainResources.PREFS_LABEL_EXPLAIN_MNEMONIC)).intValue());
		explain_button.setToolTipText(ExplainResources.getString(ExplainResources.PREFS_LABEL_EXPLAIN_TOOLTIP));

		buttonPanel.add(explain_button, BorderLayout.NORTH);

		analyze_button = new JRadioButton(ExplainResources.getString(ExplainResources.PREFS_LABEL_ANALYZE));
		analyze_button.addItemListener(cbl);
		analyze_button.setMnemonic(((Integer)ExplainResources.getObject(ExplainResources.PREFS_LABEL_ANALYZE_MNEMONIC)).intValue());
		analyze_button.setToolTipText(ExplainResources.getString(ExplainResources.PREFS_LABEL_ANALYZE_TOOLTIP));

		buttonPanel.add(analyze_button, BorderLayout.SOUTH);
		
		ButtonGroup bg = new ButtonGroup();
		bg.add(explain_button);
		bg.add(analyze_button);
		
		gbl.setConstraints(buttonPanel, gbc);
		panel.add(buttonPanel);

		gbc.gridy = 4;
		gbc.gridheight = 1;

		tooltips_button = new JCheckBox(ExplainResources.getString(ExplainResources.PREFS_LABEL_TOOLTIPS));
		tooltips_button.addItemListener(cbl);
		tooltips_button.setMnemonic(((Integer)ExplainResources.getObject(ExplainResources.PREFS_LABEL_TOOLTIPS_MNEMONIC)).intValue());
		tooltips_button.setToolTipText(ExplainResources.getString(ExplainResources.PREFS_LABEL_TOOLTIPS_TOOLTIP));

		gbl.setConstraints(tooltips_button, gbc);
		panel.add(tooltips_button);

		gbc.gridy = 5;
		gbc.weighty = 1.0;
		gbc.insets = new Insets(11, 0, 0, 0);

		reload_button = new JCheckBox(ExplainResources.getString(ExplainResources.PREFS_LABEL_RELOAD));
		reload_button.addItemListener(cbl);
		reload_button.setMnemonic(((Integer)ExplainResources.getObject(ExplainResources.PREFS_LABEL_RELOAD_MNEMONIC)).intValue());
		reload_button.setToolTipText(ExplainResources.getString(ExplainResources.PREFS_LABEL_RELOAD_TOOLTIP));

		gbl.setConstraints(reload_button, gbc);
		panel.add(reload_button);

		// some glue to position buttons at the top
		panel.add(Box.createGlue());

		return panel;
	}

	private JPanel createDisplayPanel()
	{
		JPanel panel;

		panel = new JPanel();
		panel.setBorder(BorderFactory.createEmptyBorder(12, 12, 11, 11));

		GridBagLayout gbl = new GridBagLayout();
		GridBagConstraints gbc = new GridBagConstraints();

		gbc.gridx = 0;
		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.weighty = 0.0;
		gbc.weightx = 1.0;
		gbc.ipadx = 0;
		gbc.ipady = 0;

		panel.setLayout(gbl);

		gbc.gridy = 0;
		gbc.anchor = GridBagConstraints.NORTHWEST;

		antialias_button = new JCheckBox(ExplainResources.getString(ExplainResources.PREFS_LABEL_ANTIALIAS));
		antialias_button.addItemListener(cbl);
		antialias_button.setMnemonic(((Integer)ExplainResources.getObject(ExplainResources.PREFS_LABEL_ANTIALIAS_MNEMONIC)).intValue());
		antialias_button.setToolTipText(ExplainResources.getString(ExplainResources.PREFS_LABEL_ANTIALIAS_TOOLTIP));

		gbl.setConstraints(antialias_button, gbc);
		panel.add(antialias_button);

		gbc.gridy = 1;
		gbc.weighty = 1.0;
		gbc.insets = new Insets(11, 0, 0, 0);

		largeicons_button = new JCheckBox(ExplainResources.getString(ExplainResources.PREFS_LABEL_LARGEICONS));
		largeicons_button.addItemListener(cbl);
		largeicons_button.setMnemonic(((Integer)ExplainResources.getObject(ExplainResources.PREFS_LABEL_LARGEICONS_MNEMONIC)).intValue());
		largeicons_button.setToolTipText(ExplainResources.getString(ExplainResources.PREFS_LABEL_LARGEICONS_TOOLTIP));

		gbl.setConstraints(largeicons_button, gbc);
		panel.add(largeicons_button);

		// some glue to position buttons at the top
		panel.add(Box.createGlue());

		return panel;
	}

	//
	// check box listener
	//

	private class CheckBoxListener implements ItemListener {
		public void itemStateChanged(ItemEvent e)
		{
			Object source = e.getSource();
			int state = e.getStateChange();

			if (source == tooltips_button)
			{
				if (state == ItemEvent.SELECTED && !cm.isShowToolTips())
				{
					cm.setShowToolTips(true);
				}
				else if (state == ItemEvent.DESELECTED && cm.isShowToolTips())
				{
					cm.setShowToolTips(false);
				}
			}
			else if (source == reload_button)
			{
				if (state == ItemEvent.SELECTED && !cm.isReload())
				{
					cm.setReload(true);
				}
				else if (state == ItemEvent.DESELECTED && cm.isReload())
				{
					cm.setReload(false);
				}
			}
			else if (source == explain_button)
			{
				if (state == ItemEvent.SELECTED && cm.isAnalyze())
				{
					cm.setAnalyze(false);
				}
			}
			else if (source == analyze_button)
			{
				if (state == ItemEvent.SELECTED && !cm.isAnalyze())
				{
					cm.setAnalyze(true);
				}
			}
			else if (source == antialias_button)
			{
				if (state == ItemEvent.SELECTED && !cm.isAntiAlias())
				{
					cm.setAntiAlias(true);
				}
				else if (state == ItemEvent.DESELECTED && cm.isAntiAlias())
				{
					cm.setAntiAlias(false);
				}
			}
			else if (source == largeicons_button)
			{
				if (state == ItemEvent.SELECTED && !cm.isUseLargeIcons())
				{
					cm.setUseLargeIcons(true);
				}
				else if (state == ItemEvent.DESELECTED && cm.isUseLargeIcons())
				{
					cm.setUseLargeIcons(false);
				}
			}
		}
	}

}// PreferencesDialog
