/* ================================================================
 * JCommon : a general purpose, open source, class library for Java
 * ================================================================
 *
 * Project Info:  http://www.object-refinery.com/jcommon/index.html
 * Project Lead:  David Gilbert (david.gilbert@object-refinery.com);
 *
 * (C) Copyright 2000-2002, by Simba Management Limited.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * ---------------
 * MonthTests.java
 * ---------------
 * (C) Copyright 2001, 2002, by Simba Management Limited.
 *
 * Original Author:  David Gilbert (for Simba Management Limited);
 * Contributor(s):   -;
 *
 * $Id: MonthTests.java,v 1.3 2002/06/26 16:00:34 mungady Exp $
 *
 * Changes
 * -------
 * 16-Nov-2001 : Version 1 (DG);
 * 14-Feb-2002 : Order of parameters in Month(int, int) constructor changed (DG);
 * 26-Jun-2002 : Removed unnecessary import (DG);
 *
 */

package com.jrefinery.data.junit;

import java.util.Date;
import java.util.TimeZone;
import junit.framework.*;
import com.jrefinery.data.*;

/**
 * Tests for the Month class.
 */
public class MonthTests extends TestCase {

    protected Month jan_1900;
    protected Month feb_1900;
    protected Month nov_9999;
    protected Month dec_9999;

    /**
     * Returns the tests as a test suite.
     */
    public static Test suite() {
        return new TestSuite(MonthTests.class);
    }

    /**
     * Constructs a new set of tests.
     * @param The name of the tests.
     */
    public MonthTests(String name) {
        super(name);
    }

    /**
     * Common test setup.
     */
    protected void setUp() {
        jan_1900 = new Month(Month.JANUARY, 1900);
        feb_1900 = new Month(Month.FEBRUARY, 1900);
        nov_9999 = new Month(Month.NOVEMBER, 9999);
        dec_9999 = new Month(Month.DECEMBER, 9999);
    }

    /**
     * Test that a Month instance is equal to itself.
     *
     * SourceForge Bug ID: 558850.
     */
    public void testEqualsSelf() {
        Month month = new Month();
        this.assertTrue(month.equals(month));
    }

    public void testEquals() {
        Month m1 = new Month(Month.MAY, 2002);
        Month m2 = new Month(Month.MAY, 2002);
        this.assertTrue(m1.equals(m2));
    }

    /**
     * In GMT, the end of Feb 2000 is java.util.Date(951,868,799,999L).  Use this to check the
     * Month constructor.
     */
    public void testDateConstructor1() {

        TimeZone zone = TimeZone.getTimeZone("GMT");
        Month m1 = new Month(new Date(951868799999L), zone);
        Month m2 = new Month(new Date(951868800000L), zone);

        this.assertEquals(Month.FEBRUARY, m1.getMonth());
        this.assertEquals(951868799999L, m1.getEnd(zone));

        this.assertEquals(Month.MARCH, m2.getMonth());
        this.assertEquals(951868800000L, m2.getStart(zone));

    }

    /**
     * In Auckland, the end of Feb 2000 is java.util.Date(951,821,999,999L).  Use this to check the
     * Month constructor.
     */
    public void testDateConstructor2() {

        TimeZone zone = TimeZone.getTimeZone("Pacific/Auckland");
        Month m1 = new Month(new Date(951821999999L), zone);
        Month m2 = new Month(new Date(951822000000L), zone);

        this.assertEquals(Month.FEBRUARY, m1.getMonth());
        this.assertEquals(951821999999L, m1.getEnd(zone));

        this.assertEquals(Month.MARCH, m2.getMonth());
        this.assertEquals(951822000000L, m2.getStart(zone));

    }

    /**
     * Set up a month equal to Jan 1900.  Request the previous month, it should be null.
     */
    public void testJan_1900_previous() {
        Month previous = (Month)jan_1900.previous();
        this.assertNull(previous);
    }

    /**
     * Set up a month equal to Jan 1900.  Request the next month, it should be Feb 1900.
     */
    public void testJan_1900_next() {
        Month next = (Month)jan_1900.next();
        this.assertEquals(feb_1900, next);
    }

    /**
     * Set up a month equal to Dec 9999.  Request the previous month, it should be Nov 9999.
     */
    public void testDec_9999_previous() {
        Month previous = (Month)dec_9999.previous();
        this.assertEquals(nov_9999, previous);
    }

    /**
     * Set up a month equal to Dec 9999.  Request the next month, it should be null.
     */
    public void testDec_9999_next() {
        Month next = (Month)dec_9999.next();
        this.assertNull(next);
    }

    /**
     * Test the string parsing code...
     */
    public void testParseMonth() {

        Month month = null;

        // test 1...
        try {
            month = Month.parseMonth("1990-01");
        }
        catch (TimePeriodFormatException e) {
            month = new Month(1, 1900);
        }
        this.assertEquals(1, month.getMonth());
        this.assertEquals(1990, month.getYear().getYear());

        // test 2...
        try {
            month = Month.parseMonth("02-1991");
        }
        catch (TimePeriodFormatException e) {
            month = new Month(1, 1900);
        }
        this.assertEquals(2, month.getMonth());
        this.assertEquals(1991, month.getYear().getYear());

        // test 3...
        try {
            month = Month.parseMonth("March 1993");
        }
        catch (TimePeriodFormatException e) {
            month = new Month(1, 1900);
        }
        this.assertEquals(3, month.getMonth());
        this.assertEquals(1993, month.getYear().getYear());

    }

}