/**********************************************************************
 ** Status class: is used to mark a certain status on an object.  It
 **               could be used to mark an object as used up or anything
 **               else to indicate a certain status
 **
 ** Reviewed through:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef STATUS_C
#define STATUS_C

#include "config.h"
#include "sysdep.h"
#include "status.h"
#include "newfuncts.h"

/***********************************************************************
 ** Status (constructor) - starts up the status element
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Status::Status()
{
   status_list = NULL;
}


/***********************************************************************
 ** ~Status (destructor)  - destroys the status list
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Status::~Status()
{
   status_type *tmp_list;
  
   tmp_list = status_list;
   while (tmp_list != NULL)
   {
      tmp_list = tmp_list->next_status;
      delete_status_type(status_list);
      status_list = tmp_list;
   }
}


/***********************************************************************
 ** add_status - adds a status element to the list
 **
 ** Parameters: the_name - the name of the new status element
 **
 ** Returns:  1 if success
 **           0 if already there
 **          -1 for failure
 **
 ***********************************************************************/

int Status::add_status(char *the_name)
{
   return add_status(the_name, NULL);
}

/***********************************************************************
 ** add_status - adds a status element to the list with string data
 **
 ** Parameters: the_name - the name of the new status element
 **             the_string - the string data to add
 **
 ** Returns:  1 if success
 **           0 if already there
 **          -1 for failure
 **
 ***********************************************************************/

int Status::add_status(char *the_name, char *the_string)
{
   status_type *new_status;
   status_type *tmp_status;

   /* first search the list for this status */
   tmp_status = status_list;
   while (tmp_status != NULL)
   {
      if (!STRCASECMP(tmp_status->name.str_show(), the_name))
         return 0;
      tmp_status = tmp_status->next_status;
   }

   /* now we create the new status and add it to the head of the list */
   new_status = new_status_type();
   new_status->name = the_name;
   new_status->status_string = the_string;
   new_status->next_status = status_list;
   status_list = new_status;
   return 1;
}


/***********************************************************************
 ** has_status - finds a status element in the list
 **
 ** Parameters: the_name - the name of the status element to find
 **
 ** Returns:  1 if it exists
 **           0 if not there
 **          -1 for failure
 **
 ***********************************************************************/

int Status::has_status(char *the_name)
{
   status_type *tmp_status;

   /* first search the list for this status */
   tmp_status = status_list;
   while (tmp_status != NULL)
   {
      if (!STRCASECMP(tmp_status->name.str_show(), the_name))
         return 1;
      tmp_status = tmp_status->next_status;
   }
   return 0;
}


/***********************************************************************
 ** get_status_string - finds a status element in the list and returns the
 **                     data string attached to it
 **
 ** Parameters: the_name - the name of the status element to find
 **
 ** Returns:  pointer to the string if successful, NULL if failed
 **
 ***********************************************************************/

char *Status::get_status_string(char *the_name)
{
   status_type *tmp_status;

   /* first search the list for this status */
   tmp_status = status_list;
   while (tmp_status != NULL)
   {
      if (!STRCASECMP(tmp_status->name.str_show(), the_name))
         return tmp_status->status_string.str_show();
      tmp_status = tmp_status->next_status;
   }
   return NULL;
}


/***********************************************************************
 ** remove_status - removes a status element from the list
 **
 ** Parameters: the_name - the name of the status element to remove
 **
 ** Returns:  1 if found and removed
 **           0 if not found
 **          -1 for failure
 **
 ***********************************************************************/

int Status::remove_status(char *the_name)
{
   status_type *prev_status = NULL;
   status_type *tmp_status;

   /* first search the list for this status */
   tmp_status = status_list;
   while (tmp_status != NULL)
   {
      if (!STRCASECMP(tmp_status->name.str_show(), the_name))
      {
         if (prev_status == NULL)
	 {
            status_list = tmp_status->next_status;
	 }
         else
	 {
            prev_status->next_status = tmp_status->next_status;
	 }
         delete_status_type(tmp_status);
         return 1;
      }
      prev_status = tmp_status;
      tmp_status = tmp_status->next_status;
   }
   return 0;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Status::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Status::get_mem_size_dynamic()
{
   int  size = 0;
   status_type *tmp_status;

   tmp_status = status_list;
   while (tmp_status != NULL)
   {
      size += tmp_status->name.get_mem_size_dynamic();
      tmp_status = tmp_status->next_status;
   }

   return size;
}


#endif
