// ---------------------------------------------------------------------------
// - Resolver.hpp                                                            -
// - aleph engine - file path resolver class definition                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_RESOLVER_HPP
#define  ALEPH_RESOLVER_HPP

#ifndef  ALEPH_STRVEC_HPP
#include "Strvec.hpp"
#endif

#ifndef  ALEPH_INPUT_HPP
#include "Input.hpp"
#endif

namespace aleph {

  /// The Resolver class is a special class that is used to resolve a 
  /// a particular file by name. The resolver maintains a list of path
  /// or librarian object. When a file is requested, the list is search
  /// and an input stream is returned by the resolver.
  /// @author amaury darsch

  class Resolver : public Object {
  private:
    /// the list of path
    struct s_rpath* p_rpath;

  public:
    /// create an empty resolver
    Resolver (void);

    /// create a resolver with a set of path
    /// @param paths the resolver paths
    Resolver (const Strvec& paths);

    /// destroy this resolver
    ~Resolver (void);

    /// @return the class name
    String repr (void) const;

    /// add a path to this resolver
    /// @param path the path to add
    void add (const String& path);

    /// @return true if the name is valid
    bool valid (const String& name) const;

    /// @return the resolved full path
    String getpath (const String& name) const;

    /// @return an input stream by name
    Input* get (const String& name) const;

    /// @return an input stream by name or throw an exception
    Input* lookup (const String& name) const;

    /// @return true if the name is a valid aleph file
    bool alpvld (const String& name) const;

    /// @return the aleph file resolved name
    String alpname (const String& name) const;

    /// @return an input stream by an aleph name
    Input* alpget (const String& name) const;

    /// @return an input stream by an aleph name or throw an exception
    Input* alplkp (const String& name) const;

    /// generate a new symbol
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// apply this symbol with a set of arguments and a quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);

  private:
    // make the copy constructor private
    Resolver (const Resolver&);
    // make the assignment operator private
    Resolver& operator = (const Resolver&);
  };
}

#endif
