// ---------------------------------------------------------------------------
// - Syscalls.cpp                                                            -
// - aleph:sys library - system call implementation                          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Vector.hpp"
#include "System.hpp"
#include "Integer.hpp"
#include "Syscalls.hpp"
#include "Exception.hpp"
 
namespace aleph {

  // exit this process with an exit code

  Object* asys_exit (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 1) throw Exception ("argument-error", 
				    "invalid number of arguments with exit");
    long val = argv->getint (0);
    delete argv;
    System::exit (val);
    return nilp;
  }

  // return the process id

  Object* asys_getpid (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    long argc = (args == nilp) ? 0 : args->length ();
    if (argc != 0) throw Exception ("argument-error", 
				    "invalid number of arguments with getpid");
    return new Integer (System::getpid ());
  }

  // pause for a certain time

  Object* asys_sleep (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 1) throw Exception ("argument-error", 
				    "invalid number of arguments with sleep");
    long val = argv->getint (0);
    delete argv;
    // pause for a while
    System::sleep (val);
    return nilp;
  }

  // return an environment variable value

  Object* asys_getenv (Runnable* robj, Nameset* nset, Cons* args) {
    // get the arguments
    Vector* argv = Vector::eval (robj, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 1) throw Exception ("argument-error", 
				    "invalid number of arguments with getenv");
    String val = argv->getstring (0);
    delete argv;
    // get the environment variable
    return new String (System::getenv (val));
  }

  // return a random number

  Object* asys_random (Runnable* robj, Nameset* nset, Cons* args) {
    return new Integer (Integer::random ());
  }

  // return the host name

  Object* asys_hostname (Runnable* robj, Nameset* nset, Cons* args) {
    return new String (System::hostname ());
  }

  // return the user name

  Object* asys_username (Runnable* robj, Nameset* nset, Cons* args) {
    return new String (System::username ());
  }
}
