/*
 *  CaSU - communications & status utilities.
 *  Copyright (C) 1992, 1993, 1994 Luke Mewburn <lm@rmit.edu.au>
 *	incorporating:
 *	   flon - lists your friends who are logged on.
 *	   to - send a short message to a friend
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "casu.h"

/*
 *	print_output
 *
 * print the output according to the command and template buffer.
 */

void
print_output(yes_str, no_str)
    char *yes_str, *no_str;
{
    static char	term[sizeof(_PATH_DEV) + UT_LINESIZE + 1] = _PATH_DEV;
    char	*myname;
    time_t	today;
    int		entlp;
    int		me_on;
    int		offset, width;

    static struct one_ent
    {
	int	on;		/* times this person on */
	int	avail;		/* if any term has mesg y */
	int	stres;		/* result of stat for this tty */
	int	mesg;		/* mesg of cur ent */
	time_t	idle, login; 	/* idle & login of cur ent */
	char	*tty;		/* tty of cur ent */
	char	*host;		/* remote host */
	char	*name;		/* name of user */
	int	valid;		/* is this entry valid */
    } cur_ent, null_ent;


    me_on = 0;
    today=time(NULL);
    if (!(flags & NO_HEADER))
    {
    	printf("\
                Fast Lon v%-5.5s  %s\
             -----------------------------------------------\n\
", VERSION, ctime(&today) );
    }
    myname = NULL;
    if (!(flags & NO_TAILER))
	myname=get_username();

    for (entlp=0; entlp <utmp_count; entlp++)
    {
	int		comlp, fpos;
	frend		*curfrend, tmpfrend;
	struct stat	stbufr;

	if (NULL_UTMP_ENTRY(&utmp_list[entlp]))
	    continue;		/* for who emulation - skip null entries */

    	if (!(flags & NO_TAILER) && (myname != NULL))
	{
	    if (strncmp(utmp_list[entlp].ut_name, myname, UT_NAMESIZE) == 0)
		me_on++;
	}

    	if (friends_list != NULL)
    	{
	    strnc0py(tmpfrend.name, utmp_list[entlp].ut_name, UT_NAMESIZE);
	    curfrend = nbsearch(&tmpfrend, friends_list, friends_count,
				sizeof(frend), compare_friends);
	    if (   ((!(flags & ALL_ON)) && (curfrend == NULL))
		|| (curfrend && (flags & NO_MATES)))
	    {
		cur_ent = null_ent;	/* previous info isn't valid anymore */
		continue;
	    }
    	}
    	else
	    curfrend = NULL;

    	if (flags & NEED_STAT)
    	{
	    strcpy(&term[sizeof(_PATH_DEV) - 1], utmp_list[entlp].ut_line);
	    cur_ent.stres=stat(term, &stbufr);
    	}
	else
	    cur_ent.stres= -1;	/* fool the code to skip the stat stuff */

    	if (entlp < utmp_count-1)		/* at least 1 to go */
	{
	    if (strncmp(utmp_list[entlp].ut_name,
			utmp_list[entlp+1].ut_name, UT_NAMESIZE) == 0)
	    {
		if (cur_ent.stres != -1)
		{
		    time_t idle = today - stbufr.st_atime;
			/* if not valid, or idle < least_idle */
		    if ((! cur_ent.valid) || (idle < cur_ent.idle))
		    {
			cur_ent.idle=	idle;
			cur_ent.tty=	utmp_list[entlp].ut_line;
			cur_ent.login=	utmp_list[entlp].UT_TIME;
			cur_ent.name=	utmp_list[entlp].ut_name;
#ifdef HAVE_UT_HOST
			cur_ent.host=	utmp_list[entlp].ut_host;
#endif
			cur_ent.mesg=	(stbufr.st_mode & MESGS_ON);
			cur_ent.valid++;
		    }
		}
		cur_ent.on++;
		if (stbufr.st_mode & MESGS_ON)
		    cur_ent.avail++;
		if (flags & ONE_ONLY)		/* skip multiple entries */
		    continue;
	    }
	}
	if (cur_ent.stres != -1)
	{
		/* get valid info for last entry of a user */
	    time_t idle = today - stbufr.st_atime;

	    if ((! cur_ent.valid) || (idle < cur_ent.idle))
	    {
		cur_ent.idle= idle;
		cur_ent.tty=  utmp_list[entlp].ut_line;
		cur_ent.login=utmp_list[entlp].UT_TIME;
		cur_ent.name= utmp_list[entlp].ut_name;
#ifdef HAVE_UT_HOST
		cur_ent.host= utmp_list[entlp].ut_host;
#endif
		cur_ent.mesg= (stbufr.st_mode & MESGS_ON);
		cur_ent.valid++;
	    }
	    if (stbufr.st_mode & MESGS_ON)
		cur_ent.avail++;
	    if (flags & NO_IDLE)
		if (cur_ent.idle >= max_idle)
		{
		    cur_ent = null_ent;
		    continue;	/* don't print entry if idle limit exceeded */
		}
	}
	else
	{
	    cur_ent.tty=  utmp_list[entlp].ut_line;
	    cur_ent.login=utmp_list[entlp].UT_TIME;
	    cur_ent.name= utmp_list[entlp].ut_name;
#ifdef HAVE_UT_HOST
	    cur_ent.host= utmp_list[entlp].ut_host;
#endif
	}

    	offset=fpos=0;
    	for (comlp=0; printdat.cmds[comlp]; comlp++)
    	{
	    char tmpbuf[UT_HOSTSIZE + 2 + 1], *outs;
	    int	len, tmpwid;


	    outs = NULL;
	    width=printdat.cmds[comlp];
	    tmpwid=width & C_WIDMASK;
	    switch (printdat.cmds[comlp] & C_CMDMASK)
	    {
	    case C_NULL:
		    outs = &printdat.form[fpos];
		    break;
	    case C_USER:
		    strnc0py(tmpbuf, cur_ent.name, UT_NAMESIZE);
		    outs = tmpbuf;
		    break;
	    case C_PSEUDO:
		    if (curfrend != NULL)
		    {
			outs = curfrend->pseudo;
			break;
		    }
		    if ((flags & BEST_NAME) == 0)
			break;	/* don't fall thru if 'best' mode off */
	    case C_REAL:
		    { 
#if USE_GETPWENT
			struct passwd *curupwd;
			strnc0py(tmpfrend.name, cur_ent.name, UT_NAMESIZE);
			curupwd = getpwnam(tmpfrend.name);
			if (curupwd != NULL)
			    outs = convert_realname(curupwd->pw_gecos,
						    curupwd->pw_name,
						    curupwd->pw_uid);
#else /* !USE_GETPWENT */
#   if 0	/* lsearch */
			int  i;
			for (i=0; i<pw_count; i++)
			    if (strncmp(pw_list[i].username, cur_ent.name,
				    UT_NAMESIZE) ==0)
				break;
			if (i != pw_count)
			    outs = convert_realname(pw_list[i].gcos,
						    pw_list[i].username,
						    pw_list[i].uid);
#   else	/* nbsearch */
			upwd *curp, tmpp;
			strnc0py(tmpfrend.name, cur_ent.name, UT_NAMESIZE);
			tmpp.username = tmpfrend.name;
			curp = nbsearch(&tmpp, pw_list, pw_count, sizeof(upwd),
					compare_users);
			if (curp)
			    outs = convert_realname(curp->gcos,
						    curp->username,
						    curp->uid);
#   endif
#endif /* !USE_GETPWENT */
		    }
		    break;
	    case C_COUNT:
		    if (!cur_ent.on)
			break;
		    sprintf(tmpbuf, "%-d", ++cur_ent.on);
		    outs = tmpbuf;
		    break;
	    case C_X:
		    tmpbuf[0] = cur_ent.on ? 'x' : ' ';
		    tmpbuf[1] = '\0';
		    outs = tmpbuf;
		    break;
	    case C_LOGIN:
		    outs = 4 + ctime(&cur_ent.login);
		    break;
	    case C_IDLE:
		    if (cur_ent.stres == -1)	/* print nothing if stat died */
			break;
		    if (cur_ent.idle >= MAXIDLE)
			outs = " >>> ";
		    else
			if (cur_ent.idle < min_idle)
			    outs = " ... ";
			else
			{
			    sprintf(tmpbuf, "%02d:%02d",
					(int) (cur_ent.idle / 60),
					(int) (cur_ent.idle % 60));
			    outs = tmpbuf;
			}
		    break;
	    case C_TTY:
		    outs = cur_ent.tty;
		    break;
	    case C_MESG:
		    if (cur_ent.stres == -1) 	/* print nothing if stat died */
			break;
		    outs = cur_ent.mesg ? yes_str : no_str;
		    break;
	    case C_AVAIL:
		    if (cur_ent.stres == -1) 	/* print nothing if stat died */
			break;
		    outs = cur_ent.avail ? yes_str : no_str;
		    break;
#ifdef HAVE_UT_HOST
	    case C_HOST:
		    strnc0py(tmpbuf, cur_ent.host, UT_HOSTSIZE);
		    outs = tmpbuf;
		    break;
	    case C_HOSTBRK:
		    if (cur_ent.host[0] == '\0')
			break;
		    tmpbuf[0] = '(';
		    strnc0py(tmpbuf + 1, cur_ent.host, UT_HOSTSIZE);
		    if (strlen(tmpbuf) >= tmpwid)
			tmpbuf[tmpwid - 1] = '\0';
		    strcat(tmpbuf, ")");
		    outs=tmpbuf;
		    break;
#endif /* HAVE_UT_HOST */
	    default:
		    break;
	    } /* switch */
	    if (outs == NULL)
		len = 0;
	    else
	    {
		len=strlen(outs);
		if (len>=tmpwid)			/* need to truncate */
		{
		    strncpy(&printdat.buf[offset], outs, tmpwid);
		    len = tmpwid;
		}
		else if (width & C_RIGHT)		/* right align */
		{
		    tmpwid-= len;
		    strncpy(&printdat.buf[offset+tmpwid], outs, len);
		    len = 0;
		}
		else					/* left align */
		    strncpy(&printdat.buf[offset], outs, len);
	    }
	    if (width & C_VARIENT)			/* varient width */
	    {
		width &= ~C_WIDMASK;
		width += len & C_WIDMASK;
	    }
	    else
	    {
		int lp;
		for (lp = len; lp < tmpwid; lp++)	/* clear unused areas */
		    printdat.buf[offset+lp] = ' ';
	    }
	    offset += width & C_WIDMASK;
	    fpos += printdat.cmds[comlp] & C_WIDMASK;
    	}
	printdat.buf[offset] = '\0';
    	puts(printdat.buf);
	cur_ent = null_ent;	/* previous info isn't valid anymore */
    }
    if (!(flags & NO_TAILER))
    {
    	switch (me_on)
    	{
    	case 0:
#if 0
    		puts("\nYou technically aren't logged on.");
#else
		puts("\nYou're not logged on.");
#endif
    		break;
    	case 1:
    		puts("\nYou are logged on once.");
    		break;
    	case 2:
    		puts("\nYou are logged on twice.");
    		break;
    	default:
    		printf("\nYou are logged on %d times.\n", me_on);
    		break;
    	}
    	if (utmp_count == 1)
	    puts("You are the only person logged on.");
    	else
	    printf("There are %d users logged on.\n", utmp_count);
    }
} /* print_output */


/*
 *	who_am_I
 *
 * Emulates 'who am I'.
 */

void
who_am_I()
{
    char    *tty, *tmptty;
    int	    slp;
    time_t  ttytime;

    if ((tty = ttyname(0)) == NULL)	/* who am I < somefile */
    {
	ttytime=time(NULL);
	tty=WHO_NULLTTY;
    }
    else				/* just who am I */
    {
	tmptty=strrchr(tty, '/');
	if (tmptty != NULL)
	    tty = tmptty + 1;
	
		/* search for user on tty */
	for (slp=0; slp <utmp_count; slp++)
	{
	    if (NULL_UTMP_ENTRY(&utmp_list[slp]))
		continue;
	    if (strncmp(utmp_list[slp].ut_line, tty, UT_LINESIZE) == 0)
		break;
	}

	if (slp == utmp_count)
	    errexit("not found in utmp file", tty);
	ttytime=utmp_list[slp].UT_TIME;
    }
	
    printf(WHO_AM_I_FMT, (int) UT_NAMESIZE, get_username(),
	    tty, 4 + ctime(&ttytime));
} /* who_am_I */
