/*
 * FreeType font (truetype fonts)
 */
#include	"defs.h"
#include	"global.h"
#include	"bifont.h"
#ifdef KPATHSEA
#include	<kpathsea/tex-file.h>
#endif

#ifdef FREETYPE

#include	"freetype.h"

/* misc. functions for FreeType library
 */

static BOOLEAN ft_init = FALSE;
static TT_Engine engine;
static TT_Face face;
static TT_Instance instance;
static TT_Glyph glyph;
static TT_CharMap cmap;
static TT_Face_Properties properties;
static TT_Instance_Metrics imetrics;
static TT_Raster_Map bitmap;
static TT_Outline outline;
static TT_Matrix oscale = {(1<<16)/64, 0, 0, (1<<16)/64};
static int chxmin, chxmax, chymin, chymax;

ft_initialize()
{
    if (!ft_init) {
#ifdef DEBUG
	if (Debuguser)
	    fprintf(stderr, "FreeType initializing\n");
#endif
	if (TT_Init_FreeType(&engine))
	    Fatal("FreeType cannot be initialized");
	ft_init = TRUE;
    }
}

ft_glyph(filename)
char *filename;
{
    if (TT_Open_Face(engine, filename, &face)) {
	Warning("FreeType font %s cannot be opened", filename);
	return FALSE;
    }
    if (TT_New_Instance(face, &instance)) {
	Warning("FreeType font %s cannot make instance", filename);
	return FALSE;
    }
    if (TT_New_Glyph(face, &glyph)) {
	Warning("FreeType font %s cannot get glyph", filename);
	return FALSE;
    }
    return TRUE;
}

#define	floor(x)	((x)&~0x3f)
#define	ceiling(x)	(((x)+63)&~0x3f)

fitd64(d, s)
TT_FWord d;
TT_Fixed s;
{
    if (d >= 0)
	return floor(d*s>>16);
    else
	return -ceiling((-d)*s>>16);
}

fitu64(d, s)
TT_FWord d;
TT_Fixed s;
{
    if (d >= 0)
	return ceiling(d*s>>16);
    else
	return -floor((-d)*s>>16);
}

ft_size(filename, fsize)
char *filename;
int fsize;
{
    if (TT_Set_Instance_Resolutions(instance, resolution, resolution)) {
	Warning("FreeType font %s cannot set resolution", filename);
	return FALSE;
    }
    if (TT_Set_Instance_CharSize(instance, fsize/1024)) {
	Warning("FreeType font %s cannot set charsize", filename);
	return FALSE;
    }
    if (TT_Get_Instance_Metrics(instance, &imetrics)) {
	Warning("FreeType font %s cannot get metrics", filename);
	return FALSE;
    }
    if (TT_Get_Face_Properties(face, &properties)) {
	Warning("FreeType font %s cannot get properties", filename);
	return FALSE;
    }
    chxmin = fitd64(properties.header->xMin, imetrics.x_scale);
    chxmax = fitu64(properties.header->xMax, imetrics.x_scale);
    chymin = fitd64(properties.header->yMin, imetrics.y_scale);
    chymax = fitu64(properties.header->yMax, imetrics.y_scale);
    bitmap.width = (chxmax-chxmin)/64;
    bitmap.rows = (chymax-chymin)/64;
    bitmap.cols = (bitmap.width+7)/8;
    bitmap.size = bitmap.rows * bitmap.cols;
    bitmap.flow = TT_Flow_Down;
    return TRUE;
}

ft_char(c, i)
int c, i;
{
    if (TT_Load_Glyph(instance, glyph, c, TTLOAD_DEFAULT))
	Fatal("FreeType cannot load glyph for char %d", i);
    if (TT_Get_Glyph_Bitmap(glyph, &bitmap, -chxmin, -chymin))
	Fatal("FreeType cannot get bitmap for char %d", i);
}

fto_size1000(filename)
char *filename;
{
    /* make 1000x1000 coordinate */
    if (TT_Set_Instance_Resolutions(instance, 600, 600)) {
	Warning("FreeType font %s cannot set resolution", filename);
	return FALSE;
    }
    if (TT_Set_Instance_CharSize(instance, 120*64)) {
	Warning("FreeType font %s cannot set charsize", filename);
	return FALSE;
    }
    return TRUE;
}

fto_char(c, i)
int c, i;
{
    if (TT_Load_Glyph(instance, glyph, c, TTLOAD_DEFAULT))
	Fatal("FreeType cannot load glyph for char %d", i);
    if (TT_Get_Glyph_Outline(glyph, &outline))
	Fatal("FreeType cannot get outline for char %d", i);
    TT_Transform_Outline(&outline, &oscale);
}

/* char width in 1000x1000 coordinate */
#define	cwid(cw,fw)	(int)((float)(cw)*1000/(float)(fw))


/* tfm
 */
void
init_ft_fontinfo(fe)
struct font_entry *fe;
{
    struct font_entry *repfe;
    int	mm_markchar();
    void read_ft_fontinfo();

    if (dev_mf_kind(tfmfinfo(fe)->tfm_bf) == MF_KIND_FT)
	biinifinfo(fe) = alloc_biinif(biaccinfo(fe)->bf);
    else if ((repfe = dev_get_repfe(biaccinfo(fe)->bf)) == NULL) {
	biinifinfo(fe) = alloc_biinif(biaccinfo(fe)->bf);
	dev_set_repfe(biinifinfo(fe)->bf, fe);
    } else
	biinifinfo(fe) = biinifinfo(repfe);
    fe->fnt_markchar = mm_markchar;
    fe->fnt_readfontinfo = read_ft_fontinfo;
}

void
ft_reencode(bii, remap)
struct biinitfontinfo *bii;
unsigned char remap[];
{
    char *enc;
    int i;
    char *glyphs[NTFMCHARS];

    if ((enc = enc_read(bii->bf)) == NULL) {
	for (i = 0; i < NTFMCHARS; i++)
	    remap[i] = i;
	return;
    }
    Warning("reencoding of TrueType is supported in the next version\n");
    for (i = 0; i < NTFMCHARS; i++)
	remap[i] = i;
    return;
#if 0
    t1_get_encode(enc, glyphs);
    reencode(glyphs, bii->mark, bii->maxc, remap);
    t1_free_encode(glyphs);
#endif
}

void
read_ft_fontinfo(fe)
struct font_entry *fe;
{
    struct biinitfontinfo *bii;
    struct tfmfntinfo *tfmfi;
    char *fn, *filename;
    void read_tfm_finfo();
    BOOLEAN raster;
    int cmcount, cmid;
    short pid, eid;
    unsigned char remap[NTFMCHARS];
    int tfmw;
    int nwidth, nheight, nxoff, ndepth;
    char *npixel;
    int i;
    extern int hconv, vconv;

    ft_initialize();

    bii = biinifinfo(fe);
    tfmfi = NEW(struct tfmfntinfo, "tfmfont info");
    tfmfi->tfm_bf = bii->bf;
    tfmfinfo(fe) = tfmfi;
    read_tfm_finfo(fe);
    dev_tfm_initfe(fe);

    raster = dev_mf_kind(tfmfi->tfm_bf) == MF_KIND_FT;
    if (!raster && dev_get_repfe(bii->bf) == NULL) {
	dev_fto_refontdict(fe);
	return;
    }
    fn = dev_fontpath(tfmfinfo(fe)->tfm_bf);
#ifdef KPATHSEA
    if ((filename = kpsearch_file(fn, fe->n, kpse_truetype_format)) == NULL) {
	Warning("FreeType font file %s not found", fn);
	read_null_fontinfo(fe);
	return;
    }
#else
    filename = fn;
#endif
#ifdef DEBUG
    if (Debuguser)
	fprintf(stderr, "FreeType openfont %s (for %s, %s)\n", filename, fe->n,
		raster ? "raster" : "outline");
#endif
    if (!ft_glyph(filename)) {
	read_null_fontinfo(fe);
	return;
    }
    cmcount = TT_Get_CharMap_Count(face);
    for (cmid = 0; cmid < cmcount; cmid++) {
	if (TT_Get_CharMap_ID(face, cmid, &pid, &eid)) {
	    Warning("FreeType font %s does not have charmap id", filename);
	    read_null_fontinfo(fe);
	    return;
	}
	/* TODO */
	if (pid == TT_PLATFORM_MICROSOFT && eid == TT_MS_ID_UNICODE_CS)
	    break;
	/*if (pid == TT_PLATFORM_MACINTOSH && eid == TT_MAC_ID_ROMAN)*/
    }
    if (cmid == cmcount || TT_Get_CharMap(face, cmid, &cmap)) {
	Warning("FreeType font %s does not have charmap", filename);
	read_null_fontinfo(fe);
	return;
    }

    ft_reencode(bii, remap);
    if (raster) {
	tfmw = dev_ft_begfontdict(fe, bii->maxc);
	if (!ft_size(filename, tfmw)) {
	    read_null_fontinfo(fe);
	    return;
	}
	if ((bitmap.bitmap = malloc(bitmap.size)) == NULL)
	    Fatal("Unable to allocate memory for FreeType char\n");
	for (i = 0; i <= bii->maxc; i++) {
	    if (!(bii->mark[i]))
		continue;
	    bzero(bitmap.bitmap, bitmap.size);
	    ft_char(TT_Char_Index(cmap, remap[i]), i);
	    trim_bitmap(bitmap.width, bitmap.cols, bitmap.rows,
			(-chxmin)/64, (-chymin)/64, bitmap.bitmap,
			&nwidth, &nheight, &nxoff, &ndepth, &npixel);
	    dev_ft_initfontdict(fe, tfmfi, i,
				nwidth, nheight, nxoff, ndepth, npixel);
	}
	free(bitmap.bitmap);
    } else {
	if (!fto_size1000(filename)) {
	    read_null_fontinfo(fe);
	    return;
	}
	tfmw = dev_fto_begfontdict(fe, bii->mark, bii->maxc, remap);
	for (i = 0; i <= bii->maxc; i++) {
	    if (!(bii->mark[i]))
		continue;
	    fto_char(TT_Char_Index(cmap, remap[i]), i);
	    dev_fto_initfontdict(remap[i], &outline,
				 cwid((tfmfi->ch+i)->tfmw,tfmw));
	}
	dev_fto_endfontdict(fe);
	dev_set_repfe(bii->bf, NULL);
    }

    TT_Close_Face(face);
}


/* code encoding and conversion map
 */
int jis_to_uni(), gb_to_uni(), ks_to_uni(), jsp_to_uni(),
    bg51_to_uni(), bg52_to_uni();
static struct {
    unsigned short *map;
    char *name;
    int (*conv)();
} cc_maps[] = {
    {NULL, "jis", jis_to_uni},
    {NULL, "gb", gb_to_uni},
    {NULL, "ks", ks_to_uni},
    {NULL, "jsp", jsp_to_uni},
    {NULL, "bg51", bg51_to_uni},
    {NULL, "bg52", bg52_to_uni},
};
char *unitab = "2uni";

find_enc(enc)
char *enc;
{
    int i;

    for (i = CS_ENC_FIRST; i <= CS_ENC_LAST; i++)
	if (strcmp(enc, cc_maps[i].name) == 0)
	    return i;
    return CS_ENC_JIS;
}

init_cc_map(enc, conv)
int enc;
int (**conv)();
{
    char tabname[STRSIZE], file[PATHLEN];
    FILE *f;
    int i, c1, c2;

    if (cc_maps[enc].map != NULL) {
	*conv = cc_maps[enc].conv;
	return TRUE;
    }
#ifdef DEBUG
    if (Debuguser)
	fprintf(stderr, "trying to find cconv map for %s\n",
		cc_maps[enc].name);
#endif
    strcpy(tabname, cc_maps[enc].name);
    strcat(tabname, unitab);
    if (findcconvmap(tabname, file)) {
	if ((f = BINARYOPEN(file)) == NULL) {
#ifdef DEBUG
	    if (Debuguser)
		fprintf(stderr, "-- cconv map %s cannot be opened\n", file);
#endif
	    return FALSE;
	}
	if ((cc_maps[enc].map = (unsigned short *)malloc(94*94*2)) == NULL)
	    Fatal("Unable to allocate memory for cconv\n");
	for (i = 0; i < 94*94; i++) {
	    c1 = getc(f);
	    c2 = getc(f);
	    if (c1 == EOF || c2 == EOF)
		Fatal("cconv map %s incomplete\n", file);
	    (cc_maps[enc].map)[i] = (c1<<8)|c2;
	}
	*conv = cc_maps[enc].conv;
#ifdef DEBUG
    if (Debuguser)
	fprintf(stderr, "-- cconv map %s found\n", file);
#endif
	return TRUE;
    } else {
#ifdef DEBUG
	if (Debuguser)
	    fprintf(stderr, "-- cconv map not found\n");
#endif
	return FALSE;
    }
}

jis_to_uni(idx)
int idx;
{
    return (int)(cc_maps[CS_ENC_JIS].map)[idx];
}

gb_to_uni(idx)
int idx;
{
    return (int)(cc_maps[CS_ENC_GB].map)[idx];
}

ks_to_uni(idx)
int idx;
{
    return (int)(cc_maps[CS_ENC_KS].map)[idx];
}

jsp_to_uni(idx)
int idx;
{
    return (int)(cc_maps[CS_ENC_JSP].map)[idx];
}

bg51_to_uni(idx)
int idx;
{
    return (int)(cc_maps[CS_ENC_BG51].map)[idx];
}

bg52_to_uni(idx)
int idx;
{
    return (int)(cc_maps[CS_ENC_BG52].map)[idx];
}


/* jfm
 */
void
init_jft_fontinfo(fe)
struct font_entry *fe;
{
    int	jmm_markchar();
    void read_jft_fontinfo();

    biinifinfo(fe) = alloc_jbiinif(biaccinfo(fe)->bf);
    fe->fnt_markchar = jmm_markchar;
    fe->fnt_readfontinfo = read_jft_fontinfo;
}

struct jftfntinfo *
read_jft_finfo(fe, bii, tfmw, settfmw)
struct font_entry *fe;
struct biinitfontinfo *bii;
int tfmw;
BOOLEAN settfmw;
{
    struct jftfntinfo *jftfi;
    char *fn, *filename;
    BOOLEAN raster;
    int cmcount, cmid;
    short pid, eid;
    char enc;
    int (*codeconv)();
    int nwidth, nheight, nxoff, ndepth;
    char *npixel;
    int nchars, i, jis;
    extern int idx94_to_sjis();
    extern int idx94_to_std();
    extern int hconv, vconv;

    jftfi = (struct jftfntinfo *)
	alloc_check(malloc((unsigned)sizeof(struct jftfntinfo)+
			   bii->maxc*sizeof(struct jftchar_entry)),
		    "jftfont info");
    jftfi->jft_bf = bii->bf;

    for (i = 0; i <= bii->maxc; i++)
	jftfi->ch[i].dev_font = DEV_NULLFONT;

    raster = dev_mf_kind(jftfi->jft_bf) == MF_KIND_FT;
    fn = dev_fontpath(jftfi->jft_bf);
#ifdef KPATHSEA
    if ((filename = kpsearch_file(fn, fe->n, kpse_truetype_format)) == NULL) {
	Warning("FreeType font file %s not found", fn);
	read_null_fontinfo(fe);
	return;
    }
#else
    filename = fn;
#endif
#ifdef DEBUG
    if (Debuguser)
	fprintf(stderr, "FreeType openfont %s (for %s, %s)\n", filename, fe->n,
		raster ? "raster" : "outline");
#endif
    if (!ft_glyph(filename))
	return jftfi;
    cmcount = TT_Get_CharMap_Count(face);
    for (cmid = 0; cmid < cmcount; cmid++) {
	if (TT_Get_CharMap_ID(face, cmid, &pid, &eid)) {
	    Warning("FreeType font %s does not have charmap id", filename);
	    return jftfi;
	}
	/* TODO */
	if (pid == TT_PLATFORM_MICROSOFT) {
	    if (eid == TT_MS_ID_UNICODE_CS) {
		if (init_cc_map(dev_cs_enc(jftfi->jft_bf), &codeconv))
		    break;
	    } else if (eid == TT_MS_ID_SJIS) {
		codeconv = idx94_to_sjis;
		break;
	    } else if (eid == TT_MS_ID_GB2312) {
		codeconv = idx94_to_std;
		break;
	    }
	}
    }
    if (cmid == cmcount || TT_Get_CharMap(face, cmid, &cmap)) {
	Warning("FreeType font %s does not have charmap", filename);
	return jftfi;
    }

    if (raster) {
	if (!ft_size(filename, tfmw))
	    return jftfi;
	if ((bitmap.bitmap = malloc(bitmap.size)) == NULL)
	    Fatal("Unable to allocate memory for FreeType char\n");
	for (i = 0; i <= bii->maxc; i++) {
	    if (!(bii->mark[i]))
		continue;
	    if (settfmw)
		(jftfi->ch+i)->tfmw = tfmw;
	    bzero(bitmap.bitmap, bitmap.size);
	    ft_char(TT_Char_Index(cmap, codeconv(i)), jis = idx94_to_std(i));
	    trim_bitmap(bitmap.width, bitmap.cols, bitmap.rows,
			(-chxmin)/64, (-chymin)/64, bitmap.bitmap,
			&nwidth, &nheight, &nxoff, &ndepth, &npixel);
	    dev_jft_initfontdict(fe, jftfi, i, jis, tfmw, 
				 nwidth, nheight, nxoff, ndepth, npixel);
	}
	free(bitmap.bitmap);
    } else {
	if (!fto_size1000(filename))
	    return jftfi;
	for (nchars = 0, i = 0; i <= bii->maxc; i++)
	    if (bii->mark[i])
		nchars++;
	dev_jfto_begfontdict();
	for (i = 0; i <= bii->maxc; i++) {
	    if (!(bii->mark[i]))
		continue;
	    if (settfmw)
		(jftfi->ch+i)->tfmw = tfmw;
	    fto_char(TT_Char_Index(cmap, codeconv(i)), idx94_to_std(i));
	    dev_jfto_initfontdict(fe, jftfi, i, nchars--, &outline, 1000);
	}
	dev_jfto_endfontdict();
    }

    TT_Close_Face(face);
    return jftfi;
}

void
read_jft_fontinfo(fe)
struct font_entry *fe;
{
    struct biinitfontinfo *bii;
    struct jfmfntinfo *jfmfi;
    void read_jfm_finfo();
    int tfmw;

    ft_initialize();

    bii = biinifinfo(fe);
    jfmfi = NEW(struct jfmfntinfo, "jfmfont info");
    jfmfi->jfm_bf = bii->bf;
    jfmfinfo(fe) = jfmfi;
    read_jfm_finfo(fe);

    dev_jft_initfe(fe);
    tfmw = dev_jft_begfontdict(fe);
    jftfinfo(fe) = read_jft_finfo(fe, bii, tfmw, TRUE);

    free((char *)bii);
    free((char *)jfmfi->ctype);
    free((char *)jfmfi->ch);
    free((char *)jfmfi);
}

DEV_FONT
jft_fontdict(fe, c)
struct font_entry *fe;
int c;
{
    return jftfinfo(fe)->ch[jis_to_idx94(c)].dev_font;
}


/* jstfm
 */
void
init_jsft_fontinfo(fe)
struct font_entry *fe;
{
    struct jsubshare *jss;
    int	jsmm_markchar();
    void read_jsft_fontinfo();

    jss = jstfmfinfo(fe)->js_share;
    if (jss->jss_stat < JSS_INIT) {
	jss->jss_info = (struct jssinfo *)alloc_jbiinif(jss->jss_bf);
	jss->jss_stat = JSS_INIT;
    }
    jstfmfinfo(fe)->js_info = jss->jss_info;
    fe->fnt_markchar = jsmm_markchar;
    fe->fnt_readfontinfo = read_jsft_fontinfo;
}

void
read_jsft_fontinfo(fe)
struct font_entry *fe;
{
    struct biinitfontinfo *bii;
    struct jsubshare *jss;
    struct jstfmfntinfo *jstfmfi;
    void read_jstfm_finfo();
    int tfmw;

    ft_initialize();

    read_jstfm_finfo(fe);

    dev_jsft_initfe(fe);
    jss = jstfmfinfo(fe)->js_share;
    if (jss->jss_stat < JSS_READ) {
	bii = (struct biinitfontinfo *)jss->jss_info;
	jstfmfi = jstfmfinfo(fe);
	tfmw = dev_jsft_begfontdict(fe);
	jss->jss_info = (struct jssinfo *)
	    read_jft_finfo(fe, bii, tfmw, FALSE);
	jss->jss_stat = JSS_READ;
	free((char *)bii);
    }
    jstfmfinfo(fe)->js_info = jss->jss_info;
}

DEV_FONT
jsft_fontdict(fe, c)
struct font_entry *fe;
int c;
{
    struct jstfmfntinfo *jsfi = jstfmfinfo(fe);

    return jsftfinfo(jsfi)->ch[jsub_to_idx94(jsfi->jsubfont,c)].dev_font;
}


/*
 */
trim_bitmap(wid, bwid, hgt, xo, dep, pixel, nwid, nhgt, nxo, ndep, npixel)
int wid, bwid, hgt, xo, dep;
char *pixel;
int *nwid, *nhgt, *nxo, *ndep;
char **npixel;
{
    int ht, dt, lwt, rwt, i, j, nbwid;
    char *pixelend, *npix, *p0, *p1, *q;

    for (ht = 0; ht < hgt; ht++)
	for (i = ht*bwid; i < (ht+1)*bwid; i++)
	    if (*(pixel+i))
		goto htend;
 htend:
    hgt -= ht;
    npix = pixel+ht*bwid;

    for (dt = 0, pixelend = npix+hgt*bwid-1; dt < hgt ; dt++)
	for (i = dt*bwid; i < (dt+1)*bwid; i++)
	    if (*(pixelend-i))
		goto dtend;
 dtend:
    dep -= dt;
    hgt -= dt;

    for (lwt = 0; lwt < bwid; lwt++)
	for (i = lwt; i < hgt*bwid; i += bwid)
	    if (*(npix+i))
		goto lwtend;
 lwtend:
    for (rwt = 0; rwt < bwid; rwt++)
	for (i = bwid-1-rwt; i < hgt*bwid; i += bwid)
	    if (*(npix+i))
		goto rwtend;
 rwtend:
    npix += lwt;
    nbwid = bwid-(lwt+rwt);
    if (lwt+rwt > 0)
	for (p0 = npix+nbwid, p1 = npix+bwid, i = 1; i < hgt; p1 += bwid, i++)
	    for (q = p1, j = 0; j < nbwid; j++)
		*p0++ = *q++;

    *nwid = wid-((lwt+rwt)<<3);
    *nhgt = hgt;
    *nxo = xo-(lwt<<3);
    *ndep = dep;
    *npixel = npix;
}

#else	/* not FreeType */

find_enc(enc)
char *enc;
{
    return CS_ENC_JIS;
}

void
init_ft_fontinfo(fe)
struct font_entry *fe;
{
    int null_markchar();
    void read_null_fontinfo();

    Warning("This %s is not compiled with FreeType option.\nFont %s is ignored",
	    G_progname, fe->n);
    fe->fnt_markchar = null_markchar;
    fe->fnt_readfontinfo = read_null_fontinfo;
}

void
init_jft_fontinfo(fe)
struct font_entry *fe;
{
    init_ft_fontinfo(fe);
}

void
init_jsft_fontinfo(fe)
struct font_entry *fe;
{
    init_ft_fontinfo(fe);
}

#endif
