/*
 *
 *   (C) Copyright IBM Corp. 2001, 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: internalAPI.c
 *
 */

#define _GNU_SOURCE     /* Used to pick up definition of lseek64 to avoid */
			/* compiler warning messages. */
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <pthread.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/time.h>

#include "fullengine.h"
#include "internalAPI.h"
#include "engine.h"
#include "copy.h"
#include "crc.h"
#include "discover.h"
#include "handlemgr.h"
#include "commit.h"
#include "common.h"
#include "volume.h"
#include "expand.h"
#include "shrink.h"
#include "memman.h"
#include "message.h"
#include "dm.h"
#include "dm-ioctl.h"
#include "config.h"
#include "cluster.h"
#include "remote.h"
#include "lists.h"


/*-----------*\
| Global data |
\*-----------*/

static name_list_entry_t * name_registry = NULL;


static void engine_get_plugin_api_version(evms_version_t * version) {
	version->major      = ENGINE_PLUGIN_API_MAJOR_VERION;
	version->minor      = ENGINE_PLUGIN_API_MINOR_VERION;
	version->patchlevel = ENGINE_PLUGIN_API_PATCH_LEVEL;
}


/*-----------------------------*\
| Functions for getting plug-ins |
\*-----------------------------*/

/*
 * engine_get_plugin_list returns a list_anchor_t of plugin_record_t structures,
 * optionally filtering on the plug-in type.  If the type is 0, all
 * plugin_record_t structures will be returned.
 */
int engine_get_plugin_list(plugin_type_t         type,
			   plugin_search_flags_t flags,
			   list_anchor_t       * plugin_list) {
	int rc = 0;

	list_anchor_t new_plugin_list;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Filters:\n");
	LOG_DEBUG("  Plug-in type:     %#x\n", type);
	LOG_DEBUG("  Flags:            %#x\n", flags);
	LOG_DEBUG("Destination list:   %p\n", plugin_list);

	new_plugin_list = copy_list(&plugins_list);

	if (new_plugin_list != NULL) {

		/*
		 * If the caller specified a plug-in type or flags, remove from
		 * the list plug-ins that are not of the specified type.
		 */
		if ((type != 0) || (flags != 0)) {
			list_element_t iter1;
			list_element_t iter2;
			plugin_record_t * plugin;

			LIST_FOR_EACH_SAFE(new_plugin_list, iter1, iter2, plugin) {
				boolean remove = FALSE;

				if (type != 0) {
					if (GetPluginType(plugin->id) != type) {
						/* The type doesn't match.  Delete this object from the list. */
						remove = TRUE;
					}
				}

				/* Any filter flags set? */
				if (flags != 0) {

					/*
					 * If the caller only wants plug-ins that support containers, then
					 * filter out this plug-in if it doesn't have a container_functions
					 * table.
					 */
					if (flags & SUPPORTS_CONTAINERS) {
						if (plugin->container_functions == NULL) {
							remove = TRUE;
						}
					}
				}

				if (remove) {
					delete_element(iter1);
				}
			}
		}

		*plugin_list = new_plugin_list;

	} else {
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * engine_get_plugin_by_ID will return a pointer to the plugin_record_t that
 * has the specified plugin_ID_t.
 */
int engine_get_plugin_by_ID(plugin_id_t         pluginID,
			    plugin_record_t * * plugin) {
	int rc = 0;
	plugin_record_t * pPlugRec;
	list_element_t iter;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Search for plug-in with ID %d (%#x).\n", pluginID, pluginID);

	LIST_FOR_EACH(&plugins_list, iter, pPlugRec) {
		if (pPlugRec->id == pluginID) {
			break;
		}
	}

	if (pPlugRec == NULL) {
		rc = ENOENT;
	}

	*plugin = pPlugRec;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * engine_get_plugin_by_name will return a pointer to the plugin_record_t that
 * has the specified short name.
 */
int engine_get_plugin_by_name(char              * plugin_short_name,
			      plugin_record_t * * plugin) {
	int rc = 0;
	plugin_record_t * pPlugRec;
	list_element_t iter;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Search for plug-in with short name %s.\n", plugin_short_name);

	LIST_FOR_EACH(&plugins_list, iter, pPlugRec) {
		if ((strcmp(pPlugRec->short_name, plugin_short_name) == 0)) {
			break;
		}
	}

	if (pPlugRec == NULL) {
		rc = ENOENT;
	}

	*plugin = pPlugRec;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*-----------------------------*\
| Functions for getting volumes |
\*-----------------------------*/

/*
 * engine_get_volume_list returns a list_anchor_t of logical_volume_t structures,
 * optionally filtering on the FSIM that manages the volume.  If the FSIM
 * is NULL, all logical_volume_t structures will be returned.
 */
int engine_get_volume_list(plugin_record_t     * fsim,
			   storage_container_t * disk_group,
			   volume_search_flags_t flags,
			   list_anchor_t       * volume_list) {
	int rc = 0;

	list_anchor_t new_volume_list;
	list_element_t iter;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Filters:\n");
	LOG_DEBUG("  FSIM:        %s\n", (fsim == NULL) ? "(none)" : fsim->short_name);
	LOG_DEBUG("  Disk group:  %s\n", (disk_group == NULL) ? "(none)" : disk_group->name);
	LOG_DEBUG("  Flags:       %#x\n", flags);
	LOG_DEBUG("Destination list:   %p\n", volume_list);

	/* Update the mount points for the volumes. */
	LIST_FOR_EACH(&volumes_list, iter, vol) {
                is_volume_mounted(vol);
	}

	new_volume_list = copy_list(&volumes_list);

	if (new_volume_list != NULL) {
		if ((fsim != NULL) || (disk_group != NULL) || (flags != 0)) {
			list_element_t iter1;
			list_element_t iter2;
			logical_volume_t * vol;

			LIST_FOR_EACH_SAFE(new_volume_list, iter1, iter2, vol) {
				boolean remove = FALSE;

				if (fsim != NULL) {
					if (vol->file_system_manager != fsim) {
						/* The FSIM doesn't match.  Delete this volume from the list. */
						remove = TRUE;
					}
				}

				if ((disk_group != NULL) || (flags & VOL_NO_DISK_GROUP)) {
					if (vol->disk_group != disk_group) {
						/*
						 * The volume is not in the requested disk group.
						 * Delete it from the list.
						 */
						remove = TRUE;
					}
				}

				if (flags & VOL_CHANGES_PENDING) {
					/*
					 * If the volume does not have changes pending,
					 * delete it from the list.
					 */
					if (!is_volume_change_pending(vol, NULL)) {
						remove = TRUE;
					}
				}

				if (remove) {
					delete_element(iter1);
				}
			}
		}

		*volume_list = new_volume_list;

	} else {
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*-----------------------------*\
| Functions for getting objects |
\*-----------------------------*/

/*
 * engine_get_object_list returns a list_anchor_t of storage_object_t structures,
 * optionally filtering on the data_type, plug-in, and search flags.
 * feature is NULL, all storage_object_t structures will be returned.
 */
int engine_get_object_list(object_type_t         object_type,
			   data_type_t           data_type,
			   plugin_record_t     * plugin,
			   storage_container_t * disk_group,
			   object_search_flags_t flags,
			   list_anchor_t       * objects) {
	int rc = 0;

	list_anchor_t new_object_list = allocate_list();

	LOG_PROC_ENTRY();

	LOG_DEBUG("Filters:\n");
	LOG_DEBUG("  Object type:  %#x\n", object_type);
	LOG_DEBUG("  Data type:    %#x\n", data_type);
	LOG_DEBUG("  Plug-in:      %s\n", (plugin == NULL) ? "(none)" : plugin->short_name);
	LOG_DEBUG("  Disk grouP:   %s\n", (disk_group == NULL)   ? "(none)" : disk_group->name);
	LOG_DEBUG("  Flags:        %#x\n", flags);
	LOG_DEBUG("Destination list:   %p\n", objects);

	if (new_object_list != NULL) {

		if ((object_type == 0) || (object_type & DISK)) {
			rc = concatenate_lists(new_object_list, &disks_list);
		}

		if ((rc == 0) && ((object_type == 0) || (object_type & SEGMENT))) {
			rc = concatenate_lists(new_object_list, &segments_list);
		}

		if ((rc == 0) && ((object_type == 0) || (object_type & REGION))) {
			rc = concatenate_lists(new_object_list, &regions_list);
		}

		if ((rc == 0) && ((object_type == 0) || (object_type & EVMS_OBJECT))) {
			rc = concatenate_lists(new_object_list, &EVMS_objects_list);
		}

		if (rc == 0) {
			list_element_t iter1;
			list_element_t iter2;
			storage_object_t * obj;

			LIST_FOR_EACH_SAFE(new_object_list, iter1, iter2, obj) {
				boolean remove = FALSE;

				if (object_type != 0) {
					if (!(object_type & obj->object_type)) {
						/*
						 * The object type doesn't match.
						 * Delete this object from the list.
						 */
						remove = TRUE;
					}
				}

				if (data_type != 0) {
					if (!(data_type & obj->data_type)) {
						/*
						 * The data type doesn't match.
						 * Delete this object from the list.
						 */
						remove = TRUE;
					}
				}

				if (plugin != NULL) {
					if (plugin != obj->plugin) {
						/*
						 * The plug-in doesn't match.
						 * Delete this object from the list.
						 */
						remove = TRUE;
					}
				}

				if ((disk_group != NULL) || (flags & NO_DISK_GROUP)){
					if (obj->disk_group != disk_group) {
						/*
						 * The object is not in the requested disk group.
						 * Delete it from the list.
						 */
						remove = TRUE;
					}
				}

				if (flags & TOPMOST) {
					if (!is_top_object(obj)) {
						/*
						 * The object is not the topmost.
						 * Delete this object from the list.
						 */
						remove = TRUE;
					}
				}

				if (flags & NOT_MUST_BE_TOP) {
					if (obj->flags & SOFLAG_MUST_BE_TOP) {
						/*
						 * The object insists on being topmost, but the caller doesn't
						 * want objects like that.  Delete it from the list.
						 */
						remove = TRUE;
					}
				}

				if (flags & WRITEABLE) {
					if (obj->flags & (SOFLAG_READ_ONLY | SOFLAG_IO_ERROR | SOFLAG_CORRUPT)) {
						/*
						 * The object is either read-only, has had an I/O error, or is
						 * corrupt.  It is not writeable.  Delete it from the list.
						 */
						remove = TRUE;
					}
				}

				if (flags & OBJ_CHANGES_PENDING) {
					/*
					 * If the object does not have changes pending,
					 * Delete it from the list.
					 */
					if (!is_object_change_pending(obj, NULL)) {
						remove = TRUE;
					}
				}

				if (remove) {
					delete_element(iter1);
				}
			}
		}

		*objects = new_object_list;

	} else {
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*--------------------------------*\
| Functions for getting containers |
\*--------------------------------*/

/*
 * engine_get_container_list returns a list_anchor_t of storage_container_t
 * structures, optionally filtering on the plug-in that manages the container.
 * If plugin is NULL, all storage_container_t structures will be returned.
 */
int engine_get_container_list(plugin_record_t        * plugin,
			      storage_container_t    * disk_group,
			      container_search_flags_t flags,
			      list_anchor_t          * container_list) {
	int rc = 0;

	list_anchor_t new_container_list;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Filters:\n");
	LOG_DEBUG("  Plug-in:    %s\n", (plugin == NULL) ? "(none)" : plugin->short_name);
	LOG_DEBUG("  Disk group: %s\n", (disk_group == NULL)   ? "(none)" : disk_group->name);
	LOG_DEBUG("  Flags:      %#x\n", flags);
	LOG_DEBUG("Destination list:   %p\n", container_list);

	new_container_list = copy_list(&containers_list);
	if (new_container_list != NULL) {

		if ((plugin != NULL) || (disk_group != NULL) || (flags != 0)) {
                        list_element_t iter1;
			list_element_t iter2;
			storage_container_t * con;

			LIST_FOR_EACH_SAFE(new_container_list, iter1, iter2, con) {
				boolean remove = FALSE;

				if (plugin != NULL) {
					if (con->plugin != plugin) {
						/*
						 * The plug-in doesn't match.  Delete this container from the list.
						 */
						remove = TRUE;
					}
				}

				if ((disk_group != NULL) || (flags & CON_NO_DISK_GROUP)) {
					if (con->disk_group != disk_group) {
						/*
						 * The container is not in the requested disk group.
						 * Delete it from the list.
						 */
						remove = TRUE;
					}
				}

				if (flags & CON_CHANGES_PENDING) {
					changes_pending_parms_t changes_pending;

					is_container_change_pending(con, &changes_pending);

					/*
					 * If the container does not have changes pending,
					 * delete it from the list.
					 */
					if (!changes_pending.result) {
						remove = TRUE;
					}
				}

				if (remove) {
					delete_element(iter1);
				}
			}
		}

	} else {
		rc = ENOMEM;
	}

	*container_list = new_container_list;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*-----------------------------------------------------------*\
| Functions for allocating and freeing Engine data structures |
\*-----------------------------------------------------------*/

/*
 * allocate_new_storage_object takes care of the common functions needed to
 * allocate any kind of storage object.  It allocates zero filled memory,
 * creates the lists, and initializes the flags.
 */
int allocate_new_storage_object(storage_object_t * * new_object) {
	int rc = 0;

	storage_object_t * object = NULL;

	LOG_PROC_ENTRY();

	object = (storage_object_t *) engine_alloc(sizeof(storage_object_t));
	if (object != NULL) {

                object->parent_objects = allocate_list();
		if (object->parent_objects != NULL) {

			object->child_objects = allocate_list();
			if (object->child_objects != NULL) {

				object->associated_parents = allocate_list();
				if (object->associated_parents != NULL) {

					object->associated_children = allocate_list();
					if (object->associated_children != NULL) {

						/*
						 * The default data type is
						 * DATA_TYPE.  The plug-in can
						 * change it if necessary.
						 */
						object->data_type = DATA_TYPE;

						/*
						 * If we are not in discovery,
						 * this must be a new object, so
						 * mark it new and dirty.
						 */
						if (!discover_in_progress) {
							object->flags |= SOFLAG_NEW | SOFLAG_DIRTY;
						}

					} else {
						LOG_CRITICAL("Unable to get memory for the storage object's associated children list.\n");
						rc = ENOMEM;
					}

				} else {
					LOG_CRITICAL("Unable to get memory for the storage object's associated parents list.\n");
					rc = ENOMEM;
				}

			} else {
				LOG_CRITICAL("Unable to get memory for the storage object's child list.\n");
				rc = ENOMEM;
			}

		} else {
			LOG_CRITICAL("Unable to get memory for the storage object's parent list.\n");
			rc = ENOMEM;
		}

		if (rc != 0) {
			/* destroy_list() handles NULL list pointers. */
			destroy_list(object->associated_parents);
			destroy_list(object->child_objects);
			destroy_list(object->parent_objects);
			engine_free(object);
			object = NULL;
		}

	} else {
		LOG_CRITICAL("Unable to get memory for a storage object.\n");
		rc = ENOMEM;
	}

	*new_object = object;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * free_old_storage_object takes care of the functions that are common for
 * cleaning up any type of storage object.  It destroys the app-handle, if one
 * was created, destroys the lists in the object, and frees the object.
 */
void free_old_storage_object(storage_object_t * obj) {

	LOG_PROC_ENTRY();

	if (obj->app_handle != 0) {
		destroy_handle(obj->app_handle);
	}

	destroy_list(obj->parent_objects);
	obj->parent_objects = NULL;
	destroy_list(obj->child_objects);
	obj->child_objects = NULL;
	destroy_list(obj->associated_parents);
	obj->associated_parents = NULL;
	destroy_list(obj->associated_children);
	obj->associated_children = NULL;

	if ((!(engine_mode & ENGINE_DAEMON)) &&
	    (obj->flags & SOFLAG_ACTIVE) &&
	    (obj->data_type == DATA_TYPE)) {

		/*
		 * Order is important in the deactivate_list.  Objects must
		 * be appended to the list so that they are deactivated in
		 * the order they were deleted.
		 */
		insert_thing(&deactivate_list,
			     obj,
			     INSERT_AFTER,
			     NULL);

		obj->flags |= SOFLAG_NEEDS_DEACTIVATE;
	} else {
		engine_free(obj);
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Free a storage_object_t for a logical disk.
 */
int engine_free_logical_disk(storage_object_t * disk) {
	int rc = 0;

	LOG_PROC_ENTRY();

	engine_unregister_name(disk->name);

	remove_thing(&disks_list,
		     disk);

	free_old_storage_object(disk);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Allocate a storage_object_t for a logical disk.
 * If a name is specified, register the name.
 */
int engine_allocate_logical_disk(char * name, storage_object_t * * new_disk) {
	int rc = 0;

	storage_object_t * disk;

	LOG_PROC_ENTRY();

	*new_disk = NULL;

	if (name != NULL) {
		rc = engine_validate_name(name);
	}

	if (rc == 0) {
		rc = allocate_new_storage_object(&disk);

		if (rc == 0) {
			list_element_t el;

			disk->object_type = DISK;

			el = insert_thing(&disks_list,
					  disk,
					  INSERT_AFTER,
					  NULL);

			if (el != NULL) {
				if (name != NULL) {
					rc = engine_register_name(name);

					if (rc == 0) {
						strcpy(disk->name, name);
					} else {
						remove_thing(&disks_list, disk);
					}
				}

			} else {
				LOG_CRITICAL("Error inserting new disk into the disk_list.\n");
				rc = ENOMEM;
			}

			if (rc != 0){
				engine_free_logical_disk(disk);
				disk = NULL;
			}
		}

		*new_disk = disk;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Free a storage_object_t for a disk segment.
 */
int engine_free_segment(storage_object_t * segment) {
	int rc = 0;

	LOG_PROC_ENTRY();

	engine_unregister_name(segment->name);

	remove_thing(&segments_list,
		     segment);

	free_old_storage_object(segment);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Allocate a storage_object_t for a disk segment.
 * If a name is specified, register the name.
 */
int engine_allocate_segment(char * name, storage_object_t * * new_segment) {
	int rc = 0;

	storage_object_t * segment;

	LOG_PROC_ENTRY();

	*new_segment = NULL;

	if (name != NULL) {
		rc = engine_validate_name(name);
	}

	if (rc == 0) {
		rc = allocate_new_storage_object(&segment);

		if (rc == 0) {
			list_element_t el;

			segment->object_type = SEGMENT;

			el = insert_thing(&segments_list,
					  segment,
					  INSERT_AFTER,
					  NULL);

			if (el != NULL) {
				if (name != NULL) {
					rc = engine_register_name(name);

					if (rc == 0) {
						strcpy(segment->name, name);
					} else {
						remove_thing(&segments_list, segment);
					}
				}

			} else {
				LOG_CRITICAL("Error inserting new disk into the disk_list.\n");
				rc = ENOMEM;
			}

			if (rc != 0) {
				engine_free_segment(segment);
				segment = NULL;
			}
		}

		*new_segment = segment;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Free a storage_object_t for a region.
 */
int engine_free_region(storage_object_t * region) {
	int rc = 0;

	LOG_PROC_ENTRY();

	engine_unregister_name(region->name);

	remove_thing(&regions_list,
		     region);

	free_old_storage_object(region);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Allocate a storage_object_t for a region.
 * If a name is specified, register the name.
 */
int engine_allocate_region(char * name, storage_object_t * * new_region) {
	int rc = 0;

	storage_object_t * region;

	LOG_PROC_ENTRY();

	*new_region = NULL;

	if (name != NULL) {
		rc = engine_validate_name(name);
	}

	if (rc == 0) {
		rc = allocate_new_storage_object(&region);

		if (rc == 0) {
			list_element_t el;

			region->object_type = REGION;

			el = insert_thing(&regions_list,
					  region,
					  INSERT_AFTER,
					  NULL);

			if (el != NULL) {
				if (name != NULL) {
					rc = engine_register_name(name);

					if (rc == 0) {
						strcpy(region->name, name);
					} else {
						remove_thing(&regions_list, region);
					}
				}

			} else {
				LOG_CRITICAL("Error inserting new disk into the disk_list.\n");
				rc = ENOMEM;
			}

			if (rc != 0) {
				engine_free_region(region);
				region = NULL;
			}
		}

		*new_region = region;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Free a storage_object_t for an EVMS object.
 */
int engine_free_evms_object(storage_object_t * object) {
	int rc = 0;

	LOG_PROC_ENTRY();

	engine_unregister_name(object->name);

	remove_thing(&EVMS_objects_list,
		     object);

	free_old_storage_object(object);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Allocate a storage_object_t for an EVMS object.
 * If a name is specified, register the name.
 */
int engine_allocate_evms_object(char * name, storage_object_t * * new_object) {
	int rc = 0;

	storage_object_t * object;

	LOG_PROC_ENTRY();

	*new_object = NULL;

	if (name != NULL) {
		rc = engine_validate_name(name);
	}

	if (rc == 0) {
		rc = allocate_new_storage_object(&object);

		if (rc == 0) {
			list_element_t el;

			object->object_type = EVMS_OBJECT;

			el = insert_thing(&EVMS_objects_list,
					  object,
					  INSERT_AFTER,
					  NULL);

			if (el != NULL) {
				if (name != NULL) {
					rc = engine_register_name(name);

					if (rc == 0) {
						strcpy(object->name, name);
					} else {
						remove_thing(&EVMS_objects_list, object);
					}
				}

			} else {
				LOG_CRITICAL("Error inserting new disk into the disk_list.\n");
				rc = ENOMEM;
			}

			if (rc != 0) {
				engine_free_evms_object(object);
				object = NULL;
			}
		}

		*new_object = object;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Free any kind of storage_object_t.
 */
int engine_free_storage_object(storage_object_t * obj) {
	int rc = 0;

	LOG_PROC_ENTRY();

	switch (obj->object_type) {
	case DISK:
		rc = engine_free_logical_disk(obj);
		break;
	case SEGMENT:
		rc = engine_free_segment(obj);
		break;
	case REGION:
		rc = engine_free_region(obj);
		break;
	case EVMS_OBJECT:
		rc = engine_free_evms_object(obj);
		break;
	default:
		LOG_ERROR("Object %s has an object_type of %#x which is not valid.\n", obj->name, obj->object_type);
		rc = EINVAL;
		break;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Allocate a storage_container structure filled with zeros, create the list_anchor_t
 * anchors for the segments and regions_exported lists, mark the structure
 * dirty, and add it to ContainerList, the global list of storage containers.
 * If a name is specified, register the name.
 */
int engine_allocate_container(char * name, storage_container_t * * new_container) {
	int rc = 0;

	storage_container_t * container;

	LOG_PROC_ENTRY();

	*new_container = NULL;

	if (name != NULL) {
		rc = engine_validate_name(name);
	}

	if (rc == 0) {
		container = (storage_container_t *) engine_alloc(sizeof(storage_container_t));
		if (container != NULL) {

			container->type = CONTAINER;

			container->objects_consumed = allocate_list();
			if (container->objects_consumed != NULL) {

				container->objects_produced = allocate_list();
				if (container->objects_produced != NULL) {
					list_element_t el;

					/*
					 * If we are not in discovery, this must be a new container
					 * so mark it dirty.
					 */
					if (!discover_in_progress) {
						container->flags |= SCFLAG_DIRTY;
					}

					el = insert_thing(&containers_list,
							  container,
							  INSERT_AFTER,
							  NULL);

					if (el != NULL) {
						if (name != NULL) {
							rc = engine_register_name(name);

							if (rc == 0) {
								strcpy(container->name, name);
							} else {
								remove_thing(&containers_list, container);
							}
						}

					} else {
						LOG_CRITICAL("Error inserting new disk into the disk_list.\n");
						rc = ENOMEM;
					}

					if (rc != 0) {
						destroy_list(container->objects_produced);
					}

				} else {
					rc = ENOMEM;
				}

				if (rc != 0) {
					destroy_list(container->objects_consumed);
				}

			} else {
				rc = ENOMEM;
			}

			if (rc != 0) {
				engine_free(container);
				container = NULL;
			}

		} else {
			rc = ENOMEM;
		}

		*new_container = container;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Free a storage_container structure: Destroy the app_handle if we created one,
 * destroy the segments and regions_exported list_anchor_ts, and remove the structure
 * from ContainerList, the global list of storage containers.
 */
int engine_free_container(storage_container_t * container) {
	int rc = 0;

	LOG_PROC_ENTRY();

	if (container->app_handle != 0) {
		destroy_handle(container->app_handle);
	}

	destroy_list(container->objects_consumed);
	destroy_list(container->objects_produced);

	engine_unregister_name(container->name);

	remove_thing(&containers_list,
		     container);

	engine_free(container);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/******************************************\
* Functions for I/O to object device nodes *
\******************************************/

/*
 * Open the device node for the storage object.
 */
static int engine_open_object(storage_object_t * obj, int oflags, ...) {

	int rc = 0;
	int fd;
	va_list args;
	mode_t mode;
	char dev_name[EVMS_OBJECT_NODE_PATH_LEN + EVMS_NAME_SIZE + 1];

	LOG_PROC_ENTRY();

	LOG_DEBUG("Request to open object %s.  flags: %#x\n", obj->name, oflags);

	if (obj->dev_major == 0) {
		LOG_DEBUG("Object %s does not have a non-zero major.\n", obj->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return -EINVAL;
	}

	va_start(args, oflags);
	mode = va_arg(args, mode_t);
	va_end(args);

	strcpy(dev_name, EVMS_OBJECT_NODE_PATH);
	strcat(dev_name, obj->name);
	rc = ensure_dev_node(dev_name, obj->dev_major, obj->dev_minor);
	if (rc == 0) {
		fd = open(dev_name, oflags, mode);

		if (fd < 0) {
			LOG_WARNING("Open of %s failed with error code %d: %s\n", dev_name, errno, strerror(errno));
			fd = -errno;
		} else {
			/* Close this file for all exec'd processes. */
			fcntl(fd, F_SETFD, FD_CLOEXEC);
		}

	} else {
		fd = -rc;
	}

	LOG_PROC_EXIT_INT(fd);
	return fd;
}


/*
 * Read from the device node for the storage object.
 */
static int32_t engine_read_object(storage_object_t * obj,
				  int                fd,
				  void             * buffer,
				  int32_t            bytes,
				  u_int64_t          offset) {

	int32_t rc = 0;
	off64_t curr_offset;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Read %d bytes at offset %"PRIu64" from object %s using file descriptor %d into buffer %p.\n", bytes, offset, obj->name, fd, buffer);

	if (obj->dev_major == 0) {
		LOG_ERROR("Object %s does not have a non-zero major.\n", obj->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return -EINVAL;
	}

	curr_offset = lseek64(fd, offset, SEEK_SET);
	if (curr_offset == offset) {
		rc = read(fd, buffer, bytes);
		if (rc < 0) {
			LOG_WARNING("Read from object %s failed with error code %d: %s\n", obj->name, errno, strerror(errno));
			rc = -errno;
		}

	} else {
		LOG_WARNING("lseek64 to offset %"PRIu64" on object %s failed with error code %d: %s\n", offset, obj->name, errno, strerror(errno));
		rc = -errno;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Write to the device node for the storage object.
 */
static int32_t engine_write_object(storage_object_t * obj,
				   int                fd,
				   void             * buffer,
				   int32_t            bytes,
				   u_int64_t          offset) {

	int32_t rc = 0;
	off64_t curr_offset;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Write %d bytes at offset %"PRIu64" to object %s using file descriptor %d from buffer %p.\n", bytes, offset, obj->name, fd, buffer);

	if (obj->dev_major == 0) {
		LOG_ERROR("Object %s does not have a non-zero major.\n", obj->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return -EINVAL;
	}


	curr_offset = lseek64(fd, offset, SEEK_SET);
	if (curr_offset == offset) {
		rc = write(fd, buffer, bytes);
		if (rc < 0) {
			LOG_WARNING("Write to object %s failed with error code %d: %s\n", obj->name, errno, strerror(errno));
			rc = -errno;
		}

	} else {
		LOG_WARNING("lseek64 to offset %"PRIu64" on object %s failed with error code %d: %s\n", offset, obj->name, errno, strerror(errno));
		rc = -errno;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Issue an ioctl to the device node for the storage object.
 */
static int engine_ioctl_object(storage_object_t * obj,
			       int                fd,
			       unsigned long int  command,
			       ...) {

	int32_t rc = 0;
	va_list args;
	void * arg;

	LOG_PROC_ENTRY();

	LOG_DEBUG("ioctl to object %s using file descriptor %d command %#lx.\n", obj->name, fd, command);

	if (obj->dev_major == 0) {
		LOG_ERROR("Object %s does not have a non-zero major.\n", obj->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	va_start(args, command);
	arg = va_arg(args, void *);
	va_end(args);

	rc = ioctl(fd, command, arg);

	if (rc < 0) {
		LOG_WARNING("ioctl to object %s failed with error code %d: %s\n", obj->name, errno, strerror(errno));
		rc = errno;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Close the device node for the storage object.
 */
static int engine_close_object(storage_object_t * obj, int fd) {

	int rc = 0;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Close object %s using file descriptor %d.\n", obj->name, fd);

	rc = close(fd);

	if (rc != 0) {
		rc = errno;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/******************************************\
* Functions for I/O to volume device nodes *
\******************************************/

/*
 * Open the device node for the logical volume.
 */
static int engine_open_volume(logical_volume_t * vol, int oflags, ...) {

	int fd;
	va_list args;
	mode_t mode;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Request to open volume %s.  flags: %#x\n", vol->name, oflags);

	if (vol->dev_node[0] == '\0') {
		LOG_ERROR("Volume %s does not have a valid dev node.\n", vol->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return -EINVAL;
	}

	va_start(args, oflags);
	mode = va_arg(args, mode_t);
	va_end(args);

	fd = open(vol->dev_node, oflags, mode);

	if (fd < 0) {
		LOG_WARNING("Open of %s failed with error code %d: %s\n", vol->dev_node, errno, strerror(errno));
		fd = -errno;
	} else {
		/* Close this file for all exec'd processes. */
		fcntl(fd, F_SETFD, FD_CLOEXEC);
	}

	LOG_PROC_EXIT_INT(fd);
	return fd;
}


/*
 * Read from the device node for the logical volume.
 */
static int32_t engine_read_volume(logical_volume_t * vol,
				  int                fd,
				  void             * buffer,
				  int32_t            bytes,
				  u_int64_t          offset) {

	int32_t rc = 0;
	off64_t curr_offset;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Read from volume %s using file descriptor %d %d bytes at offset %"PRIu64".\n", vol->name, fd, bytes, offset);

	curr_offset = lseek64(fd, offset, SEEK_SET);
	if (curr_offset == offset) {
		rc = read(fd, buffer, bytes);
		if (rc < 0) {
			LOG_WARNING("Read from volume %s failed with error code %d: %s\n", vol->dev_node, errno, strerror(errno));
			rc = -errno;
		}

	} else {
		LOG_WARNING("lseek64 to offset %"PRIu64" on volume %s failed with error code %d: %s\n", offset, vol->dev_node, errno, strerror(errno));
		rc = errno;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Write to the device node for the logical volume.
 */
static int32_t engine_write_volume(logical_volume_t * vol,
				   int                fd,
				   void             * buffer,
				   int32_t            bytes,
				   u_int64_t          offset) {

	int32_t rc = 0;
	off64_t curr_offset;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Write to volume %s using file descriptor %d %d bytes at offset %"PRIu64".\n", vol->name, fd, bytes, offset);

	curr_offset = lseek64(fd, offset, SEEK_SET);
	if (curr_offset == offset) {
		rc = write(fd, buffer, bytes);
		if (rc < 0) {
			LOG_WARNING("Write to volume %s failed with error code %d: %s\n", vol->dev_node, errno, strerror(errno));
			rc = -errno;
		}

	} else {
		LOG_WARNING("lseek64 to offset %"PRIu64" on volume %s failed with error code %d: %s\n", offset, vol->dev_node, errno, strerror(errno));
		rc = errno;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Issue an ioctl to the device node for the logical volume.
 */
static int engine_ioctl_volume(logical_volume_t * vol,
			       int                fd,
			       unsigned long int  command,
			       ...) {

	int32_t rc = 0;
	va_list args;
	void * arg;

	LOG_PROC_ENTRY();

	LOG_DEBUG("ioctl to volume %s using file descriptor %d command %#lx.\n", vol->name, fd, command);

	va_start(args, command);
	arg = va_arg(args, void *);
	va_end(args);

	rc = ioctl(fd, command, arg);

	if (rc < 0) {
		LOG_WARNING("ioctl to volume %s failed with error code %d: %s\n", vol->dev_node, errno, strerror(errno));
		rc = errno;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Close the device node for the logical volume.
 */
static int engine_close_volume(logical_volume_t * vol, int fd) {

	int rc = 0;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Close volume %s using file descriptor %d.\n", vol->name, fd);

	rc = close(fd);
	
	if (rc != 0) {
		rc = errno;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static boolean engine_commit_in_progress() {

	LOG_PROC_ENTRY();

	LOG_PROC_EXIT_BOOLEAN(commit_in_progress);
	return(commit_in_progress);
}


static int plugin_write_log_entry(debug_level_t     level,
				  plugin_record_t * plugin,
				  char            * fmt,
				  ...) {
	int rc = 0;
	int len;
	va_list args;

	/*
	 * Don't log entry into this function.  It gets in the way of what the
	 * plug-in is trying to write.
	 */

	if (dm_device_suspended) {
		return rc;
	}

	if (level <= debug_level) {
		if (log_file_fd > 0) {
			pthread_mutex_lock(&log_mutex);
			timestamp(log_buf, LOG_BUF_SIZE, level);

			if (plugin != NULL) {
				strcat(log_buf, plugin->short_name);
			} else {
				strcat(log_buf, "Bad plug-in pointer");
			}

			strcat(log_buf, ": ");
			len = strlen(log_buf);

			va_start(args, fmt);
			len += vsprintf(log_buf + strlen(log_buf), fmt, args);
			va_end(args);

			if (write(log_file_fd, log_buf, len) < 0) {
				rc = errno;
			}
			pthread_mutex_unlock(&log_mutex);

		} else {
			rc = ENOENT;
		}
	}

	/*
	 * Don't log exit from this function.  It gets in the way of what the
	 * plug-in is trying to write.
	 */
	return rc;
}


/*
 * Call the kernel to calculate a checksum.
 */
static int engine_calculate_checksum(unsigned char * buffer,
				     int             buffer_size,
				     unsigned int    insum,
				     unsigned int  * outsum) {
	return ENOSYS;
}


/*
 * Tell the Engine to put some sectors on its kill list.
 */
static int engine_add_sectors_to_kill_list(storage_object_t * disk, lba_t lba, sector_count_t count) {

	kill_sector_record_t * ksr = engine_alloc(sizeof(kill_sector_record_t));
	int                    rc = 0;

	LOG_PROC_ENTRY();

	if (ksr != NULL) {
		if (disk != NULL) {
			if (lba <= disk->size) {
				if (lba + count <= disk->size) {
					list_element_t el;

					ksr->logical_disk  = disk;
					ksr->sector_offset = lba;
					ksr->sector_count  = count;

					el = insert_thing(&kill_sectors_list,
							  ksr,
							  INSERT_BEFORE,
							  NULL);

					if (el != NULL) {
						LOG_DEBUG("Request queued to kill %"PRIu64" sector%s on disk %s at LBA %"PRIu64".\n", ksr->sector_count, (ksr->sector_count == 1) ? "" : "s", ksr->logical_disk->name, ksr->sector_offset);

					} else {
						LOG_CRITICAL("Error inserting a kill sector record into the kill sectos list.\n");
						rc = ENOMEM;
					}

				} else {
					LOG_ERROR("The count of sectors (%"PRIu64" at LBA %"PRIu64") goes past the end of the disk (disk size is %"PRIu64").\n", ksr->sector_count, ksr->sector_offset, disk->size);
					rc = EINVAL;
				}

			} else {
				LOG_ERROR("The starting LBA of the kill sectors (%"PRIu64") is past the end of the disk (disk size is %"PRIu64").\n", ksr->sector_offset, disk->size);
				rc = EINVAL;
			}

		} else {
			LOG_ERROR("The pointer for the disk is NULL.\n");
			rc = EINVAL;
		}

		if (rc != 0) {
			engine_free(ksr);
		}

	} else {
		LOG_CRITICAL("Error allocating memory for a kill sector record.\n");
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Validate that an object name is not too long and that there is no other
 * object of the same type with the same name.
 */
int engine_validate_name(char * name) {

	int rc = 0;
	name_list_entry_t * pNameEntry = name_registry;

	LOG_PROC_ENTRY();

	if (name != NULL) {

		LOG_DEBUG("Name to validate is %s.\n", name);

		if (strlen(name) <= EVMS_NAME_SIZE) {

			while ((pNameEntry != NULL) && (strcmp(pNameEntry->name, name) != 0)) {
				pNameEntry = pNameEntry->next;
			}

			if (pNameEntry != NULL) {
				/*
				 * We found a matching name.
				 */
				LOG_DEBUG("Name %s is already in the registry.\n", name);
				rc = EEXIST;
			}

		} else {
			/* The name is too long. */
			LOG_DEBUG("The name is too long.  It must be %d or fewer characters.\n", EVMS_NAME_SIZE);
			rc = EOVERFLOW;
		}

	} else {
		LOG_ERROR("Pointer to name is NULL.\n");
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Add a name to the name registry.
 */
int engine_register_name(char * name) {

	int rc = 0;
	name_list_entry_t * * pNameEntry = &name_registry;
	name_list_entry_t * new_entry;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Name to register is %s.\n", name);

	rc = engine_validate_name(name);

	if (rc == 0) {
		/*
		 * Create a new name registry entry and add it to the appropriate list.
		 */

		new_entry = (name_list_entry_t *) engine_alloc(sizeof(name_list_entry_t));

		if (new_entry != NULL) {
			new_entry->name = engine_strdup(name);

			if (new_entry->name != NULL) {
				new_entry->next = *pNameEntry;
				*pNameEntry = new_entry;

			} else {
				/* Could not get memory for the name. */
				LOG_CRITICAL("Error getting memory for the name in the new name registry entry.\n");
				engine_free(new_entry);
				rc = ENOMEM;
			}

		} else {
			LOG_CRITICAL("Error getting memory for the new name registry entry.\n");
			rc = ENOMEM;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Remove a name from the name registry.
 */
int engine_unregister_name(char * name) {
	int rc = 0;

	name_list_entry_t * * pNameEntry = &name_registry;

	LOG_PROC_ENTRY();

	if (name != NULL) {

		LOG_DEBUG("Name to unregister is %s.\n", name);

		while ((*pNameEntry != NULL) && (strcmp((*pNameEntry)->name, name) != 0)) {
			pNameEntry = &((*pNameEntry)->next);
		}

		if (*pNameEntry != NULL) {
			/*
			 * We found the name.  Remove it from the list.
			 */
			name_list_entry_t * entry = *pNameEntry;

			*pNameEntry = (*pNameEntry)->next;

			engine_free(entry->name);
			engine_free(entry);

		} else {
			/* We did not find the name. */
			LOG_DEBUG("Name %s is not in the registry.\n", name);
			rc = ENOENT;
		}

	} else {
		LOG_ERROR("Pointer to name is NULL.\n");
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Free up all the entries in the name registry.
 */
void clear_name_registry(void) {

	LOG_PROC_ENTRY();

	while (name_registry != NULL) {
		name_list_entry_t * entry = name_registry;

		LOG_DEBUG("Free name registry entry for \"%s\".\n", entry->name);

		name_registry = name_registry->next;

		engine_free(entry->name);
		engine_free(entry);
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Check if there are any restrictions that would prevent this object from
 * being renamed.
 */
int engine_can_rename(storage_object_t * obj) {

	boolean is_top_object = TRUE;
	boolean has_associative_feature = FALSE;

	LOG_PROC_ENTRY();

	/*
	 * If the object is not part of a volume, then it is free to change its
	 * name.
	 */
	if (obj->volume == NULL) {
		LOG_PROC_EXIT_INT(0);
		return 0;
	}

	/*
	 * Check if the object is a top level object, i.e., it has no parent;
	 * or its parent is an associative feature that snuk in on top of the
	 * object, in which case it is virtually a top level object.
	 */
	if (list_empty(obj->parent_objects)) {
		is_top_object = TRUE;

	} else {
		/*
		 * The object is not a top level object.  Check if its parent is an
		 * associative feature that snuck in on top of the object, and the
		 * associative feature has no parents.
		 */
		storage_object_t * parent;

		parent = first_thing(obj->parent_objects, NULL);
		
		if (parent != NULL) {
			if ((GetPluginType(parent->plugin->id) == EVMS_ASSOCIATIVE_FEATURE) &&
			    (list_empty(parent->parent_objects))) {
				is_top_object = TRUE;
				has_associative_feature = TRUE;

			} else {
				is_top_object = FALSE;
			}

		} else {
			/*
			 * We couldn't get the parent object.  Assume the most restrictive
			 * case, i.e., it is a top object.
			 */
			is_top_object = TRUE;
		}
	}

	if (is_top_object) {

		/*
		 * If the object is an EVMS_OBJECT, then it must be part of an
		 * EVMS volume and its name is allowed to change since it doesn't
		 * affect the volume name.
		 */
		if (obj->object_type == EVMS_OBJECT) {
			LOG_PROC_EXIT_INT(0);
			return 0;
		}

		/* If the object has  a feature header on it, that feature header is
		 * for making an EVMS volume.  An object that is part of an EVMS
		 * volume can always change its name since that doesn't affect the
		 * volume name.
		 */
		if (obj->feature_header != NULL) {
			LOG_PROC_EXIT_INT(0);
			return 0;

		} else {
			/*
			 * The object is the top level object in a compatibility volume.
			 * Check to see if the volume is opened.  If it is, don't allow
			 * the object name to be changed since that would change the
			 * volume name.  If the volume is not opened, it's OK to change
			 * the object name.
			 */
			if (is_volume_opened(obj->volume)) {
				LOG_PROC_EXIT_INT(EPERM);
				return EPERM;

			} else {

				/*
				 * If this object has an associative feature above it, then
				 * the name cannot be changed.  It is the top object of a
				 * compatibility volume which means the volume name changes
				 * with the object name.  Associative features are tied to
				 * the volume name.  We cannot let the volume name change
				 * and therefore cannot let the object name change.
				 */
				if (has_associative_feature) {
					LOG_PROC_EXIT_INT(EPERM);
					return EPERM;

				} else {
					/*
					 * The object is the top level object of a compatibility
					 * volume that is not mounted.  The name can be changed.
					 * the evms_set_info() code will take care of changing the
					 * volume name if the object name changes.
					 */
					LOG_PROC_EXIT_INT(0);
					return 0;
				}
			}
		}

	} else {

		/* The object is not a top level object.  Its name can be changed. */
		LOG_PROC_EXIT_INT(0);
		return 0;
	}

	/*
	 * We shouldn't get here.  All of the if branches above should return.
	 * But just in case, fail the query.
	 */
	LOG_PROC_EXIT_INT(EPERM);
	return EPERM;
}


static int is_valid_adopt_child(debug_level_t      level,
				storage_object_t * parent,
				storage_object_t * child) {

	LOG_PROC_ENTRY();

	if (!list_empty(child->parent_objects)) {
		int rc = 0;
		list_element_t iter;
		storage_object_t * obj;

		LIST_FOR_EACH(child->parent_objects, iter, obj) {
			if (obj == parent) {
				rc = EEXIST;
			}
		}

		switch (rc) {
			case EEXIST:
				break;

			case 0:
				LOG(level, "%s has parent(s) none of which is %s.\n", child->name, parent->name);
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;

			default:
				LOG(level, "When searching for %s in the parent list of %s, received error code %d: %s.\n",
				    parent->name, child->name, rc, evms_strerror(rc));
				LOG_PROC_EXIT_INT(rc);
				return rc;

		}
	}

	if (child->flags & SOFLAG_MUST_BE_TOP) {
		LOG(level, "%s insists on being a top object.\n", child->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LOG_PROC_EXIT_INT(0);
	return 0;
}


static int engine_can_adopt_child(storage_object_t * parent,
				  storage_object_t * child) {

	int rc;
	LOG_PROC_ENTRY();

	rc = is_valid_adopt_child(DETAILS, parent, child);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int engine_adopt_child(storage_object_t * parent,
			      storage_object_t * child) {
	int rc = 0;

	LOG_PROC_ENTRY();

	rc = is_valid_adopt_child(ERROR, parent, child);

	if (rc == 0) {
		list_element_t el;

		el = insert_thing(child->parent_objects,
				  parent,
				  INSERT_AFTER | EXCLUSIVE_INSERT,
				  NULL);

		if (el != NULL) {
			el = insert_thing(parent->child_objects,
					  child,
					  INSERT_AFTER | EXCLUSIVE_INSERT,
					  NULL);

			if (el != NULL) {

				if ((parent->object_type == EVMS_OBJECT) &&
				    (child->feature_header == NULL)) {
					child->feature_header = engine_alloc(sizeof(evms_feature_header_t));
				}

				mark_feature_headers_dirty(child);

				set_volume_in_object(child, parent->volume);

				rc = 0;

			} else {
				LOG_CRITICAL("insert_thing() to put child %s into the child list of parent %s didn't work.\n",
					     child->name, parent->name);

				remove_thing(child->parent_objects, parent);
			}

		} else {
			LOG_CRITICAL("insert_thing() to put parent %s into the parent list of child %s didn't work.\n",
				     parent->name, child->name);
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int engine_orphan_child(storage_object_t * child) {

	list_element_t iter;
	storage_object_t * parent;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(child->parent_objects, iter, parent) {
		remove_thing(parent->child_objects, child);
	}

	delete_all_elements(child->parent_objects);

	mark_feature_headers_dirty(child);

	set_volume_in_object(child, NULL);

	LOG_PROC_EXIT_INT(0);
	return 0;
}


static boolean engine_is_mounted(char * volume_name, char * * mount_name) {

	boolean result = FALSE;
	list_element_t iter;
	logical_volume_t * vol = NULL;

	LOG_PROC_ENTRY();

	/*
	 * If the caller wants a mount name, initialize it to NULL in case we don't
	 * find a mount point for the volume.
	 */
	if (mount_name != NULL) {
		*mount_name = NULL;
	}

	/* Search the volumes_list for a volume whose dev node matches volume_name. */
	LIST_FOR_EACH(&volumes_list, iter, vol) {
		if (strcmp(volume_name, vol->dev_node) == 0) {
			break;
		}
	}

	if (vol == NULL) {
		/*
		 * Didn't find it in the regular volume list. Search the
		 * volume_delete_list list as well.
		 */
		LIST_FOR_EACH(&volume_delete_list, iter, vol) {
			if (strcmp(volume_name, vol->dev_node) == 0) {
				break;
			}
		}
	}

	if (vol != NULL) {
		/* Found the volume with the matching name. */
		result = is_volume_mounted(vol);

		if (result && (mount_name != NULL)) {
			/* The caller wants a copy of the mount name. */
			*mount_name = engine_strdup(vol->mount_point);
		}

	} else {
		result = is_mounted(volume_name, 0, 0, mount_name);
	}

	LOG_PROC_EXIT_BOOLEAN(result);
	return result;
}


static int engine_assign_fsim_to_volume(plugin_record_t  * fsim,
					logical_volume_t * volume) {

	int rc = 0;

	LOG_PROC_ENTRY();

	if (volume->file_system_manager == NULL) {
		volume->file_system_manager = fsim;
		volume->flags |= VOLFLAG_PROBE_FS;

	} else {
		LOG_ERROR("Volume %s is currently being managed by the %s FSIM.\n", volume->name, volume->file_system_manager->short_name);
		rc = EBUSY;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int engine_unassign_fsim_from_volume(logical_volume_t * volume) {

	int rc = 0;

	LOG_PROC_ENTRY();

	/*
	 * If this FSIM is the original owner of the volume, save its
	 * current private data pointer.
	 */
	if (volume->file_system_manager == volume->original_fsim) {
		volume->original_fsim_private_data = volume->private_data;
	}

	volume->file_system_manager = NULL;

	/* Reinitialize the file system and volume limits. */
	volume->fs_size = volume->vol_size;
	volume->min_fs_size = 0;
	volume->max_fs_size = UINT64_MAX;
	volume->max_vol_size = UINT64_MAX;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static engine_mode_t engine_get_engine_mode(void) {
	return engine_mode;
}


static boolean engine_is_2_4_kernel(void) {

	LOG_PROC_ENTRY();

	LOG_PROC_EXIT_BOOLEAN(is_2_4_kernel);
	return is_2_4_kernel;
}


static int engine_get_clusterid(ece_clusterid_t * pclusterid) {

	LOG_PROC_ENTRY();

	memcpy(pclusterid, &clusterid, sizeof(ece_clusterid_t));

	if (cluster_manager == NULL) {
		LOG_PROC_EXIT_INT(ENODEV);
		return ENODEV;
	} else {
		LOG_PROC_EXIT_INT(0);
		return 0;
	}
}


static int engine_get_nodeid(ece_nodeid_t * nodeid) {

	LOG_PROC_ENTRY();

	if (cluster_manager == NULL) {
		LOG_PROC_EXIT_INT(ENODEV);
		return ENODEV;

	} else {
		memcpy(nodeid, current_nodeid, sizeof(ece_nodeid_t));

		LOG_PROC_EXIT_INT(0);
		return 0;
	}
}


static int engine_nodeid_to_string(ece_nodeid_t * nodeid,
				   const char * * string) {

	int rc = 0;

	LOG_PROC_ENTRY();

	if (nodeid == NULL) {
		LOG_ERROR("No nodeid given.\n");
		rc = EINVAL;
	}

	if (string == NULL) {
		LOG_ERROR("No string destination given.\n");
		rc = EINVAL;

	} else {
		*string = NULL;
	}

	if (rc == 0) {
		if (num_config_nodes == 0) {
			LOG_ERROR("The Engine is not running in a clustered environment.\n");
			*string = NULL;
			rc = ENODEV;

		} else {
			*string = nodeid_to_string(nodeid);

			if (*string == NULL) {
				LOG_ERROR("Node ID is not valid.\n");
				rc = EINVAL;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int engine_string_to_nodeid(const char   * string,
				   ece_nodeid_t * nodeid) {

	int rc = 0;

	LOG_PROC_ENTRY();

	if (string == NULL) {
		LOG_ERROR("No string given.\n");
		rc = EINVAL;
	}

	if (nodeid == NULL) {
		LOG_ERROR("No nodeid destination given.\n");
		rc = EINVAL;

	} else {
		memcpy(nodeid, &no_nodeid, sizeof(ece_nodeid_t));
	}

	if (rc == 0) {
		if (num_config_nodes == 0) {
			LOG_ERROR("The Engine is not running in a clustered environment.\n");
			rc = ENODEV;

		} else {
			int i = 0;

			while (i < num_config_nodes) {
				if (strcmp(config_node_names->node_info[i].node_name, string) == 0) {
					break;
				} else {
					i++;
				}
			}

			if (i < num_config_nodes) {
				memcpy(nodeid, &config_nodes[i], sizeof(ece_nodeid_t));

			} else {
				LOG_ERROR("There is no node ID to match node name \"%s\".\n", string);
				rc = EINVAL;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static boolean engine_have_quorum(void) {

	LOG_PROC_ENTRY();
	if (membership != NULL) {
		LOG_PROC_EXIT_BOOLEAN(membership->quorum_flag);
		return membership->quorum_flag;
	} else {
		LOG_PROC_EXIT_BOOLEAN(FALSE);
		return FALSE;
	}
}


boolean engine_is_offline(storage_object_t   * obj,
			  logical_volume_t * * pvol) {

	LOG_PROC_ENTRY();

	if (obj->volume != NULL) {
		if (is_volume_mounted(obj->volume)) {
			if (pvol != NULL) {
				*pvol = obj->volume;
			}
			LOG_PROC_EXIT_BOOLEAN(FALSE);
			return FALSE;
		} else {
			if (pvol != NULL) {
				*pvol = NULL;
			}
			LOG_PROC_EXIT_BOOLEAN(TRUE);
			return TRUE;
		}
	} else {
		list_element_t iter;
		storage_object_t * parent;

		LIST_FOR_EACH(obj->parent_objects, iter, parent) {
			if (!engine_is_offline(parent, pvol)) {
				LOG_PROC_EXIT_BOOLEAN(FALSE);
				return FALSE;
			}
		}

		if (pvol != NULL) {
			*pvol = NULL;
		}

		LOG_PROC_EXIT_BOOLEAN(TRUE);
		return TRUE;
	}
}


static pid_t do_fork_and_execvp(char * argv[],
				int    stdin_pipe[2],
				int    stdout_pipe[2],
				int    stderr_pipe[2]) {

	pid_t pid;

	LOG_PROC_ENTRY();

	/* Fork and execute the program. */
	pid = fork();
	
	if (pid == 0) {
		/* Child process */

		int dev_null_fd = -1;

		if ((stdout_pipe == NULL) || (stderr_pipe == NULL)) {
			LOG_DEBUG("Open /dev/null\n");
			dev_null_fd = open("/dev/null", O_WRONLY);
			LOG_DEBUG("dev_null_fd is %d\n", dev_null_fd);
		}

		if (stdin_pipe != NULL) {
			dup2(stdin_pipe[0], 0);	/* Route stdin to pipe's read fd */
		}

		if (stdout_pipe != NULL) {
			dup2(stdout_pipe[1], 1);/* Route stdout to pipe's write fd */
		} else {
			if (dev_null_fd > 0) {
				LOG_DEBUG("Send stdout to the bit bucket.\n");
				dup2(dev_null_fd, 1);/* Route stdout to the bit bucket */
			}
		}

		if (stderr_pipe != NULL) {
			dup2(stderr_pipe[1], 2);/* Route stderr to pipe's write fd */
		} else {
			if (dev_null_fd > 0) {
				LOG_DEBUG("Send stderr to the bit bucket.\n");
				dup2(dev_null_fd, 2);/* Route stderr to the bit bucket */
			}
		}

		if (debug_level >= DEBUG) {
			int i;
			char buff[128];
			char * ptr = buff;

			strcpy(buff, "execvp => ");
			ptr += strlen(buff);
			for (i = 0; argv[i] != NULL; i++) {
				strcpy(ptr, argv[i]);
				ptr += strlen(argv[i]);
				*ptr = ' ';
				ptr++;
			}

			*ptr = '\0';

			LOG_DEBUG("%s\n", buff);
		}

		execvp(argv[0], argv);

		/* If the execvp() works it won't return here. */
		LOG_WARNING("execvp() failed.  errno is %d: %s\n", errno, strerror(errno));

		/* Using exit() can hang GUI, use _exit */
		_exit(errno);
	}

	LOG_PROC_EXIT_INT(pid);
	return pid;
}


pid_t engine_fork_and_execvp(logical_volume_t * vol,
			     char             * argv[],
			     int                stdin_pipe[2],
			     int                stdout_pipe[2],
			     int                stderr_pipe[2]) {

	pid_t child_pid;

	LOG_PROC_ENTRY();

	child_pid = do_fork_and_execvp(argv, stdin_pipe, stdout_pipe, stderr_pipe);

	LOG_PROC_EXIT_INT(child_pid);
	return child_pid;
}


static int engine_discover(list_anchor_t objects) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = discover(objects, FALSE);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static boolean no_data_objects(list_anchor_t list) {

	list_element_t iter;
	storage_object_t * obj;

	LIST_FOR_EACH(list, iter, obj) {
		if (obj->data_type == DATA_TYPE) {
			return FALSE;
		}
	}

	return TRUE;
}


static int can_discard_volume(logical_volume_t * vol) {

	int rc = 0;

	LOG_PROC_ENTRY();

	if (vol->flags & (VOLFLAG_DIRTY | VOLFLAG_NEW |
			  VOLFLAG_NEEDS_ACTIVATE | VOLFLAG_NEEDS_DEACTIVATE |
			  VOLFLAG_MKFS | VOLFLAG_UNMKFS | VOLFLAG_FSCK |
			  VOLFLAG_SYNC_FS | VOLFLAG_PROBE_FS)) {
		rc = EBUSY;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_discard_object(storage_object_t * obj) {

	int rc = 0;
	list_element_t iter;
	storage_object_t * parent;

	LOG_PROC_ENTRY();

	if (no_data_objects(obj->parent_objects)) {
		if (obj->volume != NULL) {
			rc = can_discard_volume(obj->volume);
		}

	} else {
		LIST_FOR_EACH(obj->parent_objects, iter, parent) {
			if (parent->data_type == DATA_TYPE) {
				rc = can_discard_object(parent);

				if (rc != 0) {
					break;
				}
			}
		}
	}

	if (rc == 0) {
		if (obj->flags & (SOFLAG_DIRTY | SOFLAG_NEW |
				  SOFLAG_FEATURE_HEADER_DIRTY |
				  SOFLAG_NEEDS_ACTIVATE | SOFLAG_NEEDS_DEACTIVATE)) {
			rc = EBUSY;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int engine_can_discard(list_anchor_t objects) {

	int rc = 0;
	list_element_t iter;
	storage_object_t * obj;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(objects, iter, obj) {
		if (obj->data_type == DATA_TYPE) {
			rc = can_discard_object(obj);

			if (rc != 0) {
				break;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static void clear_volume_in_object(storage_object_t * obj) {

	list_element_t iter;
	storage_object_t * child;

	LOG_PROC_ENTRY();

	obj->volume = NULL;

	if (obj->producing_container == NULL) {
		LIST_FOR_EACH(obj->child_objects, iter, child) {
			clear_volume_in_object(child);
		}
	}

	LOG_PROC_EXIT_VOID();
	return;
}


int discard_volume(logical_volume_t * vol) {

	int rc = 0;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Request to discard volume %s.\n", vol->name);

	if (vol->file_system_manager != NULL) {
		rc = vol->file_system_manager->functions.fsim->discard(vol);
	}

	if (rc == 0) {
		engine_unregister_name(vol->name);

		/*
		 * Clear out the volume pointer on this object and on all the child
		 * objects in the tree.
		 */
		clear_volume_in_object(vol->object);

		remove_thing(&volumes_list, vol);

		if (vol->flags & VOLFLAG_NEW) {

			/*
			 * This volume hasn't been committed yet.  Just throw away the
			 * volume structure.
			 */
			LOG_DEBUG("Volume is new, so just toss it out.\n");
			engine_free(vol);

		} else {
			list_element_t el;

			LOG_DEBUG("Volume exists.  Put it on the delete list.\n");
			el = insert_thing(&volume_delete_list,
					  vol,
					  INSERT_AFTER,
					  NULL);

			if (el == NULL) {
				LOG_CRITICAL("Error putting volume %s on the volume delete list.\n", vol->name);
				LOG_PROC_EXIT_INT(ENOMEM);
				return ENOMEM;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int engine_discard(list_anchor_t objects) {

	int rc = 0;
	list_element_t iter;
	storage_object_t * obj;

	LOG_PROC_ENTRY();

	if (list_empty(objects)) {
		/* Nothing to do. */
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(objects, iter, obj) {
		LOG_DEBUG("Request to discard object %s.\n", obj->name);

		if (obj->data_type == DATA_TYPE) {
			if (obj->consuming_container != NULL) {
				if (!no_data_objects(obj->consuming_container->objects_produced)) {
					rc = engine_discard(obj->consuming_container->objects_produced);
					if (rc != 0) {
						break;
					}
				}
				rc = obj->consuming_container->plugin->container_functions->discard_container(obj->consuming_container);
				if (rc != 0) {
					break;
				}
			} else {
				if (no_data_objects(obj->parent_objects)) {
					if (obj->volume != NULL) {
						rc = discard_volume(obj->volume);
						if (rc != 0) {
							break;
						}
					}
				} else {
					rc = engine_discard(obj->parent_objects);
					if (rc != 0) {
						break;
					}
				}

				if (!no_data_objects(obj->associated_parents)) {
					rc = engine_discard(obj->associated_parents);
					if (rc != 0) {
						break;
					}
				}
			}
		}
	}

	if (rc == 0) {
		/*
		 * Pass a copy of the objects list rather than the list
		 * itself.  The list is most likely a literal parent_objects
		 * list from an object below the objects.  As the plug-in
		 * discards thing on the list it will be removing itself
		 * from the parent_objects list of its child object. It
		 * would then be modifying the list of objects that it is
		 * given to discard which could ruin its processing of the
		 * objects list it was given.
		 */
		list_anchor_t list_copy = copy_list(objects);

		if (list_copy != NULL) {
			obj = (storage_object_t *) first_thing(objects, NULL);
			rc = obj->plugin->functions.plugin->discard(list_copy);

			destroy_list(list_copy);

		} else {
			LOG_CRITICAL("Error getting memory for a copy of the objects list.\n");
			rc = ENOMEM;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int engine_rediscover_objects(list_anchor_t objects) {

	int rc = 0;
	list_element_t iter;
	storage_object_t * obj;

	LOG_PROC_ENTRY();

	if (objects != NULL) {
		LIST_FOR_EACH(objects, iter, obj) {
			int tmp_rc;

			tmp_rc = remote_mark_for_rediscover(obj->name);

			/* Return the first bad error code. */
			if (tmp_rc != 0) {
				rc = tmp_rc;
			}
		}

	} else {
		rc = remote_mark_for_rediscover(NULL);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int engine_rediscover_containers(list_anchor_t containers){

	int rc = 0;
	list_element_t iter;
	storage_container_t * con;

	LOG_PROC_ENTRY();

	if (containers != NULL) {
		LIST_FOR_EACH(containers, iter, con) {
			int tmp_rc;

			tmp_rc = remote_mark_for_rediscover(con->name);

			/* Return the first bad error code. */
			if (tmp_rc != 0) {
				rc = tmp_rc;
			}
		}

	} else {
		rc = remote_mark_for_rediscover(NULL);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int engine_save_metadata(char         * parent,
			 char         * child,
			 sector_count_t offset,
			 sector_count_t length,
			 void         * metadata) {
	int rc = 0;
	int bytes_written;
	metadata_db_entry_t metadata_entry;

	LOG_PROC_ENTRY();

	memset(&metadata_entry, 0, sizeof(metadata_entry));

	strncpy(metadata_entry.signature, EVMS_METADATA_DB_SIGNATURE, sizeof(metadata_entry.signature));
	metadata_entry.metadata_crc = CPU_TO_DISK32(evms_calculate_crc(0xffffffff,
								       metadata,
								       (u_int32_t) (length << EVMS_VSECTOR_SIZE_SHIFT)));
	if (parent != NULL) {
		strcpy(metadata_entry.parent, parent);
	}
	if (child != NULL) {
		strcpy(metadata_entry.child, child);
	}
	metadata_entry.offset = CPU_TO_DISK64(offset);
	metadata_entry.length = CPU_TO_DISK64(length);
	metadata_entry.header_crc = CPU_TO_DISK32(evms_calculate_crc(0xffffffff, &metadata_entry, sizeof(metadata_entry)));

	bytes_written = write(metadata_db_fd, &metadata_entry, sizeof(metadata_entry));
	if (bytes_written < 0) {
		rc = errno;
		LOG_SERIOUS("Error writing the metadata DB entry header.  Return code is %d: %s\n",
			    rc, strerror(rc));

		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (length != 0) {
		bytes_written = write(metadata_db_fd, metadata, length << EVMS_VSECTOR_SIZE_SHIFT);
		if (bytes_written < 0) {
			rc = errno;
			LOG_SERIOUS("Error writing the metadata.  Return code is %d: %s\n",
				    rc, strerror(rc));

			LOG_PROC_EXIT_INT(rc);
			return rc;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


engine_functions_t engine_functions = {
	get_engine_plugin_api_version: engine_get_plugin_api_version,
	get_plugin_list:               engine_get_plugin_list,
	get_plugin_by_ID:              engine_get_plugin_by_ID,
	get_plugin_by_name:            engine_get_plugin_by_name,
	get_volume_list:               engine_get_volume_list,
	get_object_list:               engine_get_object_list,
	get_container_list:            engine_get_container_list,
	allocate_logical_disk:         engine_allocate_logical_disk,
	free_logical_disk:             engine_free_logical_disk,
	allocate_segment:              engine_allocate_segment,
	free_segment:                  engine_free_segment,
	allocate_container:            engine_allocate_container,
	free_container:                engine_free_container,
	allocate_region:               engine_allocate_region,
	free_region:                   engine_free_region,
	allocate_evms_object:          engine_allocate_evms_object,
	free_evms_object:              engine_free_evms_object,
	free_storage_object:	       engine_free_storage_object,
	open_object:                   engine_open_object,
	read_object:                   engine_read_object,
	write_object:                  engine_write_object,
	ioctl_object:                  engine_ioctl_object,
	close_object:                  engine_close_object,
	open_volume:                   engine_open_volume,
	read_volume:                   engine_read_volume,
	write_volume:                  engine_write_volume,
	ioctl_volume:                  engine_ioctl_volume,
	close_volume:                  engine_close_volume,
	engine_alloc:                  engine_alloc,
	engine_realloc:                engine_realloc,
	engine_strdup:                 engine_strdup,
	engine_free:                   engine_free,
	commit_in_progress:            engine_commit_in_progress,
	write_log_entry:               plugin_write_log_entry,
	calculate_CRC:                 evms_calculate_crc,
	calculate_checksum:            engine_calculate_checksum,
	save_metadata:                 engine_save_metadata,
	add_sectors_to_kill_list:      engine_add_sectors_to_kill_list,
	validate_name:                 engine_validate_name,
	register_name:                 engine_register_name,
	unregister_name:               engine_unregister_name,
	can_expand_by:                 engine_can_expand_by,
	can_shrink_by:                 engine_can_shrink_by,
	user_message:                  plugin_user_message,
	user_communication:            plugin_user_communication,
	progress:                      plugin_progress,
	can_rename:                    engine_can_rename,
	can_adopt_child:               engine_can_adopt_child,
	adopt_child:                   engine_adopt_child,
	orphan_child:                  engine_orphan_child,
	is_mounted:                    engine_is_mounted,
	assign_fsim_to_volume:         engine_assign_fsim_to_volume,
	unassign_fsim_from_volume:     engine_unassign_fsim_from_volume,
	get_engine_mode:               engine_get_engine_mode,
	dm_get_version:                dm_get_version,
	dm_allocate_target:            dm_allocate_target,
	dm_add_target:                 dm_add_target,
	dm_deallocate_targets:         dm_deallocate_targets,
	dm_suspend:                    dm_suspend,
	dm_suspend_volume:             dm_suspend_volume,
	dm_set_suspended_flag:         dm_set_suspended_flag,
	dm_activate:                   dm_activate,
	dm_deactivate:                 dm_deactivate,
	dm_rename:                     dm_rename,
	dm_create:                     dm_create,
	dm_update_status:              dm_update_status,
	dm_get_targets:                dm_get_targets,
	dm_load_targets:               dm_load_targets,
	dm_clear_targets:              dm_clear_targets,
	dm_get_info:                   dm_get_info,
	dm_get_devices:                dm_get_devices,
	dm_deallocate_device_list:     dm_deallocate_device_list,
	dm_wait:                       dm_wait,
	is_2_4_kernel:                 engine_is_2_4_kernel,
	get_config_bool:               evms_get_config_bool,
	get_config_uint32:             evms_get_config_uint32,
	get_config_uint32_array:       evms_get_config_uint32_array,
	get_config_uint64:             evms_get_config_uint64,
	get_config_uint64_array:       evms_get_config_uint64_array,
	get_config_string:             evms_get_config_string,
	get_config_string_array:       evms_get_config_string_array,
	get_clusterid:                 engine_get_clusterid,
	get_nodeid:                    engine_get_nodeid,
	nodeid_to_string:              engine_nodeid_to_string,
	string_to_nodeid:              engine_string_to_nodeid,
	get_node_list:		       engine_get_node_list,
	have_quorum:		       engine_have_quorum,
	is_offline:                    engine_is_offline,
	offline_copy:                  engine_offline_copy,
	can_online_copy:               engine_can_online_copy,
	copy_setup:                    engine_copy_setup,
	copy_start:                    engine_copy_start,
	copy_wait:                     engine_copy_wait,
	copy_cleanup:                  engine_copy_cleanup,
	strerror:                      evms_strerror,
	fork_and_execvp:               engine_fork_and_execvp,
	discover:                      engine_discover,
	can_discard:                   engine_can_discard,
	discard:                       engine_discard,
	rediscover_objects:            engine_rediscover_objects,
	rediscover_containers:         engine_rediscover_containers,
	allocate_list:                 allocate_list,
	list_count:                    list_count,
	list_empty:                    list_empty,
	find_in_list:                  find_in_list,
	delete_all_elements:           delete_all_elements,
	destroy_list:                  destroy_list,
	get_thing:                     get_thing,
	next_element:                  next_element,
	next_thing:                    next_thing,
	previous_element:              previous_element,
	previous_thing:                previous_thing,
	first_thing:                   first_thing,
	last_thing:                    last_thing,
	insert_element:                insert_element,
	insert_thing:                  insert_thing,
	remove_element:                remove_element,
	delete_element:                delete_element,
	remove_thing:                  remove_thing,
	replace_thing:                 replace_thing,
	copy_list:                     copy_list,
	concatenate_lists:             concatenate_lists,
	merge_lists:                   merge_lists,
	sort_list:                     sort_list
};

