/* -*- Mode: C; -*- */

/* Copyright (C) beingmeta inc, 2001-2002
   Implemented by Ken Haase as part of FramerD

   This implements optimized repacking of file pools including
   scheduling of disk accesses.

   $Id: repack-file-pool.c,v 1.10 2002/06/03 21:51:21 haase Exp $
*/

#include "framerd/indextools.h"
#include <limits.h>

static char *usage=_("Usage: repack-file-pool [-q|-v] <in> [out]\n");
static char *not_readable=_("Couldn't open file for reading: %s");
static char *not_writable=_("Couldn't open file for writing: %s");

struct READ_SCHEDULE {unsigned int index, offset;};

static int read_schedule_compare_fn(const void *x,const void *y)
{
    struct READ_SCHEDULE *ix=(struct READ_SCHEDULE *)x;
    struct READ_SCHEDULE *iy=(struct READ_SCHEDULE *)y;
    if (ix->offset == iy->offset) return 0;
    else if (ix->offset == 0) return 1;
    else if (iy->offset == 0) return -1;
    else if (ix->offset<iy->offset) return -1;
    else return 1;
}

static int read_schedule_index_compare_fn(const void *x,const void *y)
{
    struct READ_SCHEDULE *ix=(struct READ_SCHEDULE *)x;
    struct READ_SCHEDULE *iy=(struct READ_SCHEDULE *)y;
    if (ix->index == iy->index) return 0;
    else if (ix->index < iy->index) return -1;
    else return 1;
}

static void copy_binary_file(char *from,char *to)
{
  FILE *in=fd_fopen_locked(from,"r+b",0), *out;
  int  bufsize=65536, ret_value=0, bytes=0;
  char *buf=fd_xmalloc(bufsize), *realname=NULL;
  if (fd_symbolic_linkp(to)) {
    realname=fd_get_real_pathname(to);
    out=fd_fopen_locked(realname,"wb",0);}
  else out=fd_fopen_locked(to,"wb",0);
  if (errno) {perror("Start of binary copy"); FD_CLEAR_ERR();}
  if (in == NULL) 
    fd_raise_detailed_exception(fd_FileOpenFailed,from);
  else if (out == NULL) 
    fd_raise_detailed_exception(fd_FileOpenWFailed,to);
  else while ((ret_value=fread(buf,sizeof(char),bufsize,in)) ||
	      (!(feof(in)))) {
    bytes=bytes+ret_value; fwrite(buf,sizeof(char),ret_value,out);}
  fclose(out); fclose(in); free(buf);
  if (realname) fd_xfree(realname);
}

int main(int argc,char *argv[])
{
  fd_lisp header, metadata;
  time_t make, repack, change;
  unsigned int major_version, minor_version;
  unsigned int code, oid_hi, oid_lo, capacity, load;
  unsigned int header_off, chunk, base, n_args, need_copy=0;
  FILE *in, *out;
  char tmpbuf[PATH_MAX]="/tmp/fdrfpXXXXXX";
  unsigned int i, write_pos, errors=0; struct READ_SCHEDULE *read_schedule;
  if (argc < 2) {
    fprintf(stderr,usage); fd_exit(1); return 1;}
  else if ((strcmp(argv[1],"-q") == 0) || (strcmp(argv[1],"-v") == 0))
    if (argc < 3) {
      fprintf(stderr,usage); fd_exit(1); return 1;}
    else base=2;
  else base=1;
  n_args=argc-base;
  if ((strcmp(argv[1],"-q")) == 0) fd_disable_notifications();
  fd_initialize_dtypes(); in=fd_fopen_locked(argv[base],"r+b",0);
  if (in == NULL) {
    fprintf(stderr,not_readable,argv[base]); exit(1);}
  code=fd_fread_4bytes(in);
  if (code == FD_FILE_POOL_MAGIC_NUMBER) {
    metadata=fd_read_file_pool_metadata
      (in,&major_version,&minor_version,&make,&repack,&change);
    /* Fixup times if neccessary */
    fd_notify(_("Repacking file pool %s with version id %d:%d"),
	      argv[base],major_version,minor_version);
    if (n_args == 2) {
      fd_notify(_("Writing new file pool to %s"),argv[base+1]);
      out=fd_fopen_locked(argv[base+1],"wb",0);
      if (out == NULL) {
	fprintf(stderr,not_writable,argv[base+1]);
	exit(1);}}
    else {
      fd_notify(_("Writing new file pool to temporary file %s"),tmpbuf);
      out=fd_fopen_tmpfile(tmpbuf,"wb"); need_copy=1;
      if (out == NULL) {
	fprintf(stderr,not_writable,argv[base+1]);
	exit(1);}}}
  else {
    fclose(in);
    fprintf(stderr,_("Error: %s is not a file pool\n"),argv[base]);
    fd_exit(1); return 1;}
  /* Read header information, set up read schedule */
  oid_hi=fd_fread_4bytes(in); oid_lo=fd_fread_4bytes(in);
  capacity=fd_fread_4bytes(in); load=fd_fread_4bytes(in);
  header_off=fd_fread_4bytes(in);
  read_schedule=fd_malloc(sizeof(struct READ_SCHEDULE)*load);
  fd_notify(_("Reading and sorting seek schedule for %d OIDs"),load);
  i=0; while (i < load) {
    read_schedule[i].index=i;
    read_schedule[i].offset=fd_fread_4bytes(in);
    i++;}
  /* Sort the read schedule */
  qsort(read_schedule,load,sizeof(struct READ_SCHEDULE),
	read_schedule_compare_fn);
  fd_notify(_("Writing new file pool"));
  /* Write header code, base OID, capacity, and load */
  fd_fwrite_4bytes(code,out);
  fd_fwrite_4bytes(oid_hi,out); fd_fwrite_4bytes(oid_lo,out);
  fd_fwrite_4bytes(capacity,out); fd_fwrite_4bytes(load,out);
  /* Get the LISP header when you're through with the data */
  fd_fwrite_4bytes(0,out); 
  /* Write an empty offset table (we'll rewrite it later) */
  i=0; while (i < capacity) {fd_fwrite_4bytes(0,out); i++;}
  write_pos=(6+capacity)*4;  chunk=load/10;
  /* Write out the metadata segment */
  fd_fwrite_4bytes(0xFFFFFFFE,out); write_pos=write_pos+4;
  fd_fwrite_4bytes(40,out); write_pos=write_pos+4;
  fd_fwrite_4bytes(major_version+1,out); write_pos=write_pos+4;
  /* Copy creation timestamp */
  if (make < 0) {
    fd_fwrite_4bytes(0,out); fd_fwrite_4bytes((int)0,out);
    write_pos=write_pos+8;}
  else {
    fd_fwrite_4bytes(0,out); fd_fwrite_4bytes((int)make,out);
    write_pos=write_pos+8;}
  /* Write repack timestamp */
  fd_fwrite_4bytes(0,out); fd_fwrite_4bytes((int)time(NULL),out);
  write_pos=write_pos+8;
  /* Copy change timestamp */
  fd_fwrite_4bytes(0,out); fd_fwrite_4bytes((int)change,out);
  write_pos=write_pos+8;
  /* Write out the metadata itself */
  if (FD_EMPTYP(metadata)) {
    fd_fwrite_4bytes(0,out); write_pos=write_pos+4;}
  else { 
    fd_fwrite_4bytes(write_pos+4,out); write_pos=write_pos+4;
    write_pos=write_pos+fd_fwrite_dtype(metadata,out);}
  /* Read and write the OIDs themselves in chunks  */
  i=0; while (i < load) {
    fd_lisp entry; unsigned int value_pos=write_pos;
    if ((chunk>0) && (((i+1)%chunk) == 0)) {
      char buf[16]; sprintf(buf,"%.2f",(i*100.0)/load);
      fd_notify(_("%s: %s%%: Copied %d of %d OIDs (%d bytes)"),
		argv[base],buf,i,load,write_pos-(6+capacity)*4);}
    fseek(in,read_schedule[i].offset,SEEK_SET);
    {FD_WITH_HANDLING 
       entry=fd_fread_dtype(in);
     FD_ON_EXCEPTION {
       fd_exception ex=fd_theException();
       fd_u8char *details=fd_exception_details();
       fd_notify(_("Error copying @%x/%x: %s (%s)"),
		 oid_hi,oid_lo+read_schedule[i].index,ex,details);
       errors=1;
       entry=fd_make_exception
	 (FD_MAKE_LIST(2,fd_make_string(ex),fd_make_string(details)));
       fd_clear_exception();}
     FD_END_HANDLING;}
    if (FD_PRIM_TYPEP(entry,mystery_type)) {
      fd_notify(_("Mystery copying @%x/%x"),
		oid_hi,oid_lo+read_schedule[i].index);
      write_pos=write_pos+fd_fwrite_dtype(FD_FALSE,out);}
    else write_pos=write_pos+fd_fwrite_dtype(entry,out);
    read_schedule[i].offset=value_pos;
    fd_decref(entry); i++;}
  if (header_off) {
    fd_lisp header;
    fseek(in,header_off,SEEK_SET); header=fd_fread_dtype(in);
    fd_fwrite_dtype(header,out);}
  else write_pos=0;
  fclose(in); /* All done with it */
  /* Put things back in order */
  qsort(read_schedule,load,sizeof(struct READ_SCHEDULE),
	read_schedule_index_compare_fn);
  fseek(out,20,SEEK_SET); fd_fwrite_4bytes(write_pos,out); 
  i=0; while (i < load) {
    fd_fwrite_4bytes(read_schedule[i].offset,out); i++;}
  fclose(out);
  if (need_copy) {
    if (errors) {
      fd_notify(_("Errors repacking file, output left in %s"),tmpbuf);}
    else {
      fd_notify(_("Copying output file back into original"));
      copy_binary_file(tmpbuf,argv[base]);}}
  return 0;
}


/* File specific stuff */

/* The CVS log for this file
   $Log: repack-file-pool.c,v $
   Revision 1.10  2002/06/03 21:51:21  haase
   Progress reports now provide more context

   Revision 1.9  2002/05/28 08:16:36  haase
   Added better messages to repack-file-*

   Revision 1.8  2002/05/13 06:51:01  haase
   Fix exception alloc/free bug

   Revision 1.7  2002/04/22 17:56:32  haase
   Fixed divide by zero case for very small file pools

   Revision 1.6  2002/04/22 14:23:08  haase
   Added extended metadata to file pools and indices

   Revision 1.5  2002/04/12 15:42:08  haase
   Repacking a linked file now copies into the target rather than overwriting the link; also moved code out of FD_SOURCE

   Revision 1.4  2002/04/10 03:02:10  haase
   Added version information to file pools and indices

   Revision 1.3  2002/04/02 21:39:32  haase
   Added log and emacs init entries to C source files

*/

/* Emacs local variables
;;;  Local variables: ***
;;;  compile-command: "cd ../..; make" ***
;;;  End: ***
*/
