/**
 ** mp3list.c - central list of mp3s available to gnomp3
 **
 ** Copyright (C) 2000 Matthew Pratt <mattpratt@yahoo.com>
 **
 ** This software is licensed under the terms of the GNU General 
 ** Public License (GPL). Please see the file LICENSE for details.
 **/ 

#include "mp3list.h"
#include "utility.h"
#include "song_list.h"
#include "all_list.h"
#include "song_find.h"
#include "dir_list.h"

/*
 * Build a text representation of the play time of the mp3 and return it in a
 * static buffer.
 */
char *mp3list_build_time(MP3 *mp3)
{
    static char ntext[30];
    if( mp3->play_time > 0 ){
	g_snprintf(ntext, sizeof(ntext), "%d:%02d", mp3->play_time/60, 
		   mp3->play_time%60 );
    }else if( mp3->play_time < 0 ){
	g_snprintf(ntext, sizeof(ntext), "?:??");
    }else{
	ntext[0] = 0;
    }
    return ntext;
}

MP3 *mp3list_new_basic_entry()
{
    MP3 *new;

    new = g_new0( MP3, 1 );
    new->row_playlist = -1;
    new->row_alllist  = -1;
    new->row_timelist = -1;
    new->row_dirlist  = -1;

    new->id3_title   = "";
    new->id3_artist  = "";
    new->id3_album   = "";
    new->id3_year    = "";
    new->id3_comment = "";
    new->id3_genre   = 255;
    
    return new;
}

/*
 * Create a new mp3list node from the given filename. All other fields are set
 * to default values.
 */
MP3 *mp3list_build_entry(char *filename)
{
    MP3 *new;
    char *line_copy;

    new = mp3list_new_basic_entry();

    new->filename = g_strdup(filename);
    line_copy     = g_strdup(filename);

    strip_underscores(line_copy);
    path_breakdown( line_copy, &new->album_name, &new->display_name_from_filename );
    new->display_name = new->display_name_from_filename;

    return new;
}

/*
 * Build a new mp3list node from a series of tokens. 
 */
MP3 *mp3list_build_full_entry(char **tokens)
{
    MP3 *new;
    char *line_copy;

    new = mp3list_new_basic_entry();

    new->filename = g_strdup(tokens[0]);
    line_copy     = g_strdup(tokens[0]);
    strip_underscores(line_copy);
    path_breakdown( line_copy, &new->album_name, 
		    &new->display_name_from_filename );

    new->play_time    = atoi(tokens[1]);
    new->file_time    = atoi(tokens[2]);
    new->no_plays     = atoi(tokens[3]);
    new->pref_level   = atoi(tokens[4]);
    new->cache_expiry = atoi(tokens[5]);

    new->id3_title    = g_strdup(tokens[6]);
    new->id3_artist   = g_strdup(tokens[7]);
    new->id3_album    = g_strdup(tokens[8]);
    new->id3_year     = g_strdup(tokens[9]);
    new->id3_comment  = g_strdup(tokens[10]);
    if(tokens[11])new->id3_genre    = atoi(tokens[11]);

    if(new->id3_artist[0] != 0)
	new->display_name_from_id3 = g_strdup_printf("%s - %s", 
						     new->id3_artist,
						     new->id3_title);

    if( gnomp3.use_id3 && new->display_name_from_id3 )
	new->display_name = new->display_name_from_id3;
    else
	new->display_name = new->display_name_from_filename;

    return new;
}

void mp3list_remove(MP3 *mp3)
{
 
    
}

/*
 * Save the current mp3list to a file for later loading.
 */
void mp3list_save_to_file(char *filename)
{
    FILE *fp;
    GList *ptr = mp3list;
    MP3 *mp3;

    fp = fopen( filename, "w" );
    if( !fp ){
	return;
    }

    while(ptr){
	mp3 = ptr->data;
	/* dont save songs for the additional direcotry (dir_clist) */
	if( mp3->row_dirlist == -1 ){
	    fprintf( fp, "%s\t", mp3->filename );
	    fprintf( fp, "%d\t", mp3->play_time);
	    fprintf( fp, "%ld\t", mp3->file_time);
	    fprintf( fp, "%d\t", mp3->no_plays);
	    fprintf( fp, "%d\t", mp3->pref_level);
	    fprintf( fp, "%ld\t", mp3->cache_expiry);
	    fprintf( fp, "%s\t", mp3->id3_title );
	    fprintf( fp, "%s\t", mp3->id3_artist );
	    fprintf( fp, "%s\t", mp3->id3_album );
	    fprintf( fp, "%s\t", mp3->id3_year );
	    fprintf( fp, "%s\t", mp3->id3_comment );
	    fprintf( fp, "%d\t", mp3->id3_genre );
	    fprintf( fp, "\n" );
	}
	ptr = ptr->next;
    }

    fclose(fp);
}

/**** mp3list searching functions ********************************************/

/*
 * To sto find an entry in the mp3list with the given path/filename.
 * Start seraching from the start pointer and return the match else NULL
 */
GList *mp3list_search(GList *start, char *filename)
{
    GList *ptr = start;
    MP3 *mp3;
    
    /* search from the start ptr to the end of the list */
    while(ptr){
	mp3 = ptr->data;
	if( !strcmp( mp3->filename, filename))
	    return ptr;
	ptr = ptr->next;
    }
    /* search from the begining to the start ptr */
    ptr = mp3list;
    while(ptr && ptr != start){
	mp3 = ptr->data;
	if( !strcmp( mp3->filename, filename))
	    return ptr;
	ptr = ptr->next;
    }

    return NULL;
}

/*
 * Serach based on the last part of the full path filname.
 */
GList *mp3list_search_by_file(char *filename)
{
    GList *ptr;
    MP3 *mp3;
    char *name, *last;
    char *file;

    file = filename;
    while(*filename++)
	if(*filename == '/')file = filename;
    file++;
    
    for(ptr = mp3list; ptr; ptr = ptr->next){
	mp3 = ptr->data;
	name = last = mp3->filename;
	while(*name++)
	    if(*name == '/')last = name;
	last++;

	if( !strcmp(last, file))
	    return ptr;
    }
    return NULL;
}

/*
 * Search the mp3list for the given text. Call the "found_callback" for each
 * match.
 * Search text looks as follows:  "foo !bar" -> will match any mp3 with foo but
 * not bar in its path name.
 */
void mp3list_search_by_name(char *text, void (*found_callback)(MP3 *mp3))
{
    char *text_copy, *song_data;
    char **tokens;
    GList *ptr;
    int all_found, i;

    text_copy = g_strdup(text);
    g_strup(text_copy );
    
    tokens = g_strsplit( text_copy, " ", 10 );
    
    for( ptr = mp3list; ptr; ptr = ptr->next ){
	song_data = g_strdup(MP3_NODE(ptr)->filename);
	g_strup(song_data);

	/* now break up the search string and only add if all are found */
	all_found = TRUE;
	for( i = 0; tokens[i]; i++ ){
	    //printf("Token: %s\n", tokens[i] );
	    if(tokens[i][0] != '!'){
		if( !strstr( song_data, tokens[i] ) )
		    all_found = FALSE;
	    }else{
		if( strstr( song_data, &tokens[i][1]) )
		    all_found = FALSE;
	    }
	}
	
	g_free(song_data);
	
	/* only accept when all tokens are found */
	if( all_found ){
	    found_callback(ptr->data);
	}
    }
    
    g_strfreev( tokens );
    g_free( text_copy );
}
