/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.  
                  Let a thousand taggers bloom!
   
   Copyright (C) Robert Kaye 2003
   
   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: analyzer.cpp,v 1.40 2004/03/18 22:21:31 robert Exp $

----------------------------------------------------------------------------*/
#ifdef WIN32
#if _MSC_VER == 1200
#pragma warning(disable:4786)
#endif
#endif

#include <stdio.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include "../config.h"
#include "analyzer.h"
#include "tunepimp.h"
#include "submit.h"
#include "file_meta.h"
#include "mp3_trm.h"
#include "ov_trm.h"
#include "wav_trm.h"
#include "flac_trm.h"
#include "flac_meta.h"
#include "id3_meta.h"
#include "vorbis_meta.h"
#include "ape_trm.h"
#include "ape_meta.h"
#include "watchdog.h"

extern "C"
{
   #include "fstrcmp.h"
}   

const int iDataFieldLen = 255;

#define DB printf("%s:%d\n", __FILE__, __LINE__);

//---------------------------------------------------------------------------

Analyzer::Analyzer(TunePimp       *tunePimpArg,
                   FileCache      *cacheArg,
                   SubmitInfo     *submitInfoArg,
                   WatchdogThread *watchdog) : Thread()
{
   tunePimp = tunePimpArg;
   cache = cacheArg;
   submitInfo = submitInfoArg;
   dog = watchdog;
   exitThread = false;
   sem = new Semaphore();
}

//---------------------------------------------------------------------------

Analyzer::~Analyzer(void)
{
   exitThread = true;
   sem->signal();
   join();
   delete sem;
}

//---------------------------------------------------------------------------

void Analyzer::wake(void)
{
    sem->signal();
}

//---------------------------------------------------------------------------

void Analyzer::threadMain(void)
{
    string  fileName, status, trm;
    Track  *track;

    dog->setAnalyzerThread(getId());
    setPriority(tunePimp->context.getAnalyzerPriority());
    for(; !exitThread;)
    {
        track = cache->getNextItem(ePending);
        if (track == NULL)
        {
           sem->wait();
           continue;
        }

        track->lock();

        dog->setAnalyzerTask(cache->getFileIdFromTrack(track));
        
        track->getFileName(fileName);

        readMetadata(track);

        // Check to see if we pulled up a musicbrainz id
        Metadata data, id3;
        track->getServerMetadata(data);
        track->getLocalMetadata(id3);
        track->getTRM(trm);
        if (trm == string(ANALYZER_REDO_STRING))
        {
            data.clear();
            data.trackId = "";
            track->setServerMetadata(data);

            track->getLocalMetadata(id3);
            id3.trackId = "";
            track->setLocalMetadata(id3);
        }
        trm = "";
        if (!id3.trackId.empty())
        {
            // Yup, toss it into saved
            track->getLocalMetadata(data);
            track->setServerMetadata(data);

            if (track->hasChanged())
                track->setStatus(eRecognized);
            else
                track->setStatus(eSaved);
        }
        else
        {
            unsigned long  duration = 0;
            char          *ptr;
            TRMGenerator  *gen;
            TRMResult      ret = eOtherError;

            ptr = strrchr(fileName.c_str(), '.');
#ifdef HAVE_LIBMAD
            if (ptr && strcasecmp(ptr, ".mp3") == 0)
                gen = new TRMGeneratorMP3(tunePimp);
            else
#endif
#ifdef HAVE_OGGVORBIS
            if (ptr && strcasecmp(ptr, ".ogg") == 0)
                gen = new TRMGeneratorOggVorbis(tunePimp);
            else
#endif
#ifdef HAVE_FLAC
            if (ptr && strcasecmp(ptr, ".flac") == 0)
                gen = new TRMGeneratorFLAC(tunePimp);
            else
#endif
#ifdef HAVE_LIBAPE
            if (ptr && strcasecmp(ptr, ".ape") == 0)
                gen = new TRMGeneratorApe(tunePimp);
            else
#endif
            if (ptr && strcasecmp(ptr, ".wav") == 0)
                gen = new TRMGeneratorWav(tunePimp);
            else
                gen = NULL;

            if (gen)
            {
                track->unlock();

                status = "Analyzing " + fileName;
                tunePimp->setStatus(status);

                // The only thing the try block should catch is something
                // we know nothing about or a segfault and even then only
                // under windows. Thus, exit the thread and let the
                // watchdog skip this file and start a new analyzer.
                try
                {
                    ret = gen->generate(fileName.c_str(), trm, duration);
                }
                catch(...)
                {
                    terminate();
                }

                track->lock();

                // Check to make sure no one has messed with this track since we started
                if (track->getStatus() == ePending)
                {
                    if (ret != eOk)
                    {
                        string err;

                        tunePimp->setStatus("Failed to generate trm from " + fileName);
                        gen->getError(err);
                        if (err.empty())
                            setError(track, ret);
                        else
                            track->setError(err);
                        track->setStatus(eError);
                    }
                    else
                        tunePimp->setStatus(string(" "));

                    delete gen;

                    if (duration > 0)
                    {
                        Metadata fileData;

                        track->getLocalMetadata(fileData);
                        fileData.duration = duration;
                        track->setLocalMetadata(fileData);
                    }

                    if (!trm.empty())
                    {
                        if (trm.substr(14, 1) != string("4"))
                        {
                            string err;

                            err = string("Unable to retrieve TRM from fingerprint server.");
                            tunePimp->setStatus(err);
                            track->setStatus(eError);
                            track->setError(err);
                        }
                        else
                        {
                            if (strcmp("c457a4a8-b342-4ec9-8f13-b6bd26c0e400", trm.c_str()) == 0)
                            {
                                track->setStatus(eError);
                                setError(track, eSigServerBusy);
                                tunePimp->setStatus("TRM calculation failed: TRM server is too busy.");
                            }
                            else
                            {
                                track->setTRM(trm);
                                track->setStatus(eTRMLookup);
                            }
                        }
                    }
                }
            }
            else
            {
                string err;

                if (ptr)
                    err = string(ptr) + string(" is not a supported filetype.");
                else
                    err = string(fileName) + string(": cannot determine filetype.");
                tunePimp->setStatus(err);
                track->setStatus(eError);
                track->setError(err);
            }
        }
        track->unlock();
        tunePimp->wake(track);
        cache->release(track);
        dog->setAnalyzerTask(-1);
    }
}

//---------------------------------------------------------------------------

void Analyzer::setError(Track *track, TRMResult retVal)
{
    switch(retVal)
    {
        case eFileNotFound:
           track->setError("Audio file not found.");
           break;
        case eDecodeError:
           track->setError("Cannot decode audio file.");
           break;
        case eCannotConnect:
           track->setError("Cannot connect to the TRM signature server.");
           break;
        case eSigServerBusy:
           track->setError("The TRM signature server is too busy to process your request.");
           break;
        default:
           track->setError("Unknown error. Sorry, this program sucks.");
           break;
    }
}

//---------------------------------------------------------------------------

// This whole scheme should get replaced with something more dynamic.
// But that takes a lot of time and effort, which I don't really have
// right now. But then again, do you ever?
void Analyzer::getSupportedExtensions(vector<string> &extList)
{
    extList.push_back(string(".wav"));
#ifdef HAVE_LIBMAD
    extList.push_back(string(".mp3"));
#endif
#ifdef HAVE_OGGVORBIS
    extList.push_back(string(".ogg"));
#endif
#ifdef HAVE_FLAC
    extList.push_back(string(".flac"));
#endif
#ifdef HAVE_APE
    extList.push_back(string(".ape"));
#endif
}

//---------------------------------------------------------------------------
// note: The track is locked when this function is called.
bool  Analyzer::readMetadata(Track *track)
{
    string        fileName, trm, ext;
    Metadata      data, fileNameData;
    char         *ptr;
    FileMetadata *fileMeta = NULL;
    bool          ret, readRet = false;

    track->getFileName(fileName);
    ptr = strrchr(fileName.c_str(), '.');
#ifdef HAVE_LIBMAD
    if (ptr && strcasecmp(ptr, ".mp3") == 0)
        fileMeta = new ID3(true);
    else
#endif
#ifdef HAVE_OGGVORBIS
    if (ptr && strcasecmp(ptr, ".ogg") == 0)
        fileMeta = new Vorbis();
    else
#endif
#ifdef HAVE_FLAC
    if (ptr && strcasecmp(ptr, ".flac") == 0)
        fileMeta = new FLAC();
    else
#endif
#ifdef HAVE_LIBAPE
    if (ptr && strcasecmp(ptr, ".ape") == 0)
        fileMeta = new Ape();
    else
#endif
    if (ptr && strcasecmp(ptr, ".wav") == 0) 
    {
        data.fileFormat = "wav";
        track->setLocalMetadata(data);
    } 
    else
        return false;

    // For the initial values, try to parse the filename and see what we can get 
    // from it.
    parseFileName(fileName, fileNameData);

    // The only thing the try block should catch is something
    // we know nothing about or a segfault and even then only
    // under windows. Thus, exit the thread and let the
    // watchdog skip this file and start a new analyzer.
    try
    {
        track->unlock();
        if (fileMeta)
        {
            readRet = fileMeta->read(fileName, data);
        }
        track->lock();
    }
    catch(...)
    {
        terminate();
    }

    if (data.artist.empty())
        data.artist = fileNameData.artist;
    if (data.album.empty())
        data.album = fileNameData.album;
    if (data.track.empty())
        data.track = fileNameData.track;
    if (data.trackNum <= 0)
        data.trackNum = fileNameData.trackNum;

    track->setLocalMetadata(data);

    if (!readRet)
       ret = false;
    else
    {
       track->getTRM(trm);
       if (trm != string(ANALYZER_REDO_STRING) && !data.fileTrm.empty())
          track->setTRM(data.fileTrm);

       ret = true;
    }

    delete fileMeta;
    
    return ret;
}

void Analyzer::parseFileName(const string &fileName, Metadata &data)
{
    FileNameMaker maker(&tunePimp->context);
    string        name;
    int           ret;
    char          field1[iDataFieldLen], field2[iDataFieldLen];
    char          field3[iDataFieldLen], field4[iDataFieldLen];

    name = maker.extractFileBase(fileName);
    ret = sscanf(name.c_str(), "%254[^-]-%254[^-]-%254[^-]-%254[^\n\r]",
            field1, field2, field3, field4);

    switch(ret)
    {
        case 4:
            data.artist = field1;
            data.album = field2;
            data.trackNum = atoi(field3);
            data.track = field4;
            break;
        case 3:
            data.artist = field1;
            if (atoi(field2) > 0)
                data.trackNum = atoi(field2);
            else
                data.album = field2;
            data.track = field3;
            break;
        case 2:
            data.artist = field1;
            data.track = field2;
            break;
        case 1:
            data.track = field1;
            break;
        default:
            break;
    }

    trimWhitespace(data.artist);
    trimWhitespace(data.album);
    trimWhitespace(data.track);

}

void Analyzer::trimWhitespace(string &field)
{
    for(; field.size() > 0;)
        if (field[0] == ' ' || field[0] == '\t' || field[0] == '\r')
            field.erase(0, 1);
        else
            break;

    for(; field.size() > 0;)
    {
        int last = field.size() - 1;
        if (field[last] == ' ' || field[last] == '\t' || field[last] == '\r')
            field.erase(last, 1);
        else
            break;
    }
}
