/*
 * Copyright (c) 2001-2003 Shiman Associates Inc. All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

#ifndef _MAS_COMMON_H
#define _MAS_COMMON_H

/***********************************************************************
 * mas_common.h
 *
 * These are MAS functions common to server, client, and device.
 * These are prefixed with "masc_".
 *
 ***********************************************************************/

#include <stdio.h> /* for size_t */
#ifdef _POSIX_C_SOURCE
# include <sys/time.h>
#endif
#include "mas/mas_constants.h"
#include "mas/mas_types.h"
#include "mas/mas_error.h"
#include "mas/mas_misc.h"
#include "mas/mas_logging.h"
#include "mas/mas_audio_basic.h"
#include "mas/mas_real_time_process.h"
#include "mas/mas_stats.h"

/* Packaging flags:

    EXTRACT - On setup, pull the package's header information from
              what's in the contents buffer.
    STATIC - contents member is statically allocated, don't realloc()
             or free()!
    NOFREE - do not free() contents when striking
    NOREALLOC - do not realloc() contents if we run out of room
    REALLOCED - the package contents has been realloc()ed

*/

#define MASC_PACKAGE_NOFREE    0x00000001
#define MASC_PACKAGE_EXTRACT   0x00000002
#define MASC_PACKAGE_STATIC    0x00000004
#define MASC_PACKAGE_NOREALLOC 0x00000008
#define MASC_PACKAGE_REALLOCED 0x00000010


/***********************************************************************
 * mas_event handling (mas_event.c)
 ***********************************************************************/

extern int32 masc_setup_event( struct mas_event* event );
extern int32 masc_strike_event( struct mas_event* event );
extern int32 masc_append_event(struct mas_event* head, struct mas_event* event);
extern int32 masc_insert_event(struct mas_event* insert_after, struct mas_event* event);
extern int32 masc_pack_event( struct mas_package *package, struct mas_event *event  );
extern int32 masc_unpack_event( struct mas_package *package, struct mas_event *event );

/***********************************************************************
 * error prettyprinting (mas_error.c)
 ***********************************************************************/

extern const char* masc_strmerror ( int );
extern char*       masc_strserror ( int );
extern const char* masc_strerrorlayer(int error);
extern void        masc_logerror ( int, const char* , ... );
extern int32 masc_register_statstrings_temp( char* module, int32 mi, char** statstring_list, int32 offset );
extern char* masc_statstring_temp( int32 err, int32 mi );
extern void masc_statstring_log_temp( int32 loglevel, int32 err, int32 mi, char* message );

/***********************************************************************
 * string processing (strings.c)
 ***********************************************************************/

extern void masc_trim_string(char* string);
extern int32 masc_get_string_index( char* sut, char** strings, int num );
extern size_t masc_strlcpy(char *dst, const char *src, size_t size);
extern size_t masc_strlcat(char *dst, const char *src, size_t size);

/* this oughta be in a separate library... */
extern char* masc_construct_title( const char *app_name );


/***********************************************************************
 * memory handling (mas_rtmemory.c)
 ***********************************************************************/


extern void* masc_rtalloc (size_t );
extern void* masc_rtcalloc (size_t, size_t );
extern void  masc_rtfree  (void* );
extern void* masc_rtrealloc(void* ptr, size_t size);

#define MAS_NEW(x) ( masc_rtcalloc( 1, (size_t) sizeof *(x) ) )

/***********************************************************************
 * smart data object creation and destruction (mas_data.c)
 ***********************************************************************/

extern int32   masc_setup_data( struct mas_data* data, int amount );
extern int32   masc_strike_data( struct mas_data* data );

/***********************************************************************
 * data characteristic functions (dc.c)
 ***********************************************************************/

extern int32   masc_print_dc( struct mas_data_characteristic* dc );
extern int32   masc_get_index_of_key( struct mas_data_characteristic* dc, 
				      char* key );
extern int32   masc_setup_dc( struct mas_data_characteristic* dc, int16 size );
extern int32   masc_strike_dc( struct mas_data_characteristic* dc );
extern int32   masc_append_dc_key_value( struct mas_data_characteristic* dc, const char* key, const char* value );
extern int32 masc_copy_dc            ( struct mas_data_characteristic* dc_dest, 
				       struct mas_data_characteristic* dc_src);
extern int32 masc_pack_dc( struct mas_package* package, struct mas_data_characteristic* dc  );
extern int32 masc_unpack_dc( struct mas_package* package, struct mas_data_characteristic* dc );
extern int32 masc_change_dc_value( struct mas_data_characteristic* dc, char* key, char* value );
extern int32 masc_compare_dc( struct mas_data_characteristic* dc1, struct mas_data_characteristic* dc2 );

/***********************************************************************
 * system timer functions (mas_time.c)
 ***********************************************************************/
extern int32 masc_init_timing( void );
extern int32 masc_exit_timing( void );
extern int32 masc_get_systime        ( struct mas_ntpval* systime );
extern int32 masc_get_short_usec_ts  ( u_int32* usec );
extern int32 masc_nanosleep          ( u_int32 nsec );
extern int32 masc_usleep             ( u_int32 usec );
extern int32 masc_realsleep          ( u_int32 nsec );
extern int32 masc_ntp_to_double(struct mas_ntpval* ntptime, double* doubletime);
extern int32 masc_ntp32_to_double(uint32 ntptime, double* doubletime);
#if defined(_POSIX_C_SOURCE)
extern int32 masc_timeval_to_double(struct timeval* tv, double* doubletime);
#endif

/***********************************************************************
 * system CPU measurement functions (mas_cpu.c)
 ***********************************************************************/
extern int32 masc_cpu_us( uint32* user, uint32* sys );

/***********************************************************************
 * mas packing and unpacking functions (mas_pushpull.c)
 ***********************************************************************/
extern int32 masc_push_int8( struct mas_package* package, int8 val );
extern int32 masc_push_uint8( struct mas_package* package, uint8 val );
extern int32 masc_push_int16( struct mas_package* package, int16 val );
extern int32 masc_push_uint16( struct mas_package* package, uint16 val );
extern int32 masc_push_int32( struct mas_package* package, int32 val );
extern int32 masc_push_uint32( struct mas_package* package, uint32 val );
extern int32 masc_push_float( struct mas_package* package, float val );
extern int32 masc_push_double( struct mas_package* package, double val );
extern int32 masc_push_string( struct mas_package* package, char* string );
extern int32 masc_push_strings( struct mas_package* package, char** strings, int num );
extern int32 masc_push_package( struct mas_package* package, struct mas_package* ppack );
extern int32 masc_push_payload( struct mas_package* package, void* payload, uint32 len );
extern int32 masc_pull_int8( struct mas_package* package, int8* retval);
extern int32 masc_pull_uint8( struct mas_package* package, uint8* retval );
extern int32 masc_pull_int16( struct mas_package* package, int16* retval );
extern int32 masc_pull_uint16( struct mas_package* package, uint16* retval );
extern int32 masc_pull_int32( struct mas_package* package, int32* retval );
extern int32 masc_pull_uint32( struct mas_package* package, uint32* retval );
extern int32 masc_pull_float( struct mas_package* package, float* retval );
extern int32 masc_pull_double( struct mas_package* package, double* retval );
extern int32 masc_pull_string( struct mas_package* package, char** string_retval, int copy );
extern int32 masc_pull_payload( struct mas_package* package, void** payload_retval, uint32* len_retval, int copy );
extern int32 masc_pull_package( struct mas_package* package, struct mas_package *retpack, int copy );
extern int32 masc_pushk_int8( struct mas_package* package, char* key, int8 val );
extern int32 masc_pushk_uint8( struct mas_package* package, char* key, uint8 val );
extern int32 masc_pushk_int16( struct mas_package* package, char* key, int16 val );
extern int32 masc_pushk_uint16( struct mas_package* package, char* key, uint16 val );
extern int32 masc_pushk_int32( struct mas_package* package, char* key, int32 val );
extern int32 masc_pushk_uint32( struct mas_package* package, char* key, uint32 val );
extern int32 masc_pushk_float( struct mas_package* package, char* key, float val );
extern int32 masc_pushk_double( struct mas_package* package, char* key, double val );
extern int32 masc_pushk_string( struct mas_package* package, char* key, char* string );
extern int32 masc_pushk_package( struct mas_package* package, char* key, struct mas_package* ppack );
extern int32 masc_pushk_payload( struct mas_package* package, char* key, void* payload, uint32 len );
extern int32 masc_pullk_int8( struct mas_package* package, char* key, int8* retval);
extern int32 masc_pullk_uint8( struct mas_package* package, char* key, uint8* retval );
extern int32 masc_pullk_int16( struct mas_package* package, char* key, int16* retval );
extern int32 masc_pullk_uint16( struct mas_package* package, char* key, uint16* retval );
extern int32 masc_pullk_int32( struct mas_package* package, char* key, int32* retval );
extern int32 masc_pullk_uint32( struct mas_package* package, char* key, uint32* retval );
extern int32 masc_pullk_float( struct mas_package* package, char* key, float* retval );
extern int32 masc_pullk_double( struct mas_package* package, char* key, double* retval );
extern int32 masc_pullk_string( struct mas_package* package, char* key, char** string_retval, int copy );
extern int32 masc_pullk_payload( struct mas_package* package, char* key, void** payload_retval, uint32* len_retval, int copy );
extern int32 masc_pullk_package( struct mas_package* package, char *key, struct mas_package *retpack, int copy );
extern int32 masc_push_members( struct mas_package* package, char* format, ... );
extern int32 masc_pull_members( struct mas_package* package, char* format, ... );
extern int32 masc_pushk_members( struct mas_package* package, char* format, ... );
extern int32 masc_finalize_package( struct mas_package* package );
extern int32 masc_peek_format( struct mas_package* package, char* retval_fmt );
extern int32 masc_peek_key( struct mas_package* package, char** key_ptr );
extern int32 masc_debug_package( struct mas_package* package, int recdep );
extern int32 masc_test_key( struct mas_package* package, char* key );
extern int32 masc_find_key( struct mas_package* package, char* key );
extern int32 masc_setup_package( struct mas_package* package, char *buffer, int size, int pflags );
extern int32 masc_strike_package( struct mas_package* package );
extern int32 masc_test_members( struct mas_package* package, char* format, ... );


#endif /* _MAS_COMMON_H */
