#ifndef _BUFFER_H_
#define _BUFFER_H_

#include <sys/types.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif

typedef struct {
	char *ptr;
	
	size_t used;
	size_t size;
} buffer;

typedef struct {
	buffer **ptr;
	
	size_t used;
	size_t size;
} buffer_array;

typedef struct {
	char *ptr;
	
	size_t offset; /* read-pointer */
	
	size_t used;   /* write-pointer */
	size_t size;
} read_buffer;

buffer_array* buffer_array_init();
void buffer_array_free(buffer_array *b);
buffer *buffer_array_append_get_buffer(buffer_array *b);

buffer* buffer_init();
buffer* buffer_init_string(const char *str);
void buffer_free(buffer *b);
void buffer_reset(buffer *b);
	
int buffer_prepare_copy(buffer *b, size_t size);
int buffer_prepare_append(buffer *b, size_t size);

int buffer_copy_string(buffer *b, const char *s);
int buffer_copy_string_len(buffer *b, const char *s, size_t s_len);
int buffer_copy_string_buffer(buffer *b, const buffer *src);

int buffer_copy_long(buffer *b, long l);

int buffer_copy_memory(buffer *b, const char *s, size_t s_len);

int buffer_append_string(buffer *b, const char *s);
int buffer_append_string_len(buffer *b, const char *s, size_t s_len);
int buffer_append_string_buffer(buffer *b, const buffer *src);
int buffer_append_string_lfill(buffer *b, const char *s, size_t maxlen);
int buffer_append_string_rfill(buffer *b, const char *s, size_t maxlen);

int buffer_append_long(buffer *b, long l);

int buffer_append_memory(buffer *b, const char *s, size_t s_len);

char * buffer_search_string_len(buffer *b, char *needle, size_t len);

int buffer_is_empty(buffer *b);
int buffer_is_equal(buffer *a, buffer *b);

int buffer_append_string_hex(buffer *b, char *in, size_t in_len);
int buffer_append_string_url_encoded(buffer *b, const char *s);
int buffer_append_string_html_encoded(buffer *b, const char *s);

int ultostr(char *s, unsigned long l);
int ultohex(char *s, unsigned long l);
int ltostr(char *s, long l);
char hex2int(unsigned char c);

#define BUFFER_APPEND_STRING_CONST(x, y) \
	buffer_append_string_len(x, y, sizeof(y) - 1)

#define BUFFER_COPY_STRING_CONST(x, y) \
	buffer_copy_string_len(x, y, sizeof(y) - 1)

#define BUFFER_APPEND_SLASH(x) \
	if (x->used > 1 && x->ptr[x->used - 2] != '/') { BUFFER_APPEND_STRING_CONST(x, "/"); }


#define SEGFAULT() *(char *)(0) = 0
#define MAX(a,b) ( (a) > (b) ? (a) : (b) )
#define MIN(a,b) ( (a) < (b) ? (a) : (b) )

#endif
