/*
 * session-scuba.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/rtp/session-scuba.cc,v 1.20 2002/02/03 04:15:47 lim Exp $";
#endif

#include "config.h"
#include <math.h>
#include <errno.h>
#include <string.h>
#include "source.h"
#include "tclcl.h"
#include "media-timer.h"
#include "crypt.h"
#include "timer.h"
#include "ntp-time.h"
#include "session-scuba.h"

struct scuba_hdr {
	u_int32_t sender_id;
	u_int32_t repid;
	u_int32_t nent;
};

struct scuba_entry {
	u_int32_t srcid;
	u_int32_t val;
};

class ScubaSession : public ScubaSessionManager, public TclObject {
public:
	ScubaSession();
	~ScubaSession();
	virtual int command(int argc, const char*const* argv);
protected:
	virtual void recv(ScubaHandler*);
	virtual void build_report(ScubaHandler*);
	virtual void announce(ScubaHandler*);
	ScubaHandler sh_;
	scuba_entry* curent_;
	u_char* pktbuf_;
	SourceManager* sm_;
	int repid_;
};

class RTPGWScubaSession : public ScubaSession {
public:
	RTPGWScubaSession() : curhdr_(0) {}
	virtual int command(int argc, const char*const* argv);
protected:
	virtual void build_report(ScubaHandler*);
	scuba_hdr* curhdr_;
};

static class ScubaSessionClass : public TclClass {
    public:
	ScubaSessionClass() : TclClass("Session/Scuba") {}
	TclObject* create(int, const char*const*) {
		return (new ScubaSession);
	}
} scuba_class;

static class RTPGWScubaSessionClass : public TclClass {
    public:
	RTPGWScubaSessionClass() : TclClass("Session/Scuba/RTPGW") {}
	TclObject* create(int, const char*const*) {
		return (new RTPGWScubaSession);
	}
} rtpgw_scuba_class;

ScubaSession::ScubaSession()
	     :curent_(0), sm_(0), repid_(0)
{
	sh_.manager(this);
	pktbuf_ = new u_char[2 * RTP_MTU];
}

ScubaSession::~ScubaSession()
{
	delete pktbuf_;
}

void ScubaSession::announce(ScubaHandler* sh)
{
	build_report(sh);
}

int ScubaSession::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();

	if (argc == 2) {
		if (strcmp(argv[1], "source-manager") == 0) {
			tcl.resultf("%s", sm_->name());
			return (TCL_OK);
		}
		if (strcmp(argv[1], "max-bandwidth") == 0) {
			tcl.resultf("%lf", sh_.bandwidth() * 8);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "start-control") == 0) {
			sh_.startctrl();
			return(TCL_OK);
		}
	}
	else if (argc == 3) {
		if (strcmp(argv[1], "source-manager") == 0) {
			sm_ = (SourceManager*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "scuba-net") == 0) {
			sh_.net((Network*)TclObject::lookup(argv[2]));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "max-bandwidth") == 0) {
			double bw = atof(argv[2]) / 8.;
			sh_.bandwidth(bw);
			return (TCL_OK);
		}
	} else if (argc == 4) {
		if (strcmp(argv[1], "add-scuba-entry") == 0) {
			curent_->srcid = htonl(strtoul(argv[2], 0, 10));
			curent_->val = htonl(strtoul(argv[3], 0, 10));
			curent_++;
			return (TCL_OK);
		}
	}
	return (TclObject::command(argc, argv));
}

void ScubaSession::recv(ScubaHandler* sh)
{
	u_int32_t src;
	int port;
	int cc = sh->recv(pktbuf_, 2 * RTP_MTU, src, port);
	if (cc <= 0)
		return;

	scuba_hdr* h = (scuba_hdr*)pktbuf_;
	sh->sample_size(cc);
	u_int32_t ssrc = h->sender_id;

	Source* ps = sm_->consult(ssrc);
	if (ps == 0)
		return;

	int nent = ntohl(h->nent);
	Tcl& tcl = Tcl::instance();

	scuba_entry* e = (scuba_entry*)(h + 1);

	tcl.evalf("%s clean_scoretab %s", name(), ps->name());

	for (int i = 0; i < nent; i++) {
		tcl.evalf("%s recv_scuba_entry %s %u %lu", name(), ps->name(),
			  (u_long)ntohl(e->srcid), (u_long)ntohl(e->val));
		e++;
	}
	tcl.evalf("%s set_allocation", name());
}

void ScubaSession::build_report(ScubaHandler* sh)
{
	scuba_hdr* h = (scuba_hdr*)pktbuf_;
	h->sender_id = sm_->localsrc()->srcid();
	h->repid = htonl(repid_);
	++repid_;

	Tcl& tcl = Tcl::instance();
	curent_ = (scuba_entry*)(h + 1);
	tcl.evalf("%s build_report", name());
	int n = atoi(tcl.result());
	h->nent = htonl(n);

	int len = sizeof(scuba_hdr) + n * sizeof(scuba_entry);
	sh->send(pktbuf_, len);
	sh->adapt(len);
	sh->sample_size(len);
}

void RTPGWScubaSession::build_report(ScubaHandler* sh)
{
	Tcl& tcl = Tcl::instance();
	curhdr_ = (scuba_hdr*)pktbuf_;
	curent_ = (scuba_entry*)(curhdr_ + 1);
	tcl.evalf("%s build_report", name());
	int size = atoi(tcl.result());
	sh->adapt(size);
	sh->sample_size(size);
}

int RTPGWScubaSession::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 3) {
		if (strcmp(argv[1], "sender-id") == 0) {
			curhdr_->sender_id = htonl(strtoul(argv[2], 0, 10));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "send-report") == 0) {
			int n = atoi(argv[2]);
			curhdr_->nent = htonl(n);
			int len = sizeof(scuba_hdr) + n * sizeof(scuba_entry);
			sh_.send(pktbuf_, len);
			curhdr_ = (scuba_hdr*)pktbuf_;
			curent_ = (scuba_entry*)(curhdr_ + 1);
			tcl.resultf("%d", len);
			return (TCL_OK);
		}
	}
	return (ScubaSession::command(argc, argv));
}

ScubaHandler::ScubaHandler()
	: bw_(0.), ctrl_inv_bw_(0.),
	  ctrl_avg_size_(128.),
	  manager_(0)
{
}

inline void ScubaHandler::schedule_timer()
{
	msched(int(fmod(double(random()), rint_) + rint_ * .5 + .5));
}

void ScubaHandler::net(Network* n)
{
	DataHandler::net(n);
}

void ScubaHandler::startctrl()
{
	cancel();
	if (net_ != 0) {
		/*
		 * schedule a timer for our first report using half the
		 * min ctrl interval.  This gives us some time before
		 * our first report to learn about other sources so our
		 * next report interval will account for them.  The avg
		 * ctrl size was initialized to 128 bytes which is
		 * conservative.
		 */
		double rint = ctrl_avg_size_ * ctrl_inv_bw_;
		if (rint < SCUBA_MIN_RPT_TIME / 2. * 1000.)
			rint = SCUBA_MIN_RPT_TIME / 2. * 1000.;
		rint_ = rint;
		schedule_timer();
	}
}

void ScubaHandler::sample_size(int cc)
{
	ctrl_avg_size_ += SCUBA_SIZE_GAIN * (double(cc + 28) - ctrl_avg_size_);
}

void ScubaHandler::adapt(int size)
{
	double ibw = ctrl_inv_bw_;
	double rint = size * ibw;
	if (rint < SCUBA_MIN_RPT_TIME * 1000.)
		rint = SCUBA_MIN_RPT_TIME * 1000.;
	rint_ = rint;
}

void ScubaHandler::timeout()
{
	manager_->announce(this);
	schedule_timer();
}

void ScubaHandler::dispatch(int)
{
	manager_->recv(this);
}

