# ers.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import Network UDPChannel

#
# Together with the ERSClient class, implements expanding ring
# search.  An instance of ERSServer or some class derived from
# it listens for queries from clients and responds when a query
# is received.<p>
#
# Derived classes may implement custom ERS protocols (i.e., they
# may piggyback information in ERS packets) by overriding the
# <i>build_reply</i> method in the ERSServer class and the
# <i>build_request</i> and <i>response</i> methods in the
# ERSClient class.
#
Class ERSServer -superclass UDPChannel

#
# Instantiate a new ERS server.  Queries are listened for on
# <i>spec</i> (in the form <i>address</i>/<i>port</i>).
#
ERSServer public init {spec} {
	$self next $spec
}

#
# Invoked by the UDPChannel base class when a packet is received.
# Builds a response packet and sends it to the client.
#
ERSServer private recv {addr port data len} {
	#FIXME first line of data should be udp port
	# to send reply to
	set l [split $data \n]
	set port [lindex $l 0]
	set data [join [lrange $l 1 end] \n]

	set reply [$self build_reply $data]
	$self sendto $addr $port $reply
}

#
# Sends <i>data</i> in a UDP datagram to address <i>addr</i>
# and port <i>port</i>.
#
ERSServer private sendto {addr port data} {
	# FIXME
	set c [new UDPChannel $addr/$port]
	$c send $data
	delete $c
}

#
# Invoked when a request is received from a client.  The payload
# of the packet from the client is in <i>data</i>.  The string
# returned by this method is used as the payload of the response
# packet.<p>
#
# The default implementation in the ERSServer base class simply
# returns the IP address of the server.
#
ERSServer public build_reply {data} {
	$self instvar net_
	return [$net_ interface]
}


#
# Together with the ERSServer class, implements expanding ring
# search.  An instance of ERSClient or some class derived from
# it sends request packets with increasing TTLs until a reply
# is received.
#
Class ERSClient -superclass UDPChannel

#
# Instantiate a new ERSClient object.  Requests are sent to the
# address/port specified in <i>spec</i>.
#
ERSClient public init {spec} {
	$self next
	$self instvar chan_ timer_ delay_ maxttl_

	set chan_ [new UDPChannel $spec]
	set timer_ ""
	set delay_ 200
	set maxttl_ 63
}

#
ERSClient public destroy {} {
	$self instvar chan_ timer_
	delete $chan_
	catch {after cancel $timer_}
	$self next
}

#
# Initiate an expanding ring search.  Custom search protocols
# (i.e., with data piggybacked in ERS packets) can be implementing
# by overriding the <i>build_request</i> and <i>response</i>
# methods.
#
ERSClient public search {} {
	$self instvar net_

	set port [$net_ rport]
	set request [$self build_request]
	set pkt "$port\n$request"

	$self send_request $pkt 1
}

#
# Sends the packet <i>pkt</i> with TTL <i>ttl</i> and schedules
# another transmission with an increased TTL.
#
ERSClient private send_request {pkt ttl} {
	$self instvar chan_

	$self ttl $ttl
	$chan_ send $pkt

	$self instvar maxttl_ timer_ delay_
	incr ttl
	if {$ttl > $maxttl_} {
		$self response ""
		return
	}
	set timer_ [after $delay_ "$self send_request \"$pkt\" $ttl"]
}

#
# Generates the payload of a request packet.  Can be overridden
# in derived classes to implement a custom ERS protcol.  The
# default implementation returns an empty string.
#
ERSClient public build_request {} {
	return ""
}

#
# Invoked by the UDPChannel base class when a packet arrives
# (a response from a server).  Cancels any pending transmissions
# and invokes the <i>response</i> method which indiciates a
# successful expanding ring search.
#
ERSClient private recv {addr data len} {
	$self instvar timer_
	after cancel $timer_

	$self response $data
}

#
# Invoked when an expanding ring search has been completed.  If
# <i>data</i> is an empty string, the search has failed (i.e., no
# server responded).  If <i>data</i> is non-empty, it contains the
# payload of the packet received from the server.
#
ERSClient public response {data} {
	puts "got response \"$data\""
}
