# ui-mediaedit.tcl --
#
#      Constructs and manages a window that is used to edit the list of
#      media streams in an SDP session description.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import PopupMenu
import RTP/Audio
import RTP/Video

# Constructs and manages a window that is used to edit the list of
# media streams in an SDP session description.  This window contains
# a listbox on the left side that lists all streams current in the
# announcement.  Streams can be added and deleted from this list.
# The right side of the window contains a series of entry widgets
# that are used to edit the format/address/port/etc.. of the stream
# that is currently selected in the listbox.
Class SDPMediaEditWindow

# Creates the user interface elements needed for this window.
# The streams in the SDPMessage object <i>msg</i> will be displayed
# initially or defaults will be displayed if <i>msg</i> is an empty
# string.  The ProgramSource <i>src</i> is queried to find out what
# administrative scope zones are available.
SDPMediaEditWindow public init {w msg src} {
    $self set win_ $w
    $self set msg_ $msg
    $self set src_ $src

    $self instvar simple_
    set simple_ [$self yesno simpleInterface]
    if {$simple_} {
	$self set simplewin_ [new SDPMediaEditWindow/Simple $self]
    } else {
	$self set advancedwin_ [new SDPMediaEditWindow/Advanced $self]
    }
}

SDPMediaEditWindow public destroy {} {
    $self instvar simplewin_ advancedwin_ win_

    if [info exists simplewin_] { delete $simplewin_ }
    if [info exists advancedwin_] { delete $advancedwin_ }
    destroy $win_
}

SDPMediaEditWindow public build-simple {} {
    $self instvar simplewin_ advancedwin_ simple_

    if [info exists advancedwin_] {
	$advancedwin_ hide
    }

    if [info exists simplewin_] {
	$simplewin_ show
    } else {
	set simplewin_ [new SDPMediaEditWindow/Simple $self]
    }
    set simple_ 1
}

SDPMediaEditWindow public build-advanced {} {
    $self instvar simplewin_ advancedwin_ simple_

    if [info exists simplewin_] {
	$simplewin_ hide
    }

    if [info exists advancedwin_] {
	$advancedwin_ show
    } else {
	set advancedwin_ [new SDPMediaEditWindow/Advanced $self]
    }
    set simple_ 0
}

SDPMediaEditWindow public buildmsgs {} {
    $self instvar simple_ simplewin_ advancedwin_
    if $simple_ {
	return [list [$simplewin_ buildmsg]]
    } else {
	return [$advancedwin_ buildmsgs]
    }
}

#
SDPMediaEditWindow proc is-number {n} {
    if [catch {expr $n}] {
	return 0
    }
    return 1
}

#
SDPMediaEditWindow proc valid-addr {a} {
    set l [split $a .]
    if {[llength $l] != 4} { return 1 }
    foreach i $l {
	if {![$self is-number $i] || $i<0 || $i>255} { return 1 }
    }
    return 0
}

#
SDPMediaEditWindow proc valid-port {p} {
    if {![$self is-number $p] || $p<0 || $p>65535} { return 1}
    return 0
}


#
#
#
Class SDPMediaEditWindow/Advanced

SDPMediaEditWindow/Advanced public init {parent} {
    $self instvar win_ parent_ msg_ src_

    set win_ "[$parent set win_].advanced"
    frame $win_

    set parent_ $parent
    set msg_ [$parent set msg_]
    set src_ [$parent set src_]

    $self buildwin
    $self show
}

SDPMediaEditWindow/Advanced public hide {} {
    $self instvar win_
    pack forget $win_
}

SDPMediaEditWindow/Advanced public show {} {
    $self instvar win_
    pack $win_ -fill both -expand yes
}

SDPMediaEditWindow/Advanced private buildwin {} {
    $self instvar win_ msg_ src_ scopes_
    $self instvar typemenu_ formatmenu_ protomenu_ attrentries_

    set scopes_ [$src_ scopes]

    $self set streams_ {}
    $self set current_ -1

    set font [$self get_option smallfont]

    frame $win_.list
    label $win_.list.l -text "Streams"
    set h [expr ([winfo reqheight $win_.list.l]/2) + 2]
    frame $win_.list.pad -height $h

    set f $win_.list.f
    frame $f -relief groove -bd 2
    frame $f.pad -height $h
    pack $f.pad -side top

    frame $f.top
    $self instvar lb_
    set lb_ [listbox $f.top.lb -width 15 -height 5 \
		 -exportselection no -selectmode single \
		 -yscrollcommand "$f.top.sb set"]
    bind $f.top.lb <Button-1> "$self streamclick %x %y"
    pack $f.top.lb -side right -fill both -expand yes
    scrollbar $f.top.sb -width 10 -command "$f.top.lb yview"
    pack $f.top.sb -side right -before $f.top.lb -fill y
    pack $f.top -side top -fill both -expand yes

    frame $f.b
    button $f.b.add -text "Add" -command "$self add-stream"
    button $f.b.del -text "Delete" -command "$self delete-stream"
    pack $f.b.add $f.b.del -side left
    pack $f.b

    pack $win_.list.pad -side top
    pack $f -fill both -expand yes -padx 4 -pady 0
    place $win_.list.l -x 6 -y 2
    raise $win_.list.l
    pack $win_.list -side left -fill both -expand no

    frame $win_.prop
    label $win_.prop.l -text "Properties"
    set h [expr ([winfo reqheight $win_.prop.l])/2 + 2]
    frame $win_.prop.pad -height $h

    frame $win_.prop.f -relief groove -bd 2
    frame $win_.prop.f.pad -height $h
    pack $win_.prop.f.pad -side top
    set f $win_.prop.f.f
    frame $f

    frame $f.left

    label $f.left.typet -text "Type:" -anchor e
    grid $f.left.typet -row 0 -column 0 -sticky ew
    set typemenu_ [new PopupMenu $f.left.typem]
    foreach type [UserApplication media] {
	$typemenu_ add $type "$self set-type $type"
    }
    grid $f.left.typem -row 0 -column 1 -sticky ew

    label $f.left.formatt -text "Format:" -anchor e
    grid $f.left.formatt -row 1 -column 0 -sticky ew
    set formatmenu_ [new PopupMenu $f.left.formatm]
    # menu gets filled in in set-type
    grid $f.left.formatm -row 1 -column 1 -sticky ew

    label $f.left.protot -text "Protocol:" -anchor e
    grid $f.left.protot -row 2 -column 0 -sticky ew
    set protomenu_ [new PopupMenu $f.left.protom]
    # filled in in set-type
    grid $f.left.protom -row 2 -column 1 -sticky ew

    # attributes created in set-type
    set attrentries_ {}

    pack $f.left -side left -fill y -expand yes

    frame $f.right

    $self instvar layeredcb_
    set layeredcb_ [new CheckButton $f.right.cb -text "Use Layering" \
			-command "$self toggle-layering"]
    pack $f.right.cb -side top -fill x

    frame $f.right.addr
    $self set addrframe_ $f.right.addr
    $self set addrblocks_ {}
    pack $f.right.addr -side top -fill both -expand yes

    pack $f.right -fill both -expand yes

    pack $win_.prop.pad -side top -fill both
    pack $win_.prop.f -fill both -expand yes
    pack $f -side left -fill both -expand yes -padx 4 -pady 0
    place $win_.prop.l -x 6 -y 2
    raise $win_.prop.l
    pack $win_.prop -side left -fill both -expand yes
    pack propagate $win_.prop no

    pack $win_ -fill both -expand yes -pady 2

    if {$msg_ == ""} {
	$self add-stream
    } else {
	foreach m [[$msg_ base] set allmedia_] {
	    $self add-stream $m
	}
    }

    $self set layered_ 0
    $self select-stream 0
    $lb_ selection set 0
}

# Callback when the mouse is clicked in the listbox containing
# all streams.  Saves the state of the current stream and selects
# a new stream.
SDPMediaEditWindow/Advanced private streamclick {x y} {
    $self instvar current_ lb_

    if {$current_ >= 0} {
	$self save-stream
    }

    set i [$lb_ nearest $y]
    $self select-stream $i
}

#
SDPMediaEditWindow/Advanced private save-stream {} {
    $self instvar typemenu_ formatmenu_ protomenu_ \
	addrblocks_ attrentries_ streams_ current_

    set media [$typemenu_ get]
    set fmt [$formatmenu_ get]
    set proto [$protomenu_ get]

    set addrs {}
    foreach b $addrblocks_ {
	lappend addrs [$b get]
    }

    set attrs {}
    foreach e $attrentries_ {
	lappend attrs [$e entry-value]
    }

    if {$proto=={RTP/AVP}} {
      set lfmt [string tolower $fmt]
      if {$media=={video}} {
         set v [new RTP/Video]
         set fmt [$v rtp_fmt_number $lfmt]
         delete $v
      } elseif {$media=={audio}} {
         set a [new RTP/Audio]
         set fmt [$a rtp_fmt_number $lfmt]
         delete $a
      } else {
         puts "nsdr: unknown format named $fmt"
      }
    }

    set stream [list $media $fmt $proto $addrs $attrs]
    set streams_ [lreplace $streams_ $current_ $current_ $stream]
}

# Updates the entry widgets on the right side of the window when a
# new stream is selected in the listbox.
SDPMediaEditWindow/Advanced private select-stream {i {layered -1}} {
    $self set current_ $i

    #fill in the properties for this stream
    $self instvar current_ streams_ typemenu_ formatmenu_ \
	protomenu_ layeredcb_ addrblocks_ attrentries_ layered_
    set stream [lindex $streams_ $i]
    set type [lindex $stream 0]
    set fmt [lindex $stream 1]
    set proto [lindex $stream 2]
    set addrs [lindex $stream 3]
    set attrs [lindex $stream 4]

    $self set-type $type
    set i [lsearch -exact [UserApplication media] $type]
    $typemenu_ select $i

    set fmts [UserApplication formats $type]
    if {($proto == {RTP/AVP}) && ([string is digit $fmt])} {
        if {$type == {video}} {
            set fmt [string toupper [RTP/Video set default_ptoa_($fmt)]]
        } elseif {$type == {audio}} {
            set fmt [string toupper [RTP/Audio set default_ptoa_($fmt)]]
        } else {
            puts "nsdr: unknown format number $fmt"
        }
    }
    set i [lsearch -exact $fmts $fmt]
    $formatmenu_ select $i

    set protos [UserApplication protos $type]
    set i [lsearch -exact $protos $proto]
    $protomenu_ select $i

    if {$layered == -1} {
	set layered 0
	if {[llength $addrs] > 1 || [lindex [lindex $addrs 0] 2] > 1} {
	    set layered 1
	}
    }

    # kill old addrblocks
    foreach b $addrblocks_ {
	delete $b
    }

    # make sure we have the right ui components
    if {$layered_ == 0 && $layered == 1} {
	# not previously layered, now layered
	$self build-layers

	set layered_ 1
	$layeredcb_ set-val 1
    } elseif {$layered_ == 1 && $layered == 0} {
	# previously layered, not any more
	$self destroy-layers

	set layered_ 0
	$layeredcb_ set-val 0
    }

    # now set the addresses
    if $layered {
	foreach a $addrs {
	    set b [$self add-addrblock]
	    $b set $a
	}
    } else {
	$self instvar addrframe_
	set addrblocks_ [new AddrBlockWindow $addrframe_ 0 $self]
	$addrblocks_ set-type [lindex $stream 0]
	    $addrblocks_ set [lindex $addrs 0]
    }

    set i [llength $attrs]
    while {$i>0} {
	incr i -1
	set e [lindex $attrentries_ $i]
	set v [lindex $attrs $i]
	$e set-value $v
    }
}

# Called when the Add button is pressed, adds a new stream to
# the list of exisitng streams.
SDPMediaEditWindow/Advanced private add-stream {{m ""}} {
    $self instvar streams_ lb_

    if {$m == ""} {
	set media [lindex [UserApplication media] 0]
	set fmt [lindex [UserApplication formats $media] 0]
	set proto [lindex [UserApplication protos $media] 0]
	set attrs [UserApplication attrs $media]
	set addr [list [$self alloc-addrs 1] [$self alloc-port $media] 1]
	set addrs [list $addr]
	set new [list $media $fmt $proto $addrs $attrs]
    } else {
	set media [$m set mediatype_]
	set fmt [$m set fmt_]
	set proto [$m set proto_]

	#FIXME
	set addr [list [$m set caddr_] [$m set port_] ]
	set addrs [list $addr]

	#FIXME
	set attrs ""
	set new [list $media $fmt $proto $addrs $attrs]
    }
    lappend streams_ $new

    $lb_ insert end $media
}

# Deletes the currently selected stream from this announcement.
SDPMediaEditWindow/Advanced private delete-stream {} {
    $self instvar streams_ lb_ current_
    set streams_ [lreplace $streams_ $current_ $current_]
    $lb_ delete $current_

    set end [expr [$lb_ index end] -1]
    if {$current_ > $end} {
	set new $end
    } else {
	set new $current_
    }
    $self select-stream $new
    $lb_ selection set $new
}

Class AddrBlockWindow

AddrBlockWindow public init {w layered parent} {
    $self next

    $self instvar win_ layered_ parent_ scope_
    $self instvar scopemenu_ addrentry_ portentry_ layerentry_

    set win_ $w
    set layered_ $layered
    set parent_ $parent

    label $w.sl -text "Address Scope:" -anchor e
    grid $w.sl -row 0 -column 0 -sticky ew

    set scopemenu_ [new PopupMenu $w.sm]
    $w.sm config -width 16 -anchor w
    grid $w.sm -row 0 -column 1 -sticky e

    set i 0
    foreach s [$parent_ set scopes_] {
	$scopemenu_ add [$s name] "$self choose-scope $i"
	incr i
    }
    $scopemenu_ select 0
    set scope_ 0

    label $w.al -text "Address:" -anchor e
    grid $w.al -row 1 -column 0 -sticky ew

    set addrentry_ [new TextEntry "SDPMediaEditWindow valid-addr" $w.ae ""]
    $w.ae config -width 16
    grid $w.ae -row 1 -column 1 -sticky ew

    label $w.pl -text "Port:" -anchor e
    grid $w.pl -row 2 -column 0 -sticky ew

    set portentry_ [new TextEntry "SDPMediaEditWindow valid-port" $w.pe ""]
    $w.pe config -width 8
    grid $w.pe -row 2 -column 1 -sticky ew

    set row 3
    if {$layered} {
	label $w.ll -text "Layers:" -anchor e
	grid $w.ll -row 3 -column 0 -sticky ew

	set layerentry_ [new TextEntry "SDPMediaEditWindow valid-port" \
			     $w.le "1"]
	$w.le config -width 8
	grid $w.le -row 3 -column 1 -sticky ew
	incr row
    }

    grid rowconfigure $w $row -weight 1

    #FIXME alloc default address/port
}

AddrBlockWindow public destroy {} {
    $self instvar scopemenu_ addrentry_ portentry_ layerentry_ win_
    delete $addrentry_
    delete $portentry_
    catch {delete $layerentry_}

    foreach w [winfo children $win_] { destroy $w }
}

AddrBlockWindow private choose-scope {i} {
    $self instvar scope_ addrentry_ layerentry_ parent_
    if {$i == $scope_} { return }

    set scope_ $i

    # reallocate address
    set l 1
    if [info exists layerentry_] {
	set l [$layerentry_ entry-value]
    }
    set addr [$parent_ alloc-addrs $l [lindex [$parent_ set scopes_] $i]]

    $addrentry_ set-value $addr
}

AddrBlockWindow public set-type {type} {
    $self instvar portentry_ parent_

    set p [$parent_ alloc-port $type]
    $portentry_ set-value $p
}

AddrBlockWindow public get {} {
    $self instvar addrentry_ portentry_ layerentry_

    set addr [$addrentry_ entry-value]
    set port [$portentry_ entry-value]
    set layers 1
    if [info exists layerentry_] {
	set layers [$layerentry_ entry-value]
    }
    return [list $addr $port $layers]
}

AddrBlockWindow public set {l} {
    $self instvar addrentry_ portentry_ layerentry_

    $addrentry_ set-value [lindex $l 0]
    $portentry_ set-value [lindex $l 1]

    if [info exists layerentry_] {
	$layerentry_ set-value [lindex $l 2]
    }
}

SDPMediaEditWindow/Advanced private build-layers {} {
    $self instvar addrframe_ addrcanvas_ addrblocks_

    set f $addrframe_.top
    frame $f

    button $addrframe_.add -text "Add Block" -command "$self add-addrblock"
    pack $addrframe_.add -side bottom

    set addrcanvas_ $f.c
    set sb $f.sb
    canvas $addrcanvas_ -yscrollcommand "$sb set"
    scrollbar $sb -relief groove -bd 2 -width 10 \
	-command "$addrcanvas_ yview"
    pack $addrcanvas_ -side right -fill y
    set addrblocks_ {}

    pack $sb -side right -before $addrcanvas_ -fill y
    pack $f -side bottom  -fill both -expand yes
}

SDPMediaEditWindow/Advanced private destroy-layers {} {
    $self instvar addrframe_
    destroy $addrframe_.top $addrframe_.add
}

SDPMediaEditWindow/Advanced private toggle-layering {} {
    $self instvar streams_ current_ layeredcb_

    $self save-stream
    set stream [lindex $streams_ $current_]
    set addrs [lindex $stream 3]

    set stream [lreplace $stream 3 3 $addrs]
    set streams_ [lreplace $streams_ $current_ $current_ $stream]
    $self select-stream $current_ [$layeredcb_ get-val]
}

SDPMediaEditWindow/Advanced private add-addrblock {} {
    $self instvar addrcanvas_ numblocks_ addrblocks_ type_

    if ![info exists numblocks_] {
	set numblocks_ 0
	set y 0
    } else {
	set bb [$addrcanvas_ bbox all]
	set y [expr [lindex $bb 3] + 4]
    }

    set w $addrcanvas_.addr$numblocks_
    frame $w -relief groove -bd 2

    $addrcanvas_ create window 2 $y -window $w -anchor nw
    set b [new AddrBlockWindow $w 1 $self]
    $b set-type $type_
    lappend addrblocks_ $b

    incr numblocks_

    update
    set bb [$addrcanvas_ bbox all]
    $addrcanvas_ config -width [lindex $bb 2]
    $addrcanvas_ config -scrollregion "0 0 2.5i [lindex $bb 3]"

    return $b
}

# Called when a selection is made in the media type menu.  Updates
# the formats menu so that the choices are appropriate for the new
# media type (for example, if the media type is set to audio, the
# format menu should contain audio formats and not video formats...)
SDPMediaEditWindow/Advanced private set-type {type} {
    $self instvar type_ formatmenu_ protomenu_ win_ \
	current_ lb_ addrblocks_ attrentries_

    set type_ $type

    # update entry in listbox
    if {$current_ == [$lb_ index end]} {
	set current_ end
    }
    $lb_ delete $current_
    $lb_ insert $current_ $type
    $lb_ selection set $current_
    set current_ [$lb_ index $current_]

    # update format menu for new type
    $formatmenu_ clear
    set fmts [UserApplication formats $type]
    foreach f $fmts {
	$formatmenu_ add $f
    }
    $formatmenu_ select 0

    # update protocol menu
    $protomenu_ clear
    foreach proto [UserApplication protos $type] {
	$protomenu_ add $proto
    }
    $protomenu_ select 0

    # reallocate ports
    foreach b $addrblocks_ {
	$b set-type $type
    }

    # attributes
    set f $win_.prop.f.f.left

    # kill old attribute entries
    foreach e $attrentries_ {
	delete $e
    }
    set row 3
    while {[winfo exists $f.l$row]} {
	destroy $f.l$row
	catch {destroy $f.e$row} ;#FIXME
	incr row
    }

    # create appropriate attribute entries
    set attrentries_ {}
    set row 3
    set attrs [UserApplication attrs $type]
    foreach attr $attrs {
	if {[llength $attr] == 1} {
	    set label $attr
	    set value ""
	} else {
	    set label [lindex $attr 0]
	    set value [lindex $attr 1]
	}
	append label :
	label $f.l$row -text $label -anchor e
	grid $f.l$row -row $row -column 0 -sticky ew
	lappend attrentries_ [new TextEntry "" $f.e$row $value]
	$f.e$row config -width 10
	grid $f.e$row -row $row -column 1 -sticky ew
	incr row
    }

    grid rowconfigure $f $row -weight 1
}

#
SDPMediaEditWindow/Advanced private alloc-addrs {n {scope ""}} {
    if {$scope == ""} {
	$self instvar scopes_
	set scope [lindex $scopes_ 0]
    }

    return [[AddressAllocator instance] alloc $scope $n]
}

#
SDPMediaEditWindow/Advanced private alloc-port {media} {
    switch $media {
	audio { set base 0x4000 }
	whiteboard { set base 0x8000 }
	video - default { set base 0xc000 }
    }
    set port [expr ([random]&0x3ffe) + $base]
    return $port
}

#
SDPMediaEditWindow/Advanced private find-scope {addr} {
    $self instvar scopes_
    foreach s $scopes_ {

	if [$s contains $addr] {
	    return $s
	}
    }
    return ""
}

#
SDPMediaEditWindow/Advanced public buildmsgs {} {
    $self save-stream

    # first, find all scopes
    set scopes {}
    $self instvar streams_
    foreach s $streams_ {
	foreach a [lindex $s 3] {

	    set scope [$self find-scope [lindex $a 0]]
	    if {$scope == ""} {
		#FIXME

		$self warn "Can't find scope for address $a"
		return {}
	    }

	    if { [lsearch -exact $scopes $scope] < 0 } {
		lappend scopes $scope
	    }
	}
    }

    # find the "base" scope -- the one that comes first in
    # scopes_ for which we have layers -- and move it to the
    # front of the list
    $self instvar scopes_
    foreach s $scopes_ {
	set i [lsearch -exact $scopes $s]
	if {$i == 0} {
	    break
	} elseif {$i>0} {
	    set scopes [concat $s [lreplace $scopes $i $i]]
	    break
	}
    }

    # build a media message fragment for each scope in which we
    # have layers
    set msgs {}
    foreach scope $scopes {
	set text ""
	foreach stream $streams_ {
	    set media [lindex $stream 0]
	    set fmt [lindex $stream 1]
	    set proto [lindex $stream 2]
	    set addrs [lindex $stream 3]
	    set attrs [lindex $stream 4]

	    set layer 0
	    foreach a $addrs {
		set addr [lindex $a 0]
		set port [lindex $a 1]
		set layers [lindex $a 2]

		if [$scope contains $addr] {

		    #FIXME
		    set ttl 127

		    append text "m=$media $port $proto $fmt\n"
		    append text "c=IN IP4 $addr/$ttl/$layers\n"
		    if {$layers > 1 || [llength $addrs] > 1} {
			if {$layers == 1} {
			    append text "a=layers:$layer"
			} else {
			    set last [expr $layer + $layers - 1]
			    append text "a=layers:$layer-$last\n"
			}
		    }

		    set i 0
		    set names [UserApplication attrs $media]
		    foreach v $attrs {
			if {$v != ""} {
			    set name [lindex $names $i]
			    if {[llength $name] > 1} {
				set name [lindex $name 0]
			    }
			    append text "a=$name:$v\n"
			    incr i
			}
		    }
		}
		incr layer $layers
	    }
	}
	lappend msgs $text
    }

    return $msgs
}


#
# FIXME needs lots of work
#
Class SDPMediaEditWindow/Simple

SDPMediaEditWindow/Simple public init {parent} {
    $self instvar win_ msg_ src_

    set win_ "[$parent set win_].simple"
    frame $win_

    set msg_ [$parent set msg_]
    set src_ [$parent set src_]

    $self buildwin
    $self show
}

SDPMediaEditWindow/Simple public hide {} {
    $self instvar win_
    pack forget $win_
}

SDPMediaEditWindow/Simple public show {} {
    $self instvar win_
    pack $win_ -fill both -expand yes
}

SDPMediaEditWindow/Simple private buildwin {} {
    $self instvar win_ msg_ media_ buttons_ menus_ addrs_ ports_

    set buttons_ {}
    set menus_ {}
    set addrs_ {}
    set ports_ {}

    label $win_.lt -text "Media"
    grid $win_.lt -row 0 -column 0 -columnspan 2 -sticky ew

    label $win_.ft -text "Format"
    grid $win_.ft -row 0 -column 2 -sticky ew

    label $win_.addrt -text "Address"
    grid $win_.addrt -row 0 -column 3 -sticky ew

    label $win_.portt -text "Port"
    grid $win_.portt -row 0 -column 4 -sticky ew

    set media_ [UserApplication media]
    set i 1
    foreach type $media_ {
	if {$msg_ == ""} {
	    set present 1
	    set addr [$self alloc-addr]
	    set port [$self alloc-port $type]
	} else {
	    #FIXME set defaults
	}

	set button [new CheckButton $win_.c$i]
	$button set-val $present
	lappend buttons_ $button
	grid $win_.c$i -row $i -column 0 -padx 2 -pady 2

	label $win_.l$i -text $type -anchor w
	grid $win_.l$i -row $i -column 1 -sticky ew

	set menu [new PopupMenu $win_.f$i]
	lappend menus_ $menu
	foreach fmt [UserApplication formats $type] {
	    $menu add $fmt
	}
	grid $win_.f$i -row $i -column 2 -sticky ew

	set entry [new TextEntry "$self valid-addr" $win_.addr$i ""]
	$win_.addr$i configure -width 16
	$win_.addr$i insert 0 $addr
	lappend addrs_ $entry
	grid $win_.addr$i -row $i -column 3 -sticky ew

	set entry [new TextEntry "$self valid-port" $win_.port$i ""]
	$win_.port$i configure -width 8
	$win_.port$i insert 0 $port
	lappend ports_ $entry
	grid $win_.port$i -row $i -column 4 -sticky ew

	incr i
    }

    #FIXME scoping stuff
    message $win_.m \
	-text "WARNING: The simple interface really doesn't work yet."
    grid $win_.m -row $i -column 0 -columnspan 5 -sticky news

    update
    $win_.m configure -width [winfo width $win_.m]
}

#FIXME
SDPMediaEditWindow/Simple private alloc-addr {} { return "224.2.3.4" }
SDPMediaEditWindow/Simple private alloc-port {media} { return "1234" }
SDPMediaEditWindow/Simple public buildobjs {} { return {} }
