# ui-list.tcl --
#
#       Defines a listbox widget
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

# Similar to the Tk listbox widget.
Class VEListbox

# Instantiate a new listbox widget.  The window <i>w</i> should already
# exists and be otherwise empty (this is different from the regular Tk
# convention for widget creation).

VEListbox public init {w} {
    $self next
    $self instvar canvas_ sb_
    set canvas_ $w.c
    set sb_ $w.s

    $self set ids_ {}

    canvas $canvas_ -relief groove -bd 0 -yscrollcommand "$sb_ set"\
	    -height 200 -width 200 
    scrollbar $sb_ -relief groove -bd 2 -command "$canvas_ yview"

    # don't pack scrollbar until needed
    pack $canvas_ -side right -fill both -expand yes

    bind $w <Configure> "$self fix-scrollbar 1"
}

# Map the scrollbar if it is needed, unmap it if it is not needed.

VEListbox private fix-scrollbar {{update 0}} {
    $self instvar canvas_ sb_
    if {$update != 0} { update }
    set yv [$canvas_ yview]
    if {[lindex $yv 0] != 0 || [lindex $yv 1] != 1} {
	pack $sb_ -side right -before $canvas_ -fill y
    } else {
	pack forget $sb_
    }
}

# Like the insert method on a regular Tk listbox; inserts an item
# with the text <i>item</i> right before the <i>i</i>th entry or at
# the end if <i>i</i> is the string "end".  Arranges for the Tcl
# command <i>callback</i> to be invoked when this item is clicked.

VEListbox public insert {i item callback} {
    $self instvar ids_ canvas_ bottom_
    set l [llength $ids_]
    if {$i >= $l } { set i "end" }
    # find the top coordinate for this item
    if {$i == 0} {
	set top 2
    } else {
	if {$i == "end"} {
	    set last [lindex $ids_ "end"]
	} else {
	    set last [lindex $ids_ [expr $i-1]]
	}
	set top [expr [lindex [$canvas_ bbox $last] 3] + 2]
    }

    # create the new item (for now it overlaps with others)
    set id [$canvas_ create text 5 $top -text $item -anchor nw]
    set bb [$canvas_ bbox $id]
    set height [expr [lindex $bb 3] - [lindex $bb 1] + 2]

    # insert the new item in to ids_ and move everything down
    set ids_ [linsert $ids_ $i $id]
    if {$i != "end"} {
	incr i
	set l [llength $ids_]
	while {$i < $l} {
	    $canvas_ move [lindex $ids_ $i] 0 $height
	    incr i
	}
    }
    # set up bindings
    $canvas_ bind $id <Enter> "$canvas_ itemconfigure $id -fill \#ff3030"
    $canvas_ bind $id <Leave> "$canvas_ itemconfigure $id -fill black"
    $canvas_ bind $id <Button-1> $callback

    set bottom [lindex [$canvas_ bbox [lindex $ids_ end]] 3]

    $canvas_ config -scrollregion "0 0 2.5i $bottom"
    $self fix-scrollbar
}

# Like the delete method on a regular Tk listbox; removes the <i>i</i>th
# entry or the last one if <i>i</i> is the string "end".

VEListbox public delete {i} {
    $self instvar ids_ canvas_
    set id [lindex $ids_ $i]
    set ids_ [lreplace $ids_ $i $i]
    set bb [$canvas_ bbox $id]
    set height [expr [lindex $bb 3] - [lindex $bb 1] + 2]
    $canvas_ delete $id

    set l [llength $ids_]
    while {$i < $l} {
	$canvas_ move [lindex $ids_ $i] 0 -$height
	incr i    
    }
    
    set bottom [lindex [$canvas_ bbox [lindex $ids_ end]] 3]

    if {$ids_ == {}} {
	set bottom 0
    }

    $canvas_ config -scrollregion "0 0 2.5i $bottom"
    $self fix-scrollbar
}



