/*
 * tg-hash.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#include <Pilot.h>
#include "tg-hash.h"


/*
 * assumption: hash table size will never grow to more than 16M entries
 * which is WAY more than the Pilot's memory size today
 */
static DWord primes[] = {
	3,
	7,
	13,
	31,
	61,
	127,
	251,
	509,
	1021,
	2039,
	4093,
	8191,
	16381,
	32749,
	65521,
	131071,
	262139,
	524287,
	1048573,
	2097143,
	4194301,
	8388593,
	16777213
};


struct TG_HashChunkHeader {
	Word entriesInThisChunk;
	Word deletedInThisChunk;
	Word maxInThisChunk;
	ULong nextChunkId;
};


struct TG_HashChunkInfo {
	VoidHand handle;
	TG_HashChunkHeader header;
};


#define HASH_SMALL_HASH_TABLE     16
//#define HASH_ENTRIES_PER_CHUNK    4096
#define OFFSET(cls,field) (Word)(&((cls*)0)->field)



/*
 * Initialize a new or pre-existing hash table
 * Side-effect: marks all records associated with the hash table as busy
 *              the programmer MUST call Close() to release all records
 */
void
TG_HashTable::Open(DmOpenRef db, Word sizeofKey, ULong firstChunk)
{
	UInt index;
	TG_HashChunkHeader *header;

	db_ = db;
	deletedEntries_ = 0;
	numEntries_  = 0;
	maxEntries_  = 0;
	numChunks_   = 0;
	maxChunks_   = 2;
	ErrFatalDisplayIf(sizeofKey%4,
			  "Hash table keys must be multiples of 4");
	sizeofKey_   = sizeofKey;
	chunks_ = new TG_HashChunkInfo[maxChunks_];
	firstChunkId_ = firstChunk;

	while (firstChunk!=0xFFFFFFFF) {
		if (numChunks_ >= maxChunks_) {
			TG_HashChunkInfo *newChunks;
			newChunks = new TG_HashChunkInfo[maxChunks_+2];
			MemMove(newChunks, chunks_,
				sizeof(TG_HashChunkInfo)*maxChunks_);
			delete [] chunks_;
			chunks_ = newChunks;
			maxChunks_ += 2;
		}

		DmFindRecordByID(db, firstChunk, &index);
		chunks_[numChunks_].handle = DmGetRecord(db, index);
		header = (TG_HashChunkHeader*)
			MemHandleLock(chunks_[numChunks_].handle);
		numEntries_ += header->entriesInThisChunk;
		maxEntries_ += header->maxInThisChunk;
		deletedEntries_ += header->deletedInThisChunk;
		firstChunk   = header->nextChunkId;
		chunks_[numChunks_].header = *header;
		MemHandleUnlock(chunks_[numChunks_].handle);
		numChunks_++;
	}

	if (numChunks_==0) {
		// this is a brand new hash table; we should create a
		// record for it

		firstChunkId_ = CreateNewChunk(HASH_SMALL_HASH_TABLE);
	}

	DWord i;
	primesArrayIndex_ = 0;
	for (i=4; i<maxEntries_; i*=2)
		primesArrayIndex_++;
}


ULong
TG_HashTable::CreateNewChunk(Word maxEntries)
{
	UInt index=0xFFFF;
	VoidPtr pointer;
	ULong id;

	if (numChunks_ >= maxChunks_) {
		TG_HashChunkInfo *newChunks;
		newChunks = new TG_HashChunkInfo[maxChunks_+2];
		if (maxChunks_>0) {
			MemMove(newChunks, chunks_,
				sizeof(TG_HashChunkInfo)*maxChunks_);
			delete [] chunks_;
		}
		chunks_ = newChunks;
		maxChunks_ += 2;
	}

	chunks_[numChunks_].handle =
		DmNewRecord(db_, &index,
			    sizeof(TG_HashChunkHeader) +
			    maxEntries*SizeofHashEntry());
	if (chunks_[numChunks_].handle == 0) return 0xFFFFFFFF;

	DmRecordInfo(db_, index, NULL, &id, NULL);
	if (numChunks_ > 0) {
		pointer = MemHandleLock(chunks_[numChunks_-1].handle);
		DmWrite(pointer, OFFSET(TG_HashChunkHeader, nextChunkId),
			&id, sizeof(ULong));
		chunks_[numChunks_-1].header.nextChunkId = id;
		MemHandleUnlock(chunks_[numChunks_-1].handle);
	}

	pointer = MemHandleLock(chunks_[numChunks_].handle);
	chunks_[numChunks_].header.entriesInThisChunk = 0;
	chunks_[numChunks_].header.deletedInThisChunk = 0;
	chunks_[numChunks_].header.maxInThisChunk = maxEntries;
	chunks_[numChunks_].header.nextChunkId = 0xFFFFFFFF;
	DmWrite(pointer, 0, &chunks_[numChunks_].header,
		sizeof(TG_HashChunkHeader));

	DmSet(pointer,sizeof(TG_HashChunkHeader),SizeofHashEntry()*maxEntries,
	      0xFF);
	/*Word size = SizeofHashEntry(), i;
	Byte empty[size];
	for (i=0; i<size; i++) empty[i] = 0xFF;
	for (i=0; i<maxEntries; i++) {
		DmWrite(pointer, sizeof(TG_HashChunkHeader) +
			i * size, &empty, size);
	}*/

	MemHandleUnlock(chunks_[numChunks_].handle);
	maxEntries_ += chunks_[numChunks_].header.maxInThisChunk;
	numChunks_++;

	return id;
}


ULong
TG_HashTable::Close(Boolean deleteAllEntries)
{
	Word i=0;
	UInt index;
	ULong retval=firstChunkId_;

	for (i=0; i<numChunks_; i++) {
		DmFindRecordByID(db_, firstChunkId_, &index);
		DmReleaseRecord(db_, index, true);
		if (deleteAllEntries==true) DmDeleteRecord(db_, index);
		firstChunkId_ = chunks_[i].header.nextChunkId;
	}
	return ((deleteAllEntries==true) ? 0xFFFFFFFF:retval);
}


void
TG_HashTable::Insert(void *key, TG_Pointer *ptr)
{
	if (numEntries_ >= (maxEntries_/4)*3) {
		// we need to grow the hash table
		Grow();
	}

	DWord h1, h2;
	Word idx, prevIdx=0xFFFF, offset;
	Byte *bucket;
	VoidPtr pointer=NULL;
	Hash(key, h1, h2);

	while (1) {
		// since the hash table is at most 75% full, we will always
		// find an empty slot and this loop will exit

		LocateChunk(h1, idx, offset);
		if (prevIdx!=idx) {
			if (prevIdx!=0xFFFF)
				MemHandleUnlock(chunks_[prevIdx].handle);
			pointer = MemHandleLock(chunks_[idx].handle);
		}

		bucket = (Byte*) (pointer + offset);
		if (IsPtrEmpty(bucket)) {
			// found an empty/deleted slot
			Boolean isDeleted = !IsKeyEmpty(bucket);

			// write the entry
			DmWrite(pointer, offset, key, sizeofKey_);
			DmWrite(pointer, offset+sizeofKey_,
				ptr, sizeof(TG_Pointer));
			// update counters
			if (isDeleted) {
				deletedEntries_--;
				chunks_[idx].header.deletedInThisChunk--;
			} else {
				numEntries_++;
				chunks_[idx].header.entriesInThisChunk++;
			}
			DmWrite(pointer, 0, &chunks_[idx].header,
				sizeof(TG_HashChunkHeader));
			MemHandleUnlock(chunks_[idx].handle);
			return;
		}

		h1 = (h1 + h2) % maxEntries_;
		prevIdx = idx;
	}
}


Boolean
TG_HashTable::FindUpdateDelete(void *key, TG_Pointer *ptr,
				 FindUpdateDeleteType what)
{
	DWord h1, h2;
	Word idx, prevIdx=0xFFFF, offset;
	Byte *bucket;
	Hash(key, h1, h2);
	VoidPtr pointer=NULL;

	while (1) {
		// since the hash table is at most 75% full, we will always
		// find either an empty slot, or the entry and this loop
		// will exit

		LocateChunk(h1, idx, offset);
		if (prevIdx!=idx) {
			if (prevIdx!=0xFFFF)
				MemHandleUnlock(chunks_[prevIdx].handle);
			pointer = MemHandleLock(chunks_[idx].handle);
		}

		bucket = (Byte*) (pointer + offset);
		if (IsKeyEmpty(bucket)) {
			// found a totally empty slot
			MemHandleUnlock(chunks_[idx].handle);
			return false;
		}
		if (Match(bucket, key) && !IsPtrEmpty(bucket)) {
			// found the entry
			switch(what) {
			case htFind:
			case htDelete:
				if (ptr) *ptr = *((TG_Pointer*)
						  (bucket+sizeofKey_));
				if (what==htDelete) {
					// to delete, only set the TG_Pointer
					// field to 0xFF
					// we will reap this slot when we
					// have to grow the buffer
					DmSet(pointer,offset + sizeofKey_,
					      sizeof(TG_Pointer), 0xFF);

					// update counters
					deletedEntries_++;
					chunks_[idx].header.
						deletedInThisChunk++;
					DmWrite(pointer, 0,
						&chunks_[idx].header,
						sizeof(TG_HashChunkHeader));
				}
				break;
			case htUpdate:
				DmWrite(pointer, offset+sizeofKey_, ptr,
					sizeof(TG_Pointer));
				break;
			}
			MemHandleUnlock(chunks_[idx].handle);
			return true;
		}

		h1 = (h1 + h2) % maxEntries_;
		prevIdx = idx;
	}
}


void
TG_HashTable::Grow()
{
	// first move the old chunks out of the way
	TG_HashChunkInfo *oldChunks = chunks_;
	VoidPtr pointer;
	Word oldNumChunks=numChunks_, chunkSize;
	DWord newMaxEntries;
	ULong oldFirstChunkId = firstChunkId_, id;
	Word i, j, maxInThisChunk;

	{
		Word powerof2=8, size=SizeofHashEntry();
		while (powerof2 < size) powerof2 *= 2;
		chunkSize = (32768/powerof2);
	}

	if (deletedEntries_ < numEntries_/2) {
		// there aren't enough deleted entries; we must grow the
		// table
		newMaxEntries = maxEntries_*2;

		// update primesArrayIndex_
		primesArrayIndex_++;
	} else {
		// there are too many deleted entries
		// don't grow the table; simply rehash it
		newMaxEntries = maxEntries_;
	}

	chunks_ = NULL;
	numChunks_ = 0; maxChunks_ = 0;
	numEntries_ = maxEntries_ = deletedEntries_ = 0;
	firstChunkId_ = 0xFFFFFFFF;

	// now create the new chunks
	while (maxEntries_ < newMaxEntries) {
		id = CreateNewChunk(((newMaxEntries-maxEntries_ < chunkSize) ?
				     (newMaxEntries-maxEntries_): chunkSize));
		if (id==0xFFFFFFFF) {
			chunkSize /= 2;
			if (chunkSize < 4) {
				ErrFatalDisplayIf(1, "Out of memory in hash "
						  "table");
			}
		}
		else if (firstChunkId_==0xFFFFFFFF) firstChunkId_ = id;
	}

	// reinsert old entries into the new table
	Byte *bucket;
	for (i=0; i<oldNumChunks; i++) {
		pointer = MemHandleLock(oldChunks[i].handle);
		maxInThisChunk = oldChunks[i].header.maxInThisChunk;
		for (j=0; j<maxInThisChunk; j++) {
			bucket = (Byte*)(pointer + sizeof(TG_HashChunkHeader)
					 + j * SizeofHashEntry());
			if (!IsPtrEmpty(bucket)) {
				// this is a useful slot
				Insert(bucket,
				       (TG_Pointer*)(bucket+sizeofKey_));
			}
		}
		MemHandleUnlock(oldChunks[i].handle);
	}

	// delete old entries
	UInt index;
	i = 0;
	while (oldFirstChunkId != 0xFFFFFFFF) {
		DmFindRecordByID(db_, oldFirstChunkId, &index);
		DmReleaseRecord(db_, index, true);
		DmDeleteRecord(db_, index);
		oldFirstChunkId = oldChunks[i++].header.nextChunkId;
	}
}


void
TG_HashTable::Hash(void *key, DWord &h1, DWord &h2)
{
	DWord fourbytes;
	Byte *bytes=(Byte*)key;
	fourbytes = ((DWord)bytes[sizeofKey_-1]) |
		(((DWord)bytes[sizeofKey_-2]) << 8)  |
		(((DWord)bytes[sizeofKey_-3]) << 16) |
		(((DWord)bytes[sizeofKey_-4]) << 24);
	h1 = fourbytes % maxEntries_;
	h2 = 2 * (fourbytes % primes[primesArrayIndex_]) + 1;
}


void
TG_HashTable::LocateChunk(DWord bucketNo, Word &idx, Word &offset)
{
	idx = offset = 0xFFFF;
	Word size=SizeofHashEntry();
	for (Word i=0; i<maxChunks_; i++) {
		if (bucketNo < chunks_[i].header.maxInThisChunk) {
			idx = i;
			offset = sizeof(TG_HashChunkHeader) +
				bucketNo * size;
			return;
		}
		bucketNo -= chunks_[i].header.maxInThisChunk;
	}
}
