/*
 * Copyright (c) 1998 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the Computer Systems
 *       Engineering Group at Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * ---------------------------
 *
 * Filename: manager-srmv2.cc
 *   -- Author: Suchitra Raman <suchi@cs.berkeley.edu>
 * @(#) $Header: /usr/mash/src/repository/srmv2/srmv2/manager-srmv2.cc,v 1.16 1999/01/20 18:49:38 kpatel Exp $
 * 
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "srmv2.h"
#include "ns-srmv2.h"
#include "source-srmv2.h"
#include "manager-srmv2.h"
#include "session-srmv2.h"
#include "srmv2-api.h"


void
SRMv2_Manager::handle_data(SRMv2_Source *src, SRMv2_pktbuf *pb, u_int32_t cid, 
			   int seqno, int more, u_short *ttl,
			   const srm_adu_info *info) 
{
	adubuf *ab=0;
	SRMv2_NameSpace *nsp=0;
	NS_Node *node=0;

	if (src) nsp = src->name_space();
	if (nsp) node = nsp->getnode(cid);

	/* Don't deal with data previously unknown. */
	if (!node) return;

	/* Aggressively set dirty signature flag in namespace. */
	if (nsp) {
	  nsp->dirty_signature();
	}

	/*
	 * Replicate and send to reassemble. Set length, since
	 * reassembly does not look into headers.
	 */	
	SRMv2_pktbuf *cp = (SRMv2_pktbuf *)pb->copy(SRMv2_DATA);
	cp->len = pb->len;  
	node->writef(cp, src, session_, seqno, !more, ttl);

	if (!more) node->set_lastseen(seqno);
	
	/* Reassemble ADU and pass up to the application */
	if (node->lastseen(seqno) == 1) {
		ab = node->reassemble(seqno);
		if (ab) {
			srm_recv_proc recv_proc =
				session_->callbacks()->recv_proc;
			if (recv_proc)
				(*recv_proc)((srm_source_t)src, cid, seqno,
					     ab->data, ab->len, info);
			srm_trace(srmSRM, ("R <%d:%d> [%d B]\n", cid, seqno,
					   ab->len));
			delete [] ab->data;
		}
	}
}

void
SRMv2_Manager::handle_rreq(SRMv2_Source *src, u_int32_t ss, u_int32_t sbytes, 
			   u_int32_t es, u_int32_t ebytes, u_int32_t cid, 
			   u_int32_t addr) 
{
	SRMv2_Edge left(ss, sbytes);
	SRMv2_Edge right(es, ebytes);

	SRMv2_NameSpace *nsp = src->name_space();
	NS_Node *node        = nsp->getnode(cid);

	/* Data for an undiscovered node is discarded. */
	if (!node) return;
	node->requests()->backoff(cid, left, right);
	node->replies()->cancel(cid, left, right);

	/* 
	 * Schedule a reply for the requested data. 
	 * We just cancelled any previously scheduled 
	 * replies. 
	 */
	SRMv2_Request *rep = new SRMv2_Request(cid, left, right);
	rep->uaddr(addr);
	rep->session(session_);
	rep->source(src);
	node->sched_reply(rep);
	return;
}
