#ifndef _SF_H
#define _SF_H

/* sf.h */

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

#ifndef FALSE
#define FALSE 0
#endif

#ifndef TRUE
#define TRUE  1
#endif

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

typedef enum
{
  c_VIEW_GEOMETRY    = 'v',
  c_CONTROL_GEOMETRY = 'c',
  c_MINIMIZE         = 'm',
  c_SIZE             = 'z',
  c_SYMMETRY         = 'y',
  c_EDGES            = 'e',
  c_COLOUR_SCHEME    = 'r',
  c_BANDING          = 'b',
  c_CENTRE           = 't',
  c_NAME             = 'n',
  c_KEY              = 'k',
  c_LIST             = 'l',
  c_SPEED            = 's',
  c_GENERATION       = 'g',
  c_LOOP             = 'p'
} command_option_e;

extern struct option long_options[];
#define COMMAND_OPTIONS ( ( sizeof( long_options ) \
                          / sizeof( struct option ) ) - 1 )

extern char         short_options[];

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

typedef enum
{
  cmd_NULL     = 0,
  cmd_PING,
  cmd_PONG,
  cmd_GENERATE,
  cmd_VIEW,
  cmd_ACK,
  cmd_TIMEOUT,
  cmd_CLOSE,
  cmd_QUIT
} snowflake_cmd;

#define SNOWFLAKE_CMD_LEN  sizeof( snowflake_cmd )

/* image generation key length */

#define MAX_KEY          ( 16 * sizeof( int ) )

#define GENERATE_CMD_FORMAT     "%s,%d,%d,%d,%d,%d,%d"
#define MAX_GENERATE_CMD_LEN    (MAX_KEY+1+1+1+1+1+1+1+1+1+1+1+1)
#define GENERATE_SCAN_FORMAT    "%[0-9A-F],%d,%d,%d,%d,%d,%d"
#define GENERATE_SCAN_FORMAT_NL "%[0-9A-F],%d,%d,%d,%d,%d,%d%[ \t\n]"

#define VIEW_CMD_FORMAT         "%s,%d,%d"
#define MAX_VIEW_CMD_LEN        (MAX_KEY+1+1+1+1)
#define VIEW_SCAN_FORMAT        "%[0-9A-F],%d,%d"

#define MAX_MSG_DATA_LEN   128
/*      MAX_MSG_DATA_LEN must be >= MAX_GENERATE_CMD_LEN */
/*      MAX_MSG_DATA_LEN must be >= MAX_VIEW_CMD_LEN */

typedef struct
{
  snowflake_cmd  cmd;
  union
  {
    int          i;
    char         s[MAX_MSG_DATA_LEN+1];
  } data;
} snowflake_message;

#define SNOWFLAKE_MSG_LEN  sizeof( snowflake_message )

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

/* defines */

typedef struct
{
  int snd;
  int rcv;
} msg_pipe;

/* pipe read/write indices */
#define MP_READ  0
#define MP_WRITE 1

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

/* shared memory creation flags */
#define IFLAGS  (IPC_CREAT|IPC_EXCL)

#ifndef SHM_RW
#define SHM_RW  0600
#endif

/* keys will have the following form:
 *
 * kkkkkkkk  32 bits
 * iiiiii..  process = id << 8 (id*256)
 * ......t.  type: { semaphore,shared memory }
 * .......d  data: { generation message,view message, image data }
 */

#define KEY_SHIFT         8

#define SEM_KEY           0x0010
#define SHM_KEY           0x0020

#define GENERATION_SUBKEY 0x0001
#define VIEW_SUBKEY       0x0002
#define IMAGE_SUBKEY      0x0004

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

#define BLACK_PIXEL           'x'
#define BLACK_COLOUR_NAME     "black"
#define GRAY_PIXEL            '-'
#define GRAY_COLOUR_NAME      "gray"
#define WHITE_PIXEL           ' '
#define WHITE_COLOUR_NAME     "white"

#define COLOURS                3

#define IMAGE_SPEC_LINE        0
#define IMAGE_SPEC_FORMAT     "%3d %3d %1d %1d"
#define MAX_SPEC_LEN          (3+1+3+1+1+1+1 + 1)

#define COLOURS_SPEC_LINE      1
#define COLOURS_SPEC_FORMAT   "%1c c %-32s m %-32s"
#define MAX_COLOURS_LEN      ((1+1+1+1+32+1+1+1+32 + 1) * COLOURS)
#define MAX_HEADER_LEN       (MAX_SPEC_LEN + MAX_COLOURS_LEN)

#define IMAGE_DATA_LINE      (COLOURS_SPEC_LINE+COLOURS)
#define IMAGE_DATA_OFFSET     MAX_HEADER_LEN

#define MAX_IMAGE_SIZE        513
#define MAX_IMAGE_WIDTH       MAX_IMAGE_SIZE
#define MAX_IMAGE_HEIGHT      MAX_IMAGE_SIZE
#define MAX_IMAGE_LEN        (MAX_IMAGE_HEIGHT*(MAX_IMAGE_WIDTH + 1))

#define MAX_IMAGE_DATA_LINES (IMAGE_DATA_LINE + MAX_IMAGE_HEIGHT)
#define MAX_IMAGE_DATA_LEN   (MAX_HEADER_LEN + MAX_IMAGE_LEN)
#define MAX_IMAGE_DATA_PTRS  (MAX_IMAGE_DATA_LINES * sizeof( char* ))

/* first line index of image data */
#define FIRST_IMAGE_DATA_LINE (COLOURS+1)

typedef struct
{
  char* line[MAX_IMAGE_DATA_LINES];
  char  data[MAX_IMAGE_DATA_LEN];
  int   size;  /* == width (== height) */
} image_data;

#define MAX_IMAGE_DATA_SIZE  (sizeof( image_data ))

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

/* size setting */
typedef enum
{
  size_032x032 = 0,
  size_064x064 = 1,
  size_128x128 = 2,
  size_256x256 = 3,
  size_512x512 = 4
} size_e;

#define SIZES 5

/* symmetry setting */

typedef enum
{
  symmetry_VERTICAL   = 0,
  symmetry_HORIZONTAL = 1
} symmetry_e;

#define SYMMETRIES 2

/* edges setting */

typedef enum
{
  edges_NO         = 0,
  edges_GRAY       = 1,
  edges_GRAY_SOLID = 2,
  edges_YES        = 3
} edges_e;

#define EDGES 4

/* loop setting */

typedef enum
{
  loop_ONCE   = 0,
  loop_REPEAT = 1
} loop_e;

#define LOOPS 2

/* colour scheme setting */
typedef enum
{
  colour_scheme_WHITE_ON_BLACK = 0,
  colour_scheme_BLACK_ON_WHITE = 1
} colour_scheme_e;

#define COLOUR_SCHEMES 2

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

/* prototypes */

void   sem_wait( int sem_id );
int  sem_signal( int sem_id );

int    is_alive( int parent_pid );

int     snd_msg( msg_pipe mp,snowflake_message* sf_msg_p );
int     rcv_msg( msg_pipe mp,snowflake_message* sf_msg_p );

int    get_image_size   ( size_e size );
void write_image_header ( char* data,int size );
void write_image_pixel  ( char* data,int x,int y,char pixel );
int  write_image_to_file( char* data,char* filespec,char* key );

/* main processes */

void generation_main( int                 argc,
                      char*               argv[],
                      msg_pipe            control,
                      int                 image_semid,
                      char*               image );
void view_main(       int                 argc,
                      char*               argv[],
                      msg_pipe            control,
                      int                 image_semid,
                      char*               image );
void control_main(    int                 argc,
                      char*               argv[],
                      msg_pipe            generation,
                      msg_pipe            view,
                      int                 image_semid,
                      char*               image );

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

#endif /* _SF_H */
