/* dvi2misc -  DVI to Misc (PS, PGM) Files
 * by Hirotsugu Kakugawa
 *
 *  22 Jul 1998   First implementation (to PS).
 *  31 Jul 1998   Changed to use frame bufer object in DVIlib.
 *  30 Oct 1998   Changed print offset.
 *  10 Dec 1998   Added dvi2img (PGM output) feature.
 *  19 Jan 1999   Added dvi2escpg (EPSON ESC/Page output) feature
 *                (snarfed from dvi2escpage.c) and dvi2rpdl (RICOH RPDL
 *                printer output, e.g., NX100, NX700) feature.
 */
/*
 * Copyright (C) 1998-1999  Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#if HAVE_STDARG_H
#  include <stdarg.h>
#else
#  include <varargs.h>
#endif
#if STDC_HEADERS
#  include <string.h>
#else
#  include <strings.h>
#endif
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif
#if HAVE_SIGNAL_H
#  include <signal.h>
#endif

#include "dvi-2_6.h"
#include "defs.h"
#include "cf-dvi2misc.h"
#include "../params.h"
#include "escpg.h"


DVI               dvi        = NULL;
DVI_DEVICE        dev        = NULL;
DVI_FRAME_BUFFER  framebuff  = NULL;
int               interrupted = 0;

int           dvi2X_dpi  = 0;
char         *dvi2X_mode = NULL;

char         *program_name      = PROGRAM; 
char         *program_desc      = "DVI to misc file format"; 
int           param_program     = -1; 
char         *param_vflibcap    = DEFAULT_VFLIBCAP;
char         *param_mode        = NULL;
int           param_dpi         = 0;
int           param_landscape   = FALSE;
double        param_offset_x    = 0;
double        param_offset_y    = 0;
char         *param_device      = NULL;
char         *param_dvi_file    = NULL;
char         *param_paper_name  = "A4";
int           param_paper_id    = 0;
int           param_num_copies  = 1;
int           param_page_from   = -1;
int           param_page_to     = -1;
int           param_page_rev    = 0;
int           param_verbose     = 0;
int           param_debug       = 0;

char         *param_dvi2img_title  = NULL;
char         *param_dvi2img_ofile  = NULL;
int           param_dvi2img_out    = DEFAULT_OUT_FORMAT;
double        param_dvi2img_shrink = -1;
int           param_dvi2img_aa     = 1;
int           param_dvi2img_bbx    = 0;
int           param_dvi2img_page   = 1;


FILE         *output_fp;
char          temp_file[1024];
long          offset_x    = 0;
long          offset_y    = 0;


char *color_fg = "black";
unsigned char  color_fg_r = 0;
unsigned char  color_fg_g = 0;
unsigned char  color_fg_b = 0;

char *color_bg = "white";
unsigned char  color_bg_r = 0xff;
unsigned char  color_bg_g = 0xff;
unsigned char  color_bg_b = 0xff;



struct s_paper  paper_list[] = {
  {"A1",     11,   23.40,  33.10 },
  {"A2",     12,   16.50,  23.40 },
  {"A3",     13,   11.70,  16.50 },
  {"A4",     14,    8.27,  11.70 },
  {"A5",     15,    5.85,   8.27 },
  {"A6",     16,    4.13,   5.85 },
  {"A7",     17,    2.92,   4.13 },
  {"B1",     21,   27.80,  39.50 },
  {"B2",     22,   19.70,  27.80 },
  {"B3",     23,   13.90,  19.70 },
  {"B4",     24,    9.84,  13.90 },
  {"B5",     25,    6.93,   9.84 },
  {"B6",     26,    5.32,   6.93 },
  {"B7",     27,    3.47,   5.32 },
  {"LETTER", 31,    8.50,  14.00 },
  {"LEGAL",  32,    8.50,  14.00 }, 
  {"US",     33,    8.50,  11.00 },  
  {NULL,      -1,    -1,     -1}
};


void   parse_args(int argc, char **argv);
void   usage(void);
void   print_defaults(void);
char  *get_input_filename(void);
int    find_page1(DVI,int);
int    find_page2(DVI,int);
void   clean_up(int);
#if HAVE_SIGNAL_H
RETSIGTYPE  sig_handler(int);
#endif

void  DEV_put_bitmap(DVI_DEVICE,DVI,DVI_BITMAP,int,long,long,long,long);
void  DEV_put_rectangle(DVI_DEVICE,DVI,long,long,long,long);
void  DEV_put_pixmap(DVI_DEVICE,DVI,DVI_PIXMAP_RGB,int,long,long,long,long);
void  DEV_color_rgb(DVI_DEVICE,DVI,int,double,double,double);
void  DEV_bgcolor_rgb(DVI_DEVICE,DVI,int,double,double,double);

void  DEV_message_ignore(DVI_DEVICE,DVI,char*,...);
void  DEV_message(DVI_DEVICE,DVI,char*,...);

void  dvi2misc(char *dvi_file);
extern void  dvi2pgm(char *dvi_file, long w, long h);
extern void  dvi2ppm(char *dvi_file, long w, long h);
extern void  dvi2bdps(char *dvi_file, long w, long h);
extern void  dvi2escpg(char *dvi_file, long w, long h);
extern void  dvi2rpdl(char *dvi_file, long w, long h);




int
main(int argc, char **argv)
{
  char   *dvi_file; 

  interrupted = 0;
#if HAVE_SIGNAL_H
  if (signal(SIGINT, SIG_IGN) == SIG_DFL)
    signal(SIGINT, sig_handler);
#endif

  DVI_SETUP();

  parse_args(argc, argv);

  if (interrupted == 1)
    clean_up(1);

  if ((dvi_file = get_input_filename()) == NULL)
    clean_up(1);
  if (access(dvi_file, R_OK) < 0){
    fprintf(stderr, "%s: Can't read DVI file: %s\n", program_name, dvi_file);
    clean_up(1);
  }

  if (param_device == NULL){
    output_fp = stdout;
  } else {
    if ((output_fp = fopen(param_device, "w")) == NULL){
      fprintf(stderr, "%s: Can't open %s\n", 
	      program_name, param_device);
      clean_up(1);
    }
  }

  if (interrupted == 1)
    clean_up(1);

  dvi2misc(dvi_file);

  if (interrupted == 1)
    clean_up(1);
  else
    clean_up(0);

  return 0;
}


void
parse_args(int argc, char **argv)
{
  int    i;
  double s, d;

  param_dpi  = dvi2X_dpi  = DEFAULT_DPI;
  param_mode = dvi2X_mode = DEFAULT_DEV_MODE;
  param_page_from  = -1;
  param_page_to    = -1;
  param_page_rev   = 0;
  
  for (--argc, argv++; argc > 0;  argc--, argv++){

    if ( (strcmp(*argv, "-dvi2img") == 0)
	 || (strcmp(*argv, "-img") == 0)
	 || (strcmp(*argv, "-dvi2pgm") == 0)
	 || (strcmp(*argv, "-pgm") == 0)){
      program_name = "dvi2pgm";
      program_desc = "DVI to a PGM file";
      param_program = PROGRAM_TYPE_DVI2PGM;
      param_dpi  = dvi2X_dpi  = 300;
      param_mode = dvi2X_mode = "cx";

    } else if ( (strcmp(*argv, "-dvi2ppm") == 0)
		|| (strcmp(*argv, "-ppm") == 0)){
      program_name = "dvi2ppm";
      program_desc = "DVI to a PPM file";
      param_program = PROGRAM_TYPE_DVI2PPM;
      param_dpi  = dvi2X_dpi  = 300;
      param_mode = dvi2X_mode = "cx";

    } else if ( (strcmp(*argv, "-dvi2bdps") == 0)
		|| (strcmp(*argv, "-bdps") == 0)){
      program_name = "dvi2bdps";
      program_desc = "DVI to a brain-damaged PostScript file";
      param_program = PROGRAM_TYPE_DVI2BDPS;
      param_dpi  = dvi2X_dpi  = 300;
      param_mode = dvi2X_mode = "cx";

    } else if ( (strcmp(*argv, "-dvi2escpg") == 0)
		|| (strcmp(*argv, "-escpg") == 0)){
      program_name = "dvi2escpg";
      program_desc = "DVI to EPSON ESC/Page printer";
      param_program = PROGRAM_TYPE_DVI2ESCPG;
      param_dpi  = dvi2X_dpi  = 300;
      param_mode = dvi2X_mode = "cx";

    } else if ( (strcmp(*argv, "-dvi2rpdl") == 0)
		|| (strcmp(*argv, "-rpdl") == 0)){
      program_name = "dvi2rpdl";
      program_desc = "DVI to RICOH RPDL page printer";
      param_program = PROGRAM_TYPE_DVI2RPDL;
      param_dpi  = dvi2X_dpi  = 600;
      param_mode = dvi2X_mode = "ljfour";

    } else if (strcmp(*argv, "-defaults") == 0){
      print_defaults();
      exit(0);

    } else if ((strcmp(*argv, "-v") == 0) && (argc > 1)){
      argc--;  argv++;
      param_vflibcap = *argv;

    } else if (DVI_parse_device_mode(&argv[0][1], &d, NULL, NULL) >= 0){
      param_mode = &argv[0][1];
      param_dpi = d;
      
    } else if ((strcmp(*argv, "-mode") == 0) && (argc > 1)){
      argc--;  argv++;
      param_mode = *argv;
      if (DVI_parse_device_mode(param_mode, &d, NULL, NULL) >= 0)
	param_dpi = d;

    } else if ((strcmp(*argv, "-dpi") == 0) && (argc > 1)){
      argc--;  argv++;
      param_dpi = atoi(*argv);

    } else if ((strcmp(*argv, "-dev") == 0) && (argc > 1)){
      argc--;  argv++;
      param_device = *argv;

    } else if (   (strcmp(*argv, "-land") == 0)
	       || (strcmp(*argv, "-landscape") == 0)){
      param_landscape = TRUE;

    } else if (strcmp(*argv, "-port") == 0){
      param_landscape = FALSE;

    } else if ((strcmp(*argv, "-ox") == 0) && (argc > 1)){
      argc--;  argv++;
      param_offset_x = atof(*argv);

    } else if ((strcmp(*argv, "-oy") == 0) && (argc > 1)){
      argc--;  argv++;
      param_offset_y = atof(*argv);

    } else if (strcmp(*argv, "-debug") == 0){
      param_debug = 1;

    } else if (strcmp(*argv, "-DEBUG") == 0){
      param_debug = 2;

    } else if (strcmp(*argv, "-verbose") == 0){
      param_verbose = 1;

    } else if ((strcmp(*argv, "-f") == 0) && (argc > 1)){
      argc--;  argv++;
      param_dvi_file = *argv;

    } else if ((strcmp(*argv, "-from") == 0) && (argc > 1)){
      argc--;  argv++;
      param_page_from = atoi(*argv);

    } else if ((strcmp(*argv, "-to") == 0) && (argc > 1)){
      argc--;  argv++;
      param_page_to = atoi(*argv);

    } else if (strcmp(*argv, "-rev") == 0){
      param_page_rev = 1;
      
    } else if ((strcmp(*argv, "-title") == 0) && (argc > 1)){    /* pgm/ppm */
      argc--;  argv++;
      param_dvi2img_title = *argv;

    } else if ((strcmp(*argv, "-ofile") == 0) && (argc > 1)){    /* pgm/ppm */
      argc--;  argv++;
      param_dvi2img_ofile = *argv;

    } else if ((strcmp(*argv, "-s") == 0) && (argc > 1)){        /* pgm/ppm */
      argc--;  argv++;
      param_dvi2img_shrink = atof(*argv);
      if (param_dvi2img_shrink <= 0)
	param_dvi2img_shrink = -1;

    } else if ((strcmp(*argv, "-aa") == 0) && (argc > 1)){       /* pgm/ppm */
      argc--;  argv++;
      param_dvi2img_aa = atoi(*argv);
      if (param_dvi2img_aa <= 0)
	param_dvi2img_aa = 1;

    } else if (strcmp(*argv, "-bbx") == 0){                      /* pgm/ppm */
      param_dvi2img_bbx = 1;

    } else if (((strcmp(*argv, "-page") == 0) && (argc > 1))
	       || ((strcmp(*argv, "-p") == 0) && (argc > 1))){   /* pgm/ppm */
      argc--;  argv++;
      param_dvi2img_page  = atoi(*argv);
      if (param_dvi2img_page <= 0)
	param_dvi2img_page = 1;

    } else if (strcmp(*argv, "-allpages") == 0){                 /* pgm/ppm */
      param_dvi2img_page  = -1;

    } else if ((strcmp(*argv, "-fg") == 0) && (argc > 1)){       /* ppm */
      argc--;  argv++;
      DVI_rgb_lookup(*argv, &color_fg_r, &color_fg_g, &color_fg_b);

    } else if ((strcmp(*argv, "-bg") == 0) && (argc > 1)){       /* ppm */
      argc--;  argv++;
      DVI_rgb_lookup(*argv, &color_bg_r, &color_bg_g, &color_bg_b);

    } else if (strcmp(*argv, "-300") == 0){                        /* escpg */
      param_dpi = 300;
      param_mode = "cx";

    } else if (strcmp(*argv, "-600") == 0){                        /* escpg */
      param_dpi = 600;
      param_mode = "ljfour";

    } else if ((strcmp(*argv, "-n") == 0) && (argc > 1)){     /* escpg/rpdl */
      argc--;  argv++;
      param_num_copies = atoi(*argv);
      if (param_num_copies <= 0)
	param_num_copies = 1;

    } else if ((strcmp(*argv, "-h") == 0) || (strcmp(*argv, "--help") == 0)){
      usage();
      exit(0);

    } else if (**argv != '-'){
      param_dvi_file = *argv;

    } else {

      for (i = 0; paper_list[i].name != NULL; i++){
	if (strcmp(&argv[0][1], paper_list[i].name) == 0){
	  param_paper_name = &argv[0][1];
	  break;
	}
      }
      if (paper_list[i].name == NULL){
	fprintf(stderr, "%s: Unknown option: %s\n", program_name, *argv);
	exit(1);
      }
    }
  }

  for (i = 0; paper_list[i].name != NULL; i++){
    if (strcmp(param_paper_name, paper_list[i].name) == 0){
      param_paper_id = i;
      break;
    }
  }

  if ((s = param_dvi2img_shrink) <= 0)
    s = 1.0;
  offset_x = (param_offset_x + SYS_OFFSET_X) * param_dpi / s;
  offset_y = (param_offset_y + SYS_OFFSET_Y) * param_dpi / s;
}


char*
get_input_filename(void)
{
  FILE  *fp;
  int   fd, ch;

  if (param_dvi_file != NULL){
    if (access(param_dvi_file, R_OK) == 0)
      return param_dvi_file;
    sprintf(temp_file, "%s.dvi", param_dvi_file);
    if (access(temp_file, R_OK) == 0)
      return temp_file;
    sprintf(temp_file, "%sdvi", param_dvi_file);
    if (access(temp_file, R_OK) == 0)
      return temp_file;
  }

  fp = NULL;
#ifdef HAVE_MKSTEMP
  sprintf(temp_file, "%s/dvi2misc-XXXXXX", TEMP_DIR);
  if ((fd = mkstemp(temp_file)) >= 0)
    fp = fdopen(fd, "w");
#else
  sprintf(temp_file, "%s/dvi2misc-XXXXXX", TEMP_DIR);
  mktemp(temp_file);
  fp = fopen(temp_file, "w");
#endif
  if (fp == NULL){
    fprintf(stderr, "%s: Can't create temporary file\n", program_name);
    clean_up(1);
  }
  while ((ch = getc(stdin)) != EOF)
    putc(ch, fp);
  fclose(fp);

  return temp_file;
}
  
void 
clean_up(int exit_val)
{
  if (param_dvi_file == NULL)
    unlink(temp_file);
  exit(exit_val);
}

#if HAVE_SIGNAL_H
RETSIGTYPE
sig_handler(int sig)
{
  fprintf(stderr, "%s: Interrupted.\n", PROGRAM);
  interrupted = 1;

  switch (param_program){
  case PROGRAM_TYPE_DVI2PGM:
  case PROGRAM_TYPE_DVI2PPM:
  case PROGRAM_TYPE_DVI2BDPS:
    exit(1);
  }
}
#endif



void  
usage(void)
{
  PR3("%s version %s --- %s\n", program_name, VERSION, program_desc);
  PR1("Usage: %s [Output] [Device] [Options] [DVI-FILE]\n", program_name);
  if (param_program == -1){
    PR0("Output:\n");
    PR0("  -dvi2pgm     DVI to PGM files\n");
    PR0("  -dvi2ppm     DVI to PPM files\n");
    PR0("  -dvi2bdps    DVI to PostScript file\n");
    PR0("  -dvi2escpg   DVI to EPSON ESC/Page printer\n");
    PR0("  -dvi2rpdl    DVI to RICOH RPDL printer\n");
    PR0("  -dvi2pgm -h     Print dvi2pgm specific options\n");
    PR0("  -dvi2ppm -h     Print dvi2ppm specific options\n");
    PR0("  -dvi2bdps -h    Print dvi2bdps specific options\n");
    PR0("  -dvi2escpg -h   Print dvi2escpg specific options\n");
  }
  PR0("Device : select device mode name for font search by kpathsea\n");
  PR0("  -cx, -sparcptr, -ljfour, ..., -MODE\n");
  PR1("  -mode MODE   Same as -MODE [%s]\n", dvi2X_mode);
  PR1("  -dpi DPI     Device resolution [%d]\n", dvi2X_dpi);
  PR0("Options:\n");
  PR1("  -v VFLIBCAP  Path name for vflibcap [%s]\n", DEFAULT_VFLIBCAP);
  PR0("  -ox OFFSET   Set horizontal offset (in inch)\n");
  PR0("  -oy OFFSET   Set vertical offset (in inch)\n");
  PR0("  -from N      Starting page number\n");
  PR0("  -to N        Ending page number\n");
  PR0("  -rev         Reverse output page ordering\n");
  PR0("  -A3, -A4, -B5, -B4, -LETTER, -LEGAL\n");
  PR0("               Select paper size [A4]\n");
  PR0("  -land        Landscape mode\n");
  PR0("  -port        Portrait mode\n");
  if ((param_program == PROGRAM_TYPE_DVI2PGM)
      || (param_program == PROGRAM_TYPE_DVI2PPM)){
    PR0("  -title TTL   Format of image title\n"); 
    PR0("               %%p - page number, %%f - DVI file name, %%%% - %% itself\n"); 
    PR0("  -ofile FILE  Format of output file name\n"); 
    PR0("               %%p - page number, %%%% - %% itself\n"); 
    PR0("  -aa N        Anti-aliasing factor [1]\n");
    PR0("  -s S         Shrink factor [1]\n");
    PR0("               (Resulting image is shrinked by N * S in total)\n");
    PR0("  -bbx         Minimum bounding box is shipped to an image file\n");
    PR0("  -p P         Page number in DVI file to be shipped out [1]\n");
    PR0("  -allpages    Make iamges for each page\n");
  }
  if (param_program == PROGRAM_TYPE_DVI2BDPS){
    PR0("  -dev DEV     Printer device file name\n");
  }
  if (param_program == PROGRAM_TYPE_DVI2ESCPG){
    PR0("  -300         300 dpi mode, same as \"-dpi 300 -mode cx\"\n");
    PR0("  -600         600 dpi mode, same as \"-dpi 600 -mode ljfour\"\n");
  }
  if (param_program == PROGRAM_TYPE_DVI2RPDL){
    PR0("  -n N         Number of copies [1]\n");
  }
  PR0("  -defaults    Print defaut values\n");
  PR0("  -h           Print this\n");
  exit(0);
}

void
print_defaults(void)
{
  fprintf(stderr, "%s version %s default values\n", program_name, VERSION);
  fprintf(stderr, "Default vflibcap: %s\n", DEFAULT_VFLIBCAP);
  fprintf(stderr, "Default device resolution: %d\n", dvi2X_dpi);
  fprintf(stderr, "Default device mode name: %s\n", dvi2X_mode);
}



void
dvi2misc(char *dvi_file)
{
  char    params[1024];
  long    w, h;
  double  s;

  sprintf(params, "%s=%d, %s=%s, %s=%s", 
	  PARAM_NAME_DPI,  param_dpi,
	  PARAM_NAME_MODE, param_mode,
	  PARAM_NAME_PROG, program_name);

  if (DVI_INIT(param_vflibcap, params) < 0){
    fprintf(stderr, "%s: Can't initialize DVI interpreter\n", program_name);
    clean_up(1);
  }

  if ((s = param_dvi2img_shrink) <= 0)
    s = 1.0;

  if (param_landscape == FALSE){
    w = (int)(paper_list[param_paper_id].w * (double)param_dpi / s);
    h = (int)(paper_list[param_paper_id].h * (double)param_dpi / s);
  } else {
    w = (int)(paper_list[param_paper_id].h * (double)param_dpi / s);
    h = (int)(paper_list[param_paper_id].w * (double)param_dpi / s);
  }

  switch (param_program){
  default:
    framebuff = DVI_fb_create_mono(w, h);
    break;
  case PROGRAM_TYPE_DVI2PPM:
    framebuff = DVI_fb_create_rgb(w, h);
    break;
  }

  if (framebuff == NULL){
    fprintf(stderr, "%s: Cannot allocate frame buffer\n", program_name);
    clean_up(1);
  }

  if (param_program == PROGRAM_TYPE_DVI2PPM){
    DVI_fb_rgb_foreground(framebuff, color_fg_r, color_fg_g, color_fg_b);
    DVI_fb_rgb_background(framebuff, color_bg_r, color_bg_g, color_bg_b);
  }
  
  DVI_fb_clear(framebuff);

  if ((dev = DVI_DEVICE_ALLOC()) == NULL){
    fprintf(stderr, "%s: Can't initialize device\n", program_name);
    clean_up(1);
  }

  dev->h_dpi = dev->v_dpi   = param_dpi;
  dev->message_advice       = DEV_message_ignore;
  dev->message_warning      = DEV_message_ignore;
  dev->message_error        = DEV_message;
  dev->message_fatal        = DEV_message;

  switch (param_program){
  default:
    abort();
    /* devices that accumulate glyphs in a frame buffer (mono/gray) */
  case PROGRAM_TYPE_DVI2BDPS:
  case PROGRAM_TYPE_DVI2PGM:
  case PROGRAM_TYPE_DVI2RPDL:
    dev->put_rectangle      = DEV_put_rectangle;
    dev->put_bitmap         = DEV_put_bitmap;
    break;
    /* devices that accumulate glyphs in a frame buffer (color) */
  case PROGRAM_TYPE_DVI2PPM:
    dev->put_rectangle      = DEV_put_rectangle;
    dev->put_bitmap         = DEV_put_bitmap;
    dev->put_pixmap_rgb     = DEV_put_pixmap;
    dev->color_rgb          = DEV_color_rgb;
    dev->bgcolor_rgb        = DEV_bgcolor_rgb;
    break;
    /* devices that each glyph is sent to device, without frame buffer */
  case PROGRAM_TYPE_DVI2ESCPG:
    dev->put_rectangle      = DEV_put_rectangle_escpg;
    dev->put_bitmap         = DEV_put_bitmap_escpg;
    dev->device_polling     = DEV_poll_escpg;
    break;
  }

  if ((dvi = DVI_CREATE(dev, dvi_file, NULL)) == NULL){
    fprintf(stderr, "%s: Can't open a DVI file: %s\n", program_name, dvi_file);
    clean_up(1);
  }

  if (interrupted == 1){
    clean_up(1);
    exit(0);
  }

  if (DVI_OPEN_FONT(dvi, dev) < 0){
    fprintf(stderr, "%s: Cannot find all fonts\n", program_name);
    clean_up(1);
  }


  if (param_page_from >= 1)
    param_page_from = find_page1(dvi, param_page_from);
  if (param_page_from <  1)
    param_page_from = 1;

  if (param_page_to >= 1)
    param_page_to = find_page2(dvi, param_page_to);
  if (param_page_to <  1)
    param_page_to = dvi->pages;

  if (param_page_from > param_page_to){
    int  tmp = param_page_to;
    param_page_to   = param_page_from; 
    param_page_from = tmp; 
    if (param_page_rev == 0)
      param_page_rev  = 1;
    else
      param_page_rev  = 0;
  }

  if (interrupted == 1){
    clean_up(1);
    exit(0);
  }

  switch (param_program){
  default:
  case -1:
  case PROGRAM_TYPE_DVI2PGM:
    dvi2pgm(dvi_file, w, h);
    break;
  case PROGRAM_TYPE_DVI2PPM:
    dvi2ppm(dvi_file, w, h);
    break;
  case PROGRAM_TYPE_DVI2BDPS:
    dvi2bdps(dvi_file, w, h);
    break;
  case PROGRAM_TYPE_DVI2ESCPG:
    dvi2escpg(dvi_file, w, h);
    break;
  case PROGRAM_TYPE_DVI2RPDL:
    dvi2rpdl(dvi_file, w, h);
    break;
  }
}

int
find_page1(DVI dvi, int p)
{
  int  i, x;

  x = -1;
  for (i = 0; i < dvi->pages; i++){
    if (dvi->page_no[i].c[0] == p){
      x = i + 1;
      break;
    }
  }
  return x;
}


int
find_page2(DVI dvi, int p)
{
  int  i, x;

  x = -1;
  for (i = dvi->pages - 1; i >= 0; i--){
    if (dvi->page_no[i].c[0] == p){
      while (dvi->page_no[i].c[0] == p)
	i++;
      x = i;
      break;
    }
  }
  return x;
}



void  
DEV_put_bitmap(DVI_DEVICE dev, DVI dvi, DVI_BITMAP bm, 
	       int font_id, long key2, long code_point, 
	       long pos_x, long pos_y)
{
  DVI_fb_put_bitmap(framebuff, bm, pos_x + offset_x, pos_y + offset_y);
}


void  
DEV_put_pixmap(DVI_DEVICE dev, DVI dvi, DVI_PIXMAP_RGB bm, 
	       int font_id, long key2, long code_point, 
	       long pos_x, long pos_y)
{
  DVI_fb_put_pixmap_rgb(framebuff, bm, pos_x + offset_x, pos_y + offset_y);
}

void  
DEV_put_rectangle(DVI_DEVICE dev, DVI dvi, 
		  long pos_x, long pos_y, long rect_w, long rect_h)
{
  DVI_fb_put_rectangle(framebuff, pos_x + offset_x, pos_y + offset_y, 
		       rect_w, rect_h);
}

void
DEV_color_rgb(DVI_DEVICE dev, DVI dvi, int page_resume_flag,
	      double r, double g, double b)
{
#if 0
  printf("FOREGROUND COLOR %.3f %.3f %.3f\n", r, g, b);
#endif
  color_fg_r = 255 * r;
  color_fg_g = 255 * g; 
  color_fg_b = 255 * b;
  DVI_fb_rgb_foreground(framebuff, color_fg_r, color_fg_g, color_fg_b);
  return; 
}

void
DEV_bgcolor_rgb(DVI_DEVICE dev, DVI dvi, int page_resume_flag,
		double r, double g, double b)
{
#if 0
  printf("BACKGROUND COLOR %.3f %.3f %.3f\n", r, g, b);
#endif
  color_bg_r = 255 * r;
  color_bg_g = 255 * g; 
  color_bg_b = 255 * b;
  DVI_fb_rgb_background(framebuff, color_bg_r, color_bg_g, color_bg_b);
  DVI_fb_clear(framebuff);
  return;
}


void  
DEV_message(DVI_DEVICE dev, DVI dvi, char *fmt,...)
{
  va_list ap;
  
  va_start(ap, fmt);
  fprintf(stderr, "%s: ", program_name);
  vfprintf(stderr, fmt, ap);  
  fprintf(stderr, "\n");
  va_end(ap);
}


void  
DEV_message_ignore(DVI_DEVICE dev, DVI dvi, char *fmt,...)
{
  va_list ap;
  
  va_start(ap, fmt);
  va_end(ap);
}


/*EOF*/
