/*
 * ui-print.c - a module for print UI
 * by Hirotsugu Kakugawa
 */
/*
 * Copyright (C) 1996-1997 Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#include <pwd.h>
#ifdef HAVE_STDARG_H
#  include <stdarg.h>
#else
#  include <vararg.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#endif
#ifdef HAVE_STRINGS_H
# include <strings.h>
#endif
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_SIGNAL_H
# include <signal.h>
#endif
#ifdef HAVE_MALLOC_H
# include <malloc.h>
#endif
#include <sys/wait.h>
#include <sys/param.h>
#include <X11/Xlib.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/List.h>
#include <Xm/MessageB.h>
#include <Xm/Text.h>

#include "cf-xmdvi.h"
#include "dvi-2_6.h"
#include "defs.h"
#include "resource.h"
#include "window.h"
#include "dev.h"
#include "ui.h"

#if HAVE_GETCWD
#else
# if HAVE_GETWD
#  define getcwd(s,n)  getwd(s)
# else
#  define getcwd(s,n)  NULL
# endif
#endif

#ifndef HAVE_RINDEX
#  define rindex(s,c)  strrchr(s,c)
#endif
#ifndef HAVE_INDEX
#  define index(s,c)   strchr(s,c)
#endif


Private Widget  dialog_print; 
Private Widget  dialog_confirm_file; 
Private Widget  dialog_confirm_page; 
Private Widget  x_print_message;
Private Widget  abort_button; 

Private struct printer_list {
  char  *lp_name;
  int   lp_type;
  char  *lp_type_s;
  char  *lp_comment;
  char  *lp_data;
  char  *lp_pr_spec, *lp_pg_spec, *lp_qu_spec, *lp_rm_spec ;
} PrinterList[32];
Private int   current_printer_id;
#define LPTYPE_DVI    1
#define LPTYPE_PS     2
#define LPTYPE_USR1   3 
#define LPTYPE_USR2   4 
#define LPTYPE_DVI_S  "DVI"
#define LPTYPE_PS_S   "PostScript"
#define LPTYPE_USR1_S "User Defined 1" 
#define LPTYPE_USR2_S "User Defined 2" 

#define LPCMD_FILE    0 
#define LPCMD_PAGE    1 
#define LPCMD_QUEUE   2 
#define LPCMD_CANCEL  3

Private XtInputId  input_id_read = -1;
Private int        input_fd      = -1;

Private Widget make_lp_selection(Widget,Widget);
Private void     cb_print_print_file(Widget,caddr_t,caddr_t);
Private void     cb_print_print_page(Widget,caddr_t,caddr_t);
Private void     cb_print_queue(Widget,caddr_t,caddr_t);
Private void     cb_print_cancel(Widget,caddr_t,caddr_t);
Private void     cb_print_set_printer(Widget,int,caddr_t);
Private Widget make_print_confirm_panel(Widget,int);
Private void     print_confirm(Widget);
Private void     cb_print_confirm_yes(Widget,int,caddr_t);
Private void     cb_print_confirm_no(Widget,int,caddr_t);
Private void   run_print_command(int);
#if HAVE_PIPE
Private void     cb_print_abort(Widget,caddr_t,caddr_t);
Private void   kill_print_command(void);
#endif
Private void   print_message(char*);
Private void   print_vmessage(char*,...);
Private void   print_message_str(char*);
Private void   print_message_char(int);
Private void   print_message_flush(void);

Private char*  pr_get_printer_name(void);
Private char*  pr_get_printer_type_s(void);
Private char*  pr_get_printer_data(void);
Private int    command_running_asynch_process_id = -1;
Private int    command_include_output(char*, char*);
Private void   command_end(void);
Private int    command_spec_sprint(char*,int,char*);



Public Widget
x_make_panel_print(Widget parent)
{
  Widget  xpop, xfm, xtitle, xb_close, xsep1;
  Widget  xl_cmd, xrc1;
  Widget  xb_cmd_pr, xb_cmd_prpg, xb_cmd_qu, xb_cmd_ca;
  Widget  xsep2, xpr_label, xlplist; 
  Widget  xsep3, xmsg_label; 
  Arg     args[25];
  int     i;

  i = 0;
  XtSetArg(args[i], XmNdialogStyle, XmDIALOG_MODELESS); i++;
  xpop = XmCreateBulletinBoardDialog(parent, "Print", args, i);
  dialog_print = xpop;

  xfm = XmCreateForm(xpop, "print", NULL, 0);
  XtManageChild(xfm);

  i = 0;
  XtSetArg(args[i], XmNlabelString, 
	   XmStringCreate("Print DVI file",
			  XmSTRING_DEFAULT_CHARSET));  i++;
  XtSetArg(args[i], XmNtopAttachment, XmATTACH_FORM);  i++;
  XtSetArg(args[i], XmNleftAttachment, XmATTACH_FORM);  i++;
  xtitle = XmCreateLabel(xfm, "title", args, i);
  XtManageChild(xtitle);
  i = 0;
  XtSetArg(args[i], XmNlabelString, 
	   XmStringCreate("Close", XmSTRING_DEFAULT_CHARSET)); i++;
  XtSetArg(args[i], XmNrightAttachment, XmATTACH_FORM);  i++;
  XtSetArg(args[i], XmNleftAttachment, XmATTACH_NONE);  i++;
  XtSetArg(args[i], XmNtopAttachment, XmATTACH_FORM);  i++;
  xb_close  = XmCreatePushButton(xfm, "close", args, i);
  XtAddCallback(xb_close, XmNactivateCallback, 
		(XtCallbackProc)cb_print_close, NULL);
  XtManageChild(xb_close);

  xsep1 = x_p_make_hsep(xfm);
  x_p_put_below(xsep1, xtitle);
  x_p_put_below(xsep1, xb_close);

  i = 0;
  XtSetArg(args[i], XmNlabelString, 
	   XmStringCreate("Print Commands", XmSTRING_DEFAULT_CHARSET));  i++;
  XtSetArg(args[i], XmNtopAttachment, XmATTACH_WIDGET);  i++;
  XtSetArg(args[i], XmNtopWidget, xsep1);  i++;
  XtSetArg(args[i], XmNleftAttachment, XmATTACH_FORM);  i++;
  XtSetArg(args[i], XmNrightAttachment, XmATTACH_NONE);  i++;
  xl_cmd = XmCreateLabel(xfm, "cmdLabel", args, i);
  XtManageChild(xl_cmd);

  i = 0;
  XtSetArg(args[i], XmNorientation, XmHORIZONTAL);  i++;
  XtSetArg(args[i], XmNpacking, XmPACK_COLUMN);  i++;
  XtSetArg(args[i], XmNentryAlignment, XmALIGNMENT_CENTER);  i++;
  XtSetArg(args[i], XmNnumColumns, 3);  i++;
  xrc1 = XmCreateRowColumn(xfm, "printCommands", args, i);
  XtManageChild(xrc1);

  x_p_put_below(xrc1, xl_cmd);

  XtSetArg(args[0], XmNlabelString, 
	   XmStringCreate("Print File!", XmSTRING_DEFAULT_CHARSET));
  xb_cmd_pr = XmCreatePushButton(xrc1, "printFile", args, 1);
  XtManageChild(xb_cmd_pr);
  dialog_confirm_file = make_print_confirm_panel(xb_cmd_pr, LPCMD_FILE);
  XtAddCallback(xb_cmd_pr, XmNactivateCallback, 
		(XtCallbackProc)cb_print_print_file, NULL);
  XtSetArg(args[0], XmNlabelString, 
	   XmStringCreate("Print Current Page!", XmSTRING_DEFAULT_CHARSET));
  xb_cmd_prpg = XmCreatePushButton(xrc1, "printPage", args, LPCMD_PAGE);
  XtManageChild(xb_cmd_prpg);
  XtAddCallback(xb_cmd_prpg, XmNactivateCallback, 
		(XtCallbackProc)cb_print_print_page, (XtPointer)LPCMD_PAGE);
  dialog_confirm_page = make_print_confirm_panel(xb_cmd_prpg, LPCMD_PAGE);
  XtSetArg(args[0], XmNlabelString, 
	   XmStringCreate("Check Print Queue", XmSTRING_DEFAULT_CHARSET));
  xb_cmd_qu = XmCreatePushButton(xrc1, "printQueue", args, 1);
  XtAddCallback(xb_cmd_qu, XmNactivateCallback, 
		(XtCallbackProc)cb_print_queue, NULL);
  XtManageChild(xb_cmd_qu);
  XtSetArg(args[0], XmNlabelString, 
	   XmStringCreate("Cancel Print jobs", XmSTRING_DEFAULT_CHARSET));
  xb_cmd_ca = XmCreatePushButton(xrc1, "printCancel", args, 1);
  XtAddCallback(xb_cmd_ca, XmNactivateCallback, 
		(XtCallbackProc)cb_print_cancel, NULL);
  XtManageChild(xb_cmd_ca);

  XtSetArg(args[0], XmNlabelString, 
	   XmStringCreate("Abort Command", XmSTRING_DEFAULT_CHARSET));
  XtSetArg(args[1], XmNmappedWhenManaged, False);
  abort_button = XmCreatePushButton(xrc1, "printAbort", args, 2);
  XtAddCallback(abort_button, XmNactivateCallback, 
		(XtCallbackProc)cb_print_abort, NULL);
  XtManageChild(abort_button);

  xsep2 = x_p_make_hsep(xfm);
  x_p_put_below(xsep2, xrc1);

  i = 0;
  XtSetArg(args[i], XmNlabelString, 
	   XmStringCreate("Printer Selection", XmSTRING_DEFAULT_CHARSET)); i++;
  XtSetArg(args[i], XmNtopAttachment, XmATTACH_WIDGET);  i++;
  XtSetArg(args[i], XmNtopWidget, xsep2);  i++;
  XtSetArg(args[i], XmNleftAttachment, XmATTACH_FORM);  i++;
  XtSetArg(args[i], XmNrightAttachment, XmATTACH_NONE);  i++;
  xpr_label = XmCreateLabel(xfm, "printerLabel", args, i);
  XtManageChild(xpr_label);

  xlplist = make_lp_selection(xfm, xpr_label);
  x_p_put_below(xlplist, xpr_label);

  xsep3 = x_p_make_hsep(xfm);
  x_p_put_below(xsep3, xlplist);

  i = 0;
  XtSetArg(args[i], XmNlabelString, 
	   XmStringCreate("Message", XmSTRING_DEFAULT_CHARSET)); i++;
  XtSetArg(args[i], XmNtopAttachment, XmATTACH_WIDGET);  i++;
  XtSetArg(args[i], XmNtopWidget, xsep3);  i++;
  XtSetArg(args[i], XmNleftAttachment, XmATTACH_FORM);  i++;
  XtSetArg(args[i], XmNrightAttachment, XmATTACH_NONE);  i++;
  xmsg_label = XmCreateLabel(xfm, "messageLabel", args, i);
  XtManageChild(xmsg_label);

  i = 0;
  XtSetArg(args[i], XmNeditMode, XmMULTI_LINE_EDIT); i++;
  XtSetArg(args[i], XmNeditable, False); i++;
  XtSetArg(args[i], XmNwordWrap, False); i++;
  XtSetArg(args[i], XmNcolumns, RES(DviDev,print_msg_window_columns)); i++;
  XtSetArg(args[i], XmNrows,    RES(DviDev,print_msg_window_rows)); i++;
  XtSetArg(args[i], XmNcursorPositionVisible, False); i++;
  XtSetArg(args[i], XmNscrollHorizontal, True); i++;
  XtSetArg(args[i], XmNscrollVertical, True); i++;
  XtSetArg(args[i], XmNscrollLeftSide, True); i++;
  XtSetArg(args[i], XmNscrollTopSide, True); i++;
  XtSetArg(args[i], XmNtopAttachment, XmATTACH_WIDGET);  i++;
  XtSetArg(args[i], XmNtopWidget, xmsg_label);  i++;
  XtSetArg(args[i], XmNleftAttachment, XmATTACH_FORM);  i++;
  XtSetArg(args[i], XmNrightAttachment, XmATTACH_FORM);  i++;
  XtSetArg(args[i], XmNbottomAttachment, XmATTACH_FORM);  i++;
  x_print_message = XmCreateScrolledText(xfm, "printMessage", args, i);
  XtManageChild(x_print_message);

  if (RES(DviDev,debug_lpspec) == 1)
    print_message("Print debugging mode");
  cb_print_set_printer(NULL, 0, NULL);  /* Use First Printer as Default */
  
  return xpop;
}


Private Widget 
make_lp_selection(Widget parent, Widget label)
{
  char   *lplist;    /* do not free()! */
  char   *name, *type, *comm, *data;
  char   *p0, *p1, ch, lptitle[80];
  char   *pr_spec, *pg_spec, *qu_spec, *rm_spec, *type_str;
  int    i, lptype;
  Arg    args[10];
  Widget xsel, xrc, xoptm;
  Widget xlpitems[32];
  int    lpid;

  xrc = XmCreateRowColumn(parent, "lpSelect", NULL, 0);
  XtManageChild(xrc);
  xsel = XmCreatePulldownMenu(xrc, "lpSelectMenu", NULL, 0);

  if ((lplist = malloc(strlen(RES(DviDev,lplist))+1)) == NULL){
    fprintf(stderr, "No memory.");
    exit(1);
  }
  strcpy(lplist, RES(DviDev,lplist));

  p0 = lplist;
  while ((p0 = index(lplist, '\n')) != NULL)  /* '\n' -> ' ' */
    *p0 = ' '; 
  p0 = lplist;
  while ((p0 = index(lplist, '\r')) != NULL)  /* '\r' -> ' ' */
    *p0 = ' ';
  p0 = lplist;
  while ((p0 = index(lplist, '\t')) != NULL)  /* '\t' -> ' ' */
    *p0 = ' ';

  /* parse */
  lpid = 0;
  p0 = lplist;
  while (*p0 != '\0'){
    /* skip SPC before lp name */
    while ((*p0 == ' ') && (*p0 != '\0'))
      p0++;
    if (*p0 == '\0')
      break;
    /* parse NAME */
    name = p0;
    p1 = p0;
    while ((*p1 != ',') && (*p1 != ':') && (*p1 != '\0'))
      p1++;
    ch = *p1;  /* ',' or ':' or '\0' */
    *p1 = '\0';
    if ((ch == ',') || (ch == '\0')){
      type = p1;  /* null str */
      comm = p1;  /* null str */
      data = p1;  /* null str */
    } else {  /* ch == ':' */
      p1 = p1+1;
      /* parse TYPE */
      type = p1;
      while ((*p1 != ':') && (*p1 != ',') && (*p1 != '\0'))
	p1++;
      ch = *p1;
      *p1 = '\0';
      if ((ch == ',') || (ch == '\0')){
	comm = p1;  /* null str */
	data = p1;  /* null str */
      } else {  /* ch == ':' */
	p1 = p1+1;
	/* parse COMMENT */
	comm = p1;
	while ((*p1 != ':') && (*p1 != ',') && (*p1 != '\0'))
	  p1++;
	ch = *p1;
	*p1 = '\0';
	if ((ch == ',') || (ch == '\0')){
	  data = p1;  /* null str */
	} else {
	  p1 = p1+1;
	  /* parse DATA */
	  data = p1;
	  while ((*p1 != ',') && (*p1 != '\0'))
	    p1++;
	  ch = *p1;
	  *p1 = '\0';
	}
      }
    } /* ASSERT: (ch = ',') || (ch == '\0') */
    /*printf("%s %s %s %s\n", name, type, comm, data);*/

    if (strcasecmp(type, "dvi") == 0){
default_type:
      lptype = LPTYPE_DVI;
      type_str = LPTYPE_DVI_S;
      pr_spec = RES(DviDev,lpr_spec_dvi);
      pg_spec = RES(DviDev,lpr_page_spec_dvi);
      qu_spec = RES(DviDev,lpq_spec_dvi);
      rm_spec = RES(DviDev,lprm_spec_dvi);
    } else if (strcasecmp(type, "ps")== 0){
      lptype = LPTYPE_PS;
      type_str = LPTYPE_PS_S;
      pr_spec = RES(DviDev,lpr_spec_ps);
      pg_spec = RES(DviDev,lpr_page_spec_ps);
      qu_spec = RES(DviDev,lpq_spec_ps);
      rm_spec = RES(DviDev,lprm_spec_ps);
    } else if (strcasecmp(type, "usr1")== 0){
      lptype = LPTYPE_USR1;
      type_str = LPTYPE_USR1_S;
      pr_spec = RES(DviDev,lpr_spec_usr1);
      pg_spec = RES(DviDev,lpr_page_spec_usr1);
      qu_spec = RES(DviDev,lpq_spec_usr1);
      rm_spec = RES(DviDev,lprm_spec_usr1);
    } else if (strcasecmp(type, "usr2")== 0){
      lptype = LPTYPE_USR2;
      type_str = LPTYPE_USR2_S;
      pr_spec = RES(DviDev,lpr_spec_usr2);
      pg_spec = RES(DviDev,lpr_page_spec_usr2);
      qu_spec = RES(DviDev,lpq_spec_usr2);
      rm_spec = RES(DviDev,lprm_spec_usr2);
    } else {
      goto default_type;
    }
    sprintf(lptitle, "%s", comm);
    PrinterList[lpid].lp_name    = name;
    PrinterList[lpid].lp_type    = lptype;
    PrinterList[lpid].lp_type_s  = type_str;
    PrinterList[lpid].lp_comment = comm;
    PrinterList[lpid].lp_data    = data;
    PrinterList[lpid].lp_pr_spec = pr_spec;
    PrinterList[lpid].lp_pg_spec = pg_spec;
    PrinterList[lpid].lp_qu_spec = qu_spec;
    PrinterList[lpid].lp_rm_spec = rm_spec;
#if 0
    printf("Printer: %s\n", comm);
    printf("  Type: %s\n", type_str);
    printf("  Print file command spec: %s\n", pr_spec);
    printf("  Print page command spec: %s\n", pg_spec);
    printf("  Queue check command spec: %s\n", qu_spec);
    printf("  Remove queue command spec: %s\n", rm_spec);
#endif
    XtSetArg(args[0], XmNlabelString,
	     XmStringCreate(lptitle, XmSTRING_DEFAULT_CHARSET)); 
    xlpitems[lpid] = XmCreatePushButton(xsel, name, args, 1);
    XtAddCallback(xlpitems[lpid], XmNactivateCallback, 
		  (XtCallbackProc)cb_print_set_printer, (XtPointer)lpid);
    if (++lpid >= sizeof(xlpitems))
      break;
    p0 = p1; 
    if (ch == ',')
      p0++;
  }
  PrinterList[lpid].lp_name    = NULL;
  PrinterList[lpid].lp_type    = -1;
  PrinterList[lpid].lp_type_s  = NULL;
  PrinterList[lpid].lp_comment = NULL;
  PrinterList[lpid].lp_data    = NULL;
  PrinterList[lpid].lp_pr_spec = NULL;
  PrinterList[lpid].lp_pg_spec = NULL;
  PrinterList[lpid].lp_qu_spec = NULL;
  PrinterList[lpid].lp_rm_spec = NULL;

  XtManageChildren(xlpitems, lpid);

  i = 0;
  XtSetArg(args[i], XmNsubMenuId, xsel); i++;
  XtSetArg(args[i], XmNmenuHistory, xlpitems[0]); i++;
  xoptm = XmCreateOptionMenu(xrc, "lp", args, i);
  XtManageChild(xoptm);

  return xrc;
}


Private void
print_message(char *message)
{
  print_message_str(message);
  print_message_char((int)'\n');
  print_message_flush();
}

Private void
print_vmessage(char *fmt,...)
{
  va_list   ap;
  char      msg[512];

  va_start(ap, fmt);
  vsprintf(msg, fmt, ap);
  print_message(msg);
  va_end(ap);
}

Private void
print_message_char(int ch)
{
  char   msg[2];

  msg[0] = (char)ch;
  msg[1] = '\0';
  print_message_str(msg);
}

Private void
print_message_str(char *msg)
{
  int             nlines, have_nl, i;
  char            *message_buffer, *s;
  XmTextPosition  pos;

  XmTextSetInsertionPosition(x_print_message, 
			     XmTextGetLastPosition(x_print_message));

  pos = XmTextGetLastPosition(x_print_message);
  XmTextReplace(x_print_message, pos, pos, msg);

  have_nl = 0;
  for (s = msg; *s != '\0'; s++){
    if (*s == '\n'){
      have_nl = 1;
      break;
    }
  }
  if (have_nl == 0)
    return;

  message_buffer = XmTextGetString(x_print_message);
  nlines = 0;
  for (i = 0; message_buffer[i] != '\0'; i++){
    if (message_buffer[i] == '\n')
      nlines++;
  }
  while ((nlines > RES(DviDev,print_msg_history_size))
	 && (nlines >= 1)){
    for (pos = 0; message_buffer[pos] != '\0'; pos++)
      if (message_buffer[pos] == '\n')
	break;
    XmTextReplace(x_print_message, 0, pos+1, "");
    --nlines;
  }
  XtFree(message_buffer);
}

Private void
print_message_flush(void)
{
  XFlush(XtDisplay(x_print_message));
}


Public void
cb_print(Widget w, caddr_t d1, caddr_t d2)
{
  if (XtIsManaged(dialog_print)){
    XtUnmanageChild(dialog_print);
  } else {
    ui_close_panel_all();
    XtManageChild(dialog_print);
  }
}

Public void
cb_print_close(Widget w, caddr_t d1, caddr_t d2)
{
  if (XtIsManaged(dialog_print))
    XtUnmanageChild(dialog_print);
}

Private void
cb_print_set_printer(Widget w, int lp_id, caddr_t d2)
{
  if (current_printer_id == lp_id)
    return;
  print_vmessage("Current Printer:  %s",
		 PrinterList[lp_id].lp_comment);
  current_printer_id = lp_id;  
}

Private char*
pr_get_printer_name(void)
{
  return PrinterList[current_printer_id].lp_name;
}

Private char*
pr_get_printer_type_s(void)
{
  return PrinterList[current_printer_id].lp_type_s;
}

Private char*
pr_get_printer_data(void)
{
  return PrinterList[current_printer_id].lp_data;
}



Private Widget
make_print_confirm_panel(Widget parent, int cmd)
{
  Widget  xpr, xh;
  Arg     args[10];
  int     i;

  i = 0;
  switch (cmd){
  case LPCMD_FILE:   
    XtSetArg(args[i], XmNmessageString, 
	     XmStringCreate("Do you really want to print out the file?", 
			    XmSTRING_DEFAULT_CHARSET)); i++;
    break;
  case LPCMD_PAGE:   
    XtSetArg(args[i], XmNmessageString, 
	     XmStringCreate("Do you really want to print out this page?", 
			    XmSTRING_DEFAULT_CHARSET)); i++;
    break;
  default:
    fprintf(stderr, "%s: Can't happen in make_print_confirm_panel().\n",
	    "XMDVI Internal Error");
    exit(1);
  }
  XtSetArg(args[i], XmNokLabelString, 
	   XmStringCreate("Yes", XmSTRING_DEFAULT_CHARSET)); i++;
  XtSetArg(args[i], XmNcancelLabelString,
	   XmStringCreate("No", XmSTRING_DEFAULT_CHARSET)); i++;
  XtSetArg(args[i], XmNdialogStyle, XmDIALOG_MODELESS); i++;
  xpr = XmCreateQuestionDialog(parent, "printConfirm", args, i);
  XtAddCallback(xpr, XmNokCallback, 
		(XtCallbackProc)cb_print_confirm_yes, (XtPointer)cmd);
  XtAddCallback(xpr, XmNcancelCallback, 
		(XtCallbackProc)cb_print_confirm_no, (XtPointer)cmd);

  xh = XmMessageBoxGetChild(xpr, XmDIALOG_HELP_BUTTON);
  XtUnmanageChild(xh);

  return xpr;
}

Private void
print_confirm(Widget dialog)
{
  if (! XtIsManaged(dialog))
    XtManageChild(dialog);
}

Private void
cb_print_confirm_yes(Widget w, int cmd, caddr_t d2)
{
  XtUnmanageChild(w);
  run_print_command(cmd);
}

Private void
cb_print_confirm_no(Widget w, int cmd, caddr_t d2)
{
  XtUnmanageChild(w);
}


Private void  
cb_print_print_file(Widget w, caddr_t d1, caddr_t d2)
{
  if ((DviFileName == NULL) || (DviFile == NULL)){
    x_bell();
    print_message("DVI file is not selected.");
    return;
  }

  if (RES(DviDev,novice) == 1)
    print_confirm(dialog_confirm_file);
  else 
    run_print_command(LPCMD_FILE);
}

Private void  
cb_print_print_page(Widget w, caddr_t d1, caddr_t d2)
{
  if ((DviFileName == NULL) || (DviFile == NULL)){
    x_bell();
    print_message("DVI file is not selected.");
    return;
  }

  if (RES(DviDev,novice) == 1)
    print_confirm(dialog_confirm_page);
  else 
    run_print_command(LPCMD_PAGE);
}

Private void  
cb_print_queue(Widget w, caddr_t d1, caddr_t d2)
{
  run_print_command(LPCMD_QUEUE);
}

Private void  
cb_print_cancel(Widget w, caddr_t d1, caddr_t d2)
{
  run_print_command(LPCMD_CANCEL);
}

#if HAVE_PIPE
Private void  
cb_print_abort(Widget w, caddr_t d1, caddr_t d2)
{
  if (command_running_asynch_process_id <= 0){
    x_bell();
    print_vmessage("Command is not runing.");
    return;
  }

  kill_print_command();
}
#endif


Private void 
run_print_command(int cmd)
{
  int    res;
  char   cmdline[BUFSIZ], realspec[BUFSIZ], errfile[512];
  char   *spec;

  if (command_running_asynch_process_id > 0){
    x_bell();
    return;
  }

  switch (cmd){
  case LPCMD_FILE:   spec = PrinterList[current_printer_id].lp_pr_spec; break;
  case LPCMD_PAGE:   spec = PrinterList[current_printer_id].lp_pg_spec; break;
  case LPCMD_QUEUE:  spec = PrinterList[current_printer_id].lp_qu_spec; break;
  case LPCMD_CANCEL: spec = PrinterList[current_printer_id].lp_rm_spec; break;
  default:
    fprintf(stderr, "XMDVI Internal Error: Unknown print command.\n");
    return;
  }
  if (RES(DviDev,debug_lpspec) == 1)
    print_vmessage("DEBUG: Spec=%s", spec);

  if (command_spec_sprint(realspec, sizeof(realspec), spec) < 0){
    switch (cmd){
    case LPCMD_FILE:   print_message("Failed to print the file.");     break;
    case LPCMD_PAGE:   print_message("Failed to print current page."); break;
    case LPCMD_QUEUE:  print_message("Failed to check print queue.");  break;
    case LPCMD_CANCEL: print_message("Failed to cancel print jobs.");  break;
    default:
      fprintf(stderr, "XMDVI Internal Error: Unknown print command.\n");
      return;
    }
    goto end;
  }

  if (RES(DviDev,debug_lpspec) == 1){
    print_vmessage("DEBUG: Printer Class: %s", 
	      PrinterList[current_printer_id].lp_type_s);
    print_vmessage("DEBUG: Command: %s", realspec);
    return;
  }

#if HAVE_PIPE
  sprintf(cmdline, "%s", realspec);
#else
  sprintf(errfile, "%s/XXXXXX", RES(DviDev,temp_dir));
  mktemp(errfile);
  if (strcmp(errfile, "") == 0)
    return;
  sprintf(cmdline, "(%s) >%s 2>&1", realspec, errfile);
#endif /*HAVE_PIPE*/

  print_message(""); 
  switch (cmd){
  case LPCMD_FILE:
    print_vmessage("Sending %s on %s ...", 
		   DviFileNameBase, pr_get_printer_name());
    break;
  case LPCMD_PAGE:
    print_vmessage("Sending a page of %s on %s ...", 
		   DviFileNameBase, pr_get_printer_name());
    break;
  case LPCMD_QUEUE:
    print_vmessage("Checking print job queue on %s ...", 
		   pr_get_printer_name()); 
    break;
  case LPCMD_CANCEL:
    print_vmessage("Canceling print jobs on %s ...", 
		   pr_get_printer_name()); 
    break;
  }

  res = command_include_output(cmdline, errfile);

end:
  command_end();
  unlink(errfile);
}


#if HAVE_PIPE
Private void
cb_process_output(caddr_t d, int *fdp, XtInputId *id)
{
  int   n, st, p;
  char  ch;

  if ((fdp != NULL)
      && ((n = read(*fdp, &ch, 1)) > 0)){
    print_message_char(ch);
  } else {
    while (command_running_asynch_process_id > 0){
      p = wait(&st); 
      if (p == command_running_asynch_process_id){
	command_running_asynch_process_id = -1;
	XtRemoveInput(*id);
	if (fdp != NULL){
	  close(*fdp);
	  print_message("done.");
	} else {
	  close(input_fd);
	  print_message("aborted.");
	}
	XtUnmapWidget(abort_button);
	input_id_read = -1;
	input_fd      = -1;
      }
    }
  }
}

Private void
kill_print_command(void)
{
  if (command_running_asynch_process_id > 0){
    if (command_running_asynch_process_id > 0)
      kill(command_running_asynch_process_id, SIGKILL);
    cb_process_output(NULL, NULL, &input_id_read);
  }
}

Private int
command_include_output(char *cmdline, char *junk_arg)
{
  int   pipes[2], pid, devnull;

  command_running_asynch_process_id = -1;
  input_id_read = -1;
  input_fd      = -1;

  if (pipe(pipes) < 0)
    return -1;

  pid = fork();
  if (pid < 0)
    return -1;
  else if (pid > 0){  /* parent: xmdvi */
    close(pipes[1]);
    input_fd = pipes[0];
    command_running_asynch_process_id = pid;
    input_id_read = XtAddInput(input_fd, (caddr_t)XtInputReadMask, 
			       (XtInputCallbackProc)cb_process_output, NULL);
    XtMapWidget(abort_button);
  } else {            /* child:  lp command */
    close(0);
    close(1);
    close(2);
    if ((devnull = open("/dev/null", O_RDONLY)) >= 0)
      dup2(devnull, 0);
    dup2(pipes[1], 1);
    dup2(pipes[1], 2);
    close(pipes[0]);
    close(pipes[1]);
#if 0
    system(cmdline);
    close(1);
    close(2);
    exit(0);
#else
    execl("/bin/sh", "sh", "-c", cmdline, NULL);
    exit(0);
#endif
  }
  return 0;
}
#else  /*!HAVE_PIPE*/
Private int
command_include_output(char *cmdline, char *f)
{
  int   n, res;
  FILE  *fp;
  char  buff[BUFSIZ], *p;

  res = system(cmdline);
  n = 0;
  if ((fp = fopen(f, "r")) != NULL){
    for (;;){
      if (fgets(buff, sizeof(buff)-1, fp) == NULL)
	break;
      buff[sizeof(buff)-1] = '\0';
      if ((p = index(buff, '\n')) != NULL)
	*p = '\0';
      if ((n == 0) && (strcmp(buff, "") == 0))
	continue;
      n++;
      print_message(buff);
    }
    fclose(fp);
  }
  return n;
}
#endif /*HAVE_PIPE*/

Private char command_tempfile[128];

Private void
command_end(void)
{
  if (strcmp(command_tempfile, "") != 0)
    unlink(command_tempfile);
}

Private int
command_spec_sprint(char *buff, int n, char *spec)
{
  int            s, b, sval_len;
  char           cmd, *sval, temp[MAXPATHLEN];
  char           path[MAXPATHLEN], *p;
  struct passwd  *user_info;

  strcpy(command_tempfile, "");
  s = 0;
  b = 0;
  for (;;){
    sval = NULL;
    if (spec[s] != '%'){
      if (b >= n)
	goto TooLongError;
      if ((spec[s] == '\n') || (spec[s] == '\r')){
	buff[b] = ' ';   /* ignore CRLF */
	b++; s++; 
      } else if ((buff[b++] = spec[s++]) == '\0')
	return 0;
    } else {
      s++;
      cmd = spec[s++];
      switch (cmd){
      default:
	print_vmessage("Unknown spec command: %%%c.", cmd);
	sval = NULL;
	break;
      case '\0': /* "... %" */
	if (b >= n)
	  goto TooLongError;
	buff[b++] = '\0';
	return 0;
      case '%':  /* %% */
	sval = "%";
	break;
      case 'f': /* %f: DVI file name */
	sval = DviFileName;
	break;
      case 'b': /* %b: DVI file name (base name) */
      case 'B': /* %B: DVI file name without ".dvi" extension (base name) */
      case 'F': /* %F: DVI file name (full path) */
      case 'd': /* %d: Directory name of the DVI file */
	if (DviFileName[0] == '/'){  /* absolute path */
	  sprintf(path, "%s", DviFileName); 
	} else {         /* relative path */
	  if ((p = getenv("PWD")) != NULL)
	    strcpy(temp, p);
	  else if (getcwd(temp, sizeof(temp)) == NULL){
	    print_message("Failed to get current directory name.");
	    sval = NULL;
	    break;
	  }
	  if (strcmp(temp, "/") == 0)
	    sprintf(path, "/%s", DviFileName); 
	  else 
	    sprintf(path, "%s%s%s", temp, "/", DviFileName); 
	}
	sval = path;
	switch (cmd){
	case 'b':
	  if ((p = rindex(path, '/')) != NULL)
	    sval = p+1;
	  break;
	case 'B':
	  if ((p = rindex(path, '/')) != NULL)
	    sval = p+1;
	  if ((p = rindex(sval, '.')) != NULL)
	      if (strcmp(p, DVI_FILE_EXTENSION) == 0)
		*p = '\0';
	  break;
	case 'F':
	  break;
	case 'd':
	  if ((p = rindex(path, '/')) != NULL)
	    *p = '\0';
	  break;
	default:
	  print_message("Internal error #1 in command_spec_sprint()");
	  sval = NULL;
	}
	break;
      case 'c': /* %c: Current directory */
	sval = path;
	if ((p = getenv("PWD")) != NULL){
	  strcpy(path, p);
	} else if (getcwd(path, sizeof(path)) == NULL){
	  print_message("Failed to get current directory name.");
	  sval = NULL;
	}
	break;
      case 'p': /* %p: Printer name (selected by the DVI Print window) */
        sval = pr_get_printer_name();
	break;
      case 'P': /* %P: printer name. (like '%p') If environment variable 
		   PRINTER is defined, use it instead. */
	if ((sval = getenv("PRINTER")) == NULL)
	  sval = pr_get_printer_name();
	break;
      case 'R': /* %R : The value of the env var PRINTER. If undefied, 
		   DEFAULT_PRINTER is used for %R. */
	if ((sval = getenv("PRINTER")) == NULL)
	  sval = DEFAULT_PRINTER; 
	break;
      case 'a': /* %a : DATA specified by Xmdvi.lpList resource. 
		   If undefied, "" is used.  */
	if ((sval = pr_get_printer_data()) == NULL)
	  sval = "";
	break;
      case 'y': /* %y : Printer Type. (One of "DVI", "PostScript", 
		   "User Defined 1", or "Usr Defined 2".) */
	if ((sval = pr_get_printer_type_s()) == NULL)
	  sval = "";
	break;
      case 'h': /* %h: Host name */
	sval = temp;
#if HAVE_GETHOSTNAME
	if (gethostname(temp, sizeof(temp)) != 0){
	  print_message("Failed to get host name.");
	  sval = NULL;
	}
#else
	print_message("Warning: %h (host name) is not supported.");
	sprintf(temp, "");
#endif
	break;
      case 'D': /* %D: Domain name */
	sval = temp;
#if HAVE_GETDOMAINNAME
	if (getdomainname(temp, sizeof(temp)) != 0){
	  print_message("Failed to get domain name.");
	  sval = NULL;
	}
#else
	print_message("Warning: %D (domain name) is not supported.");
	strcpy(temp, "");
#endif
	break;
      case 'T': /* %t: Temporary directory specified as resource */
	sval = RES(DviDev,temp_dir);
	break;
      case 't': /* %t: Make new temporary file */
	sval = command_tempfile;
	sprintf(command_tempfile, "%s/XXXXXX", RES(DviDev,temp_dir));
	mktemp(command_tempfile);
	if (strcmp(command_tempfile, "") == 0){
	  print_message("Failed to make temporary file.");
	  sval = NULL;
	}
	break;
      case 'r': /* %r: Process id */
	sval = temp;
#if HAVE_GETPID
	sprintf(temp, "%d", (int)getpid());
#else
	print_message("Warning: %r (process is) is not supported.");
	sprintf(sval, "1");
#endif
	break;
      case 'l': /* %l: Login name */
#if HAVE_GETLOGIN
	if ((sval = getlogin()) == NULL){
	  print_message("Who are you?");
	}
#else
	print_message("Warning: %l (login name) is not supported.");
	sval = temp;
	sprintf(sval, "me");
#endif
	break;
#if HAVE_GETPWUID
# if HAVE_GETUID
#  if HAVE_PWD_H
      case 'u': /* %u: User name */
      case 'H': /* %H: Home directory */
	if ((user_info = getpwuid(getuid())) == NULL){
	  print_message("Who are you?");
	  sval = NULL;
	  break;
	}
	switch (cmd){
	case 'u':
	  sval = user_info->pw_gecos;
	  break;
	case 'H':
	  sval = user_info->pw_dir;
	  break;
	default:
	  print_message("Internal error #2 in command_spec_sprint()");
	}
	break;
#  else
      case 'u': /* %u: User name */
	print_message("Warning: %u (user name) is not supported.");
	sval = temp;
	sprintf(sval, "me");
	break;
      case 'H': /* %H: Home directory */
	print_message("Warning: %H (home directory) is not supported.");
	sval = temp;
	sprintf(sval, ".");
	break;
#  endif
# endif
#endif
      case 'i': /* %i: User id */
	sval = temp;
#if HAVE_GETUID
	sprintf(temp, "%d", (int)getuid());
#else
	print_message("Warning: %i (user id) is not supported.");
	sprintf(sval, "1");
#endif
	break;
      case 'g': /* %g: Group id */
	sval = temp;
#if HAVE_GETGID
	sprintf(temp, "%d", (int)getgid());
#else
	print_message("Warning: %g (group id) is not supported.");
	sprintf(sval, "1");
#endif
	break;
      case 'C': /* %C: Current (absolute) page number of the DVI file  
		   (1 <= %C <= %N) */
	sval = temp;
	sprintf(temp, "%d", DviCurrentPage);
	break;
      case 'N': /* %N: The number of pages in the DVI file */
	sval = temp;
	sprintf(temp, "%d", DviFile->pages);
	break;
      }
      if (sval == NULL){
	buff[b] = '\0';
	return -1;
      }
      sval_len = strlen(sval);
      if (b+sval_len >= n)
	goto TooLongError;
      strcpy(&(buff[b]), sval);
      b += sval_len;
    }
  }

TooLongError:
  print_vmessage("Print spec is too long.");
  return -1;
}
