/* -*- mode: c; c-file-style: "gnu" -*-
 * lru.c -- lru routines
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of BoneHunter Libs.
 *
 * BoneHunter Libs is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 dated June,
 * 1991.
 *
 * BoneHunter Libs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file lru.c
 * This file implements a simple LRU cache mechanism. The data is
 * internally stored in a doubly-linked list, with the most recent
 * entry on top - this makes both finding and aging out quite
 * painless.
 */

#include "compat/compat.h"
#include "lru.h"

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/** An element in the LRU cache.
 * This structure describes an element in the LRU cache. It is as
 * content-agnostic as possible. Apart from the real data, there is
 * only one more member: a pointer to the next element in the chain.
 */
typedef struct
{
  void *next; /**< Pointer to the next element. */
  void *prev; /**< Pointer to the previous element. */

  size_t size; /**< Size of the data. */
  void *data; /**< The data itself. */
} bhl_lru_entry_t;

/** Head element of an LRU object.
 * This element stores the head of an LRU object: its current and
 * maximum size, and a link to the first entry. We can walk through
 * the list from that.
 */
typedef struct
{
  size_t size; /**< Size of the LRU cache. */
  size_t max; /**< Maximum size of the LRU cache. */
  bhl_list_free_f efree; /**< Enhanced element free()ing routine. */

  bhl_lru_entry_t *head; /**< A pointer to the first entry in the
			    cache. */
} bhl_lru_t;

/** Create a new LRU object.
 * Initialises a new LRU objects, with zero initial elements.
 *
 * @param max is the maximum number of entries in the cache.
 * @param efree is either a function to free one list element, or
 * NULL, if the standard free() call should be used instead.
 *
 * @returns A newly allocated LRU object.
 */
void *
bhl_lru_new (size_t max, bhl_list_free_f efree)
{
  bhl_lru_t *lru;

  lru = (bhl_lru_t *)bhc_malloc (sizeof (bhl_lru_t));
  lru->max = max;
  lru->size = 0;
  lru->head = NULL;
  lru->efree = efree;

  return (void *)lru;
}

/** @internal Delete the last element of an LRU cache.
 * Removes the last element of the LRU cache.
 *
 * @param lru is the cache to remove the last element from.
 *
 * @note Modifies lru in-place, and does not decrement the cache size!
 * Decrementing lru->size is the job of the caller.
 */
static void
_bhl_lru_delete_last (bhl_lru_t *lru)
{
  bhl_lru_entry_t *e, *p;

  /* Find the last entry */
  p = NULL;
  e = lru->head;
  while (e->next)
    {
      p = e;
      e = e->next;
    }

  /* Free the last element */
  if (e)
    {
      if (lru->efree)
	(*(lru->efree)) (e->data);
      free (e->data);
      free (e);
    }

  /* The previous one's next should point to the void. */
  if (p)
    p->next = NULL;
  else
    lru->head = NULL;
}

/** Add a new element to the LRU cache.
 * Adds a new element to the cache, freeing the oldest one first, if
 * necessary.
 *
 * @param lru is the LRU cache object.
 * @param data is the data to store.
 * @param size is the size of the data.
 *
 * @note A copy of the data is stored in the cache, so the caller can
 * freely do whatever it wants with the original copy.
 */
void
bhl_lru_add (void *lru, const void *data, size_t size)
{
  bhl_lru_t *list = (bhl_lru_t *)lru;
  bhl_lru_entry_t *entry;

  entry = (bhl_lru_entry_t *)bhc_malloc (sizeof (bhl_lru_entry_t));
  entry->size = size;
  entry->data = bhc_malloc (size);
  entry->next = NULL;
  entry->prev = NULL;
  memcpy (entry->data, data, size);

  if (list->head)
    {
      if (list->size >= list->max)
	{
	  _bhl_lru_delete_last (list);
	  list->size--;
	}

      entry->next = list->head;
      if (list->head)
	list->head->prev = entry;
      list->head = entry;
    }
  else
    list->head = entry;

  list->size++;
}

/** Find an entry in the cache.
 * Iterates through the cache until the finder function says there is
 * a match, or if there are no more entries.
 *
 * @param lru is the LRU object.
 * @param finder is the finder function.
 * @param fdata is the data we're searching.
 * @param fsize is the size of that data.
 *
 * @returns The matching data or NULL, if none was found.
 */
void *
bhl_lru_find (void *lru, bhl_lru_finder_f finder, const void *fdata,
	      size_t fsize)
{
  bhl_lru_t *list = (bhl_lru_t *)lru;
  bhl_lru_entry_t *e, *p, *n;

  e = (bhl_lru_entry_t *)list->head;
  while (e != NULL)
    {
      if ((*finder) (e->data, e->size, fdata, fsize) != 0)
	{
	  if (e != list->head)
	    {
	      /* Ok, we found something, and it is not the first entry.
		 We pick it out, and put at the beginning. */
	      p = e->prev;
	      n = e->next;
	      if (p)
		p->next = n;
	      if (n)
		n->prev = p;

	      e->next = list->head;
	      e->prev = NULL;
	      list->head->prev = e;
	      list->head = e;
	    }
	  return e->data;
	}
      e = (bhl_lru_entry_t *)e->next;
    };

  return NULL;
}

/** Delete an entry from the cache.
 * Searches through the cache to find an element whose data pointer
 * points to the same location as the one passed to the function, then
 * simply removes it.
 *
 * We use this simple comparsion for two reasons: first, it is
 * faster. Second, the passed argument is most probably a result of a
 * bhl_lru_find(), therefore is a pointer into our own cache.
 *
 * @param lru is the cache to delete from.
 * @param data is the data to remove.
 *
 * @note Use with care.
 */
void
bhl_lru_delete (void *lru, const void *data)
{
  bhl_lru_t *list = (bhl_lru_t *)lru;
  bhl_lru_entry_t *e, *p, *n;

  e = (bhl_lru_entry_t *)list->head;
  while (e != NULL)
    {
      /* We can do a == comparsion here, since data is supposed to be
	 coming directly from us. */
      if (e->data == data)
	{
	  p = e->prev;
	  n = e->next;

	  if (p)
	    p->next = n;
	  else
	    list->head = n;
	  if (n)
	    n->prev = p;

	  if (list->efree)
	    (*(list->efree)) (e->data);
	  free (e->data);
	  free (e);

	  return;
	}
      e = e->next;
    }
}

/* arch-tag: dac668c9-5e57-4f08-b1a9-1eaf41cef331 */
