/*
 * Copyright (C) 2004  Stefan Kleine Stegemann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <Foundation/NSException.h>
#include <Foundation/NSThread.h>
#include "SearchService.h"
#include "DocumentPosition.h"


/* Possible values for stopSyncLock's condition  */
#define SERVICE_IS_RUNNING      1
#define SERVICE_HAS_TERMINATED  2
#define SERVICE_NOT_RUNNING     3


/*
 * Non-Public methods.
 */
@interface SearchService (Private)
- (void) _pickNewMatches: (NSTimer*)aTimer;
- (void) _performSearch: (id)anObject;
@end


@implementation SearchService

- (id) initWithText: (NSString*)aTextFragment
        forDocument: (PDFDocument*)aDocument
     startingAtPage: (int)aPage
         atPosition: (NSRect)aPagePosition
           delegate: (id<SearchServiceDelegate>)aDelegate
{
   NSAssert(aDocument, @"document to search cannot be nil");
   NSAssert(aTextFragment, @"text fragment to search for cannot be nil");
   NSAssert((aPage > 0) && (aPage <= [aDocument countPages]), @"start-page out of range");

   if ((self = [super init]))
   {
      document     = RETAIN(aDocument);
      text         = [aTextFragment copy];
      page         = aPage;
      pageCount    = 0;
      position     = aPagePosition;
      delegate     = aDelegate;
      results      = [[NSMutableArray alloc] init];
      resultsLock  = [[NSLock alloc] init];
      
      [delegate serviceWillStartSearch: self];

      // The resultPicker will periodically pick new matches produced
      // by the search thread and forward the matches to the service's
      // delegate.
      // TODO: switch to more intelligent inter-thread-communication
      stopPicking     = NO;
      releaseWhenDone = NO;
      resultPicker    = 
         [NSTimer scheduledTimerWithTimeInterval: 0.15
                                          target: self
                                        selector: @selector(_pickNewMatches:)
                                        userInfo: nil
                                         repeats: YES];

      // Detach a thread that performs the serach in the background.
      shouldStop = NO;
      [NSThread detachNewThreadSelector: @selector(_performSearch:)
                               toTarget: self
                             withObject: nil];
   }

   return self;
}


- (void) dealloc
{
   NSLog(@"dealloc SearchService");

   if (resultPicker)
   {
      [resultPicker invalidate];
   }
   
   RELEASE(document);
   RELEASE(text);
   RELEASE(results);
   RELEASE(resultsLock);

   [super dealloc];
}


- (void) setDelegate: (id)aDelegate
{
   delegate = aDelegate;
}


- (BOOL) isRunning
{
   return (resultPicker != nil);
}


- (void) stopAndReleaseWhenDone: (BOOL)flag
{
   if (!resultPicker)
   {
      [NSException raise: NSInternalInconsistencyException
                  format: @"SearchService is not running"];
   }

   releaseWhenDone = flag;
   shouldStop      = YES;
}


- (PDFDocument*) document
{
   return document;
}


- (int) countSearchedPages
{
   return pageCount;
}

@end


/* ----------------------------------------------------- */
/*  Category Private                                     */
/* ----------------------------------------------------- */

@implementation SearchService (Private)

/** Forward all matches that have arrived since the
 *  last invocation of this method to the service's
 *  delegate.  */
- (void) _pickNewMatches: (NSTimer*)aTimer
{
   NSArray* matches;

   //NSLog(@"pick new matches");

   [resultsLock lock];
   matches = [NSArray arrayWithArray: results];
   [results removeAllObjects];
   [resultsLock unlock];

   if ([matches count] > 0)
   {
      [delegate service: self didFoundMatches: matches];
   }
   
   if (stopPicking)
   {
      [delegate serviceDidFinishSearch: self];
      [resultPicker invalidate];
      resultPicker = nil;

      if (releaseWhenDone)
      {
         AUTORELEASE(self);
      }
   }
}


/** Perform the search in the background. Found matches
 *  are stored in the results array.  */
- (void) _performSearch: (id)anObject
{
   NSAutoreleasePool* threadPool;
   PDFSearchContext*  searchContext;
   int                startPage;
   NSString*          textContext;

   NSLog(@"search thread started");

   threadPool = [[NSAutoreleasePool alloc] init];

   searchContext = RETAIN([document createSearchContext]);
   startPage     = page;
   pageCount     = 0;
   
   do
   {
      if ([delegate service: self willStartSearchingPage: page])
      {
         //NSLog(@"start searching page %d", page);

         // find all matches on the current page
         while ([document findText: text
                usingSearchContext: searchContext
                              page: &page
                            toPage: page
                          position: &position
                       textContext: &textContext])
         {
            DocumentPosition* dpos = [DocumentPosition positionAtPage: page
                                                         boundingRect: position];
            Match* match = [[Match alloc] initWithPosition: dpos
                                                   context: textContext
                                                searchText: text];

            [resultsLock lock];
            [results addObject: match];
            [resultsLock unlock];

            RELEASE(match);

            //NSLog(@"found match at page %d", page);
            
            // move position after match
            position.origin.x = position.origin.x + position.size.width;
            position.origin.y = position.origin.y + (position.size.height / 2);
            position.size.height = 0;
         }
      }

      pageCount++;
      [delegate service: self didFinishedSearchingPage: page];

      // advance to next page
      page++;
      position = NSMakeRect(0, 0, 0, -1);
      if (page > [document countPages])
      {
         page = 1; 
      }
      
   } while (!shouldStop && (page != startPage));

   RELEASE(searchContext);

   [threadPool release];

   stopPicking = YES;

   NSLog(@"search thread will terminate");
}

@end
