/*
 * Copyright (C) 2007-2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef INCLUDE
#endif /* INCLUDE */
#ifdef STATE

// struct {
#define CPU_TLB_BITS 8
#define CPU_TLB_SIZE (1 << CPU_TLB_BITS)
	struct CPUTLBEntry {
		/*
		 * Bit max to 12 : virtual address
		 * Bit 11 to  4  : if non zero, memory io zone number
		 * Bit 3         : indicates that the entry is invalid
		 * Bit 2..0      : zero
		 */
		vaddr_t address;

		/*
		 * Addend to virtual address to get address to be accessed.
		 * WARNING: In case of RAM and ROM the address is the address
		 * where the image is mapped into host address space and not the
		 * emulated physical address!
		 */
		Haddr host_addend;

		/* Addend to virtual address to get physical address. */
		paddr_t phys_addend;
	} tlb_read[2][CPU_TLB_SIZE], tlb_write[2][CPU_TLB_SIZE], tlb_code[2][CPU_TLB_SIZE];
// } NAME;

#endif /* STATE */
#ifdef EXPORT

/*forward*/ static void
CHIP_(tlb_protect)(struct cpssp *cpssp, paddr_t pa);
/*forward*/ static void
CHIP_(tlb_unprotect)(struct cpssp *cpssp, paddr_t pa);

/*forward*/ static paddr_t
CHIP_(tlb_virt_to_phys)(struct cpssp *cpssp, vaddr_t va);

#if 80486 <= CONFIG_CPU
/*forward*/ static void
CHIP_(code_invlpg)(struct cpssp *cpssp, vaddr_t addr);
/*forward*/ static void
CHIP_(data_invlpg)(struct cpssp *cpssp, vaddr_t addr);
#endif
/*forward*/ static void
CHIP_(code_tlb_flush)(struct cpssp *cpssp, int flush_global);
/*forward*/ static void
CHIP_(data_tlb_flush)(struct cpssp *cpssp, int flush_global);
/*forward*/ static void
CHIP_(code_unmap)(struct cpssp *cpssp, paddr_t start, paddr_t len);
/*forward*/ static void
CHIP_(data_unmap)(struct cpssp *cpssp, paddr_t start, paddr_t len);

/*forward*/ static void
CHIP_(tlb_fill)(struct cpssp *cpssp, vaddr_t addr, int is_write, int is_user, void *retaddr);

#endif /* EXPORT */
#ifdef BEHAVIOR

#define TLB_INVALID_MASK        (1 << 3)
#define IO_MEM_SHIFT            4

#define IO_MEM_RAM              (0 << IO_MEM_SHIFT) /* hardcoded offset */
#define IO_MEM_CODE             (1 << IO_MEM_SHIFT) /* used internally */
#define IO_MEM_IO               (2 << IO_MEM_SHIFT)
#define IO_MEM_IO_CODE          (3 << IO_MEM_SHIFT) /* used internally */
#define IO_MEM_NB_ENTRIES       4

static uint32_t
CHIP_(mr_data_b)(struct cpssp *cpssp, paddr_t pa)
{
	udata_t val;

	assert(! (pa & 0));

	CHIP_(a20gate_mr)(cpssp, pa & ~(sizeof(udata_t) - 1),
			0b1 << (pa & (sizeof(udata_t) - 1)), &val);

	return (val >> ((pa & (sizeof(udata_t) - 1)) * 8)) & 0xff;
}

static uint32_t
CHIP_(mr_data_w)(struct cpssp *cpssp, paddr_t pa)
{
	udata_t val;

	assert(! (pa & 1));

	CHIP_(a20gate_mr)(cpssp, pa & ~(sizeof(udata_t) - 1),
			0b11 << (pa & (sizeof(udata_t) - 1)), &val);

	return (val >> ((pa & (sizeof(udata_t) - 1)) * 8)) & 0xffff;
}

static uint32_t
CHIP_(mr_data_l)(struct cpssp *cpssp, paddr_t pa)
{
	udata_t val;

	assert(! (pa & 3));

	CHIP_(a20gate_mr)(cpssp, pa & ~(sizeof(udata_t) - 1),
			0b1111 << (pa & (sizeof(udata_t) - 1)), &val);

	return (val >> ((pa & (sizeof(udata_t) - 1)) * 8)) & 0xffffffff;
}

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
static uint64_t
CHIP_(mr_data_q)(struct cpssp *cpssp, paddr_t pa)
{
	if (sizeof(udata_t) < sizeof(uint64_t)) {
		uint32_t v1;
		uint32_t v2;

		v1 = CHIP_(mr_data_l)(cpssp, pa + 0);
		v2 = CHIP_(mr_data_l)(cpssp, pa + 4);

		return v1 | ((uint64_t) v2 << 32);
	} else {
		udata_t val;

		assert(! (pa & 7));

		CHIP_(a20gate_mr)(cpssp, pa & ~(sizeof(udata_t) - 1),
				0b11111111 << (pa & (sizeof(udata_t) - 1)), &val);

		return (val >> ((pa & (sizeof(udata_t) - 1)) * 8)) & 0xffffffffffffffffULL;
	}
}
#endif /* CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT */

static void
CHIP_(mw_data_b)(struct cpssp *cpssp, paddr_t pa, uint32_t val)
{
	udata_t valX;

	assert(! (pa & 0));

	valX = ((udata_t) val) << ((pa & (sizeof(udata_t) - 1)) * 8);

	CHIP_(a20gate_mw)(cpssp, pa & ~(pa & (sizeof(udata_t) - 1)),
			0b1 << (pa & (sizeof(udata_t) - 1)), valX);
}

static void
CHIP_(mw_data_w)(struct cpssp *cpssp, paddr_t pa, uint32_t val)
{
	udata_t valX;

	assert(! (pa & 1));

	valX = ((udata_t) val) << ((pa & (sizeof(udata_t) - 1)) * 8);

	CHIP_(a20gate_mw)(cpssp, pa & ~(pa & (sizeof(udata_t) - 1)),
			0b11 << (pa & (sizeof(udata_t) - 1)), valX);
}

static void
CHIP_(mw_data_l)(struct cpssp *cpssp, paddr_t pa, uint32_t val)
{
	udata_t valX;

	assert(! (pa & 3));

	valX = ((udata_t) val) << ((pa & (sizeof(udata_t) - 1)) * 8);

	CHIP_(a20gate_mw)(cpssp, pa & ~(pa & (sizeof(udata_t) - 1)),
			0b1111 << (pa & (sizeof(udata_t) - 1)), valX);
}

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
static void
CHIP_(mw_data_q)(struct cpssp *cpssp, paddr_t pa, uint64_t val)
{
	if (sizeof(udata_t) < sizeof(val)) {
		uint32_t v1;
		uint32_t v2;

		v1 = (uint32_t) val;
		v2 = val >> 32;
		CHIP_(mw_data_l)(cpssp, pa + 0, v1);
		CHIP_(mw_data_l)(cpssp, pa + 4, v2);
	} else {
		udata_t valX;

		assert(! (pa & 7));

		valX = ((udata_t) val) << ((pa & (sizeof(udata_t) - 1)) * 8);

		CHIP_(a20gate_mw)(cpssp, pa & ~(pa & (sizeof(udata_t) - 1)),
				0b1111111 << (pa & (sizeof(udata_t) - 1)), valX);
	}
}
#endif /* CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT */

static uint32_t
CHIP_(mx_code_b)(struct cpssp *cpssp, paddr_t pa)
{
	udata_t val;

	assert(! (pa & 0));

	CHIP_(a20gate_mx)(cpssp, pa & ~(sizeof(udata_t) - 1),
			0b1 << (pa & (sizeof(udata_t) - 1)), &val);

	return (val >> ((pa & (sizeof(udata_t) - 1)) * 8)) & 0xff;
}

static uint32_t
CHIP_(mx_code_w)(struct cpssp *cpssp, paddr_t pa)
{
	udata_t val;

	assert(! (pa & 1));

	CHIP_(a20gate_mx)(cpssp, pa & ~(sizeof(udata_t) - 1),
			0b11 << (pa & (sizeof(udata_t) - 1)), &val);

	return (val >> ((pa & (sizeof(udata_t) - 1)) * 8)) & 0xffff;
}

static uint32_t
CHIP_(mx_code_l)(struct cpssp *cpssp, paddr_t pa)
{
	udata_t val;

	assert(! (pa & 3));

	CHIP_(a20gate_mx)(cpssp, pa & ~(sizeof(udata_t) - 1),
			0b1111 << (pa & (sizeof(udata_t) - 1)), &val);

	return (val >> ((pa & (sizeof(udata_t) - 1)) * 8)) & 0xffffffff;
}

/*
 * Add a new TLB entry. At most one entry for a given virtual address
 * is permitted.
 */
static void
tlb_set_page(struct cpssp *cpssp, vaddr_t vaddr, paddr_t paddr, int rwx, int is_user, Haddr haddr)
{
	TranslationBlock *first_tb;
	unsigned int index_;
	vaddr_t address;
	Haddr addend;
	paddr_t phys_addend;

	assert(! (vaddr & (CACHE_LINE_SIZE - 1)));
	assert(! (paddr & (CACHE_LINE_SIZE - 1)));

	/* NOTE: we also allocate the page at this stage */
	first_tb = CHIP_(tb_find_alloc)(cpssp, paddr);

	index_ = (vaddr / CACHE_LINE_SIZE) & (CPU_TLB_SIZE - 1);

	phys_addend = paddr - vaddr;

	switch (rwx) {
	case 0:
		/*
		 * Memory Read
		 */
		if (! haddr) {
			/* I/O Case */
			address = vaddr | IO_MEM_IO;
			addend = NULL;
		} else {
			/* Standard Memory */
			address = vaddr | IO_MEM_RAM;
			addend = haddr - vaddr;
		}
		cpssp->tlb_read[is_user][index_].address = address;
		cpssp->tlb_read[is_user][index_].host_addend = addend;
		cpssp->tlb_read[is_user][index_].phys_addend = phys_addend;
		break;

	case 1:
		/*
		 * Memory Write
		 */
		if (! haddr) {
			if (first_tb) {
				address = vaddr | IO_MEM_IO_CODE;
			} else {
				address = vaddr | IO_MEM_IO;
			}
			addend = NULL;
		} else {
			if (first_tb) {
				address = vaddr | IO_MEM_CODE;
			} else {
				address = vaddr | IO_MEM_RAM;
			}
			addend = haddr - vaddr;
		}
		cpssp->tlb_write[is_user][index_].address = address;
		cpssp->tlb_write[is_user][index_].host_addend = addend;
		cpssp->tlb_write[is_user][index_].phys_addend = phys_addend;
		break;

	case 2:
		/*
		 * Memory Execute
		 */
		if (! haddr) {
			/* I/O Case */
			address = vaddr | IO_MEM_IO;
			addend = NULL;
		} else {
			/* Standard Memory */
			address = vaddr | IO_MEM_RAM;
			addend = haddr - vaddr;
		}
		cpssp->tlb_code[is_user][index_].address = address;
		cpssp->tlb_code[is_user][index_].host_addend = addend;
		cpssp->tlb_code[is_user][index_].phys_addend = phys_addend;
		break;

	default:
		assert(0); /* Mustn't happen. */
	}
}

static void __attribute__((__noreturn__))
NAME_(mem_fault)(struct cpssp *cpssp, void *retaddr)
{
	TranslationBlock *tb;
	unsigned long pc;

	if (retaddr) {
		/* now we have a real cpu fault */
		pc = (unsigned long)retaddr;
		tb = CHIP_(tb_find_pc)(cpssp, pc);
		if (tb) {
			/*
			 * The PC is inside the translated code. It
			 * means that we have a virtual CPU fault
			 */
			CHIP_(restore_state)(cpssp, tb, pc);
		}
	}
	CHIP_(raise_exception_err)(cpssp, CPU_FAULT_PF, cpssp->error_code);
	/*NOTREACHED*/
}

/*
 * Try to fill the TLB and return an exception if error. If retaddr is
 * NULL, it means that the function was called in C code (i.e. not
 * from generated code or from helper.c)
 */
static void
CHIP_(tlb_fill)(struct cpssp *cpssp, vaddr_t addr, int rwx, int is_user, void *retaddr)
{
	paddr_t paddr;
	Haddr haddr;
	uint16_t err;
	int ret;

	switch (rwx) {
	case 0: {
		/* Read */
		int (*cf)(void *, paddr_t, unsigned int, udata_t *);
		void *cs;

		ret = CHIP_(cache1d_map_r)(cpssp, addr & ~(CACHE_LINE_SIZE - 1), is_user, &paddr, &cf, &cs, &haddr, &err);
		break;
	    }
	case 1: {
		/* Write */
		int (*cf)(void *, paddr_t, unsigned int, udata_t);
		void *cs;

		ret = CHIP_(cache1d_map_w)(cpssp, addr & ~(CACHE_LINE_SIZE - 1), is_user, &paddr, &cf, &cs, &haddr, &err);
		break;
	    }
	case 2: {
		/* Execute */
		int (*cf)(void *, paddr_t, unsigned int, udata_t *);
		void *cs;

		ret = CHIP_(cache1i_map_x)(cpssp, addr & ~(CACHE_LINE_SIZE - 1), is_user, &paddr, &cf, &cs, &haddr, &err);
		break;
	    }
	default:
		assert(0);
	}

	if (ret) {
		cpssp->error_code = err;
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		if (((int64_t) addr >> 47) != 0
		 && ((int64_t) addr >> 47) != -1) {
			cpssp->exception_index = 13; /* Proction Fault */
		} else
#endif
		{
			cpssp->exception_index = 14; /* Page Fault */
		}
		cpssp->cr[2] = addr;
		NAME_(mem_fault)(cpssp, retaddr);
	}

	assert((paddr & (CACHE_LINE_SIZE - 1)) == 0);

	tlb_set_page(cpssp, addr & ~(CACHE_LINE_SIZE - 1), paddr, rwx, is_user, haddr);
}

static inline void
cpu_tlb_protect1(struct CPUTLBEntry *tlb_entry, paddr_t pa)
{
	paddr_t tlbpa;

	if (tlb_entry->address & TLB_INVALID_MASK) {
		return;
	}
	tlbpa = (tlb_entry->address & CACHE_LINE_MASK) + tlb_entry->phys_addend;
	if (tlbpa < pa
	 || pa + TARGET_PAGE_SIZE <= tlbpa) {
		/* Different address in cache. */
		return;
	}

	switch (tlb_entry->address & ~CACHE_LINE_MASK) {
	case IO_MEM_CODE:
	case IO_MEM_IO_CODE:
		/* Already protected. */
		break;
	case 0: /* RAM */
		tlb_entry->address
			= (tlb_entry->address & CACHE_LINE_MASK)
			| IO_MEM_CODE;
		break;
	case IO_MEM_IO:
		tlb_entry->address
			= (tlb_entry->address & CACHE_LINE_MASK)
			| IO_MEM_IO_CODE;
		break;
	default:
		assert(0);
	}
}

/*
 * Update the TLBs so that writes to code in the physical page 'pa'
 * can be detected.
 */
static void
CHIP_(tlb_protect)(struct cpssp *cpssp, paddr_t pa)
{
	int i;

	pa &= TARGET_PAGE_MASK;
	for (i = 0; i < CPU_TLB_SIZE; i++) {
		cpu_tlb_protect1(&cpssp->tlb_write[0][i], pa);
		cpu_tlb_protect1(&cpssp->tlb_write[1][i], pa);
	}
}

static inline void
cpu_tlb_unprotect1(struct CPUTLBEntry *tlb_entry, paddr_t pa)
{
	paddr_t tlbpa;

	if (tlb_entry->address & TLB_INVALID_MASK) {
		return;
	}
	tlbpa = (tlb_entry->address & CACHE_LINE_MASK) + tlb_entry->phys_addend;
	if (tlbpa < pa
	 || pa + TARGET_PAGE_SIZE <= tlbpa) {
		/* Different address in cache. */
		return;
	}

	switch (tlb_entry->address & ~CACHE_LINE_MASK) {
	case IO_MEM_RAM:
	case IO_MEM_IO:
		/* Already unprotected. */
		break;
	case IO_MEM_CODE:
		tlb_entry->address = (tlb_entry->address & CACHE_LINE_MASK)
			| IO_MEM_RAM;
		break;
	case IO_MEM_IO_CODE:
		tlb_entry->address = (tlb_entry->address & CACHE_LINE_MASK)
			| IO_MEM_IO;
		break;
	default:
		assert(0);
	}
}

/*
 * Update the TLB so that writes in physical page 'pa' are no longer
 * tested self modifying code.
 */
static void
CHIP_(tlb_unprotect)(struct cpssp *cpssp, paddr_t pa)
{
	int i;

	pa &= TARGET_PAGE_MASK;
	for (i = 0; i < CPU_TLB_SIZE; i++) {
		cpu_tlb_unprotect1(&cpssp->tlb_write[0][i], pa);
		cpu_tlb_unprotect1(&cpssp->tlb_write[1][i], pa);
	}
}

static paddr_t
CHIP_(tlb_virt_to_phys)(struct cpssp *cpssp, vaddr_t va)
{
	unsigned int hash;
	unsigned int is_user;

	hash = (va / CACHE_LINE_SIZE) & (CPU_TLB_SIZE - 1);
	is_user = ((cpssp->hflags & HF_CPL_MASK) == 3);
	if (__builtin_expect(cpssp->tlb_code[is_user][hash].address
			!= (va & CACHE_LINE_MASK), 0)) {
		(void) ldub_code(cpssp, va);
	}

	return va + cpssp->tlb_code[is_user][hash].phys_addend;
}

static inline void
tlb_flush_entry(struct CPUTLBEntry *tlb_entry, vaddr_t addr)
{
	if (addr == (tlb_entry->address & (CACHE_LINE_MASK | TLB_INVALID_MASK))) {
		tlb_entry->address = TLB_INVALID_MASK;
	}
}

#if 80486 <= CONFIG_CPU
static void
CHIP_(code_invlpg)(struct cpssp *cpssp, vaddr_t addr)
{
	int i, j;
	TranslationBlock *tb;

	/*
	 * Must reset current TB so that interrupts cannot modify the
	 * links while we are modifying them.
	 */
	cpssp->current_tb = NULL;

	addr &= TARGET_PAGE_MASK;
	for (j = 0; j < TARGET_PAGE_SIZE / CACHE_LINE_SIZE; j++) {
		i = (addr / CACHE_LINE_SIZE) & (CPU_TLB_SIZE - 1);
		tlb_flush_entry(&cpssp->tlb_code[0][i], addr);
		tlb_flush_entry(&cpssp->tlb_code[1][i], addr);
		addr += CACHE_LINE_SIZE;
	}

	for (i = 0; i < TB_JMP_CACHE_SIZE; i++) {
		tb = cpssp->tb_jmp_cache[i];
		if (tb
		 && ((tb->pc & TARGET_PAGE_MASK) == addr
		  || ((tb->pc + tb->size - 1) & TARGET_PAGE_MASK) == addr)) {
			cpssp->tb_jmp_cache[i] = NULL;
		}
	}
}

static void
CHIP_(data_invlpg)(struct cpssp *cpssp, vaddr_t addr)
{
	int i, j;

	addr &= TARGET_PAGE_MASK;
	for (j = 0; j < TARGET_PAGE_SIZE / CACHE_LINE_SIZE; j++) {
		i = (addr / CACHE_LINE_SIZE) & (CPU_TLB_SIZE - 1);
		tlb_flush_entry(&cpssp->tlb_read[0][i], addr);
		tlb_flush_entry(&cpssp->tlb_write[0][i], addr);
		tlb_flush_entry(&cpssp->tlb_read[1][i], addr);
		tlb_flush_entry(&cpssp->tlb_write[1][i], addr);
		addr += CACHE_LINE_SIZE;
	}
}
#endif /* 80486 <= CONFIG_CPU */

static void
CHIP_(code_tlb_flush)(struct cpssp *cpssp, int flush_global)
{
	int i;

	/*
	 * Must reset current TB so that interrupts cannot modify the
	 * links while we are modifying them.
	 */
	cpssp->current_tb = NULL;

	for (i = 0; i < CPU_TLB_SIZE; i++) {
		cpssp->tlb_code[0][i].address = TLB_INVALID_MASK;
		cpssp->tlb_code[1][i].address = TLB_INVALID_MASK;
	}

	memset(cpssp->tb_jmp_cache, 0, sizeof(cpssp->tb_jmp_cache));
}

static void
CHIP_(data_tlb_flush)(struct cpssp *cpssp, int flush_global)
{
	int i;

	for (i = 0; i < CPU_TLB_SIZE; i++) {
		cpssp->tlb_read[0][i].address = TLB_INVALID_MASK;
		cpssp->tlb_write[0][i].address = TLB_INVALID_MASK;
		cpssp->tlb_read[1][i].address = TLB_INVALID_MASK;
		cpssp->tlb_write[1][i].address = TLB_INVALID_MASK;
	}
}

static void
CHIP_(code_unmap)(struct cpssp *cpssp, paddr_t start, paddr_t len)
{
	CHIP_(code_tlb_flush)(cpssp, 1);
}

static void
CHIP_(data_unmap)(struct cpssp *cpssp, paddr_t start, paddr_t len)
{
	CHIP_(data_tlb_flush)(cpssp, 1);
}

static void
NAME_(mw_code_b)(struct cpssp *cpssp, paddr_t pa, uint32_t val)
{
	CHIP_(tb_invalidate_phys_page_fast)(cpssp, pa, 1);
	CHIP_(mw_data_b)(cpssp, pa, val);
}

static void
NAME_(mw_code_w)(struct cpssp *cpssp, paddr_t pa, uint32_t val)
{
	CHIP_(tb_invalidate_phys_page_fast)(cpssp, pa, 2);
	CHIP_(mw_data_w)(cpssp, pa, val);
}

static void
NAME_(mw_code_l)(struct cpssp *cpssp, paddr_t pa, uint32_t val)
{
	CHIP_(tb_invalidate_phys_page_fast)(cpssp, pa, 4);
	CHIP_(mw_data_l)(cpssp, pa, val);
}

typedef void CPUWriteMemoryFunc(struct cpssp *cpssp, paddr_t pa, uint32_t val);
typedef uint32_t CPUReadMemoryFunc(struct cpssp *cpssp, paddr_t pa);

static CPUReadMemoryFunc *const CHIP_(io_mem_read)[IO_MEM_NB_ENTRIES][4] = {
	[IO_MEM_RAM >> IO_MEM_SHIFT] = {
		NULL, NULL, NULL /* Not used. */
	},
	[IO_MEM_CODE >> IO_MEM_SHIFT] = {
		NULL, NULL, NULL /* Not used. */
	},
	[IO_MEM_IO >> IO_MEM_SHIFT] = {
		CHIP_(mr_data_b), CHIP_(mr_data_w), CHIP_(mr_data_l)
	},
	[IO_MEM_IO_CODE >> IO_MEM_SHIFT] = {
		CHIP_(mr_data_b), CHIP_(mr_data_w), CHIP_(mr_data_l)
	},
};
static CPUWriteMemoryFunc *const CHIP_(io_mem_write)[IO_MEM_NB_ENTRIES][4] = {
	[IO_MEM_RAM >> IO_MEM_SHIFT] = {
		NULL, NULL, NULL /* Not used. */
	},
	[IO_MEM_CODE >> IO_MEM_SHIFT] = {
		NAME_(mw_code_b), NAME_(mw_code_w), NAME_(mw_code_l)
	},
	[IO_MEM_IO >> IO_MEM_SHIFT] = {
		CHIP_(mw_data_b), CHIP_(mw_data_w), CHIP_(mw_data_l)
	},
	[IO_MEM_IO_CODE >> IO_MEM_SHIFT] = {
		NAME_(mw_code_b), NAME_(mw_code_w), NAME_(mw_code_l)
	},
};
static CPUReadMemoryFunc *const CHIP_(io_mem_code)[IO_MEM_NB_ENTRIES][4] = {
	[IO_MEM_RAM >> IO_MEM_SHIFT] = {
		NULL, NULL, NULL /* Not used. */
	},
	[IO_MEM_CODE >> IO_MEM_SHIFT] = {
		NULL, NULL, NULL /* Not used. */
	},
	[IO_MEM_IO >> IO_MEM_SHIFT] = {
		CHIP_(mx_code_b), CHIP_(mx_code_w), CHIP_(mx_code_l)
	},
	[IO_MEM_IO_CODE >> IO_MEM_SHIFT] = {
		CHIP_(mx_code_b), CHIP_(mx_code_w), CHIP_(mx_code_l)
	},
};

#define MMUSUFFIX _mmu
#define GETPC() (__builtin_return_address(0))

#define SHIFT 0
#include "arch_gen_cpu_x86_align_template.c"

#define SHIFT 1
#include "arch_gen_cpu_x86_align_template.c"

#define SHIFT 2
#include "arch_gen_cpu_x86_align_template.c"

#define SHIFT 3
#include "arch_gen_cpu_x86_align_template.c"

#undef GETPC
#undef MMUSUFFIX


#define MMUSUFFIX _cmmu
#define GETPC() ((void *) 0)
#define CODE_ACCESS

#define SHIFT 0
#include "arch_gen_cpu_x86_align_template.c"

#define SHIFT 1
#include "arch_gen_cpu_x86_align_template.c"

#define SHIFT 2
#include "arch_gen_cpu_x86_align_template.c"

#define SHIFT 3
#include "arch_gen_cpu_x86_align_template.c"

#undef GETPC
#undef MMUSUFFIX

#endif /* BEHAVIOR */
