/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "glue.h"
#include "gdb.h"

static int gdb_inetport = -1;
static struct gdb *gdb_XX; /* FIXME */

static unsigned int
from_hex(char c)
{
	if ('0' <= c && c <= '9') {
		return c - '0';
	} else if ('A' <= c && c <= 'F') {
		return c - 'A' + 10;
	} else if ('a' <= c && c <= 'f') {
		return c - 'a' + 10;
	} else {
		assert(0);
	}
}

static char
to_hex(unsigned int x)
{
	if (x < 10) {
		return '0' + x;
	} else if (x < 16) {
		return 'A' + x - 10;
	} else {
		assert(0);
	}
}

static void
gdb_from_hex(unsigned char *mem, char *buf, unsigned long len)
{
	unsigned long i;

	for (i = 0; i < len; i++) {
		unsigned char val;

		val = from_hex(*buf++);
		val *= 16;
		val += from_hex(*buf++);

		*mem++ = val;
	}
}

static void
gdb_to_hex(unsigned char *mem, char *buf, unsigned long len)
{
	unsigned long i;

	for (i = 0; i < len; i++) {
		*buf++ = to_hex((*mem >> 4) & 0xf);
		*buf++ = to_hex((*mem >> 0) & 0xf);
		mem++;
	}
}

static void
gdb_ack(struct gdb *gdb)
{
	char c;
	int ret;

	c = '+';
	ret = write(gdb->datafd, &c, 1);
	assert(ret == 1);
}

static void
gdb_send(struct gdb *gdb, const char *str)
{
	char buf[4096];
	unsigned char sum;
	const unsigned char *p;
	int ret;

	sum = 0;
	for (p = (const unsigned char *) str; *p; p++) {
		sum += *p;
	}

	sprintf(buf, "$%s#%02X", str, sum);
fprintf(stderr, "Sending: %s\n", buf);
	ret = write(gdb->datafd, buf, strlen(buf));
	assert(ret == strlen(buf));
}

void
gdb_do(struct gdb *gdb)
{
	int ret;
	gdb = gdb_XX; /* FIXME */

	switch (gdb->state) {
	case GDB_STATE_RUNNING:
		assert(0); /* Mustn't happen. */
		break;

	case GDB_STATE_STOPPING:
		/*
		 * Target stopping.
		 */
		sprintf(gdb->ans, "S%02X", gdb->reason);
		gdb_send(gdb, gdb->ans);

		gdb->state = GDB_STATE_STOPPED;
		break;

	case GDB_STATE_STOPPED:
		if (gdb->parse == GDB_PARSE_DONE) {
			/*
			 * Got command from GDB.
			 */
			fprintf(stderr, "%s: req=%s\n", __FUNCTION__, gdb->req);

			gdb->parse = GDB_PARSE_NONE;

			/* Execute command. */
			/* See: https://sourceware.org/gdb/onlinedocs/gdb/Packets.html */
			switch (gdb->req[0]) {
			case '?':
				gdb_send(gdb, "S00");
				break;

			case '!':
				/* Enable extended mode. */
				gdb_send(gdb, ""); /* Ignored for now. */
				break;

			case 'c': {
				unsigned long addr;
				char *next;

				if (gdb->req[1]) {
					addr = strtoul(&gdb->req[1], &next, 16);
					assert(*next == '\0');
					
					/* ... FIXME */
				}

				gdb->state = GDB_STATE_RUNNING;
				sim_cont();
				break;
			    }
			case 'g': {
				unsigned char regs[256];
				unsigned long addr;
				unsigned long len;

				len = 0;
				for (addr = 0; ; addr++) {
					ret = (*gdb->funcs->reg_read)(gdb->cpssp,
							addr, regs + len,
							sizeof(regs) - len);
					if (ret < 0) {
						break;
					}
					len += ret;
				}

				gdb_to_hex(regs, gdb->ans, len);
				gdb_send(gdb, gdb->ans);
				break;
			    }
			case 'G':
				assert(0); /* FIXME */

			case 'H':
				/* FIXME */
				gdb_send(gdb, "OK");
				break;
				
			case 'k':
				/* Kill target. */
				gdb->state = GDB_STATE_EXITING;
				sim_cont();
				sim_exit();

				/* No reply! */
				break;

			case 'm': {
				unsigned char mem[256];
				unsigned long addr;
				unsigned long len;
				char *next;

				addr = strtoul(&gdb->req[1], &next, 16);
				assert(*next == ',');
				len = strtoul(next + 1, &next, 16);
				assert(*next == '\0');
				assert(len < sizeof(mem));

				len = (*gdb->funcs->mem_read)(gdb->cpssp,
						addr, mem, len);

				gdb_to_hex(mem, gdb->ans, len);
				gdb_send(gdb, gdb->ans);
				break;
			    }
			case 'M': {
				unsigned char mem[256];
				unsigned long addr;
				unsigned long len;
				char *next;

				addr = strtoul(&gdb->req[1], &next, 16);
				assert(*next == ',');
				len = strtoul(next + 1, &next, 16);
				assert(*next == ':');
				gdb_from_hex(mem, next + 1, len);

				ret = (*gdb->funcs->mem_write)(gdb->cpssp,
						addr, mem, len);

				if (ret < len) {
					gdb_send(gdb, "E00");
				} else {
					gdb_send(gdb, "OK");
				}
				break;
			    }
			case 'q':
				if (strcmp(gdb->req, "qAttached") == 0) {
					/* Attached or created? */
					gdb_send(gdb, "0"); /* Created */

				} else if (strcmp(gdb->req, "qOffsets") == 0) {
					gdb_send(gdb, "Text=00000000;Data=00000000;Bss=00000000");

				} else if (strncmp(gdb->req, "qSupported:", strlen("qSupported:")) == 0) {
					/* ? */
					sprintf(gdb->ans, "PacketSize=%X", (int) sizeof(gdb->req) - 1);
					gdb_send(gdb, gdb->ans); /* Buffer size */

				} else if (strcmp(gdb->req, "qTStatus") == 0) {
					goto unknown; /* Info missing FIXME */

				} else {
					goto unknown;
				}
				break;

			case 's':
				/* Single step. */
				/* FIXME */
				gdb_send(gdb, "S02");
				break;

			case 'S':
				/* Single step with signal. */
				goto unknown;

			case 'v':
				if (strncmp(gdb->req, "vKill;", strlen("vKill;")) == 0) {
					/* Kill given thread. */
					gdb_send(gdb, ""); /* Ignore for now. */
				} else {
					goto unknown;
				}
				break;

			case 'Z':
			case 'z': {
				/* Set/clear breakpoint. */
				int type;
				unsigned long addr;
				unsigned long len;
				char *next;

				type = strtol(&gdb->req[1], &next, 16);
				assert(*next == ',');
				addr = strtol(next + 1, &next, 16);
				assert(*next == ',');
				len = strtol(next + 1, &next, 16);
				assert(*next == '\0');

				if (gdb->req[0] == 'Z') {
					ret = (*gdb->funcs->brk_set)(gdb->cpssp,
							type, addr, len);
				} else {
					ret = (*gdb->funcs->brk_clr)(gdb->cpssp,
							type, addr, len);
				}

				if (ret < 0) {
					gdb_send(gdb, "E00");
				} else {
					gdb_send(gdb, "OK");
				}
				break;
			    }
			default:
			unknown:
				/* Unknown command. */
				fprintf(stderr, "WARNING: %s: %s: %s\n",
						"gdb", gdb->req, "Unknown request");
				gdb_send(gdb, "");
				break;
			}
		}
		break;
	case GDB_STATE_EXITING:
		break;

	default:
		assert(0);
	}
}

void
gdb_stop(struct gdb *gdb, int sig)
{
	gdb->state = GDB_STATE_STOPPING;
	gdb->reason = sig;
	sim_stop();
}

static void
gdb_read(int datafd, void *_gdb)
{
	struct gdb *gdb = _gdb;
	char c;
	int ret;

	assert(datafd == gdb->datafd);

	for (;;) {
		if (gdb->parse == GDB_PARSE_DONE) {
			/* Buffer full... */
			break;
		}

		ret = read(gdb->datafd, &c, 1);
		if (ret < 0) {
			return;
		}
		if (ret == 0) {
			/* Disconnect */
			io_unregister(gdb->datafd);

			ret = close(gdb->datafd);
			assert(0 <= ret);

			gdb->datafd = -1;

			gdb->state = GDB_STATE_RUNNING;
			sim_cont();
			return;
		}
		switch (gdb->parse) {
		case GDB_PARSE_NONE:
			switch (c) {
			case '+':
				/*
				 * Acknowledge
				 */
				/* Nothing to do... */
				break;

			case '-':
				/*
				 * Resend
				 */
				assert(0); /* FIXME */
				break;

			case 3:
				/* Ctrl-C */
				gdb_stop(gdb, 2); /* SIGINT */
				break;

			case '$':
				/* Command */
				memset(gdb->req, 0, sizeof(gdb->req));
				gdb->parse = GDB_PARSE_COMMAND;
				break;

			default:
				assert(0);
			}
			break;
		case GDB_PARSE_COMMAND:
			if (c == '#') {
				gdb->parse = GDB_PARSE_CHECKSUM0;
			} else {
				gdb->req[strlen(gdb->req)] = c;
			}
			break;
		case GDB_PARSE_CHECKSUM0:
			/* Ignore for now... */
			gdb->parse = GDB_PARSE_CHECKSUM1;
			break;
		case GDB_PARSE_CHECKSUM1:
			/* Ignore for now... */
			gdb->parse = GDB_PARSE_DONE;

			gdb_ack(gdb);
			break;
		default:
			assert(0);
		}
	}
}

static void
gdb_accept(int sockfd, void *_gdb)
{
	struct gdb *gdb = _gdb;

	fprintf(stderr, "%s...\n", __FUNCTION__);

	assert(gdb->datafd < 0);

	gdb->datafd = accept(sockfd, NULL, NULL);
	assert(0 <= gdb->datafd);

	io_register(gdb->datafd, gdb, gdb_read);

	gdb->state = GDB_STATE_STOPPED;
	sim_stop();

	gdb_read(gdb->datafd, gdb); /* In case gdb already sent a request... */
}

void
gdb_port(int port)
{
	gdb_inetport = port;
}

struct gdb *
gdb_create(const char *name, void *cpssp, struct gdb_funcs *funcs)
{
	struct gdb *gdb;
	int val;
	struct sockaddr_in addr;
	int ret;

	if (gdb_inetport < 0) {
		return NULL;
	}

	gdb = malloc(sizeof(*gdb));
	assert(gdb);

	gdb->state = GDB_STATE_STOPPED;
	gdb->parse = GDB_PARSE_NONE;

	gdb->sockfd = socket(PF_INET, SOCK_STREAM, 0);
	assert(0 <= gdb->sockfd);

	val = 1;
	ret = setsockopt(gdb->sockfd, SOL_SOCKET, SO_REUSEADDR,
			(void *) &val, sizeof(val));
	assert(0 <= ret);

	addr.sin_family = AF_INET;
	addr.sin_port = htons(gdb_inetport);
	addr.sin_addr.s_addr = 0;
	ret = bind(gdb->sockfd, (struct sockaddr *) &addr, sizeof(addr));
	assert(0 <= ret);

	ret = listen(gdb->sockfd, 0);
	assert(0 <= ret);

	io_register(gdb->sockfd, gdb, gdb_accept);

	gdb->datafd = -1;
	gdb->cpssp = cpssp;
	gdb->funcs = funcs;

	fprintf(stderr, "gdb: %s: %d\n", name, gdb_inetport);

	gdb_inetport++;

	gdb_XX = gdb; /* FIXME */

	return gdb;
}

void
gdb_destroy(struct gdb *gdb)
{
	int ret;

	if (gdb == NULL) {
		return;
	}

	if (0 <= gdb->datafd) {
		ret = close(gdb->datafd);
		assert(0 <= ret);
	}

	ret = close(gdb->sockfd);
	assert(0 <= ret);

	free(gdb);
}
