(defpackage "SB-GMP-TESTS"
  (:use "COMMON-LISP" "SB-GMP" "SB-RT"))

(in-package "SB-GMP-TESTS")

(defparameter *state* (make-gmp-rstate))
(rand-seed *state* 1234)

(defmacro defgenerator (name arguments &body body)
  `(defun ,name ,arguments
     (lambda () ,@body)))

(defgenerator gen-mpz (&key (limbs 5) sign nonzero)
  (let ((integer (random-bitcount *state*
                                  (* limbs sb-vm:n-word-bits))))
    (when (and nonzero (zerop integer))
      (setf integer 1))
    (ecase sign
      ((+ nil) integer)
      (- (- integer))
      ((t random) (if (zerop (random 2))
                      integer
                      (- integer))))))

(defun gen-mpq (&key (limbs 5) sign nonzero)
  (let ((numerator (gen-mpz :limbs limbs :sign sign
                            :nonzero nonzero))
        (denominator (gen-mpz :limbs limbs :nonzero t)))
    (lambda ()
      (/ (funcall numerator) (funcall denominator)))))

(defun maybe-apply (maybe-function &optional arguments)
  (if (typep maybe-function '(or function symbol))
      (apply maybe-function arguments)
      maybe-function))

(defun test-one-case (base tested &rest arguments)
  (let* ((initial-hashes (mapcar #'sxhash arguments))
         (base-values (let ((*gmp-disabled* t))
                        (multiple-value-list (maybe-apply base arguments))))
         (test-values (let ((*gmp-disabled* nil))
                        (multiple-value-list (apply tested arguments))))
         (final-hashes (mapcar #'sxhash arguments)))
    (unless (and (= (length base-values) (length test-values))
                 (every #'eql base-values test-values))
      (error "Failed test: ~S returned ~S; expected ~S"
             (cons tested arguments) test-values base-values))
    (unless (every #'eql initial-hashes final-hashes)
      (error "Failed test: ~S modified arguments ~{~A~^, ~} ~
              (printed modified values)"
             (cons tested arguments)
             (loop for i upfrom 0
                   for initial in initial-hashes
                   for final in final-hashes
                   unless (eql initial final)
                     collect i))))
  nil)

(defvar *iteration-count* 10)

(defun test-n-cases (base tested &rest argument-generators)
  (let ((*random-state* (sb-ext:seed-random-state 54321)))
    (loop repeat *iteration-count* do
      (apply 'test-one-case base tested
             (mapcar #'maybe-apply argument-generators)))))

(defmacro define-gmp-test ((name &key (repeat 1) limbs (gmp-seed 1234))
                           &body body)
  `(deftest ,name
       (let ((*random-state* (sb-ext:seed-random-state 54321)))
         (rand-seed *state* ,gmp-seed)
         (handler-case
             (dotimes (i ,repeat)
               ;; try to get small failures first
               (let ((limbs (case i
                              (0 ,(subst `(lambda (x)
                                            x 0)
                                         'random
                                         limbs))
                              (1 ,(subst `(lambda (x)
                                            (if (> x 1) 1 0))
                                         'random
                                         limbs))
                              (t ,limbs))))
                 (declare (ignorable limbs))
                 ,@body))
           (error (c)
             (format t "~&~A~%" c)
             nil)
           (:no-error (&rest _) _ t)))
     t))

(define-gmp-test (mpz-add :repeat 7 :limbs (+ (random #xFF) 2))
  (test-n-cases '+ 'mpz-add
                    (gen-mpz :limbs limbs :sign t)
                    (gen-mpz :limbs limbs :sign t)))

(define-gmp-test (mpz-sub :repeat 7 :limbs (+ (random #x1F) 2))
  (test-n-cases '- 'mpz-sub
                (gen-mpz :limbs limbs :sign t)
                (gen-mpz :limbs limbs :sign t)))

(define-gmp-test (mpz-mul :repeat 7 :limbs (+ (random #x25) 2))
  (test-n-cases '* 'mpz-mul
                (gen-mpz :limbs limbs :sign t)
                (gen-mpz :limbs limbs :sign t)))

(define-gmp-test (mpz-tdiv :repeat 7 :limbs (+ (random #x23) 2))
  (test-n-cases 'truncate 'mpz-tdiv
                (gen-mpz :limbs limbs :sign t)
                (gen-mpz :limbs limbs :sign t :nonzero t)))

(define-gmp-test (mpz-fdiv :repeat 7 :limbs (+ (random #x253F) 2))
  (test-n-cases 'floor 'mpz-fdiv
                (gen-mpz :limbs limbs :sign t)
                (gen-mpz :limbs limbs :sign t :nonzero t)))

(define-gmp-test (mpz-cdiv :repeat 7 :limbs (+ (random #x253F) 2))
  (test-n-cases 'ceiling 'mpz-cdiv
                (gen-mpz :limbs limbs :sign t)
                (gen-mpz :limbs limbs :sign t :nonzero t)))

(define-gmp-test (mpz-gcd :repeat 7 :limbs (+ (random #x25) 2))
  (test-n-cases 'gcd 'mpz-gcd
                (gen-mpz :limbs limbs :sign t)
                (gen-mpz :limbs limbs :sign t)))

(define-gmp-test (mpz-lcm :repeat 7 :limbs (+ (random #x25) 2))
  (test-n-cases 'lcm 'mpz-lcm
                (gen-mpz :limbs limbs :sign t)
                (gen-mpz :limbs limbs :sign t)))

(define-gmp-test (isqrt :repeat 7 :limbs (+ (random #x25) 2))
  (test-n-cases 'isqrt 'mpz-sqrt (gen-mpz :limbs limbs)))

(define-gmp-test (mpz-mod :repeat 7 :limbs (1+ (random #x25)))
  (test-n-cases 'mod 'mpz-mod
                (gen-mpz :limbs limbs :sign t)
                (gen-mpz :limbs limbs :sign t :nonzero t)))

(define-gmp-test (mpz-pow :repeat 7 :limbs (1+ (random #x25)))
  (test-n-cases (lambda (base exponent)
                  (let ((*gmp-disabled* nil)) ; atrociously slow otherwise
                    (expt base exponent)))
                'mpz-pow
                (gen-mpz :limbs limbs :sign t)
                (lambda ()
                  (1+ (random 40)))))

(define-gmp-test (mpz-powm :repeat 7 :limbs (1+ (random #x23)))
  (test-n-cases (lambda (base exponent mod)
                  (let ((*gmp-disabled* nil)) ; atrociously slow otherwise
                    (mod (expt base exponent) mod)))
                'mpz-powm
                (gen-mpz :limbs limbs :sign t)
                (lambda ()
                  (1+ (random 40)))
                (gen-mpz :limbs (ceiling limbs 2) :nonzero t)))

;; bugs that have been fixed
(define-gmp-test (sign-conversion)
  (test-one-case '+ 'mpz-add #x7FFFFFFFFFFFFFFF #x7FFFFFFFFFFFFFFF))
(define-gmp-test (truncate-1)
  (test-one-case 'truncate 'mpz-tdiv
                 30951488519636377404900619671461408624764773310745985021994671444676860083493
                 200662724990805535745252242839121922075))
(define-gmp-test (truncate-2)
  (test-one-case 'truncate 'mpz-tdiv
                 320613729464106236061704728914573914390
                 -285049280629101090500613812618405407883))

(defun fac (n)
  (loop for i from 1 to n
        for fac = 1 then (* fac i)
        finally (return fac)))

(define-gmp-test (fac1)
  (test-one-case 'fac 'mpz-fac
                 6))

(define-gmp-test (fac2)
  (test-one-case 'fac 'mpz-fac
                 63))

(define-gmp-test (pow1)
  (test-one-case (lambda (base exponent)
                   (let ((*gmp-disabled* t))
                     (expt base exponent)))
                 'mpz-pow
                 16 3))

(define-gmp-test (pow2)
  (test-one-case (lambda (base exponent)
                   (let ((*gmp-disabled* t))
                     (expt base exponent)))
                 'mpz-pow
                 -15546163094340153687 11))

(deftest intexp-1
    (sb-gmp::gmp-intexp 1 (ash 1 127))
  1)

(deftest remove-1
    (multiple-value-list (mpz-remove 28 2))
  (7 2))

#+sb-gmp-5.1
(deftest remove-2
    (multiple-value-list (mpz-remove 28 -7))
  (-4 1))

(deftest remove-3
    (multiple-value-list (mpz-remove 28000000000000000000000000000000000000000 16))
  (25465851649641990661621093750 10))
; 1701411834630158168966515027820462151030

(deftest fib
    (multiple-value-list (mpz-fib2 250))
  (7896325826131730509282738943634332893686268675876375
   4880197746793002076754294951020699004973287771475874))

(deftest binomial
    (mpz-bin 2000000000000 20)
  430998041177272843950422879590338454856322722740402365741730748431530623813012487773080486408378680853987520854296499536311275320016878730999689934464711239072435565454954447356845336730100919970769793030177499999999900000000000)

(define-gmp-test (mpz-nextprime :repeat 7
                                :gmp-seed 6234
                                :limbs (1+ (random #x10)))
  (let ((a (gen-mpz :limbs limbs)))
    (dotimes (i *iteration-count*)
      (let* ((a (funcall a))
             (p (mpz-nextprime a)))
        (assert (>= p a))
        (assert (plusp (mpz-probably-prime-p p)))))))

(define-gmp-test (mpq-add :repeat 7 :limbs (1+ (random #x3F))
                          :gmp-seed 1235)
  (test-n-cases '+ 'mpq-add
                (gen-mpq :limbs limbs :sign t)
                (gen-mpq :limbs limbs :sign t)))

(define-gmp-test (mpq-sub :repeat 7 :limbs (1+ (random #x1F))
                          :gmp-seed 1235)
  (test-n-cases '- 'mpq-sub
                (gen-mpq :limbs limbs :sign t)
                (gen-mpq :limbs limbs :sign t)))

(define-gmp-test (mpq-mul :repeat 7 :limbs (1+ (random #x5F))
                          :gmp-seed 6235)
  (test-n-cases '* 'mpq-mul
                (gen-mpq :limbs limbs :sign t)
                (gen-mpq :limbs limbs :sign t)))

(define-gmp-test (mpq-div :repeat 7 :limbs (1+ (random #x3F))
                          :gmp-seed 7235)
  (test-n-cases '/ 'mpq-div
                (gen-mpq :limbs limbs :sign t)
                (gen-mpq :limbs limbs :sign t)))

(define-gmp-test (add-1)
  (test-one-case '+ 'mpq-add
                 (/ -13209053177313216326720071994575668671625946154424946979941042235025779609406066489124822060263741415731767909095535978759309084425875537563179113298131593713396775343237383458479781949550240630412701857403503233189863607072117930640826569182009206708933257720426339611012943709281997759942473045656872630494457723826960825086582617760738451841527423714294371807083373098832437021848589728037526905908672891882725056067884256042505048001986703119531737056795446226445664725837005863750123574298000192591001893057277178437857007262375095660896699217748352172606682055924706139632861269010372898134646451737590670614560649546844481697242528214535706357927544878674268030374370278308504509252394576207386344666818237571152429009824441574054780299582577801481767459654249022803191897258173315859493825673310472505293627476855369546650359583218498516377023038245685243868968780236437619483792036441607501402092965075649435802858970715108833831163596260720848064686538519339195183094001726352453705513582309597381501641400986633247524429625086995030279106454495104883103144972006136945504038625443714829393575431723477731391536529901691154731695666918250476276904458544458920225308813979815661628422015950374787964790294728431305560901981574526733321069031075413803807919559580147250764144945690106618011831706083014149915084399785240071376864007068579410836584583640124573636564525508781814067728494858869755462333538613864275409311494135416846021419052727382769669646774316183742088736319734577932969015168552780639091132865606483317081877017019863932502252092416408465358902183952810083840135676232857108681406486228546498252506817867584563688247332110215273151817829391150607426998926468854041744560970884608178439217990221993505915120441065973024578417594697614340421542869597391944892731436704363092556227861288542154533966853578546397006975240622959448425346779466724647279240015643799101159755851939673521288390733178144929994118609073425419299771438638154675479577257608964922414393526655389054776040890897127353109186990337860645391379224816579489224213787802841647588422060321741574667225867107935341545992758460567691834834677102164126543306712069736700806387259764413531457970954153801229449751018964914808807523738864401593622739672532046389168352737493963609733273138264236810210602108213237142407172633253427427858308191860601367964093868742916150428348086935533601815990061734949289335443255625406474502500919112688955733286467584667225159924898259902443424322015574285591785177463802961642299768433955323304169824647179428285450887144401989305402699455358336862489956755941197154511191311114593457064024159546112615698065878019110371636625550987691920666797008128403458483365368131571213824939808939703491750827705258225332786724552178963564173170124332641627169399852304611511887521527613858208776877706934034658839028986454686812721986441576582538345640083288494377295752136545030030378845647222986379656035835398148857804268458136966853832117064077363965110584146936510473620820871282616637067556761611627531675227320081363307277507286698878265502547270682650862708828
                    20513114107748916213404225865244788457074409421317268073736007022331351875833624862756219402282516605499234739123890210259042897848417665114604539868868273254948688298788998302213021613923064906756450800034198611450550412754819890990174752374167903316805361554027498760235600729993563831298333173441094727731923224830092028024856604134356448524792713778602056165546782663297074097079284342477853104853964497763641859184704167439137864700572855254893938640443507173796307557848785066207502084190070941243677848148415872970426984112227143138388781146980391816967852643014559749964089362445213740678445781520407345416931182699041155590319696207409023305181829071577033026467781464157778708535067535410514618107828410799078854809043994182182513891242966598128605638721171207590595334175915256179515795767729079909987621749163917687365458820366329459365528343025952159907952014278541741375614385752430254027787458039763962980148354021017595711354474983444535795390727075604793582413720611425701009821061847193862993314927110707228039583968452336179043689984205243355533686106965429229843687723310570965586179382872671178881510560063890749790799969604375110656863922578399537470723920490589304812587620853690645052574500291509160991572359900940420724641176985308557030554912090272478685362502316357934024797972855044761873086094428264355950862436019798203732663793722180258524408701122357905680756259238827164553903541080614504438587683436150871685884268883283911854170592499833400902456121829220126566671137352815854372233474135278712361983897095540626421407462186513362543785344229962579231578886782545540253824395033344947140802911329621494871150797172785087252655178157292521158795162751736228321379594242640753289713774813788577182079164501873509243856938395077798238684195225086857610004547607931023719346097561632391139975815092002982975159546683160426362246955972077936658217594864092323585697829920668939829394800301107410233703599100424307946464487800029921136792727516126004924528206024989173681199190467183044174366289573226993387328796686353001262094065886677828612237553191296728467686124117645915020679885821098583880459512400005399643605438196485962638776350246725512917429760098432821130792531585302118917535837554162150451664904547053284426081154205894049283709132043472492948062433231732886890323081803717104257817546161455251910978584912564590124304484825003663209179656643588909568838436967306030449237270646752767124447743338839118234701364264756637068592181857304587935081380346741935676377777738000391654868635675482278059571682109768442900093568916953814629840854483385045234030286006852324973600582554130111693028897769906912575142665730244531455877465249521338449046855843855835695525125243807274003500298341019159598027990797824307900319150505057082890499198084718553708432933227009336155667500047178145987247945476571211248754056910790992752991755636297837275630733526771844363248887124090245941209079831661430017567282457161714201881877508564203445830784748962887859489164006718845423665056853294160573803367220078511760896875018875436265794666526797116713991)
                 (/ 18679644232137694835773895789796935368277093319442347009882028895983636202845338609238192673627146497565928042387055224157574139050288826655862246758264677614448028402353844741710560974837728602947105854327615172777677807366701532505782053272007434767818839012955843675198432125954639704393332610481612266824735837213574122272690607379032167676488469612066394795256393456323115729874647307324022050094919570365558303290939751605851466337737809073009233406468650259124598268624411389241168196985092309722980787131179252290893559312026443285238855953743578446404260697683876718720243998070208963196110161880695659120258496298742068955177202297231221854713889514939341338710288466093793249620509478542499217270116943730534610528631414218420777365890198159419904789404544815594578693254741245950839290769417056174832288369376710235014676146147468633129018928427491168063325782948815792849178692928887382135434681729289287967337075439807692747252894555096464608468870770272306141894543939790457871582280905906502647124459402979133843083046084613148746238605877442222978217173864099311407377701093864103693585496264088046841197293166458333808109330897750695971436430929379693868141924159750578349474196528735831027022930710200698011465050411238305883665750989820947782615767690752174603656863655235695510372750031379950324783924993662562999898391005787288706074052766163310550368263948617871014791452525375381426803827826752927165103290925907171678884968925213591369800936425974016544201074162690808540744854897729680887261315804116938589922540061828019386378590167985023192456982368984194146169494610555807299338947249571398537979070473103314582346674210413883618845411747108357789193206528225060723206424692411535736636880124631856413786992405295584419830157311803942862853074313776159067194511528927928414266465879542622955788891440153638676754727239764508736989303091478159209924874227484028795818451610680606439970077540524099704206162370096870622410572956084155250617029486154870570966638812142456113480214174492723144414953252009715373131781710069274723880618136495090451070098583243662797572407241313286841653442689485898363388204083005649951198361398468612590147786690372977367193534838948182349778146464897378808482133163629825944522445704281495970890808554290254714873933391569033738861111337892983959660609734480992012993277433865997793175736326621314652898038483105812001022395450458749805256561238577572040146462804577251468325598886411516301289338452351467886537896065545021406988061091200941643057561971430385531845109791470909518930415984599441096465126692173387505995283866721562382544618237963718834045094760633539051908412566846285956073333799948712666521093676261861167303418973611709249741242454070021816184150635280988297097917463890986448016624789958062269731291092669270798993316373736592456861706792773261332823061122660874852391531335207185069468318947590952403786907139751917687296093870144147055137535538562666514450751236638639254714837540658351799195360032379598389809413531717309390498240542341409517870564609553438235694303893655478920111421329599854382059
                    35047375165067289478184579374088753519723409541358727851367833378045381161956857862822162664157407750114046088766861757768628361860450858550612162200604740528020549735526868301563450011458255876617372435581481413877626213584152958947512852610351865749619221182400335300981664095326585454825312297691968407481156213225240040259895631250587691435815407287057903383853149474327315762925358991507160427907888884554569193526587056400916286053773407408865362995781271043549911925410762585516917061851141429216498699036671966534807428201871680758644707869709055458345849760857447496997369459512430997600004647049262418108322920507202971361167434749566827221165629531524830063260150702915506513004140215619053251853221128227039573877830371056963496851061915005076713551938040426681835756484540822137833583839287273858119692954255679012880175954568099315917680559792168668290774115965992289930491245825501336418496149153195257486784214896251536902218095859878825175930104814002335186480516826697953970017283350762453123058826001074995827392758721263396789788171838854949995547275268946535021565360713052629801500761448894400314861354423310859391743686807240892514171117787583263534162869357320748877744731428554325911583092674254089309526792474206410123702980303052860969897399493933015372688281452770610307341291480264049601427255610847963613178959192364378622499160842253921207881626181581709855849432538739896839132656376924833085425334004208124463768334575132431712512267775703706491862340888727839505443433268529379987084625482496093787809074752688019948611710332687765927010511936393170622967393012885553949858840148946705853974220230333552679372649812288028205467991322591700053013007905861724568610606853496128017958814314450986832892988026223481899589331851816885128871124803692166197269829726785503967919268160500701435937831624617814093029450809869946138434889833384333032268194378347437153894298364527885530000640988674907753972046462491022827738005865076839138904283740722398989182880398823501150929948982395291592901295075465416607929007655790650722355676164182368279922022441635530379579396127841519540366765299138907464310936858631412869302144760544631722698855436241594426632799148068619791910004407651630129367721152620976432342245466252584851082147635414151337510386583270139031454475785430451613760410780963926233601058128946480944715083701501658239619004129521741582609884569752285592082229894073389739632290612979527960898956998537733110196762265428813645010248053457222143316651370707088803028348532186655314492324589438375979080705908912991001132757969685814749516624261196677521777688416066667857217162155509052884746889529221826307232465665939634819692653774644615752505524321373928021711308907637490611431718655229495166013973268494159392003260853439666992720242593681536443289086106672585130316509972729860170549253311425331179571717512946796316828551789499616649200744710228301066546180726362338960578934089254316390186365586939632431422116920095304572927553904681045284800605022616383305331739050601840250168653669981271907611676558111666927724598500621490620608)))

(define-gmp-test (mod-1)
  (test-one-case 'mod 'mpz-mod
                 249598007630932994851905502846736256724074712953159542087465548523550109752100478387303118890742990712928353184901454701879529637159096879579386479271365050411374644809906857058361150342525797993390453185625880316798133033117605442630558315621487983857989110546702462697111450041137122781015878282503651693276655422067339833681716383891135506495283841662229329502960178743922714732155409088277902142224089938552638529149338960934382239020947602923363589113701364361657696313015380856381434014252204238232543735251577345132402442291628276877684937095374216438475149471002257617994732443192233653826861158778185544814047738630497835562035578439122067344961036040081366666547693058769326626485255725595713488531675525600103629886073615376345582878147215005421904570372171492079892576538462771300539835056615452366320468328278688867491768252549825865375155544524894819234024425065948298467407732997285092039923948934693273773708757887464972872815993683747766972312674740152292324057565537308093490468716221557765360654525333075552425546820692126249241536409552258936262877415408238571255221499328809842925429371300617192286153473882755176084813875439559104771732193753203575093557256324627924316951736447112007507216409383740660629148152193753196563653167029460781234722890175682221182226860924739770996439452455262951053623492949064116682295518500112348745829094845322431191572887160906895188878706785305348864014051907993761401962109046756828215299023913961959736276445916191440195334521539150246411433164256759289272755950001175116592552385381824831029781033767686270935554406235405159516227274576882530194339021095496841761044026781661820132933014058886262086839134535419780522370911505051655952148886370233771729373345051565553813250395694698721409894856715121326190151469812000532527497010039365255709456426965690336486822109647862308448012585313520934056430420966340465792231736450064085240346488262718940916977070140468307135854172295546307591412788398086995722811027915421666440243156210003742290896216880095185841394811672417951175767281053161696828964939861844863164737572253496550660888896937306903820054571068803501612450520754502361938238290632424993462951593360973261556208400362555363451977888355555788987425351504873599533703165332894024249874243832467486607740653764812227410668383044366771387427564259595171724765756121349973136944418155420280653984138504790126554468314332731882588101792705826112149173023352145793274600327999612793494385099078080651201060323088914794165502411988251603021423195326666584568291603941327551397704000880030989060399077677526120284097199025161308969754919858366031602541091006941755312811656696481764246315375644928380961673756346269386057349529336498423740888041844828135592499325717677204770313196588088496757988296846440832241099657042988082498015444014239102510970378597161127287226426622352641620930881718314720181597997499275216365790816495230251248147965033887987069960239866037691168816124996535110560184021753143325316225999580738490468246683302662091822783978599071868221179348415365884220253437950728730147428254523830141972804066335708196940615768495310546110673336000990305465784075075354693049110321231371901765037615510420650178538134407211339130026236857082293899207597887171440988286541694760993813312140946000578206285529329499533547964022797412165425468720689803948085932285261335246680174719264588476443036073820175583500580764674339276020029910263833412550806127311487587594131743775091104481208347197797291793811770670433438132432498357853919879978126831173543907005470713543672314457725754916535610838998910663811437908773536949503973537595566369308015850862251661019442071938752320865226562821358301960889467112257005728170640091655514488258458844015143456673266554629803857032572350143249015039084666929184483882821032676613531597429108519155199042832326508258650182834913574868071945743022813311515454761721904617964585068370669712169309501311408103758937455350611786824783612935936572171372186880469676749159610970738804230112453820822849901849267670562160197106959132676398648866860297910449246896227439973137308232180851048845083653612250994451560108844544943837538849770554397356250870867449859221100831845459446465301150631921815426126491749855951076747685062132080296394261839120209464535632489514004939937042999469657674735202438102189778148491966369706872343357588916032782529536087102902163794590980968259721696878972043030934538437505438555753533712881773626549678205641729275264936261261930086592854304643784235761517135165583733705593442055444052400394518455761067881498183206938249920986274700043899319703975011585809355246131709637692470506620097770175353383593963637718569135993602219565455312398495094764288663409387160773026629464215551872868103600843174253021945366009117716564009014001349827748662176004169981847297882926935176068523919090661559605799744554077644379511509397497803738157754147318187218598312912709004327047149008975985131727783466356036757824110985493660986346542417364632478961980576622824567085469637142929024416795590124572212956438027483952980949960539702922330379332545930989837146633226988558841169849813642525139291514468173204319079698119131411554001230422419381997405745696632611604683526687531608202118936937204335792634240694597773241135340457386160411631323536143722218684081147453849198494776367208964442406036764994343008387541270893568915208348050258860197292081113743438526193308570069936084420504658542793299185424359846980542654924899214843904381883311392649726278727815241829342798652502910018535314421243877481422687401525160270299742828918646731225165151239551333479941932055042234558401658194981733904661438764944483159001716511560680538113400252675502551736809724399661777505714192490559284137336640304579919333723464863760803089583640541003767578034146421408149821780096522608804368019772135865319330825428791039476318903392576997551863064562119856218572565351387696990690839530803297556001786720892122732998900895201158803568355583248835889526797851269898579702022694678289562082551504474957102965107000042500413369121871428878942657902838908508456738178407627862384078745263462884535148542410825911972312511070183709488932897479432813019524475152511864996873930392650925820825654513166443572842089571027964694163372602191936781393584850369407233238310276969639309711161374401023751074204564248465246870581944623241415213713248697328194635935833303957459961405036455879003360463988495991156176743328392068454786555261062741274794691384211313549905720425443048247923824169147623979920739427367596659323292480219856969405438956133713213771107291842541046062136296450381825681891598385821164556655747480771106637355699955521556828271501501870509306697230856298116519393014467574901811432487178065947758021354941426463780850255634668682546131054670973363814586272755938009064795183155129758069693594545314716927168284998980282254390018664017401165831840535433527723603568395155112250304724244735958893744377432962494420686280123652483589511632162538364430811834770833807213730716402890314653424249967017302424739678393269016117943122391093997438157436307548484603342734596493798810320202639558686753398308751950867286125338215644858982884671979949365354213512231707756187063964563754818041486020542350165273087203192233501312582798471175083904055757931290840852561754544432536723513401197017091331774453564503229857424441646793935706542888794143159700013494272653058595353458042582818123291609565666412929946462212558410202140709611058364227013689767986357073799735796249186999117691182396940350350149421501754605133286981944510606077227682665308130135165719914664531406534271303986369898827009731261653994479677916129612124249889001970992880130923269712752570978538871140489516326833001990258355721041045645344961351222322131880510733231464910485747379895846708835685315104161333297832488389105742489629339792717907171403319966586906199342181670805123695791433818207865790805006583130983975626841961527907719986512719818285283471810018893058760204472096350436730387532546050699900308911475655754633428392743733605744769562778609608092412661109580179023193879274666405855203404858302204945530436059528120252361569665247589158934225577721110826009978744007606804952833007219831087042300884377410506553959153128311138521110866076945943673269056102406050264990792285111280834883858410253802785854851965693491642528860968630534217762302622015827088459700744084693392175949346998220635252048625796915583698819990773602542402844568999049106907489636603767422817462820388265560760366164381045641133221376842849378517097301139475698549237276176154314666742748746476209731427478090818313632915385353914136203631042555962539557345537204914210068762294141918707125886654065204635002132617927654375314307013395025247031325093658755828630840475608758967133361667771011464315785209131635021275483870539065491865692478810662023336029953747654068818494428676083454624276671894931460902050187787343290608307450478206462579843843807280255834930481220806964601937541575877691117744999398416616911816850516033244632988002632252583667884462994910543139671389560977909374946491440098798246611323671943795048405368776691095848820907298228888258591244261190937781018522580023245920259320982197837458893839700823911273347981580017801340545760113778348170672459118669582677594411953041276870963463763412174767316654962648665220772467967689194495273095519499825959694963536467456350949983747795380046645268972606754896302645778704222425795039018004333006575069070563264500205603380731286049294539309552879047798337421601968699813650456531398091251665122930639805532644606014762525844484721000899088290971426782978894784497276593766395898066603585970907579404903010497039547990068298648266456589155353118690839730397428929739490925340039054179569970222028938099763469482916170780292918332243686877600809797209812471422767787941469781258003372435629925204037684300452052240914905048838738325138486208687926949371658350230352114188789725474201053710374596379099825842173957110043681043499192640544769018201018578462453610581312164964293838270482204928815143822981244569441043948368061646538762164519755540581206351739915495921909211938211061256614309020699058205047886322558779492820801465391505370180007530331800594057938860976215896145588463653020382269160878565478991414984284148509306518416556356991213135760530576928549487778261095176367191610060810945139562285209822902177750413312612523373098388237832584091337972272869093413100971166008980025406439866238804411540204913279049204433632811036802607555986688790490185381527974958576439252534686206420189513789066550848759975643368373895611212176221421071659813225842186684307913823578637032788928723634086399575155964941210912860857286197138982285008060192284415308439924603462761757725723828432186066114278018157691073490364802044239606259067375821873083974553198457033095388451247075359790760096674005772704841587776989027471318996620399347811944713622509691893634294040726569032047460617620587385757084503689860266375081436199091705212672325466783714733285478303275762485410959239505572022958216872037434702146212107415596654744645360227693071605538924316061845634539753151136151213986466885132175098535120240344273977101200907474087126730018763328376198986821169695410537005552303815949238407419082816233238989796015883022059424418801263897159466314773364975924864972465725181350913395618339275825526547353089990626005146069297427193311151473685260473824185439079496160497239190321855598454374445955024983701982064518980295275391438716949382359783649283893969066283271743469212161646576621204241055755846580147006615861935979633818684792180077245704229098183157325993576871867934501126286513478135239551382685751795296301657025019307444880593839596509858973374037956364349076285045963345323157857204295763178664860833198453418507152476642039698949794763337039714254880933807245184342057954517051515814590600747127344039535029144534028002061975645365921968558354314983838952684370332147461217216255977331845987780672251011589680278423081877405575106294711275088897951718160947668809409947016858434698248100617993713229376973404018688234348772036404079280420819112012229587679957125355923611822537680359335471567172688644401527605797556377946533408024148497637554295633081735277796829153129492030343384060448352221118381954984580368883649184895405795044195868808707331745911979962127644709165256855037050851565850175877839053735670702357185934666880722472584468078078289938616333056481517148390332137590270872170161479344749559354442327052801871624255014888827559783451200019231877809139856126676424837501295303853477420783667517442973043819437203807214325771630953798527418854826414478743982335987933328135112166629224259281527505964200853005134079107767894140963502087317272447283932326107867484176925696692602574628269804015089769458188564586609257238464340021044829567746837219139311894902889579146443916299061155498780419236948469872534583334913257985139962554290338605635974032265664768530860118668259541524645618095011316133668774438576937708606002206136606698044279466090369844520805365427150320937632943396530123650275179587036524623106051722320422655696541299029895662904902532040808196584808189243079358543771432096841294961244015894190206509995029495471783193228974889624922382051210089165467970182078411017434204875112032496080190019880103225467325204329795285781872155391131051673872826135527248784795452879183593586086361941322915599693800807609920909959885441932416699134035320913202761048854546812457430168189316957929291982330211281675000971070563074572312503093449540667941738351658961943860469734596673652499286984953024886978009275635242129014852134998110625927307941503371491925773937763240694612048015975225018295536710286945870490783793331906740238374406735270681312039093252661456197182451320520510719010080582180760839367043139557866647563701650161406404659926878700208793561772425280118289927637720402021931148447333156976725766951514407512401677781141928753100082464771632587332695307342867632998472106147169207981208191273115794430366455905336998690607178574943751908726862262897460368070384532325368735710336945645870989550327212101238760217104143055058832495827234775174802337132548409487337828927458398646042526917999668292870430938408056806262990036764309042744354402190075716500239775706304635829541389264773631395627622480411201603107217590322163190021626438122130893372033278712481332328032306306899056643840198099078383637395602496914352924336289071671562864447121405725736754870006779500334975752190723208713149654323922165815188580704302517868144138050020777547219623347793034420474515378904739591226393994021873219586742434782114421481578605148264248169430249610964628077257090829642878531102769362945064882838151141645730364115105164054808178205646021604277603575123658866779410529088604426088755929201359417135880503127897100995533214529179106616729447346892197287805946661005627849147828857524854465017838873503747793435898818273562606969724027108028546425257326415392287773884124105681634602840551553452132165573639181379414583068679926395595280301045336129196223913635509564323718184354770186279320556969364198362852913247139320496007567651483449496856805365210743885636972721699867814818267063055104767399201028899005682145451754662855805042212897169404072545363915660170178199991607746467788979621665644205686918275095749919289039898725671751073779937202796154843094199187950889760265798721128185596095970410336284228805180946658065521547339638283867276878218015145353925552346028989851089894035090443350002765028914445656637867318096926808812824820805074635833132983717521926073820724772217434481163620371774754634428621647220188895301176043106005080218225790726072381316533610528193394461596090944988764992090599718496491235727201267754899414789280454736353225219631330313869878242912239510761523280926389792827130263662731224245288804906046993174507521528268111801060769193515340725007077162686734157646471692409483465026598526408070669943716668389532978054875444926639845024284799851073787984219264552198741790448537925945607108498269959054707676195436743008332810351604334948998435408479143202503951365347732911217459630154729054137245879831661300542021368278011176257179080756242412129838584096658300487439380988546845055312777538243288454630062011446070711852547598239507196664858752750652902126922524913481451122823477470823977182049012125802423294978755957076445537866121996035850441262634553329484838798716466473867732418877827276345888353053004913182162618040528997971714153941503042457425506385619610578886800108283119571640658841972859580028185329035264076897705378668731969775408036076612929290061475312127830713110420109648563044281160109231825843325731519306953193867476164895111329989320191555843288552994963160410174301742363852421382730996114839654915503905509835765611478556239169950145181603347520527877825304773919850887795678316995250066024440907594926218937458186443204036327398838772427844472917662205052037462323075123799892040545903652599906249965498239120579167994265018969534617061524769450960555179965801571943269345347250630040897039557171942263153863638016269661764615203046107717676017286304973750699571038847983550799007656218453426193450021294957184248285597711748655172721460944252281287380575479752417580543041073054320628276860798807691899875257995335326871263720738284203783233367898191446493371091391894574485794575840723387615729752424607566491391237281331406597331642834691555105026363557311276435163254997237087572207479300425211862243181158917888889644863540277493025985286403848331545624836639955207096724072034066371223158966623584269324657862801395966411352760106016225907697444282830501957374255852882763983611995141824046929580043847068788488017339493203291543542821036291414157831294226663632484679804342580435419541973672628527967870474597022206864292867903790266104472453799393068609697545566916355673633534416655414825239426675361854423369358402499659886185359768237380539833467287807884438081376666010457953918353654061991967697355882727036421063675746498880718144018914418429031627673004572850782910254361655653613661843671491178993409134316517690710070443877378428822691881128013036158376118753654714545415629836049944911196015884925655042701785367297011232829862877840217033269618279333968193932387150857526436742374111497733233010176142657703396167230770152075117809861806181284037833890214113193001954184024901164528007940237068719408386420810094750942457007045495702122776992489464406044127340661187913022742571947291367590222826637707042804780733673477072335494645439638254525465839242282043522656725474294854083716549980453265648862755283953024492462922964202572417350617186028711092455826675503752244281286090522342318679936828609107639073562907662769983799537616114313897386384034316726668093585934122018214861423058906119222560643215657712616256307550660446534297083114707120006640783082970107072023769865670826235501213913174861055070195466446281262029542860411099144525507833760257611122837642212531945716721689311797922547406784284072192988317831589908580921003464404813261309283452940979422873387556671769167827023283297617350563630792536905105026076114249945324921197338415217040992458294659438026548792539457711022261975347035958857066365938966977904253895432786603762903125944664112622109921365454621124888867841321318065216877448198332713708994538146802201998238268729802609166748565123888484913277818434021391599747291979977062099136132936493830310536209251322904185754477512431425421417928550168718926510973944120178107352786069791435424081118458461472781331719126017512926294505012813536573314486772475058358844780085262804578699377713805108814039000383854542751774007414976297459146043043129101069385548370742262660070096156352093298915845993712344619611652694863903812534545639859182517896735834579374184533042649876757168312885547117964257415128491854525911732104927814116590860479228233781003267149790577366397644520110439123062504673105028603217439995531664804349271277395092331054857522362949198428076798965866666257070616717798821907615086103683520806927922653080107700980229397687668332445578437311963275892637340790986898631398770875556031237248423852844452155015458230168921589138041511083682243728570561404155023579437676106600682261724514482670086318231674382175511288238236913168028151673981482651438358948007083382731764662040262108126235518549471427599192290771992056997517255008081626020322059530459734910245072284200232023184097591449459554176604157969389133129140752190657753565458024222135031524322811186214096086054975596921701170416069352920371199908443026503415901550275481789107113382593030434640757486045169586862441132806070117086487067217134797287977248596625158277734237452387730308652418955221014191376124400875361132698045518648850655660312413918162892337359547516654049072460271840856048197440058842681956119987839868700197440033374941115805503157099321012428692509366371096954120647478657856042688829831231356127301175576409628282965171255923614102911216543345070523277087744655957528380732616069003807212790797283134183317104169294157133486256186984834996970920153390475452130411702874413402958076246414283231568094192428977561871383088449955041147099956375417429988370925072931925979262811603419044252434856229708838043319844955536340940990319016209011055876988456470545318968824176933676487172595828270010379260073106722170394961273129278213604151725319857073902809156047801210635195342703183519910932032382950107568741565824905554560289872293807021188474954317118265498391410049813502743740478098193876402258735944495756263303923183737888507407249859862153357260209668498393933509943686350067210232552366676649274606632007373897159522460954788747057084132353763706912326421985003333039438028778567793899116597207091209862008341230420966685536038718304068130166861687126471890767907297266116228596483502515551686843033292909212272383240817944491169119500973353826653342050056565738317960000186757038461547262467376686109372235616454793663318146257479256255797888960695063650980444886630713261768720266883104193915347689765621083880061312864654414029507616706103013771863287601613871347973571850674147859506872343366194839351348998216818470119009724659871928825237790680031632906583876554492010341899631811566147724693184802213004033071444232651998731304996760976322166574734374939384686916684233411852420170384121317631774858369465514078299174338583384814275580257192564458993010584730362198790500785455451575031823646561500820873800046876157804690707219909464921166510425648097445694429877080935097963712158919168279153924001893691372227812295213393335281569236393304414250945743958243957246984907536679243158343222717954797024513649329841049611751983762463692386080836901512581381648587015601972667907773123208031130840355813566546223312891013287521617965482900187550493433315682954133560607296315705206330348444593282558221500407544936049981267394702101631977958155358387028612206256727737755864969254096739107024679732423116264315204990360893825491093511268590393719522639989469419499849594118671783857732084651532541971694346766902758709037490804470370516375962253244396736874090979812995808734394953532509068630348344542345051450670479968088721259269174177205987319915495232944662547726809623233874698577055566234044099602980597950549107864620448298459736629986443643628836145102671105920330362469122090009742857616253694178749433927734603757103721825231409719719894582185224111453399680150553663873718810947224517468468152025050026183361376872360305730038148857110747997754902095587500040761840448196157265557757981464033741260357737770689190764739540543439366082033997796363931870285651007342811275919531733660705053030637606393573713242216885433676870035858323346138722003735468346453517543937561584299582477776220579514114945936566431090221267402979010177373504000686151513092370587778245572033486785523325509037785278741745634115236237451519388849564328191274722531317350551419026807814755109736927275404312525429816313393682723382412781620803225903586851980847899254636123964923605566523893611878092082621855405676069905337376703999378958581110512461577006139931218355413539296307482325569538183224721645291765726279011207779904929944022087111426838045660684756767708704230480350341684669842438071756899452606956481852512157141723512738979632554586106688832521818287682944229097167461592025621527285298558449906393200558507564659890606089067743611423161137001412055473425571996657457329542971836195991150926321619785290276208576044946818698427498061280590588602171382289391705011730226049974577042547858968075679260487095258059680440122374889829209699672330509459798694803693068102065045877555895734739502104952450355992000254106275875436550032599218846884899265181927094644663283381102299696458063204158136313420834831803735890845610482028649587354018581304377643167795767479938759520988545254748793447115260774532536081057029495735649536558871878266007294907635024599921355231747009106319878682358113572840335599350381946137709052293955268257403752850924009809662669131040870005991945652096866203753603885294494852585022651774854859619965338197591161965907382648573156105284977559403304559594378407620361636989121145570282438015740458656332792244566036188561768942011981362958417012690366771575984603961804411225813426916564566758784570314742769261759547191785155402988752521380904434265224097345351582782974045024194843511588044909360801173075324849814926453156488608321894696954769195887516262795519467888536704403085484482740567732293005985097344333730235020359988936515868660455393498120762667582491727703903562101910105859649751531112623594530303230229342665130993295416997487502669486690070415410358679481581858565838854634833958311701758429297348800073309470438553349286714578248727405682504757143797503545810236299154212144544382501754972821509927174503755710012410582129990477434651187930128216378737630324520007832873766554568761744536348314371709496207922229874234078644098957406689658074560287347213927399790416141449235493608965901142689968216353471523043784126513146365071951314327952994993447793130390071202049078500856812723006738921628260088397282526493802333260284449309954915991810477552347967877237297400306556049942938739178678257234220925851111701233448224283547093096676677499256639881827866488626343809965641946038265020087454925338255432181135801874811377542899053939235465051163968277501014747773068635386719249849473013531172271628279975815419414645431607250588513113747447887992280146797368596002362424186889403532270760024024609965878094869505256929203824947610024405197265900962788264272736328774974216068850896080009557953568561038000459344909620584710693348945842708659431100677456211739841463794678193501071642472057232398738766053333332411216888154451424667796658931552096996183570722412351888449921600014493241882497126091584636602221949591436361007877362556899466812527186552566808248085744487724117341770171092859704682113875217452030598886323942756460904563192452382859035223585602556796627072196335033155964630242591143250425375527128415269567717265511069551579220782178512493785157334450294898994388963323682608835539060018426306679459710220267076828287211731833329346969298383404132966933958586813161834168563168237844097895006883958292886497398916587633643187318212815101013722320619744307893021653662850587875354317894382154820858717036166529284582366252045723984842912208841049818518360028025481348497565560264339906101409918062831707412530354437459684599237077434011737471277544986038306573038294763647342493103096140597084725535403567729309153176412536416481226483681842858213587964071359662509265733143933111802181633690499896928969872662014486275213699872306444775887448233624008110811486310414907883615694652383772627311200403723499130177298949700301232073585151701042340013440045206515006467184418919605063691096708465746561045715270263400777810820908274160638060354854445261068345288842173284058716522968965042161124331699783472400104231361740675501479075598043044960812192246416974014281488689343901148005732299954817646867753654311841047442983354312939049588474249159817338505368806665526092982205312299262963756243346668125910188241961642251452004706208840985604374134214347097782229458664843944429230194345739010880262030262159106044414522243632219458070568013308262249197009529037109251420848939395646011754152385778423282433830940693193831180383805223935189598523722999518122120543592581533051384734951373602050351394884621417946550081203211447537260970856201791536093898052447091922926844282988543584921175076146496936488086650942771657682922495617023020398542130179695010125411050740030478931921011370934135502029512099237628435980897644642625613967349374979515302120652872421378047574664344853639448848737652238216950084648227171554009451812257177517767851490169602528185205615429490050843323633165800307980311935983215385585605382892000596638858792348097287135027610092012929290826378923564878350308111820378258180719668471050927582596274712874155798291766845877734160351687681313910535779566334029671472056148760161427391191088072067817007744800881081781339977336456033646247950133813485158140316965008616267033010184032895129593308628867863306021610308711127107511762474581908372297836297107599287606067671569594258405950536143102628509434306565297487601838622977975298173435190733991959691008981366552508830823158201354891963369100713399570559207999683852458130255485160205271054577732274454921016290052220968504300185983699515387099524072735619412445154147765668054142542216036824453109801097059562103272306924470907047848518983367410847833041060382820037353327194240446201167008692491222015611482598335547622733612396435524050275125307964056838315490435970788053988064672339434260281665735470735165083513585150107168642712935332048404924937608159433600810757472467161583848102248480235732210072570270851732447296558643371701241402665459977228106163316437611106459040733705744831061615339277302212634222397866852257250614424196306549831943126773033001384646128058484562653497710896435706354744884099990788041026770221667394364126475246686405684827675366212620432144022784681417688230828730535030755764473473006335331739428433977216267970304730579624464047975447195464907231058745396330859024404721513438029085891584368487580848158223734886129903221919730228248506650313149406427838678689031978527694134230111508616649812993022137785194171455843998167856784341868418398983262404933685931038563836302397751744066509009509347469659839903818569355500136571581333694418752263388214025189329886327017291871689154822308553172034159582917452564485916879947547862946748640293574780102722476806594244252850963715213066094682291429293664023832025388215539868311434905432141232400418641558991032435933368454167090513596186429257139898467062434316783541637396117214347730683763985454165405047858700818542002239881304546357374362793290093112349643184312270887174961284969932681288003755071993815023356624288775251138427837108420033318210189029528987301285448170324583139471849603133132647856499638664807789150859669723968528745412622477681930684016635517362383581625890902045144058287946326010341194442895303072799682398017464497437479025931268206647337418257179861633065943077735111356439989176463502074745986891334459972355729410657287969720680655048001917022499787425721611195115239299991756863835345322813101025919650643318094759719800848219640446045887701632286642824991404991489391886672679412280673876411194320513290537609855790067103610941428892630157633402948244113953166868811606169671219843884611365790908806276758889554307743258649986776771115143259593604834700538765002714836031491217023713797065319937045773381150298867130056671312630954471541058513036439690238692274775023860004519229216736206178746645696334843914015023635617250647275018176725331406170109280382261981192109865653868459431642024536670290501311975423297697823284159608305812454217709569973153562374886085965749688652889529230269088940462309090850312624204718568457454750470720416999880357870755176131318795757458022837450440921231062384130179677215287656812819298560826638848245737956744387405639054998200993126406836343951574463433557120789886400827025252301673266572205617200613811996036802362180851532493318280965615253435257226199959232700758980873215531886827537875703756487615391045014247185160071607026629328459409091337148224643155332830663099241508056522007917675268955504531563066106087074300794413954540960012796701625238630715180271880327989470786682294792851885698336743290522032599402477508568771414469347386202676636837527462847144645708573720818992727228129115301520551709014132059496910360438722574463004225953995223515252486354880084685502866739883868090079613698074653529537510615881010264649504034059177456424581960851205057885771946390373668378690582810777782795453988015719796231497762569463886216644277210261657898438020142824306110717439843303832338283628473915366036512395495933579670621109263065602341229973012695172699805485066411981733926068576171165875199623270529533725806174307722665482191274211396391372955672030428138198550931674049241075652966811913279320847801832580071395219840764310490061700131492185103724273802731371121859472612652842473664250338212197378783891124987373549882163086394656680298963377146723135917545844061447132734629163120708741041721739018705725590396356382056570398905738540808399857555550113532391465986526113631782053381646839040971601358940532757167789391770073048053982322983392537499634358184235329507592358437610878976991729882985221880261436851148478117508877172979597911369192772330396721012502679578634766471802849101596211276281935757924040880123106148414341582524149000599537042243054905044684083017657071918811364401617384168532134634574116439639999159853683413169506953739470296857341424630323302101491839114577941280569389886920600149134863182514912997716971926973219538068736201431021641358433417839786613972924367345326097679728254048177475919645145467376437737052142169542640236433655650228745947293172744503709388515877079943105696037205322776015500427439453614836306442547822184853455274063755691766587809404004460931057144626163856404525399880130914148185357306770286818094983806317018997624851916337419381243427377596598632893167526861405119536568484782972225875705466823481043320702645708198401989933286945856667814655650763239687994573836616333073133676815322646119957906341918094738768684419405472746067240612878997953160671968864783155369752871183454593092451397741295629717082755402027906612622600519639274713367504700826098407136811223327270189193342692548209535108424204474616102130830527326835127670498861883000127925662374041026642176747457356895919759647170238001502397714936287804834401352797728248820180857894909285628555182750664275543102171717152347830086577137886082675821112695547707037388851046738601796911867576433091488701764631892383731788650743007210166598885817199018441245597162887171977079404062590051251109564194400726473185086751343505082556769845706171120008372484882860760905714631855420675694817791076914628566924352544568854129222129048974501569875493096034780036452559974011887811336592710550421845079584048141308915530634516212228993467150953146361247238887383443777940109527542142520183355796057707150530012156013943291999468453357048711470766607629710453217132890842570751034493441993095524272021598881963714544285851127413630472223828601330859086034604757177400830311164647762316931454653685189632829572971502232187651730065364682709691104011147145664598930503433944424002809131929816437728160692345952498881724451163671313669821959285780639522115414989120547928336957995164570143551811040289369072892910368884293416461939126551977625542341435578291632658408107437638691915097301434665743805539559018896150242670580045172339752502622876544978251576170619338176646969535409402157534864689286654754418848393842310882121436624535651720412253490933157138550886873774635117805298772948589974756203540802179338994047053990382109318947331864080033344448132111626319374247323167606679616444800137920342377439763694006363280265165764158254275398514838331598572892172817202941006885876552662728135602150059726685457317434366846237829236984849071283954109133124944017964041547658172519529503077642763751156822859807349081343252136056137396502255085378395267794221141974061515935472144300690971287804951914372933088048980745986220091275982606839633316980412454540068787364507376912902653856274809046419842454956168639530048143359341999964954403223579874752464757137483257827131128893805019082713096626200933068372249250145370527564913467372067882014905654961338823531474941576209880152990706506491036379396822068474462146192643100597369573257615174003132677593812399438426224939809590506542976686351335155888132000849488687102109227405348832934778073110833567619702785977191130540151856711788377103624846546181188926529724406395390571545403724415849570642111861067000487313743481111460266483763043756871178756621215920428622912105214983319314099001179182929509323437099585708006690485509450968214050461548711827357178804775764051182072801521512370082893575595331698331094535285944504982108827824520304527587832158553415695359355911121555900080642028698744229877713460844525296715573857846604646377025322747744735587422553102059784302115098610610083193052313380001527334351852903253437554887514134565419381936377187199699998420987794780994560078161097602366127136036898403960372520838321352489001925357110592295903712098977739955812285616437952601624401046919581632141389379063668963347070749623454041943692484909982146829446000200035140964697322189124917907945384472458837856477536347505558012505858139670596043827357399205766419182680086505764592312862679201093601426364922478977453682218881947067866562108365867130521360727467043422754055013570170873562383232165180089092364452544033163356947162912607088361619126978338125869794231164127720406444414429066882662049289926821337929634860205542696488988860719464084546815825695045224167140366539475167370364532673412814621357026495343095810786435166845685001580768144246156586317316452924599306007374895549190277308296729110901596329022791940843232105509811932589234757541133204905974966744159481460586754124894199511314024122439571272743027259117527421741406326170290513315131790890860387000031719010913566176903298271547108316324649793498223564314817995744289232431086164427978544567483879455929086948331224717937898924249492697152942650486514697993528380613552522100049262940335855782120959524376209004892911068358599665932305779254585190194134490037648991073390327601946087639995952362364243209739166931564541617175959723654923279063828618830958409920842955194837531382617261547681567799744212156656475758691370072082484008476156627018127113403326335083758384573641832770950300368824466222306004770098110623445825707519404416752486229105687102535237599397376425344556610231846590353533282382525637334073851613213465905081824099250244777292775987840325078093604320074277464404177472879369997238403939342030037454020460263630320928863582764050981172160209917067256112938694916264693417994737325027572092985314328404340816339153968821464438379591323538834117291961713454492335560472703078393990954516857469078162097888286535926490901153461303936071267475403874428163486148578029387777229264946747267607301966843806343200960071889712727730461232649432223491013811392357990192270809679323220142275831572129124189280403208345764373830098440861554950475273091061173873416074386685590737372449067624941275460675083279516993477551877433423151896639979867470541813822459314646616503888163174696077472906302571708413844945413137717705121464746195442711475809882822973003952406284491395430049471458256464769763613953555666167238686035376526808411834859509366507369098911768137254093146656895317635739631857405767372724483670891166706297773351576121746414717782503713187042368269454951788410371801039553721082683449462674506112205205883573775245783373118786804093771964265513571619935617728491431658602849533564050832736175484311101898067216816421544367856189855367958255226999961257056128505241653918093995447964393466160070754483932162704019209639679372189437565899321109704878073913410709431408778541720615876982103840784042279609510105074096928113817649132826098080891436617650192691646963616465707274300629599820441676518490244246509177669523684922912201906017230860379177055287929176218841654174421287259358860143874160635424487301871693060202353105284576365171908211799508427538128133561013823433694559551381510712575614256987700402511743914254122386195271978123336937727357436396726333476055705371101974476710400143996341613263865081115337694759416161386934411847645592581776558232183200930799716688410716271976045126353709667070423521649102402799924544434615116362056535523134300697552392188533820858297527828422007224890173070594291546857174121464607100126248363723981048584303174058630319655470348751875666229793718046580745127086384975377521933400723527706873354764647662562249362896584279015630628090794049813308673718130144266567468091636700456228381262023270702034100569962176806649675352689604274793840464854587839129450506282464149909351009516112653134651715909026868299742643937942917988778501098126294705366734468494516936107924092901769775460545665951197950489154926171768183698394473035275266702956923651990221656198962127751926349600650669314471776396165723954955777378904437586197765276316240097324648618838586218403821997500036634083559835871650542212040391650243781698127555998229390439652667719587886749886601591312063700832077509570643725454501458281123210912665107273822735325097320182279930418518091727706182841375306883233318985608699602314930950517075279134165643902480501748718332849047807787951514187845475403575200957215181430471524159696548099376385675651970230791725841958257312627128368639381232762898955011582250862892982938129952367958943700259459992482818189109781597595550806598627627206321005759134073088468074309026011532451815594209406461516449126040155755637121408676866630663779195494228339122206536391172860522096387236481567072780805382550829196034252728033946902631715994730742906352867029861450949420213522404581108024837248831206749203811213376864129534594521597984212357304838807655476845412060381276854368123050180487639953963250225007966170258363112867117223026243785769740942774454965830132067070677902665584940973212235224912432419085350430058443267946903495324893974545320620224897992407507920798272674902892015995027409711924780609895626634276312486948906418263296081264626280065682682734250402085169977282380652839059438159522061408445089394050594038658646908619231077643143805898556242148859151631074189970983165015102596837872804481668838954112919146404523404900600521793175883931085088338895835295455333293877163645593966584482304498648642695254125400217693521238298148263303126034297945232174611800733948899839689001769310199106697983388702895942351450553332353177037085609169280774385979141496515891671274587478356332321302515084951096752401964224342166623679568071782025459235675337398232454757280513688311625976836344811679038467015674215058676765608950447960635269892054841161330052697130238308030254661133726505071690939973068448411658160390069217961581741373456650439909711106646632519968994735355199769133239520205697697190731283679354331800379604504288126141412998641273567206863772357448457778611377915636248311625974319798459124860593911459581391216499552678130868895248803180367380932188849707234205603836289696985603877823382238839887450386627020288649725392110563595842316686114772470010686085803251341985627713157097454458899340525366928531496984813507914381540490279998602682256341706577102579196235668233671144445453609660468906367677144920941598991637560826250864622516728595653117086262156267032523130112068143314741295738617499339385871273781931996748255844634717731643678903992484241362012452397511611382248463043480349201579386899034159667976259530457558854498183912212273923002165627066605463440183186237861971328609718833430574597749204643257090930182575173668073675427790696914117488896955860647284741384848183120103688356235085258848166323545523595843959707257169082414904698927159544986565268272124245134435449000126986763906816690554586795065832294585820981692032086620991060142474683221388200494057722600518894966561223814520799313665472560904272909394995963510086671014292066608712661861346078946309881901756831709675456061426750104412542470944753115861521174461425461438725073788309075825416019076709011315976239376088501394642515991871696089907390312144023684690121574127567405607009730542944086046562925607480946092322101105598992233517358113629399257982943461254832115987665165901437428367360261371373807387832230079569932411172702163561015760009717207665139520331644297774033995911404074648905594523582412158922679326358585453992058915958512730489909048739073287039055077995591083821699621497365305894428163186980399975832926366410940694416796570347376455307602543105447404998008581888543512774035858858952580671628306094617172862195378166021835061984977308857163508573851072725608587775896116093985546405661669942552777628412762296732345446026670087219510997095636946651306324787318869405009243347215475572282591095938852558180442550826696643040910439708075324340434097494160244671923140541026260184286723749779768905894478755203779024034319890434033484228612794909904374944620254858981871524302972184297977680623428886674358028891128392485169283802860320334064702174754037893282846916641701353392472418005738294547717708077085556422707163615228113603353043454952801982188354546941371640958315923869759312781282775107135757036394398985857047967406106859148921658312954062851352054988727816306135268626586388316304714922065700896011825448981049802236920878349217666615002577097566421759306095246476644383885723283223112828837292648857171779555764373574868737749735200846849183698079822189797224189207225216606298060817192205882090660726607295776996294448099725051051520332199964825986944137444463615976488753918389626740837203588827535090115663909617255803058465844527879600462542010091120784310730941199160618711250937633426995003287489802295869149035573445866898318487107940826041310356924078236911261221701415966839921326124344512826864498454411899352550872024425070457470982100620133287896559977961845045181042691234174776186811174770746980571047425761832129202978591302702065213550691089632100454857357440654815062625100408461267772937579510196661570513439198274771671086095896574367547236517218152524231692903013409882557569227594759620104154079503559332072388566582140786270964604587110485663425448079357281449422493196682147059096465582597178382423066815540874629431978108218748499866173780810145884036057385534609581897967161091158800728473488773929226104211657843380585643294378749336664964499302851376274258594569845535841039079914318075229790131810329437397509733141307625245153582686696963323380379308552159819389687041773790617599995852593137015671117349893887755366163195027781918788690109754739580908560818780830538175852821980187622619240682155770062084932929717062165260271642855071549548539441192625410887817442302715873833593857312772926802293480783352092539789798934134150273451966334474971978531730472123183652237689948288968772173050250815225480362927352785168879210154434015124722017548148589147626786131886252494832695363530129629860519082028305068229657822673720637376279227625662753511851666451282580024951375090719965194812084472313022637590532940566781213500201450069974561159709485435694769344074742095855558905435781872472103393821829323819718021425287275802846259306480854924093317186675579716355710902764965548159749350058294071831977879706466479265127636613219483308505433715332059028401764171451990604438685007654870670268627807758587088345584714745294691152950871328730467369701576820545052888418248745575682841045140935335607263753167954340072824255419684791997245543803121677379462132922656912278518460595043115899777641442900289102521731079379601232319100883682193090307554458403232585036177638314848387243772845292518664436570084271775306466544308852984387856406929538300373455389709878087094526384383865825754509761187476694423006977886037245577975908134202137626450979950332899764450713715290865267044873640207824062281032266341862336565906312496583584914475144589255813969080693051002610816158351683027077940629513302236191269679172872207597774027507745585087128099451244775209896828945568862244366970008436006294869613497841351750967162442962059398626503713746583988218136716152185136728323887710820674291844663128239463377970047183237613052133947461413592436549188683482535586622891614570625059819304669101418262427654208774505538240099170818339219594427918370749258950704514221928475588605573622733355370033012414126076218920577919073073669109954321618056597384579057381173236826873544887716227062960760047148053730858946235573109130170205490201426203324956468366264922278421031272337435905539337993928529905051864306026216342622025563046633912050681035098500583201198786581625693063366430207380274414944606205647915310506001509646903916588092224173767571132793222010028323544392603495441978384052087909860381215449140799900222055791412668678283603800784401315472890651178795824110910006979993413459441307456003878330226828779896770636412752833040299587686256873818061545975547161647723885771554264559424864359474432823771622498349230910367105686685674709873185634122637556916813261169647846913941462482050776271171978945407877960593334779781600370200710691717666388772152913186044851851851917350468888793783540680490173119928814153091206243822876920542995919577866690829275216766509194341298638085848341292918514350463700091353321599550193305202427718521486641496956428077628818621110945539024579132993563187909497829070779566794105147168147847331836895457537428864675284775538239495195751538615294592737535764125557045134573263962794703586548517004511693060199781848179941415145844996837825062064969940485402227385432210429302956979414485521792480631509161087014711804714636810024061470763176728320079010119262346383375943264929115632297154965341073794110856249057152323342529239520317877316645683629612537600540443493678708330125909663073169549817781427127559232025118354530711103987951848542754437565451342500580909175230285794184508285213300062617017891660702472543075892984956934689936858822220475682977832723782502737326182346053939671041319771275411989989642015813097343422484415967557759325417112631027853838417946163660315741939209208234206515346746319978032615014807099698190926061472702185463758481147707205796539906276083679846288093930334575107845336145902318743459619948970761964142499221060759289309617242554673076198106512192770241088843654691619874119569583023187091828571430774031479975993756025976976636638488288938798618406661776470343207831542943655406416937135421321240329339807646877898754496190000645773811740233821255144026179762209258926345486290324339208303366662174684030895900731566687397120704581870519661615057320508856432227
                 -144246942720230506419293193743558885954636791377022229070962814938111509993985127901528810057315150209639235246922153183027621252183565148773876020062602884012022749879094589497514159772548029519047475610435152856910647719949270253591918994841233818579657292861368204007208350108966089500406753176389993928420299364833766850895045444292805308689452994078359311276784902481933148207387472293157576570853463433459116791204028101333485970566860017017528875671002006470598116399324567293303533660615818466512318384284646443529110424043348051086762985995232766665674275020533338012121543921164735482571489935547403354889783024728544796022744570279076211209393887533960584547232026766847911622781652228579192302558762633561974586357461916140321563933985053497791604264874322841848523890574839381680027809718724624889371547183678412482360699093065473453915710786346104080984876410123825719376570971026069447227184858973188863929910457704107180447682909420958865968988205715866775579188286308568451675656377853823294610690626586019965724364186067882001875423493995969099307333147410213339481466808782548632919941935570122960557189918928595034046997446550768561426732377277567840932668451442459341390382807766665643924681089610286725065423246149749955307905257501646105159624924382842476011823894791718628814809744311800493030661946018642048460097515341908485538682616664935691431149437363128113567608020755940357403874499337391754532014085757490986410565733724543295936557594275778134163598849072833512280504171369030944030174023772306460735042246913561093168764609991551616188563445042322438677607788265114174085638793107050438613636023638053650597624484421336569007669880648167146185011118523273542019778531292710004380156064709354612565118442728217265558280667270267955644169141489272630417794072970713749868094577231365705340976793596352186471198711107516686568346964691218373952105276406587131914318015253947833880446477298536410174701894419928284616079195729576280803093835582692092769834263369032523004621608601346687515853227374961720450709147852398325037750691558876675622271082141892988386510199635660424666743801057860427324084085388704608940086918211420008547464973072559562700860941867555504207553588452873246441780225667145404539216400665845284576718293967483045306449065453267685305297200752588482838034062229061133435505550296805034165470069641831052799480562180389633991891874998543491115287039896642219137961389626411356282963398052532869739309211703960460510454520218688539370150148527420859524879270693013527219748146303809491959259776200421825942594385872504839820036570007511223747371688258397559031775744295629751569281070832689581638505657253409250012401780295568992705198253932514229921264105522975151388817415101799442288992587314943410009853330995861920753241132084233660940354229155755379356658757709272689252033319365474452410545717490197684294248603322520499628849068657011006292761202031558829039296908276445842280516830335393386259350244426920519936655990354982012205794809809602077916601350076687734474224498739125677289997133302658568701995767035521126338535235743329470482729210403478041747557170641835557806712679091703778050379551313016667789231735943031580283210544363869186958200105803589436030704488675678565318140942936190887817353532996741311069829280925285164416179529383946001525034841358042236836696767411975375906431272060314765354701418884419420177749930246449137457596203008878885037541228058642489932539199663141001566073470478535756244511948902773380429385740963188172989697314583742090490303315033547036879200764883895058166867236149156898057292350921068309466207101710615078312090410816559103001841935915029980952500445731452272606875930135041508472841374865274616030557454355676835034053521029549822467942656238735676026592087990725273042205604461356665505971281917458913743186633074908530838784198052379868115634542931356576767431373297442590483282271205434154253875987629004982112935594205724753090539120000147470445453179412285840458557206267333421000878950094794282170203404826552703823331350661762795804972218483017616261502272852206486675287956239911071647828526212690288330117524632370388585970968721161863087253124430673928062378239985028362192152232381411173726217086868064397366967317000143047196545040685116155472513304879087893826070756875128159241176370012837898344158499736684542410058083414996587666678923967682956385693610973705148938282713686294960536562539813664419265667961190984252199185466453575777228389219726074890290164758372998643035462448289618773559922066388149300202891758487354782676935173458397945290376749784807802270845807094395100949022549727382749320743017099710906586712271999917007173951651913337064484563698843468189172487277050660945208459741811090901503595163171616049800516281837499557307853729150388173047214873905222915554605349226722185321487978338012681912850810763347037376298300051638402726503699612147916567938028889717054167392059963470134699345565947989683781087271492615022856635841982573227865487448375022636153564773511869439202973027416654652640382113601528392472556699707650516698061380584075337045382901845018137079590013189370234152937750311951454662695668548517103178827141027790735614821382562318336520209280337676838818368558712421651489236798376707274720122806794211544991719532401890466772048470461948628490720120248238948504798459079417908498062824380491076096658963458356189309452039109705208356951944261868934636511639120799329613616242529472763184592951165501233071455297720176285592885105761085773920540402017453508961065592183370652465794195890953953940783171045532994236522734064773478706501748768561629413529176071663332336770716731973826697602997103818773925734260332854938084608415660894271946940714607067350939740609604078400073288246721762165971470495152599017471989309915566081145502560334287769084858073515233141236998342138272374123760538544523967746545675433120033785390927113361194123851478821088086039219874665927397473786834412945264496362713845525426016922725735561631737288041546302875405297345073570626690591146153914028151503543819187111971250744391822942635101863776357706091884407504507134630092254453543373974711125621123849720247916720613504312181052967023894482755304732651704849314668596914916281419856625249807583562385315578750632989086403996208906581634503561687356037204802135513494254662353830551213229161960553586644923346583840094012962529024566444972883756797726180943249266321139356891516107893515874926066348142267190364573151174608264602737266545446679608684094847419074768978052891785521733858752688663787107174895316533150578705960499875919480111263210207535628730891980908474375792735469809969273059911230694781955167921640308528307447660971674561016467091182267761812462531389809710103487419856842581525668614125003993765540725927325451038536962145484388708590031226208465198447550591196111181756312134919300978305920020665942062630188782701167711870766212144957223850745307362618796798133065403578557133505071590951388853415215058032549577723804212991887115819051300135647630063005208010912647361367987700564609983609232030645461320861522075812560640663660842908547014329605350253587667201365800770180032087566703850933163713383890444349179236928465539860435211873574884977511751595903794909307929452566444119566909557896600643593141453182591138480213951855495595875950367363660030945618981776466378036028878194990260531102393145819463643919148200564250220955592738042696866722559837420819434329753528587571671229978919282659154004149968087820435776630921200621968507739173081226909230407708735922283735014879051276407660948047015575135252998781888643228473703936566210096874794720793554663858877316909995912153453921667030671219306780126306775259984645125999603047805748613609412519383164555451110219626873534824330494040224305923031525332982435097229958346528114168543472518366550823248662912822175464435856818282190151461983921823401698465996560116263522309017055650810607521057577879230500762952250778421165267081863315956904209060293900549197460559296094300199837505712976350170412641647761961364672288017621432213935811721904788664023245976416378122411690585442910108099093438381906649578056245933009677179426300928059549726373974036463283966848762531032016968442264404910323659388957445747105201206981922065944435012811373184319881686023428085995281216142397082885971654162957130167221458788382666690608547991786545965378342257147617800027598941985773236810442409348509922210813542477287162373543574761944027234487012814692981736831306400263319700571553456601113723769767349029322521435306493413804285074595483677263638970435598152662570097095669987368918382713139930610409762212181806407219268344702341622201834550426604382675240759906957208086030951742482039281302324642006367037595268708996652714044638832168127369843270217380340146178255811801610434853727855414500373213338690842240020384395489280516196438967022884722879097636122418403112740890623489965447713808557949693379282032393890617124823183187959602186657888972678520655290115342991249707058264602797476457813550109387903972322176387592403622838244878407915800566571439834252042763108427517909637366323657797155090871765932160541965825906989412207825357903174386262256738374761851884079595505317903815558507077259665193900265454441085744341923291042846799043743836078410133625091936868256564319410856818034360103513348549276075193724357168993310400291829371212998676642727408628085466302705310468813944952201861645337447425246448831380294032496991554753084956074760666009288098708412015224947024650232568848847278655426676249096427824722233594558083681048287730933708282496230448591505534635245429924682464788697343983601811825169284547822742731579351825846748198374069774473479302299096378185090135075620565578431208303793572822199705690600276891032393206610041321691792559582991360888173369453838615905409484009452758047441847773333055385735639486829132218681725264179162207922586350789195593789511164602308850963064943708748868044276421778475944910398917505816138316371682635681424821955813512159338278976628549985804387017353739430503313212043010389113243336139135233682504384130777779998444409513616839360688389518103767777472912008903720923207451724498124716213296227290832101813641146448259124536676580291730211393597989962102605284522327045151444911675323705119792555779653550319907551332315829869529821334129775671347266369886179107379263438204001359756215932210842440958558058690506439368288689267668201722332282174934149884731973484746749165882010395196797965128080331389582363427856158340158310183833199279514997254364504992717712869842940631298191823299459616081233004368075124624204862704476739527188672117641540593463688632574410534166715115556757053954527218773484762562807052467702275625845054459395784536977263721536458286878474615247511930428067561591036330894369618520674333011655759993592519273819116264364034151397689524667945463987132691437998070186568638307849010325740666250921999935176661151024934671908540134212437987084590487793840174230610465092409184136524452174562076971746949068209135487434381041521607015833625196771073905843038323188208600299923835518811331361650071406932410707722018964516430169259335107047808293698970215791634703636339355555589428539721350280840780248076015992674097215883792580150563070781495866510966582125947234320192395993277277318014425549988058100611131231429166588046906488464755863611515683016297388587020438532182866238419736454615447229456131431197401053308905112101718116548301960008919113360457158727421146926408937706343086427122659179119720314374270791419526082614980196816007462516089511008757053407768026217780300722131844886393008658312451869832773904015183614009384087926299087930334019755047731581943346094760418992962384998688176680032088849046177045433986149446649863243871419080752711586727395164641736260858263836514390458724876847641878821068435880452898862604415438512228403153398880642223364376986146345476853198936332801762781945743762900346041312711441745695872595681925799829644968963935382724790842134619558112292966654341116527985942736769675728286764540926124581436808863938992139883279266907172368285093992828508233036884978456526098848635022497988440213446166659456493510215138908711327647241532798970365199092291780840001385924526560229431710890498961600230645643955000084323780083140202400771206356801446299808461424110800383888258095895686162390759790414028151128792425645439608194644840273847238769029714703322627307368384499235181007227374777451436260165342531700304271466846922074967348602432605865248704546292750792549854205172495612813058008214560937051590144945792053095221072761102977268081160876526489975087137312116536203146289973257450828491261313833266272858984609881023669856651862513076097133626084417648197967287437076633384181066132831439047985904984984035885285637011304056037140857730033303709023505793128384533815895930952093984975947042294403037704637725950756847507122930902369354210710317493536949583885121629702671393040999284529545190700975996592183782255495159355687634346663097826731795484234494434152259228096558563790560626965823855052067639635509972199075889409223066197271857714601620895909952419895945935437213535074746818229798615450112971782012271771535779168642000947969753083070378222992801444653901295328668576651770667940046887938081608375808913138838098186165967962426997370511925005603824202136963423917749264053348741050961729131353464199761816985264135523594626391557535987146377019535382278074580037821046424931726175705765881572198904872973862524010993832143271513196857966284202754071066243653051243526199428833874486087963421921431634972660428030884291423524182405425570322232279636961802307972889922490972115109099787552440430117487839607111847435398121624677556353162439285874267699130151089631005968406893990599453709869779256887798462883469214519997084977051201546139448766025379244810513333428550296787105381518663705060335263943304321430538697556649237460800934146423174106737759607486270164913010645186601122804785900304649508906638720225970340680184595577675507926517175552929629300839845465327776514770846838713641299968206911186097503962807385567674890955227867652398826404468428300943853951276694733478623288335964640837887777393913693033202236125429214890587488055463820596514095989484165013237771387107560852649050606881275649560291034258269087188516087208298088487070340080244971476395719434628463723386250612185184922805196410906239931544422491153244854214196259247264693028329439340808916873475698838944485829356351921273857926756986364692023576987849027606726647057427766543192581171261361683598627901601649286288731466103218250978138934888115855721540338590441262543644896836752766778636725361199277670784498915715508068565561677938765363229129194797951770606460517989233940893045208813808707420617892016059270831314487471082433035545848026707416513435450539944330240059147866342639867093045001573711943195093878432329215782926974467407253475619015322714563157629424665853828025004253726996044713679260130264649371228797956548714478033041996275404513403563518925997072122209604938298160216510147712440096355186859301303983436863554264774778042476373032359528301711222261726345407535176846931386802599354932790436157370584906055828373199637676219198883756345460707080291646264278306610860362328939057402290018263187888080079069309451181224048112457750105883852111463847869609865399041695318699092036197526223057612177686291345075032067787767051662579016383403789932039983984683754100675055294293559021909492998641102843619246348582575479371256274066245521163844599765979079944288563398228811332893101628645801921831320626839846080869910487981302354492472549544526756702775713401159197072582678101321772128432935084788093265428327329806978585244810656074027433399148603465713043432527200515036050761680256885334197319501865083719703748049344345986846281364868957480204237747152459795061675107208167277900285226259533577178428182978251693638162521452598654724204221056155275758723730646428517294196233078572076530674693343858100725283709751610675151956164991434333007625521189827849159369866105419953547782714649412459805537077414462680822940962640416034759832184150663037697776004751880067705721385512343773411349562384931972078098258573749659661633310029978249928804565333634852075557799663994161552631750881597156261772038378064937758859452382827337598637871883993521166009191002107940490441782174037080107962424276987322432556282518709518526589086463189483451837623708188947969881234779644930088962567065657707163244399629381641397619497170228499038656237890423600633951762056552347025293553421990668439115699019734971311342823710264002149378235593700228255498291371832968697446217465253922365869554899140096823005652118385165851652919257865717093574160706244063547003648820359047574018320755020229515899409735517874438414807652011243650808277524513434739615558304539005150904534645716287586980607188234295613723454915367789491717779921815948755817721449187132810967324700242068428672962200495456886959345232451373478550157527376806679543012125850610315751673571902960653994543082467648169822797034566976072939935540275442633466735945171674893153049194294560626143847037900435788488738768458598151626944887378788319033010545874397375754926364992168605844816190169296322854738315696398552487823218132315274712486921074753716764104376886113691997507925814198171572773191198713773093331469998069481585678731617866188400177749748420960561156039507598450625816578338928557380483403102396933865237429124338259268101057547585701820736605556631074502533764124850893262574861012515176018098525829197133301481540028279178894495233139866728267335184695434747897862002318978321076841530846470831926436220800309568094566523092272083117958578285066906297303778656186136893350903357984622768332069045155591616130413778030040127873690312570499792983733269035893135554786422086224126236140895798507739655732310342473051538390618126740401372406877652252134525173461862102738421379591955622287168888849582552454310017288931232757135979679237937578416682241598215549176583125556328612162281865496824863777585000651781370365131770238001865991651718883829547170238834817999931094435799146238426775700479517041628790263283178235808439349524397480204869200019021645526706973843767281176064716951168020856452794486450058664735205733342791149301842455476751951390868160486139217044729355634898660756728012072064414562098107033384898631354293454389580166266462573697444631616999319680205496643530390564957521147064860853768153489847949589103831110410168757377743502911224997304669485284890006823733997524074642352977027107109184660587379272252071389331121661379891674171399745857020575528566170686859783393425267734148402332784158013442378376067923139170429184465036324389440778251130554301800347643587295415811666778691533479256067047648878979845303434253846191736810777789462454489595938968849304429669285557658149355825252048401015159592830215682763640457832634784223701677112276408174605535793877439271947337186544063956018554192805741598182588230180442004683635241555920771269192188938863317004981252170479664026742769623608295415935206502224236709176971858740025665527004931662437313853568504988853806038132289391256503241673949988195644217089554830821519912474194034200537024364380330554678286244053212424529404373745688329031539603423980456133320340064949538662013029707377547430924206978899587470863098361439086565721287026620111190822707133198410782788400622530039362662418413013806126746680318523567663306099894346536182343626534199260270260040443083493397629627918018594036417914267606349010314307650821928433329836152523008152927729631318582855515266484152238444696074677686516341376709957531623205415542530902888065172990713261422980088926879333330766410410779488841743894673429028884519785600872793590258944172929702626389878262706418666135846323551208957971545180031749420887118357461933596681632670657224771971935923160483684478291525633726081500364744720331957963374761810590260097718483661337017851946135758569122177061585770521717637689676061055759477870463142694201643254002975882177591500369969215496810316355224107677228187929140934227469380587302076842729705966111799448886426042236057237464030780257461414052726659880300246511708462379581182763715106012457571216992582148601804155813109644335658954958374412407648374269151961582665538249675370320928370377869255784635462522237060659920924134443562981921855289638203740712886794767770666106412075367790416009951117262580310916536978408840740031786846984557248737629268083699769876343808614193214299728528515896295539955307887664573485870158326313278322643580045663742575809303801931430601443157872217118525393506947655384829891998493459521830184426485733605007872756864502836600837414410140506168919559521458802252043638496106916461588210961683728318475295888185033662258623014606172640406155500328120448177530019027012231784720682874139509855179727748680620915197344971513202735014291381006603274916657126247863399219793869372733232759868205331642551723347721902208823449501889928772821275092725182623055820636872316673161898006019332625989722879383330443021813188272563206588986889195611791595410558364598722413514554641746488995761287166522490266566146756029771928912446366830915870215737491493878830116068075159799572528892091906272352238201521979571461425273366595176008691693016461937138448072814979317813766000641363675777775866705197261813117599656624748047885004075580421365557515823498072741126201490945243786543007632578702950367095012843565704600477938422417242119644522403885411118768066746737987575195792710860081502296580118781962855386448703662248017495784132247942964186147708798646827385162457104866257511959054571080536289148789659897432636614889342548943146622700777028681775282117290974360163499964349370864136622958757542176345251287906301860308179071909349384257470776002246958869220138769765745349375459748354779313313867661333981552604606756251794433079400858286774139096817768860464656355860538266726357553847702134883924465748636693068094882840369785437759731792134538146418265434296005739091240998877991964299920526536348956828576640260200855853954046493132348254820320790969343074179151646585041239607336437171326353217495301506092690214421052891870110461225597116158168250821178324792260630009815159474306706422416976251331119151606726691332075606610173839775791410314559012276242913510072860586843188194249820688517241777560008607233491303300349980222756755141204064484335620467310434250494082450753563902292694669806828248744218357559038465734399761591798741543795092829478162679853066192055831455916361121235296520803115178778423019056926861584145343917469716370427730456045721416019618853186793582437211966409343546573325465373359351409484618212229987718256863972243586834266415686993841795098597720886769683943661104531180865048098506313489443884864435048853692829693999147883330751634457960897922717190095812384678896704470366040670296039337743812639320813175446435110817701070560439457303404056644533396413160506261078988554915278922776396977119060307882743279271553264619412010175184739343938410016579959590374875493705787799928803150385772765867026904812852485067859018178942605041231379874642024272040357314576718362888643692126509699347330348592793687547301344433749965011913991071149285495137565902540445407397202287282119394754632640709020718980148777980589347763429365025258766392979581591788713022408366871544020366613126699450256927371227718881182221012715573056731329661430831572937948821193835666177689346906284225185702733697381733932557603457012528973203404847192810410976767162684444157744557137649787870011174602310720088375838734665254078250626305018797993469130995912766267954298336942642461305850563496677007360088094398378949003898075912473054932059068851858879772230308627108545426346079572238476665234665839858258168962443785984422276955549238019588820647213850999042303557759412571952737172893063733660013615064712460202878819548226158153833101137497927671430380681136911411575792113782945638823129464190730527076582267457115486242524303156994700156532171246042048407451171641049709710546830249097542731024181695350826748117856145997887172631632821381940045116031303811719354658392748971550999029001518542482024229767311717656513146136038216079800773179161879146390331000932460857220937103518778652697204700961572730666405714287296985327513364102187847734189944033283586563546340111294015948041528784925647046800720837504767781669561424250260059248496120999291714432448146597102627771671590459707062912638561283175540074064869346664517752757281659895591946656750064335083967152511346296804738836107398337566467760943901557634236047171908231839241874009791214963914921113677483947569179970280914704667415941890627274892473163299224985089975983069911795565702927648580898920828946391865621162296624988434373365172362437307406066228669642483171060760667582449815594402736911654803164756379237069499168781350119258082629389261305448166604500259170694878362828125334961232511120730092594429537737181275170216186844072783981802866048635330467531595122218910143360312341537651949035790514512029371579299833704618568296607439930169296933702233779510973091514077426457645294790353569168721042787990840716403809526815204634483612908341128860304244033139280059257128892904271980555253212841947864385331417411384228485856890546115202063124548030368897746180279521675212167498151050796462162727449593723426382893777442430012389855738848661327449708640990379682215066253079460470105528921277387150953217480661586288465196508251990867902742069409002820029577993481221226155484735256417159663398845651903004557270249492233520079784869393784425937271935530087770114174155641407537376461881159851906917627436189255916795040280060862906385622559693125293491429028364016591859651218819540843424241671860569474255496268502615558538667488945720138542716568114111471327690153822409207959724243552947443270377360418507752047359844382780326840878114456120988728711947053061510206014744009379635530710404136802924630335817029702083629483354648193049019455371693198120761879543656069069643141769547380153101747398084257932367801243276180983725319423537033623451861307772084565127750203116753598952414039913611859395348811180288612963611453746269697845851517762793212442862308077605061942504441920489306851890938989123448080716954689207965670169405342664456963563578518988119178684609137303573119883800862672953749597686032379087526277937697456392337924445889186753629115199785316657717370812870555916346377320069649227406333505008652319887770852420937008281626675654344094787215981950172579442396233376446511386564364680552323339236274037983861316421378535469578936274206352424986438884049368672271423492606006868606163955043578905649454790146360804526644706322039312705912802869911167380520804201541042862907411709351755019748271471741967651461197213317684041269932311657741843224727969789269338669724485602607920108766572372807646370820282616236339221301383038059091725339353185035660405972218512038041550488286879644653135254168119934880305415616401776381639434195432085932497766510276930591613444989985669145540857683030747990545475256597399784042067923414271837185033516044278434988852244083988876027655852249464365193097773807128276073510519203591853416213430275855358553134423298431844582480164366812052406423903964305598046681752613287720579240953608551289702604392999440755313706151422323662863487805311509572030959389194386362711380603759198361706649363420938126355810698374954711158193213525191874938529563983158129571403521818717470477288575093135335537845864102819019067394192453335174997380954459875812646027920505275788696366558598622111721662858353132822483957183632541539586022057247603334331292425207180198397826481131583294088561681006962810862322371524274284954483643226891731596928086457915339462786536019486082497323505138013907902460185080017154765421310236880231963742711693327578506074493677923240879848176773587515790225776176500391150333419799857696177526121427814617841288288922613240220663411308701304998315658774499707710508618598236203558434364891857620518494276413666977720351914806623261222629527206349322079144954544378981200992036357673993661216541270519504568566254676395283002652149941458369834378418186525717116110960468555740052564862072268075863150894314962120376990583129720719683197697652072378974233436989044407103522015408384370220873967279892524613108612166219343147727583735730923141146869989342620728079320582948023882257783954527814372434567073339498138759597354491375861990095937257784205440175743246313669297327144892992220135312804909911881232129643678999167854053628476398534471487003988584437393889577600959182160530243950858677134038447565794489429923147299891329853650233413453158036746725123703822025373193598769674514190183121664606779971402334203442596773752797635145860323649231868589934641156195254713553679447977849329365394275661133000841152382433411304740973599003217586665519833440825210174894735712455181446647217905662612289996083420224630494756763394629800501226695990766626355676625303870180505516156994779321362677536014228903728958846881936166584219815320113584501344947276758166458957895004329545384144674836493287497573463090363623290642691670007165974398094023417877738672239339678873055986868927266614727913647368152027389317013060669461785745929523673515581830576123877268347135111242494646635602026763276637235388122270338256875652974888123399186394876524825311644055799853358309313479051059985121074736179580609042413160865840698638169171713500569296243861718992522802249935122851322316584229050422557671533563101251442771687803903397512292201648810941834228071550655079290366968137519236871471168019889718468272533541043963745682009959258205985104806458085289463910699250696259943589534793546891003141336984110982620346375110321874291261891373289473056048006027648838356116992621147957505447265782743172778711512299541308189341964752054102987303931781085635525759029699654021112547082157132090900456072209619981637108206475825658489800234338906293377342596194646826602014495311107248101829177759580684300754698172842313016623563363040963343771805175324801121604842647420154524848706567278452926993266950845842181815472136160769566147266423714399691136737747967783112266442471252412825845309797967856650172263878202745249184473037782423791795623634897212531641973441776909023438824956504670781912724936642776293961463603931254905459411908498602597345466631980919079385483796553586495109790756631027094127399816646593976389015019759111686468982992971227466879634984682681182632687870213132677894555393782199697258166109867575964631216087500194177960494071492189080926227106929079288269081494369149313905435859850936277321766080997264703367622980366825156806654359462979438439142942151284032648182110424227773623468637328072140286349578643052131113248116013193022075158854303655754421128827594130896337083600375298679609360746462684796760596346436886949059392879512894528115866007980408469551407242491961593853452748199276820145001850555608319521419574134538854190254753390724528915153669211437061887355250385143330367260683314196603834960618634788227673810132610629454386396192392053545633605013966092496462365010494253046705728138636875786339013419700236221714650992800155662798801953530254328691107586744005695942799255278696749761240396349918831235231191454676410040112263255418837974537037902411605516768765244477756419690145985545348093181234228590089763719116939239346514847115898901314295073681616052937342145637316526272043556217804763220557402902266071862289535834435455262984015630478519988023688564829597815049243820817306789271792515440831347917572965915730569686385085062539706743932339008227854839440961031773845836036689177571215404680134824152786620346109292167876256327094531141183294289279373329492531751905661680963552496942297181413175970633242733597353612987350237477501290901069936456209222584885250879743018344975825374318466234050883144730846940796264027048894791379592157978233954975794075659168502427115427923591349377905446943712355558696207710887792902754961624603290869230107299678479165623213482272891320476246485332168759113204206775821071836262132069237903268859915461857357190958080687665949341471578051962004791887446802295357609120255626147196966892255627885710464161120437814080859930380322950567266637082482372565104331042557548940795482237862496934575645496292109060497236945181903379512747165948761870692787687007575451256996125146152634853179847126095662468970278589097203901812917586090324093772686855579169855176975623261022652852995244910140400425700914968806086581989888309568578327525834127224026916339882247353433308099403450580452703960096850505102344389372238584307575325403453996144860246375186258263746073589019909269155283996099131319074141890497483827810857154524247861162273057770404646792412978239606647912292020362680949654821146699264243641025825088106953066758809807336783637488062399282195531150213075001483165435389073197098141938436756512977007526623780320114431928402760307020597078786088072485257656475833735660924626558971146072505705376070375804539720928721794804411989607498796632044308202518849523830009378225679040051102384916543071948281595254865756165074427531347460356802614770042614585335934816077117335063498263428192351913741613538433026327003271605086529085867241646071223708985336456783896407963737864320683396798586724458390799607109931397611890560223726491766119003207274181699691208369306913512097983613398092461647051944512788720051411150786820914817038018189715720831020931490160994437780868870460299889806521697323428284215340198577814919890681123092750004763604194895439551778629246380686866411623640346956895558650223378222212529859905652033876539410826102844559726989389612290334439707778820669913490420442490713540369833289079467031728179962957345445880526153345955119577112550519848876551575776425346194521982631712470793650969283101856326234996572358160099740023501985410663041470947640544764722526020054024129839484747174074431278876176199944318094223092676889393068451781936336541317586813690416237417401789765812244723474586121865683808684016296693138348157812817646545539050247694593159645041353023661774895816355318031730852264455947953574479916806042935553432179269023227567679148638402630133752818468684448743074118689469023878443536066590807456526455661120723548949622518815372474050674226852198536649506959368946272572603686466209706203870527801710199515839131012088202861822758981105860129546325488217083669688346848315218850392165828226575460366981306201167915254697457331888749903843034533609233024569489813718707824164945923209644078360157772223405386784039375237184156872377990618498043813522168273389890741317073490219916258647864926148038680040924811635014536802469918373274400997264686165765629015509043911680404925115509436077632624601115902025958021288794186476994475961764526532374990576701538538148533920842221793107758731192486769056884776753650282359616238877950698742918306045088585083038778330630264715488145625119431578369848151518718631389750091581301697618081259510466638665205696274813960191755204961663900852946113703012998604243498811576990500564924760059944529100142641547997191774253289692967545983200004091908362302868252432591293330822529616618709831460276077642021208919292255351976861132412291465182985837690207602857807012503536144959229922985735002005799816545713210377133514485774308474307472268413042604611770048908483262181218660770946345341384230870321590539095311981952332832068792375111454172209986968472702869486224156766243447140551927248898450653044396964451089700308647013063364912282992292455858416580916786883345320364016287450954515950725750550099362909578831489890273155048882006043784472852450368314202005674763401613289342244004858613066019907526973070531440325351784843496703299756941994143640616599126233583835394989279380986903799153604097313042830201521049086490661950868925470713835572220579020291622515087368297565494449148890349213347080773956141429944995868521327271848657145850211653535685806467808135524647919950119710380365039491917953136896408723344655923074270166158422142090553903083664688261496338386709982353710240669541565918881504605957524953101038711256642026902225597502956945243663182809679304594772111220927782302888053797152354489244842185502106234516819895545369757657053136128480370587883936016742658243855129481107956239735558874991224266108764013741958974526109659732116056176031803672858283158015958100244866383769201339634289970127307923493080277450217288068117733177915092004637353427307858681215614821765802845852900382888524788606280135262419199950347371013303893975363038647138068553867338402654723595331390831025838964306965527956303998323819132996234906083311591450002429023986017075881834382594329218344904582474184538426140931498685830765568328979006866989202570823937075773064866537467115583689539579277032062972806964980322257041569014728072725930975774468994783085487302016776624276353526403281905481238403224266597477187894327128217477546288031405893207059510709797762358494745030131244706830743658621015859450318405332118938332357360476736296702440695967131040211718965173142513728733736613291195228719584048859805186977178181757432539963454207722672331938624356185238081130220400918765704081098174881757606501250349513303605766902586121231564743525988919242978354551897157303398552865189391698073640933284393991987818080586044848556505733359101536193469134522578151264144342708038717779176864949326034321610544308370306811504414092357865227553021892046812772123485724788455450482015150089643218210508582108926389425677735716422564931089720172417515335757031492617599552043746237298765473476802023809693983102989181750128261552928498154041400419242962257978373196427014096969863261454378318303693044152867374315680868812191759792414007187255071118393936248089352518954868962161468862636684120409229621630071826197484166775135232049821141618020648718668696523489306269886278917953401804995195402968124436747110131445810984055188110213078246105195419888500324170556540212493211703924531988739989901860632401321843623887388775041254065265339852568217144291189630348377665132715578815546802607354052271359988367018369857014246255009188169140473850371454240929611223139994440515980333625592609343892534447762197596747166128632655677974295691986878029951410076754565082555848695783283538227611268351273955990602191918116441509149724846572774460877333827920400840608216356145286269719842362284913300672409039093444673355177873825469714790872399762736003147468162467683069261048927476378158233729367341416793914929538961589412517924935249422835197760569892728873246675446484966625129109540162778417894805297488828546344063044069059464167353051745347025951855730472817837943233199970254991586024745220244818913697608460729490548963799627413150188528564897575355859424085769175719662355045440766175512566324315804171345647666986914859017864070633977585222543124537195448852662203687316801681707558998841420305932044383283907628132551394952088407146125459207996344601810162539181990108915177135851831552019749198433414111504716033376854090820070048946471416351771874593059212774531654144649539139590360276649750036186273390503466281902082933635760370273566303652268221842940463885643593993806767544173445934203345463419284120906690981713041779756272431300379610142946804016822743964178039655334306154545368390007274911883208302249059553763995195058769738201388867385482912239980943041998236413344709920284287843444990916028771428462188527862816405146383507285589451433664769988923806870896147362371745782888329150141162614851967984440798129383617287263117592914317242758037200182826524047855280736487276060345350509122534009179879819513634903940510633262739695823669811613236388617642667720767858356492586909739224917117202340572752796566732995372368596020562989039128946194627707087578974979681988131531985591096242734554660083120580372493896161664498312598777031512364128940146196277737979128338653697753386059289139317376043065720410972510152556951163260093399141491887935119121482675177542814926134984137900563594854768865846655607288237277843798282058397011103045428273867908332506174339101845812845035158974187251253359794610274509154236821985373375398527173191643224441415314318037313276510682544706164799970652781734799386973914922593792976066916061461231285088872096080789357696959924390867416340113752059261990899859375328263908713321872837864985969870469872883162703520683645974659001383768407326737702682719717534730151582961866696439301674783971108267988966296584186353917269251153714947472628015116078982271365955399162747058662201075809346328977953550628027445804296128671730596915838614267340645264346358353531460479248347131216073464401315303613389003539229664553045473998318647241940634214557112686517980786359607945201405654243691746168191000005902210340284281269178177848380486678983201433216918656377756290782876860171001699530735641550064175279293512970404802913684262551632428943419375389447608424655721488741170531148776114572496951837852491730071677471194369264669583761136339547223725297281627609066544380532429910821452361767900588594175286796158287945960875964150855157716257010835816642551508010170619322710012792872190529863951890345133673257769869482088979986500510777385758080501689127436119214411503993974814495036744348093913872173156649877426324048044151661594155251874841317310656459730032969697686690439891321247783082251275267635678336676660752560395285172407031458705344081876389699284106989741872102825554509076139783815781880194678548356444860654147126317823858172384644701573822313245432611577539545960952595179691307908299768158103052707971637054372559783806213389982315242683061697894246300662925021994854815053771000664493241807306610757715183376269040695151496456711293518751992401136611087857300257943840800258042900077104041195500520453883905054970344614548469835759110072039553328012097960566590849785009869514453340638272600754984421221337849118206883041791387089031919674223443759122480306022164075514715998559966860073294517722464747292477909632706636472492831983086577124639954559197820320317725160515248164577978325539818316779577337007666175579946734127709829947548986330637623128495995788129193309595003727447682259826065111096559622406450939569910515737292373012988927930875927040825563391240317762604608501244386739780725288925869173677439030518532172094321218822818871826621504200743351718576657788088959124005166774293361850518631045014874243171984231532567548282814355730227256102913820360201736577284171163796359938369893333370119477989965530185223768832787125223029668086399155244747911069499564924082807052594753791571886889486372706883398963416146147939976546318432923958734621738049078981057748426423962543581304989509993728277489332894373219230823135130728635238176627314802262594319281733259054913116399554904859379607642776224033529906856088069005616009806350966339654735552544520546742795411994964480395792782874637407106432646435936666395864938150488029701895328942670440930069780592389072954743693832524959149926280843825061157366137179038176448094699134218875092838274517022208081118180122245421709670557079855787418934144231184694070136565636838106016482265175940582209680673076555617421710556198240608483649330941803864201501280165401156038509004623166477696914660830850596420209735043693420462876889630452729779737305010493860718935062199518813823129863619248291009764208057705382297785078772709269466216571334186655978441923826727337434506693707223959572719137134227292550859731376376447435978559605154341403848570581573009957863431843549968834251705204296969703994157175323637809898644919695031510149087902462245644928986565971783429110720686867191918391039060218975040120179060721771099444837423944892967187053020227213830579914245588064511126469928754374952301716770694970731253237260048117250919884857564950533933784627473783364189933898880345024270306138348445819202899838790254068614871536551233728477520965749697965160527465497604825624670049008765493938218481613848612897226503861346973750941323551632139130890850626330136589511505931307952932757959926120870305751399316721934430008509134069273154543752488537427967767508270578493007379731624568686046159145497222945195733173215554188104410272148255754749357470724944702408733242980946282161083347109685518078635287177569643350014641421213282175313817065344394861640923643372276838005572345200751979209303774274413774882209950909884447622820119507492615506685264313892368936778630972029925004055584926686344631009370889404432737878622757700715303063601693113154231680999831294433902361034094121718312536819350468756279461309100175713879792230457424034215459408270825613103485402656721266587673267159227458574404006065318877129203394956353117197001327073157205443821797802906274240567997792333078900625406017363562658793482046946418786192722058706169948250753701067182369153940141605221163784044320633071854727736080304891303438505393908792332148111660936070350570069631197536363383581545612908872885156982805225659128987609204753628547036537386786515366651328341121321870929396836849882296006291280111326700963083310480365624024266223802184268566031200275558305299727573129822759443292095742969086128252055806146269996877961548609064630391160113178162310237179953194602120946665174203806506834690835062256211927300234671788796197208253046064014648968261600314369508761043060165814251605137670335638958906050884122513951798803653456883656767555575956072868640928713157980809942022928148963579990878686926810635575653765835833778150157487599387211703351293053823103409006399991997979592187823178034246211811712641808362377220645796513156064558159215178789716216052365861086532158249875030483512221721265611198833353443676580740021876876136901005940622043392059617793319838801490859193407634043130052730025893149333058361695584795969998857826763220116703275347665682665640136623833095959951376851720574642040885071593017583001653955227791376713235532021659191296125848525279106113278464105388189416900325744584592880531519870380945659225927373513629337960147839937902480709462698592892413209310734597688356196366333163925112273389282680980451564411722626527281833814538054065857483252288456679412247818865812772676223892046231801464137358450622639379171514282210645583224344610147857455364571293010904973819106338456886123954587626122987138407533549052281928923175473923465129685220374909062606301412751735244320205497223798481612229695283778377370228702798781515583199885619474534691453206546215568817659418265006119506712368770485840585118624180456129434087812510086527988686841442844563642428256071989466595553067790811418331538198953329782443026134275900158613546003776777824127447910523071192402426773144528282598411463498008622537686764212311313950147728912106042151733918087562801470891472184017077940547202402827770258812673475841757368046349142205927390050445908426590485803121734823855042085074212076166193915734228387696278714519309899508041486731743608348806601442536907670997107915500062551169373969825144829284099848561033688405621048154948815458871804899529718483346090901602241096930062083490005456526994505736873751333947594295442903807211985173745477674487487440994611569051206712105657071749162870807406995560533422664342980009953253312034584163160321030035417456608593443916664431195328889542357990811305294993673876777958232812785402850752583467286932078187061084900549996548622160033531473790403444400821393598288696479599783422101261181569161081208745345051275353380446195294137683122060335481811252728677576063129891601357100080910527950130400346816091262194658011141428614045800343174525472684217575346914224893314794980153394936679854794375254464614865591888950837035578692662817174787027461608911543818730681027453736455583479211066897703116045873078888935876992627208804253659732045876170140639550276141270407134718045383264954691376291115255688978103220656988737919701769500500788113961511854432899250755189393885475007603965643943368531432225046508638695226320890743836723990494780242350623393340611611934782461187181388912061869424166433951293204178239187653902424547287720055384571025073695280901509272597451133152963747080799709954376013809702979341859021862832848382892229799648036045703142955385049844672055124068772335614390581504338417293859727672571573866900418846607922349275691440377353727269239064395808652916974794564346616425510250660940747778510205707509915699207225834242894706929732886154037956774182957725580186561520525924112300032754510212815597529387372819579660042220314113374375127133954421323452436918407109701876228695302594702939721267866828425954870243908705034424745935789564099073816902202886595912982972269414444709298158964381545684833926686495518019723007182687263010952607417654707378289932127871275260320966011323822379580723836789932102784974069754539453503668670681405162563624927598247794077285968999553107736551725594507688048001341733746897008102821019933197696976138355046079122472996903747803131629442460729805971652054016929453791020757065502411734822183154531927341169325624447767322994357678482432462740537072761139282373984980538136842041059348424244987202260284532548377173346433524363921286027663951018733136876814188797324208990307026601761732116182165486687028655406832465235084438993385116365990800148188174252726779091376767155988328312002892844801052740623396977644015215049136157445298856842413381370783949131546593474731017055475495727779854359414700903215022521755294104681171722897887511102052605556660661197502670606991847011703523292882502306646179827579268068915919673355181247056202533627258440463850432842657751345897545144537832117410996008754666881200849127689182847078832667038060322732361675171862117789133014132286533117725595217365495211256831614211696650161378265035538866105568985581153356689421758153788155321141844497038267825169183437192709446077570787427015033295520888796420404907174416036300979257120000031491975212648887960711289814998725544289080436751175921163355474498198170475571710180562102863567911553718466664844248334097395896642756804333796708817830512474041141579495306385459815798128919129486931800111798614931130964743004773763597058121593882848479803096363063736969869522988554591556581199165277514931089301462123882078469130600384875172562811166230403239129757161816170357676209669838188288140330605411041574321546521737578729119470074344863651608851104070536420241334886470684720434499636699198250294976379628686962895944632224683315618639671702810508210980573284867198851168527449696122388755394110498314718025724942298925005613368850957428508269884814189500923185615803606209201694816932924733802949410760584666692589785537066581740753167351338869155039907491147650956612739720227562726162524007784980890946641642245026019885241845516312257163195896721821723980014828756759107726342040597375172966639183554755803174676775305497907705360809504839545373422521390105715623262351720948458429665295550164256789533648052401512934786686568224432208996))
