/*****************************************************************************/
/*  dnd.c - drag and drop functions                                          */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"

static char *dnd_check_url (struct ftp_window_data *wdata, char *oldpos, char *newpos);
static int dnd_remote_file (char *url, GList **transfers);
static int dnd_local_file (char *url, GList **transfers);

void listbox_drag (GtkWidget *widget, GdkDragContext *context, 
   	GtkSelectionData *selection_data, guint info, guint32 clk_time, 
   	gpointer data) {
   struct ftp_window_data *wdata;
   struct ftp_file_data *tempfle;
   size_t totlen, oldlen;
   char *tempstr, *str;
   GList *templist;
   
   totlen = 0;
   str = NULL;
   wdata = (struct ftp_window_data *) data;
   if (!check_status (_("Drag-N-Drop"), wdata, 0, 1, 1)) return;

   if ((templist = get_next_selected_filename (wdata->hdata->files)) == NULL) {
      ftp_log (gftp_logging_misc, NULL, _("Internal gFTP Error: Could not find a selected file. This is probably a bug. Please email masneyb@seul.org about it\n"));
      return;
   }

   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (totlen != 0) totlen--;
      oldlen = totlen;
      if (strcmp (tempfle->file, "..") == 0) continue;
      if (wdata->local == 1) {
          tempstr = g_strdup_printf ("file:%s/%s%s\n", 
          	GFTP_GET_DIRECTORY (wdata->hdata->ftpdata), 
          	tempfle->file, tempfle->isdir ? "/" : "");
      }
      else {
         tempstr = g_strdup_printf ("%s://%s@%s:%d%s/%s%s\n", 
            GFTP_GET_URL_PREFIX (wdata->hdata->ftpdata),
            GFTP_GET_USERNAME (wdata->hdata->ftpdata), 
            GFTP_GET_HOSTNAME (wdata->hdata->ftpdata),
            GFTP_GET_PORT (wdata->hdata->ftpdata),
            GFTP_GET_DIRECTORY (wdata->hdata->ftpdata), tempfle->file,
            tempfle->isdir ? "/" : "");
      }
      remove_double_slashes (tempstr + 6);
      totlen += strlen (tempstr) + 1;
      str = g_realloc (str, totlen);
      strcpy (str + oldlen, tempstr);
      g_free (tempstr);
      templist = get_next_selected_filename (templist->next);
   }

   if (str != NULL) {
      if(totlen > 1) str[totlen - 2] = '\0';
      gtk_selection_data_set (selection_data, selection_data->target, 8,
      	str, strlen (str));
      g_free (str);
   }
}
/*****************************************************************************/
void listbox_get_drag_data (GtkWidget *widget, GdkDragContext *context, gint x, 
	gint y, GtkSelectionData *selection_data, guint info, guint32 clk_time, 
	struct ftp_window_data *wdata) {
	
   GList *new_file_transfers, *templist;
   struct ftp_transfer_data *tdata;
   char *newpos, *oldpos, *tempstr;
   int finish_drag, ret;
   
   if (!check_status (_("Drag-N-Drop"), &window2, 0, 0, 1)) return;

   new_file_transfers = NULL;
   finish_drag = 0;
   if ((selection_data->length >= 0) && (selection_data->format == 8)) {
      oldpos = (char *) selection_data->data;
      while ((newpos = strchr (oldpos, '\n')) || (newpos = strchr (oldpos, '\0'))) {
         if ((tempstr = dnd_check_url (wdata, oldpos, newpos)) == NULL) {
            oldpos = newpos + 1;
            continue;
         }
         ftp_log (gftp_logging_misc, NULL, "Received URL %s\n", tempstr);
         if (wdata->local == 1) ret = dnd_remote_file (tempstr, &new_file_transfers);
         else ret = dnd_local_file (tempstr, &new_file_transfers);

         if (!finish_drag && ret) finish_drag = 1;
         g_free (tempstr);
         
         if (*newpos == '\0') break;
         oldpos = newpos + 1;
      }
   }
   if (finish_drag) {
      templist = new_file_transfers;
      while (templist != NULL) {
         tdata = (struct ftp_transfer_data *) templist->data;
         dotrans (tdata);
         add_file_transfer (tdata);
         templist = templist->next;
      }
   }
   gtk_drag_finish (context, finish_drag, FALSE, clk_time);
}
/*****************************************************************************/
static char *dnd_check_url (struct ftp_window_data *wdata, char *oldpos, char *newpos) {
   char *tempstr, tempchar;
   
   tempstr = g_malloc (newpos - oldpos + 1);
   tempchar = *newpos;
   *newpos = '\0';
   strcpy (tempstr, oldpos);
   *newpos = tempchar;

   if (wdata->local != 1 && strncmp (oldpos, "ftp://", 6) == 0) {
      ftp_log (gftp_logging_misc, NULL, _("Drag-N-Drop: Ignoring file %s: Cannot drag a ftp site to a ftp site\n"), tempstr);
      return (NULL);
   }
   else if (wdata->local == 1 && strncmp (oldpos, "file:", 5) == 0) {
      ftp_log (gftp_logging_misc, NULL, _("Drag-N-Drop: Ignoring file %s: Cannot drag a local file to the local window\n"), tempstr);
      return (NULL);
   }
   return (tempstr);
}
/*****************************************************************************/
static int dnd_remote_file (char *url, GList **transfers) {
   struct ftp_transfer_data *tdata;
   gftp_request *current_ftpdata;
   struct ftp_file_data *newfle;
   char *str, *pos;
   GList *templist;

   newfle = g_malloc0 (sizeof (struct ftp_file_data));
   newfle->selected = newfle->shown = 1;
   if (url[strlen (url) - 1] == '/') {
      newfle->isdir = 1;
      url[strlen (url) - 1] = '\0';
   }
   current_ftpdata = gftp_request_new ();
   if (gftp_parse_url (current_ftpdata, url) != 0) {
      ftp_log (gftp_logging_misc, NULL, _("Drag-N-Drop: Ignoring url %s: Not a valid url\n"), url);
      gftp_request_destroy (current_ftpdata);
      free_fdata (newfle);
      return (0);
   }

   if (strcmp (GFTP_GET_USERNAME (current_ftpdata), ANON_LOGIN) == 0) {
      gftp_set_password (current_ftpdata, emailaddr);
   }

   if ((str = GFTP_GET_DIRECTORY (current_ftpdata)) != NULL) {
      if ((pos = strrchr (str, '/')) == NULL) pos = str;
      else pos++;
   }
   else pos = str = "";

   newfle->remote_file = g_malloc (strlen (str) + 1);
   strcpy (newfle->remote_file, str);
   newfle->file = g_strconcat (GFTP_GET_DIRECTORY (window1.hdata->ftpdata), "/", pos, NULL);
   newfle->ascii = get_file_transfer_mode (newfle->file);
   gftp_set_directory (current_ftpdata, "");

   tdata = NULL;
   templist = *transfers;
   while (templist != NULL) {
      tdata = (struct ftp_transfer_data *) templist->data;
      if (compare_hdata_structs (tdata->hdata->ftpdata, current_ftpdata, 1)) {
         break;
      }
      templist = templist->next;
   }

   if (tdata == NULL) {
      tdata = new_tdata ();
      tdata->direction = 1;
      gftp_request_destroy (tdata->hdata->ftpdata);
      tdata->hdata->ftpdata = current_ftpdata;
      if ((GFTP_GET_PASSWORD (tdata->hdata->ftpdata) == NULL || *GFTP_GET_PASSWORD (tdata->hdata->ftpdata) == '\0') &&
      		compare_hdata_structs (tdata->hdata->ftpdata, window2.hdata->ftpdata, 0)) {
         gftp_set_password (tdata->hdata->ftpdata, GFTP_GET_PASSWORD (window2.hdata->ftpdata));
      }
      *transfers = g_list_append (*transfers, tdata);
   }
   else {
      gftp_request_destroy (current_ftpdata);
   }

   tdata->hdata->files = g_list_append (tdata->hdata->files, newfle);
   tdata->hdata->totalfiles++;
   if (tdata->curfle == NULL) tdata->curfle = tdata->hdata->files;
   return (1);
}
/*****************************************************************************/
static int dnd_local_file (char *url, GList **transfers) {
   struct ftp_transfer_data *tdata;
   struct ftp_file_data *newfle;
   char *str, *pos;
   struct stat st;
   
   newfle = g_malloc0 (sizeof (struct ftp_file_data));
   newfle->selected = newfle->shown = 1;
   if (url[strlen (url) - 1] == '/') {
      newfle->isdir = 1;
      url[strlen (url) - 1] = '\0';
   }
   if (strncmp (url, "file:", 5) == 0) {
      str = url + 5;
   }
   else str = url;

   stat (str, &st);
   newfle->file = g_malloc (strlen (str) + 1);
   strcpy (newfle->file, str);
   newfle->size = st.st_size;

   if ((pos = strrchr (str, '/')) == NULL) pos = str;
   else pos++;
   newfle->remote_file = g_strconcat (GFTP_GET_DIRECTORY (window2.hdata->ftpdata), "/", pos, NULL);

   if (*transfers == NULL) {
      tdata = new_tdata ();
      tdata->direction = 0;
      *transfers = g_list_append (*transfers, tdata);
      copy_hdata_struct (window2.hdata, tdata->hdata);
   }
   else tdata = (struct ftp_transfer_data *) (*transfers)->data;

   tdata->hdata->files = g_list_append (tdata->hdata->files, newfle);
   tdata->hdata->totalfiles++;
   if (tdata->curfle == NULL) tdata->curfle = tdata->hdata->files;
   return (1);
}
/*****************************************************************************/
void openurl_get_drag_data (GtkWidget *widget, GdkDragContext *context, gint x, 
	gint y, GtkSelectionData *selection_data, guint info, guint32 clk_time, 
	struct ftp_window_data *wdata) {

   if ((selection_data->length >= 0) && (selection_data->format == 8)) {
      if (gftp_parse_url (window2.hdata->ftpdata, selection_data->data) == 0) {
         if (strcmp (GFTP_GET_USERNAME (window2.hdata->ftpdata), ANON_LOGIN) == 0) {
            gftp_set_password (window2.hdata->ftpdata, emailaddr);
         }
         if (window2.local != -1) disconnect (&window2);
         ftp_connect (window2.hdata, 1);
      }
      else ftp_log (gftp_logging_misc, NULL, _("Could not parse URL %s\n"), selection_data->data);
   }
}
/*****************************************************************************/
