/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999-2000 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */


#include "mymultilineedit.h"
#include "editcmd.h"

#include <qpixmap.h>
#include <qpainter.h>
#include <qvaluelist.h>

#include <kdebug.h>
#include <kglobal.h>
#include <kmessagebox.h>



// copied from QMultiLineEdit;
static int textWidthWithTabs( const QFontMetrics &fm, const QString &s, uint start, uint nChars, int align )
{
    //int tabStopDist=defaultTabStop()*fm.width( QChar('x') );
    int tabStopDist=8*fm.width( QChar('x') );

    if ( s.isEmpty() )
	return 0;

    int dist = -fm.leftBearing( s[(int)start] );
    int i = start;
    int tabDist = -1; // lazy eval
    while ( (uint)i < s.length() && (uint)i < start+nChars ) {
	if ( s[i] == '\t' && align == Qt::AlignLeft ) {
	    if ( tabDist<0 )
		tabDist = tabStopDist;
	    dist = ( dist/tabDist + 1 ) * tabDist;
	    i++;
	} else {
	    int ii = i;
	    while ( (uint)i < s.length() && (uint)i < start + nChars && ( align != Qt::AlignLeft || s[i] != '\t' ) )
		i++;
	    dist += fm.width( s.mid(ii,i-ii) );
	}
    }
    return dist;
}


MyMultiLineEdit::MyMultiLineEdit(QWidget* parent,const char* name)
                :QMultiLineEdit(parent,name),
                emitUndo(true)
{
   //setFont(KGlobal::generalFont());
   setUndoEnabled(false);
}

MyMultiLineEdit::~MyMultiLineEdit()
{
}

void MyMultiLineEdit::processCommand(EditCommand* cmd, bool undo)
{
   if(cmd->terminator()!=0)
      return;

   emitUndo=false;
   DelTextCmd* delcmd = (DelTextCmd*) cmd;
   bool ins =  true;
   if (delcmd->type() == EditCommand::Delete )
      ins = undo;
   else if (delcmd->type() == EditCommand::Insert )
      ins = !undo;
   else
   {
      return;
   }

   if ( ins )
   {
      int row, col;
      offset2Pos( delcmd->offset, row, col );
      setCursorPosition( row, col, false );
      QMultiLineEdit::insertAt( delcmd->str, row, col, false );
      offset2Pos( delcmd->offset+delcmd->str.length(), row, col );
      setCursorPosition( row, col, false );
   }
   else
   { // del
      setUpdatesEnabled(false); // avoid flickering

      int row, col, rowEnd, colEnd;
      offset2Pos( delcmd->offset, row, col );
      offset2Pos( delcmd->offset + delcmd->str.length(), rowEnd, colEnd );
      setCursorPosition( row, col, false );
      setCursorPosition( rowEnd, colEnd, true );
      QMultiLineEdit::del();

      setUpdatesEnabled(true);

      update();
   }

   emitUndo=true;

   emitCursorPosition();
}

bool MyMultiLineEdit::hasMarkedText()
{
   return QMultiLineEdit::hasMarkedText();
}

QString MyMultiLineEdit::markedText()
{
   return QMultiLineEdit::markedText();
}

int MyMultiLineEdit::beginOfMarkedText() const
{
	int beginX=0;
	int beginY=0;
	int endX=0;
	int endY=0;

	int pos=-1;
	
	if(getMarkedRegion(&beginY,&beginX,&endY,&endX))
	{
		pos = pos2Offset(beginY,beginX);
	}

	return pos;
}

void MyMultiLineEdit::emitCursorPosition()
{
    int line=1;
    int col=1;
    getCursorPosition(&line,&col);
    line++;
    col++;

    emit cursorPositionChanged(line,col);
}

void MyMultiLineEdit::dropEvent(QDropEvent* e)
{
   emit signalUndoCmd(new BeginCommand());
   QMultiLineEdit::dropEvent(e);
   emit signalUndoCmd(new EndCommand());

   emit QMultiLineEdit::textChanged();

   setFocus();

   QWidget* source=e->source();

   if(source && source->inherits("MyMultiLineEdit"))
   {
	   MyMultiLineEdit* me = dynamic_cast<MyMultiLineEdit*>(source);
	   if(me)
	   {
		   me->deselect();
	   }
   }

   emitCursorPosition();
}

void MyMultiLineEdit::mousePressEvent(QMouseEvent* e)
{
   if ( e->button() != RightButton )
   {
      QMultiLineEdit::mousePressEvent(e);

      emitCursorPosition();
   }
}

void MyMultiLineEdit::mouseMoveEvent(QMouseEvent* e)
{
    QMultiLineEdit::mouseMoveEvent(e);

    if(e->state() != 0)
    {
        emitCursorPosition();
    }
}

void MyMultiLineEdit::wheelEvent(QWheelEvent *e)
{
    e->ignore();
}

void MyMultiLineEdit::keyPressEvent(QKeyEvent *e)
{
    if( (e->state() & ControlButton) && (e->key() == Key_Z) )
    {
        e->ignore();
    }
    else
    {
        QMultiLineEdit::keyPressEvent(e);
        emitCursorPosition();
    }
}

void MyMultiLineEdit::focusInEvent(QFocusEvent *e)
{
    QMultiLineEdit::focusInEvent(e);

    emitCursorPosition();
}

void MyMultiLineEdit::setText(const QString& s)
{
    QMultiLineEdit::setText(s);

    emitCursorPosition();
}

void MyMultiLineEdit::insertAt ( const QString & s, int line, int col, bool mark )
{
   if(emitUndo)
      emit signalUndoCmd( new InsTextCmd(pos2Offset(line,col), s) );

   QMultiLineEdit::insertAt(s,line,col,mark); 

   emitCursorPosition();
}

void MyMultiLineEdit::removeLine ( int line )
{
   KMessageBox::information(this,"removeLine() was called.\n"
      "Please send a message with a detailed note what you have done\n"
      "(e.g. cut,paste,d&d,...) to kiefer@kde.org.");

   QMultiLineEdit::removeLine(line);

   emitCursorPosition();
}

void MyMultiLineEdit::clear()
{
   QString s=text();
   if(!s.isNull() && emitUndo)
   {
      emit signalUndoCmd( new BeginCommand() );
      emit signalUndoCmd( new DelTextCmd(0,s) );
      emit signalUndoCmd( new EndCommand() );
   }


   QMultiLineEdit::clear();

   emitCursorPosition();
}

void MyMultiLineEdit::paste()
{
   emit signalUndoCmd(new BeginCommand());

   QMultiLineEdit::paste();

   emit signalUndoCmd(new EndCommand());

   emitCursorPosition();
}


void MyMultiLineEdit::del()
{
   int markBeginX, markBeginY;
   int markEndX, markEndY;

   int cursorY, cursorX;
   cursorPosition( &cursorY, &cursorX );

   if( getMarkedRegion( &markBeginY, &markBeginX, &markEndY, &markEndX ) )
   {
      int offset=pos2Offset(markBeginY,markBeginX);
      QString s=markedText();

      if(emitUndo)
      {
         emit signalUndoCmd(new BeginCommand());
         emit signalUndoCmd(new DelTextCmd(offset,s));
      }

      QMultiLineEdit::del();

      if(emitUndo)
      {
         emit signalUndoCmd(new EndCommand());
      }
   }
   //else if( !atEnd() );
   else if(! (cursorY==numLines()-1 && cursorX==lineLength( cursorY )) )
   {
      if(emitUndo)
      {
         int offset = pos2Offset( cursorY, cursorX );

         QString s=stringShown(cursorY);
         if(cursorX != (int)s.length())
         {
            QString delTxt(s[cursorX]);
            emit signalUndoCmd(new DelTextCmd(offset,delTxt));
         }
         else if(isEndOfParagraph(cursorY))
         {
            emit signalUndoCmd(new DelTextCmd(offset,"\n"));
         }
      }

      QMultiLineEdit::del();

   }


   emitCursorPosition();
}

void MyMultiLineEdit::insert( const QString &s, bool mark )
{
   bool wasMarked=hasMarkedText();

   if(wasMarked)
      emit signalUndoCmd(new BeginCommand());

   QMultiLineEdit::insert(s, mark);

   if(wasMarked)
      emit signalUndoCmd(new EndCommand());

   emitCursorPosition();
}


void MyMultiLineEdit::killLine ()
{
   int curY, curX;
   cursorPosition( &curY, &curX );
   int offset = pos2Offset( curY, curX );

   if(emitUndo)
   {
      emit signalUndoCmd(new BeginCommand());

      QString s=stringShown(curY);
      if( curX==(int)s.length() )
      {
         if ( ! atEnd() && isEndOfParagraph(curY) )
         {
            emit signalUndoCmd( new DelTextCmd(offset,"\n") );
         }
      }
      else
      {
         QString str = s.mid( curX, s.length() );
         emit signalUndoCmd( new DelTextCmd( offset, str ) );
      }
   }
	
   QMultiLineEdit::killLine();

   if(emitUndo)
      emit signalUndoCmd(new EndCommand());

   emitCursorPosition();
}

void MyMultiLineEdit::offset2Pos(int offset, int &row, int &col) const
{
    if (offset <= 0)
    {
	row = 0;
	col = 0;
	return;
    }
    else
    {
       int charsLeft = offset;
       int i;

       for( i = 0; i < numLines(); ++i )
       {
           if (lineLength( i ) < charsLeft)
              charsLeft -= lineLength( i );
           else
           {
              row = i;
              col = charsLeft;
              return;
           }
           if( isEndOfParagraph(i) )
           {
               --charsLeft;
           }
       }

       //if (contents->at( i - 1) && !contents->at( i - 1 )->newline)
       if(getString(i-1) && isEndOfParagraph(i-1) )
       {
	    row = i - 1;
	    col = lineLength( i - 1 );
       }
       else
       {
          row = i;
          col = 0;
       }
       return;
    }

}

int MyMultiLineEdit::pos2Offset(int row, int col) const
{
    row = QMAX( QMIN( row, numLines() - 1), 0 ); // Sanity check
    col = QMAX( QMIN( col,  lineLength( row )), 0 ); // Sanity check
    if ( row == 0 )
	return QMIN( col, lineLength( 0 ));
    else
    {
        int lastI;
	lastI  = lineLength( row );
	int i, tmp = 0;

	for( i = 0; i < row ; i++ )
	{
	    tmp += lineLength( i );
	    if ( isEndOfParagraph(i) )
		++tmp;
	}

	tmp += QMIN( lastI, col );

	return tmp;
    }
}

void MyMultiLineEdit::setReadOnly(bool on)
{
	QMultiLineEdit::setReadOnly(on);

	// I want this backgroundmode, also when readonly==true
	if(on) 
	{
		setBackgroundMode(PaletteBase);
	}
}

MsgMultiLineEdit::MsgMultiLineEdit(QWidget* parent,const char* name)
                :MyMultiLineEdit(parent,name),
                _quotes(false),
                _cleverEditing(false),
                _highlightBg(false),
                _spacePoints(true),
                _bgColor(colorGroup().base().dark(110)),
                _hlSyntax(true),
                _quoteColor(Qt::darkGreen),
                _unquoteColor(Qt::red),
                _cformatColor(Qt::blue),
                _accelColor(Qt::darkMagenta),
                dnd_forcecursor(false),
                _showDiff(false),
                _diffUnderlineAdd(true),
                _diffStrikeOutDel(true),
                _diffAddColor(Qt::darkGreen),
                _diffDelColor(Qt::darkRed)
{
   //setFont(KGlobal::fixedFont());

   diffPos.setAutoDelete(true);
}


MsgMultiLineEdit::~MsgMultiLineEdit()
{
}


void MsgMultiLineEdit::setText(const QString& s)
{
    QString str = s;
    
    if(_showDiff)
    {
        diffPos.clear();
        int lines = s.contains('\n');
        diffPos.resize(lines+1);

        QStringList lineList = QStringList::split('\n',s,true);
        
        int lineCounter=-1;
        bool haveAdd=false;
        bool haveDel=false;
        bool multiline=false;
        QStringList::Iterator it;
        for(it = lineList.begin(); it != lineList.end(); ++it)
        {
            lineCounter++;
            
            int lastPos=0;
            bool atEnd=false;

            while(!atEnd)
            {            
                int addPos=-1;
                int delPos=-1;
         
                if(haveAdd && multiline)
                {
                    addPos=0;
                }
                else
                {
                    addPos = (*it).find("<KBABELADD>",lastPos);
                }
            
                if(haveDel && multiline)
                {
                    delPos=0;
                }
                else
                {
                    delPos = (*it).find("<KBABELDEL>",lastPos);
                }

                if(delPos >= 0 && addPos >= 0)
                {
                    if(delPos <= addPos)
                    {
                        haveDel=true;
                        haveAdd=false;
                    }
                    else
                    {
                        haveDel=false;
                        haveAdd=true;
                    }
                }
                else if(delPos >= 0)
                {
                    haveDel=true;
                    haveAdd=false;
                }
                else if(addPos >= 0)
                {
                    haveDel=false;
                    haveAdd=true;
                }
                else
                {
                    atEnd=true;
                    haveAdd=false;
                    haveDel=false;
                }
                
                DiffInfo di;
                di.begin=-1;
                
                if(haveAdd)
                {
                    if(!multiline)
                    {
                        (*it).remove(addPos,11);                    
                    }
                    
                    int endPos = (*it).find("</KBABELADD>",addPos);
                    if(endPos < 0)
                    {
                        endPos = (*it).length();
                        atEnd=true;
                        multiline=true;
                    }
                    else
                    {
                        (*it).remove(endPos,12);
                        haveAdd=false;
                        multiline=false;
                    }
                    
                    lastPos=endPos;

                    di.begin=addPos;
                    di.end=endPos-1;
                    di.add=true;
                }
                else if(haveDel)
                {
                    if(!multiline)
                    {
                        (*it).remove(delPos,11); 
                    }
                    
                    int endPos = (*it).find("</KBABELDEL>",delPos);
                    if(endPos < 0)
                    {
                        endPos = (*it).length();
                        atEnd=true;
                        multiline=true;
                    }
                    else
                    {
                        (*it).remove(endPos,12);
                        haveDel=false;
                        multiline=false;
                    }
                    
                    lastPos=endPos;

                    di.begin=delPos;
                    di.end=endPos-1;
                    di.add=false;
                }

                if(di.begin >= 0)
                {
                    //kdDebug() << "begin: " << di.begin
                    //    << " end: " << di.end << " add: " << di.add << endl;
                    
                    QValueList<DiffInfo> *list = diffPos[lineCounter];
                    if(!list)
                    {
                        list = new QValueList<DiffInfo>;
                        diffPos.insert(lineCounter,list);
                    }

                    list->append(di);
                }
                
            }
        }
        
        QRegExp reg("</?KBABELADD>");
        str.replace(reg,"");
        reg.setPattern("</?KBABELDEL>");
        str.replace(reg,"");
    }

    MyMultiLineEdit::setText(str);
}

void MsgMultiLineEdit::setQuotes(bool on)
{
   _quotes=on;

   setCellWidth(maxLineWidth());

   update();
}

void MsgMultiLineEdit::setCleverEditing(bool on)
{
   _cleverEditing=on;
}


void MsgMultiLineEdit::setHighlightBg(bool on)
{
   _highlightBg=on;
   update();
}


void MsgMultiLineEdit::setBgColor(const QColor& color)
{
   _bgColor=color;

   if(_highlightBg)
      update();
}

void MsgMultiLineEdit::setSpacePoints(bool on)
{
   _spacePoints=on;

   update();
}

void MsgMultiLineEdit::setHighlightSyntax(bool on)
{
   _hlSyntax=on;

   update();
}

void MsgMultiLineEdit::setHighlightColors(const QColor& quoteColor, const QColor& unquoteColor
                 , const QColor& cformatColor, const QColor& accelColor)
{
   _quoteColor=quoteColor;
   _unquoteColor=unquoteColor;
   _cformatColor=cformatColor;
   _accelColor=accelColor;

   update();
}


void MsgMultiLineEdit::setFont(const QFont& font)
{
   QMultiLineEdit::setFont(font);

   updateCellWidth();

   repaint();
}

void MsgMultiLineEdit::setDiffDisplayMode(bool addUnderline, bool delStrikeOut)
{
    _diffUnderlineAdd = addUnderline;
    _diffStrikeOutDel = delStrikeOut;
    
    if(_showDiff)
        update();
}

void MsgMultiLineEdit::setDiffColors(const QColor& addColor
        , const QColor& delColor)
{
    _diffAddColor = addColor;
    _diffDelColor = delColor;

    if(_showDiff)
        update();
}

void MsgMultiLineEdit::updateCellWidth()
{
    QString* s=getString(0);

    int maxW = 0;
    int w;

    for(int i=0; i < numLines(); i++)
    {
       s = getString(i);
       w = textWidth(*s);
       if ( w > maxW )
          maxW = w;	
    }

    setCellWidth(maxW);

    updateTableSize();
}



void MsgMultiLineEdit::paintCell( QPainter *painter, int row, int )
{
   int lr_marg=hMargin();
   //int marg_extra=0;
   bool markIsOn=hasMarkedText();
   int align=alignment();
   int cursorX=0;
   int cursorY=0;
   cursorPosition(&cursorY,&cursorX);

    const QColorGroup & g = colorGroup();
    QFontMetrics fm( painter->font() );

   int tabStopDist=defaultTabStop()*fm.width( QChar('x') );


    QString s = stringShown(row);
    if ( s.isNull() ) {
	qWarning( "QMultiLineEdit::paintCell: (%s) no text at line %d",
		  name( "unnamed" ), row );
	return;
    }

    if(_quotes)
    {
       s="\""+s+"\"";
    }

    QRect updateR = cellUpdateRect();
    QPixmap *buffer = new QPixmap( updateR.size() );
    ASSERT(buffer);
    buffer->fill (  g.base() );

    QPainter p( buffer );
    p.setFont( painter->font() );
    p.translate( -updateR.left(), -updateR.top() );

    p.setTabStops( tabStopDist );

    int yPos = 0;
    int markX1, markX2;				// in x-coordinate pixels
    markX1 = markX2 = 0;			// avoid gcc warning
    if ( markIsOn ) {
	int markBeginX, markBeginY;
	int markEndX, markEndY;
	getMarkedRegion( &markBeginY, &markBeginX, &markEndY, &markEndX );
	if ( row >= markBeginY && row <= markEndY ) {
	    if ( row == markBeginY ) {
		markX1 = markBeginX;
		if ( row == markEndY ) 		// both marks on same row
		    markX2 = markEndX;
		else
		    markX2 = s.length();	// mark till end of line
	    } else {
		if ( row == markEndY ) {
		    markX1 = 0;
		    markX2 = markEndX;
		} else {
		    markX1 = 0;			// whole line is marked
		    markX2 = s.length();	// whole line is marked
		}
	    }
	}
    }

    if(_highlightBg)
    {
	int sLength = s.length();
	if( (_quotes && sLength>2) || (!_quotes && sLength>0 ))
	{
	   int fillxpos1 = mapToView( 0, row );
	   int fillxpos2 = mapToView( sLength, row );
	
	   if(_quotes)
	   {
	      fillxpos1 -= fm.leftBearing(QChar('"'));
	      fillxpos2 += fm.rightBearing(QChar('"'));
	   }
	
	   p.fillRect( fillxpos1, 0, fillxpos2 - fillxpos1, cellHeight(row),
		    _bgColor );
        }
    }
    
    if(_showDiff && (!_diffUnderlineAdd || !_diffStrikeOutDel) )
    {
        QValueList<DiffInfo> *list = diffPos[row];
        if(list)
        {
            QValueList<DiffInfo>::ConstIterator it;
            for(it = list->begin(); it != list->end(); ++it)
            {
                int xpos1 = mapToView( (*it).begin, row );
                int xpos2 = mapToView( (*it).end+1, row );
	
                /*
                if(_quotes)
                {
                    xpos1 -= fm.leftBearing(QChar('"'));
                }	
                */
                
                int h=cellHeight(row);

                if( (*it).add  && !_diffUnderlineAdd)
                {
                    p.fillRect( xpos1, 0, xpos2 - xpos1, h, _diffAddColor);
                }
                else if(!(*it).add && !_diffStrikeOutDel)
                {
                    p.fillRect( xpos1, 0, xpos2 - xpos1, h, _diffDelColor );
                }
            }
        }
    }


    p.setPen( g.text() );
    int wcell = cellWidth() - 2*lr_marg;// - marg_extra;
    int wrow = textWidth(s);
    int x = lr_marg - p.fontMetrics().leftBearing(s[0]);
    if ( align == Qt::AlignCenter )
	x += (wcell - wrow) / 2;
    else if ( align == Qt::AlignRight )
	x += wcell - wrow;
    p.drawText( x,  yPos, cellWidth()-lr_marg-x, cellHeight(),
		align == AlignLeft?ExpandTabs:0, s );


    // highlight syntax
    if(_hlSyntax)
    {
       QString line=s;
       if(_quotes)
          line=s.mid(1,s.length()-2);

       int index;

       // find characters that have to be quoted, but aren't
       p.setPen(_unquoteColor);

       // color all '\' red. The one that are used for quoting are colored green anyway later.
       QString regexp="\\\\";
       index=0;
       index=line.find(QRegExp(regexp),index);
       while(index>=0)
       {
          int xPos=mapToView(index,row);

          p.drawText( xPos,  yPos, cellWidth()-lr_marg-xPos, cellHeight(),
		align == AlignLeft?ExpandTabs:0, "\\" );
		
          index=line.find(QRegExp(regexp),index+1);

       }

       // color all '"' red. The one that are used for quoting are colored green anyway later.
       regexp="\"";
       index=0;
       index=line.find(QRegExp(regexp),index);
       while(index>=0)
       {
          index=line.find(QRegExp("\""),index);
          int xPos=mapToView(index,row);

          p.drawText( xPos,  yPos, cellWidth()-lr_marg-xPos, cellHeight(),
	        align == AlignLeft?ExpandTabs:0, "\"" );
		
          index=line.find(QRegExp(regexp),index+1);
       }

       p.setPen(_quoteColor);

       // saves index of '\' that are already used for quoting
       QValueList<int> validIndex;

       QString spclChars="abfnrtv'\"\?\\\\";
       regexp="\\\\["+spclChars+"]";

       index=0;
       index=line.find(QRegExp(regexp),index);
       while(index>=0)
       {
          if( !validIndex.contains(index) )
          {
             validIndex.append(index);
             if( line[index+1] == QChar('\\') )
                validIndex.append(index+1);

             int xPos=mapToView(index,row);

             p.drawText( xPos,  yPos, cellWidth()-lr_marg-xPos, cellHeight(),
                 align == AlignLeft?ExpandTabs:0, line.mid(index,2) );
	
          }

          index=line.find(QRegExp(regexp),index+1);

       }

       // find oktal numbers
       index=0;
       index=line.find(QRegExp("\\\\\\d+"));
       while(index>=0)
       {
          if( !validIndex.contains(index) )
          {
             validIndex.append(index);

             int xPos=mapToView(index,row);

             int endIndex=line.find(QRegExp("[^\\d]"),index+1);

             p.drawText( xPos,  yPos, cellWidth()-lr_marg-xPos, cellHeight(),
                    align == AlignLeft?ExpandTabs:0, line.mid(index,endIndex-index) );
          }

          index=line.find(QRegExp("\\\\\\d+"),index+1);
       }


       // find hexadecimal numbers
       index=0;
       index=line.find(QRegExp("\\\\x[\\dabcdef]+"));
       while(index>=0)
       {
          if( !validIndex.contains(index) )
          {
             validIndex.append(index);

             int xPos=mapToView(index,row);

             int endIndex=line.find(QRegExp("[^\\dabcdef]"),index+2);

              p.drawText( xPos,  yPos, cellWidth()-lr_marg-xPos, cellHeight(),
                  align == AlignLeft?ExpandTabs:0, line.mid(index,endIndex-index) );	
          }
                          	
          index=line.find(QRegExp("\\\\x[\\dabcdef]+"),index+1);
       }

       p.setPen(_cformatColor);
       index=0;
       QString formatChars="dioxXucsfeEgGp%";
       index=line.find(QRegExp("%."));
       while(index>=0)
       {
          int endIndex=line.find(QRegExp("[^\\-\\.\\?\\dhl]"),index+1);
          if(endIndex<0)
             endIndex=line.length();
          else
          {
             if(formatChars.contains(line[endIndex]) )
                endIndex++;
          }

          int xPos=mapToView(index,row);

          p.drawText( xPos,  yPos, cellWidth()-lr_marg-xPos, cellHeight(),
                  align == AlignLeft?ExpandTabs:0, line.mid(index,endIndex-index) );	
                                    	
          index=line.find(QRegExp("%."),endIndex);
       }

       p.setPen(_accelColor);
       index=0;
       index=line.find(QRegExp("&[^\\s]"));
       while(index>=0)
       {
          int xPos=mapToView(index,row);

          p.drawText( xPos,  yPos, cellWidth()-lr_marg-xPos, cellHeight(),
                  align == AlignLeft?ExpandTabs:0, line.mid(index,2) );	
                                    	
          index=line.find(QRegExp("&[^\\s]"),index+2);
       }


       p.setPen(g.text());
    }		

	
			
    if(_spacePoints)
    {
       QString str=s;
       if(_quotes)
       {
          str=s.mid(1,s.length()-1);
       }
       int index=str.find(QChar(' '));
       while(index>=0)
       {
          int xPos=mapToView(index,row);
          xPos=xPos + ( fm.width(QChar(' '))/2 );
          int yPos=cellHeight(row)/2;
          p.drawPoint(xPos,yPos);

          index=str.find(QChar(' '),index+1);
       }
    }
    
    if( _showDiff && (_diffUnderlineAdd || _diffStrikeOutDel) )
    {
        QValueList<DiffInfo> *list = diffPos[row];
        if(list)
        {
            QPen addPen(_diffAddColor,2);
            QPen delPen(_diffDelColor,2);
            QValueList<DiffInfo>::ConstIterator it;
            for(it = list->begin(); it != list->end(); ++it)
            {
                int xpos1 = mapToView( (*it).begin, row );
                int xpos2 = mapToView( (*it).end+1, row );
	
                int h=cellHeight(row);

                if( (*it).add && _diffUnderlineAdd)
                {
                    p.setPen(addPen);
                    p.drawLine(xpos1,h-1,xpos2,h-1);
                }
                else if(!(*it).add && _diffStrikeOutDel)
                {
                    p.setPen(delPen);
                    p.drawLine(xpos1,h/2,xpos2,h/2);
                }
            }

            p.setPen(g.text());
        }
    }

//    if ( !isEndOfParagraph(row) && BREAK_WITHIN_WORDS )
//	p.drawPixmap( x + wrow - lr_marg - marg_extra, yPos, d->arrow );
	
#if 0
    if ( isEndOfParagraph(row) )
	p.drawLine( lr_marg,  yPos+cellHeight()-2, cellWidth() - lr_marg, yPos+cellHeight()-2);
#endif
    if ( markX1 != markX2 ) {
	int sLength = s.length();
	int xpos1   =  mapToView( markX1, row );
	int xpos2   =  mapToView( markX2, row );
	int fillxpos1 = xpos1;
	int fillxpos2 = xpos2;
	if ( markX1 == 0 )
	    fillxpos1 -= 2;
	if ( markX2 == sLength )
	    fillxpos2 += 3;
	p.setClipping( TRUE );
	p.setClipRect( fillxpos1 - updateR.left(), 0,
		       fillxpos2 - fillxpos1, cellHeight(row) );
	p.fillRect( fillxpos1, 0, fillxpos2 - fillxpos1, cellHeight(row),
		    g.brush( QColorGroup::Highlight ) );
	p.setPen( g.highlightedText() );
	p.drawText( x,  yPos, cellWidth()-lr_marg-x, cellHeight(),
		    align == AlignLeft?ExpandTabs:0, s );
	p.setClipping( FALSE );
    }

    if ( row == cursorY && cursorOn && !isReadOnly() ) {
	int cursorPos = QMIN( (int)s.length(), cursorX );
	int cXPos   = mapToView( cursorPos, row )-1; // added -1
	int cYPos   = 0;
	
	if ( hasFocus() || dnd_forcecursor ) {
	    p.setPen( g.text() );
	
	    p.drawLine( cXPos, cYPos,
			cXPos, cYPos + fm.height() - 2);

	    // TODO: set it other times, eg. when scrollbar moves view
	    QWMatrix wm = painter->worldMatrix();
	    setMicroFocusHint( int(wm.dx()+cXPos),
			       int (wm.dy()+cYPos),
			       1, fm.ascent() );
	}
    }
    p.end();
    painter->drawPixmap( updateR.left(), updateR.top(), *buffer,
			 0, 0, updateR.width(), updateR.height() );
	
			
    delete buffer;
}



int MsgMultiLineEdit::mapToView( int xIndex, int line )
{
    int lr_marg=hMargin();
    int align=alignment();

    QString s = stringShown( line );
    xIndex = QMIN( (int)s.length(), xIndex );
    QFontMetrics fm( font() );
    int wcell = cellWidth() - 2 * lr_marg;// - d->marg_extra;
    int wrow = textWidth(s);

    uint offset=0;
    if(_quotes)
    {
       s="\""+s;
       offset=1;
    }

    int w = textWidthWithTabs( fm, s, 0, xIndex+offset, align ) /*- 1*/;
    if ( align == Qt::AlignCenter )
	w += (wcell - wrow) / 2;
    else if ( align == Qt::AlignRight )
	w += wcell - wrow;
	
    return lr_marg + w;
}

void MsgMultiLineEdit::setCellWidth(int w)
{
   if(_quotes)
   {
      QFontMetrics fm( font() );
      w+=2*fm.width(QChar('"'));
   }

   QTableView::setCellWidth(w);
}


void MsgMultiLineEdit::dragMoveEvent(QDragMoveEvent* e)
{
   dnd_forcecursor=true;

   QMultiLineEdit::dragMoveEvent(e);

   dnd_forcecursor=false;

   emitCursorPosition();
}

void MsgMultiLineEdit::mousePressEvent(QMouseEvent* e)
{
    if(e->button() == RightButton)
        return;

    // if surrounding quotes are enabled, the position of the mouse
    // event has to be translated, because QMultiLineEdit knows
    // nothing about the extra displayed quotes
    if(_quotes)
    {
        QFontMetrics fm( font() );
        QPoint pos = e->pos();
    	pos.setX( pos.x()-fm.width("\"") );

        QMouseEvent *event = new QMouseEvent(e->type(),pos,e->button(),e->state());
        QMultiLineEdit::mousePressEvent(event);
        delete event;
    }
    else
    {
        QMultiLineEdit::mousePressEvent(e);
    }

    emitCursorPosition();
}

void MsgMultiLineEdit::mouseReleaseEvent(QMouseEvent* e)
{
    // if surrounding quotes are enabled, the position of the mouse
    // event has to be translated, because QMultiLineEdit knows
    // nothing about the extra displayed quotes
    if(_quotes)
    {
        QFontMetrics fm( font() );
        QPoint pos = e->pos();
    	pos.setX( pos.x()-fm.width("\"") );

        QMouseEvent *event = new QMouseEvent(e->type(),pos,e->button(),e->state());
        QMultiLineEdit::mouseReleaseEvent(event);
        delete event;
    }
    else
    {
        QMultiLineEdit::mouseReleaseEvent(e);
    }

    emitCursorPosition();
}

void MsgMultiLineEdit::mouseMoveEvent(QMouseEvent* e)
{
    // if surrounding quotes are enabled, the position of the mouse
    // event has to be translated, because QMultiLineEdit knows
    // nothing about the extra displayed quotes
    if(_quotes)
    {
        QFontMetrics fm( font() );
        QPoint pos = e->pos();
    	pos.setX( pos.x()-fm.width("\"") );

        QMouseEvent *event = new QMouseEvent(e->type(),pos,e->button(),e->state());
        MyMultiLineEdit::mouseMoveEvent(event);
        delete event;
    }
    else
    {
        MyMultiLineEdit::mouseMoveEvent(e);
    }
}

void MsgMultiLineEdit::mouseDoubleClickEvent(QMouseEvent* e)
{
    // if surrounding quotes are enabled, the position of the mouse
    // event has to be translated, because QMultiLineEdit knows
    // nothing about the extra displayed quotes
    if(_quotes)
    {
        QFontMetrics fm( font() );
        QPoint pos = e->pos();
    	pos.setX( pos.x()-fm.width("\"") );

        QMouseEvent *event = new QMouseEvent(e->type(),pos,e->button(),e->state());
        QMultiLineEdit::mouseDoubleClickEvent(event);
        delete event;
    }
    else
    {
        QMultiLineEdit::mouseDoubleClickEvent(e);
    }

    emitCursorPosition();
}

void MsgMultiLineEdit::keyPressEvent(QKeyEvent *e)
{
    if(!_cleverEditing || isReadOnly())
    {
        MyMultiLineEdit::keyPressEvent(e);
        return;
    }

    
    if(e->key() == Key_Return || e->key() == Key_Enter)
    { 
        emit signalUndoCmd(new BeginCommand());

        int row, col;
        getCursorPosition(&row,&col);
        QString *str=getString(row);

        if(e->state() & ShiftButton)
        {            
            if(col > 0 && str)
            {
                if(str->at(col-1) == '\\' && !isMasked(str,col-1))
                {
                    insert("n",false);
                }
                else
                {
                    insert("\\n",false);
                }
            }
            else
            {
                insert("\\n",false);
            }
        }    
        else if(!(e->state() & ControlButton))
        {
            if(col > 0 && str && !str->at(col-1).isSpace())
            {
                if(str->at(col-1)=='\\' && !isMasked(str,col-1))
                {
                    insert("\\",false);
                }
                
                // if there is not a new line at the end
                if(col < 2 || str->mid(col-2,2)!="\\n")
                {
                    insert(" ",false);
                }
            }
            else if(str->isEmpty())
            {
                insert("\\n",false);
            }
        }    
     
        if(str && !str->isEmpty())
        {
            newLine();
            emit returnPressed();
            emit textChanged();
    
            e->accept();
        }

        emit signalUndoCmd(new EndCommand());
    }
    else if(e->key() == Key_Tab)
    {
        insert("\\t",false);
        emit textChanged();
        e->accept();
    }
    else if(e->key() == Key_Delete 
            || ((e->state() & ControlButton) && e->key() == Key_D) )
    {
        emit signalUndoCmd(new BeginCommand());
        
        if(!hasMarkedText())
        {
            int row, col;
            getCursorPosition(&row,&col);
            QString *str=getString(row);

            if(str && col < (int)str->length() && str->at(col) == '\\'
                        && !isMasked(str,col))
            {
                QString spclChars="abfnrtv'\"?\\";
                if(col < (int)str->length()-1 
                        && spclChars.contains(str->at(col+1)))
                {
                    del();
                }
            }
        }
        
        del();

        emit signalUndoCmd(new EndCommand());
        emit textChanged();
        e->accept();
    }
    else if(e->key() == Key_BackSpace
            || ((e->state() & ControlButton) && e->key() == Key_H) )
    {
        emit signalUndoCmd(new BeginCommand()); 
        
        if(!hasMarkedText())
        {
            int row, col;
            getCursorPosition(&row,&col);
            QString *str=getString(row);

            QString spclChars="abfnrtv'\"?\\";
            if(str && col > 0 && spclChars.contains(str->at(col-1)))
            {
                if(col > 1 && str->at(col-2)=='\\' && !isMasked(str,col-2))
                {
                    backspace();
                }
            }

        }
        
        backspace();

        emit signalUndoCmd(new EndCommand());

        emit textChanged();
        e->accept();
    }
    else if(e->text() == "\"")   
    {
        emit signalUndoCmd(new BeginCommand());

        int row, col;
        getCursorPosition(&row,&col);
        QString *str=getString(row);
    
        if(col == 0 || str->at(col-1) != '\\' || isMasked(str,col-1) )
        {
            insert("\\\"",false);
        }
        else
        {
            insert("\"",false);
        }

        emit textChanged();
        e->accept();

        emit signalUndoCmd(new EndCommand());
    }
    else
    { 
        MyMultiLineEdit::keyPressEvent(e);
    }
}

void MsgMultiLineEdit::setDiffMode(bool on)
{
    _showDiff=on;
    
    if(!on)
    {
        diffPos.clear();
    }
}

bool MsgMultiLineEdit::isMasked(QString *str, uint col)
{
    if(col == 0 || !str)
        return false;

    uint counter=0;
    int pos=col;
    
    while(pos >= 0 && str->at(pos) == '\\')
    {
        counter++;
        pos--;
    }
    
    return !(bool)(counter%2);
}

#include "mymultilineedit.moc"
