/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLAttributeDesignator.cpp - designates a SAML attribute in a query

   Scott Cantor
   1/15/04
*/

#include "internal.h"

using namespace saml;
using namespace std;

SAMLAttributeDesignator::SAMLAttributeDesignator(const XMLCh* name, const XMLCh* ns)
    : m_name(XML::assign(name)), m_namespace(XML::assign(ns))
{
    RTTI(SAMLAttributeDesignator);
}

SAMLAttributeDesignator::SAMLAttributeDesignator(DOMElement* e) : m_name(NULL), m_namespace(NULL)
{
    RTTI(SAMLAttributeDesignator);
    fromDOM(e);
}

SAMLAttributeDesignator::SAMLAttributeDesignator(istream& in) : SAMLObject(in), m_name(NULL), m_namespace(NULL)
{
    RTTI(SAMLAttributeDesignator);
    fromDOM(m_document->getDocumentElement());
}

SAMLAttributeDesignator::~SAMLAttributeDesignator()
{
    if (m_bOwnStrings) {
        XMLString::release(&m_name);
        XMLString::release(&m_namespace);
    }
}

void SAMLAttributeDesignator::ownStrings()
{
    if (!m_bOwnStrings) {
        m_name=XML::assign(m_name);
        m_namespace=XML::assign(m_namespace);
        m_bOwnStrings=true;
    }
}

void SAMLAttributeDesignator::fromDOM(DOMElement* e)
{
    SAMLObject::fromDOM(e);

    if (SAMLConfig::getConfig().strict_dom_checking && !XML::isElementNamed(e,XML::SAML_NS,L(AttributeDesignator)))
        throw MalformedException("SAMLAttributeDesignator::fromDOM() missing saml:AttributeDesignator element at root");

    m_name=const_cast<XMLCh*>(e->getAttributeNS(NULL,L(AttributeName)));
    m_namespace=const_cast<XMLCh*>(e->getAttributeNS(NULL,L(AttributeNamespace)));
    checkValidity();
}

void SAMLAttributeDesignator::setName(const XMLCh* name)
{
    if (XML::isEmpty(name))
        throw MalformedException("name cannot be null or empty");
    
    if (m_bOwnStrings)
        XMLString::release(&m_name);
    else {
        m_name=NULL;
        ownStrings();
    }
    m_name=XML::assign(name);
    setDirty();
}

void SAMLAttributeDesignator::setNamespace(const XMLCh* ns)
{
    if (XML::isEmpty(ns))
        throw MalformedException("name cannot be null or empty");
    
    if (m_bOwnStrings)
        XMLString::release(&m_namespace);
    else {
        m_namespace=NULL;
        ownStrings();
    }
    m_namespace=XML::assign(ns);
    setDirty();
}

DOMElement* SAMLAttributeDesignator::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* a = doc->createElementNS(XML::SAML_NS, L(AttributeDesignator));
    if (xmlns)
        a->setAttributeNS(XML::XMLNS_NS, L(xmlns), XML::SAML_NS);
    return a;
}
    
DOMNode* SAMLAttributeDesignator::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLObject::toDOM(doc,xmlns);
    DOMElement* a=static_cast<DOMElement*>(m_root);
    
    if (m_bDirty) {
        a->setAttributeNS(NULL,L(AttributeName),m_name);
        a->setAttributeNS(NULL,L(AttributeNamespace),m_namespace);
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(a,XML::SAML_NS);
    }

    return m_root;
}

void SAMLAttributeDesignator::checkValidity() const
{
    if (XML::isEmpty(m_name) || XML::isEmpty(m_namespace))
        throw MalformedException("AttributeDesignator invalid, requires name and namespace");
}

SAMLObject* SAMLAttributeDesignator::clone() const
{
    return new SAMLAttributeDesignator(m_name,m_namespace);
}
