# Samizdat HTML helpers
#
#   Copyright (c) 2002-2008  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'

module ApplicationHelper

  STRIP_TAGS_PATTERN = Regexp.new(/<[^>]*?>/).freeze

  def strip_tags(text)
    text.gsub(STRIP_TAGS_PATTERN, '')
  end

  # title for logo link in the page header
  #
  def logo_link_title
    config['site']['name'] + ': ' + _('Front Page')
  end

  SIZE_UNITS = [
    [ 'K', 1024 ],
    [ 'M', 1024 ** 2 ],
    [ 'G', 1024 ** 3 ],
    [ 'T', 1024 ** 4 ],
    [ 'P', 1024 ** 5 ]
  ]

  def display_file_size(size)
    return '' unless size.kind_of? Numeric

    SIZE_UNITS.each do |name, multiplier|
      display = size.to_f / multiplier

      if display < 1
        return sprintf("%1.2f#{name}", display)

      elsif display < 10
        return sprintf("%1.1f#{name}", display)

      elsif display < 1024 or name == SIZE_UNITS[-1][0]
        return sprintf("%1.0f#{name}", display)
      end
    end
  end

  # wrap title and content into a CSS-rendered box
  #
  def box(title, content, id=nil)
    box_title = %{<div class="box-title">#{title}</div>} if title
    box_id = %{ id="#{id}"} if id
%{<div class="box"#{box_id}>
  #{box_title}<div class="box-content">
#{content}
  </div>
</div>\n}
  end

  # page number to be appended to a page title
  #
  def page_number(page)
    page > 1 ? sprintf(_(', page %s'), page) : ''
  end

  # navigation link to a given page number
  #
  def nav(dataset, name = 'page', page = (@request[name] or 1).to_i, script = @request.route + '?')
    return '' if page < 1 or dataset.size < dataset.limit

    max_page = (dataset.size.to_f / dataset.limit).ceil
    return '' if page > max_page

    link = "#{script}#{name}="
    pages = []

    range_start = page - 2
    range_start = 1 if range_start < 1
    range_end = page + 2
    range_end = max_page if range_end > max_page

    if range_start > 1
      pages << 1
      if range_start > 2
        pages << ellipsis
      end
    end

    range_start.upto(range_end) do |i|
      pages << i
    end

    if range_end < max_page
      if range_end < max_page - 1
        pages << ellipsis
      end
      pages << max_page
    end

    pages = _('pages: ') + pages.collect {|i|
      (i.kind_of?(Integer) and i != page)?
        %{<a href="#{link}#{i}">#{i}</a>} : i
    }.join(' ')

    pages << %{, <a href="#{link}#{page + 1}">} + _('next page') + '</a>' if
      page < max_page

    '<div class="nav">' << pages << "</div>\n"
  end

  # add link to RSS rendering of the page
  #
  def nav_rss(link)
    link ? %{<div class="nav"><a href="#{link}">rss 1.0</a></div>} : ''
  end

  # resource list with navigation link
  #
  def list(list, nav, foot='')
    foot = %{<div class="foot">\n#{foot + nav}</div>\n} unless
      '' == foot and '' == nav
    even = 1
    %{<ul>\n} <<
    list.collect {|li|
      even = 1 - even
      %{<li#{' class="even"' if even == 1}>#{li}</li>\n}
    }.join << %{</ul>\n} << foot
  end

  # resource table with navigation link
  #
  def table(table, nav, foot='')
    foot = %{<div class="foot">\n#{foot + nav}</div>\n} unless
      '' == foot and '' == nav
    even = 1
    %{<table>\n<thead><tr>\n} <<
    table.shift.collect {|th| "<th>#{th}</th>\n" }.join <<
    %{</tr></thead>\n<tbody>\n} <<
    table.collect {|row|
      even = 1 - even   # todo: a CSS-only way to do this
      %{<tr#{' class="even"' if even == 1}>\n} << row.collect {|td|
        "<td>#{td or '&nbsp;'}</td>\n"
      }.join << "</tr>\n"
    }.join << %{</tbody></table>\n} << foot
  end

  # type can be any of the following:
  #
  # [:label]
  #   wrap label _value_ in a <div> tag and associate it with _name_ field
  #   (caveat: ids are not unique across multiple forms)
  # [:br] line break
  # [:textarea] fixed text area 70x20 with _value_
  # [:select] _value_ is an array of options or pairs of [option, label]
  # [:submit] _value_ is a button label
  # [standard HTML input type] copy _type_ as is into <input> tag
  #
  def form_field(type, name=nil, value=nil, default=nil)
    value = CGI.escapeHTML(value) if value.class == String
    attr = %{ name="#{name}"} if name
    attr += %{ id="f_#{name}"} if name and :label != type
    attr += ' disabled="disabled"' if name and :disabled == default
    case type
    when :br then %{<br />\n}
    when :label
      %{<div class="label"><label for="f_#{name}">#{value}</label></div>\n}
    when :textarea
      %{<textarea#{attr} cols="70" rows="20">#{value}</textarea>\n}   # mind rexml
    when :select, :select_submit
      attr += ' onchange="submit()"' if :select_submit == type
      %{<select#{attr}>\n} + value.collect {|option|
        v, l = (option.class == Array)? option : [option, option]
        selected = (v == default)? ' selected="selected"' : ''
        %{    <option#{selected} value="#{v}">#{l}</option>\n}
      }.join + "</select>\n"
    when :submit, :submit_moderator
      value = _('Submit') if value.nil?
      %{<input#{attr} type="submit" value="#{value}" class="#{type}" />\n}
    else
      if :checkbox == type
        attr += ' checked="checked"' if value
        value = 'true'
      end
      %{<input#{attr} type="#{type}" value="#{value}" />\n}
    end
  end

  # normalize form action location
  #
  def action_location(action = nil)
    action ||= @request.route   # default to current location
    unless absolute_url?(action)
      action = '/' + action unless '/' == action[0, 1]   # force start with '/'
      action = @request.uri_prefix + action   # relative to site base
    end
    action
  end

  # wrap a list of form fields into a form (see form_field())
  #
  # _action_ should always be relative to site base (start with '/'),
  # default _action_ is current location
  #
  # automatically detects if multipart/form-data is necessary
  #
  def form(action, *fields)
    if fields.assoc(:file)
      enctype = ' enctype="multipart/form-data"'
    end

    %{<form action="#{action_location(action)}" method="post"#{enctype}><div>\n} <<
      fields.collect {|param| form_field(*param) }.join << "</div></form>\n"
  end

  def action_token_key
    %{action_token/#{@session.login}}
  end

  # wrapper around form() to protect logged in members against CSRF by adding a
  # hidden action confirmation hash both to the form and to member's preferences
  #
  # only one secure form per page will work
  #
  def secure_form(action, *fields)
    if @session.member
      key = action_token_key + action_location(action)
      action_token = random_digest(key)
      cache[key] = action_token
      fields.push [:hidden, 'action_token', action_token]
    end
    form(action, *fields)
  end

  # check CSRF protection prepared by secure_form()
  #
  # compares an action token submitted in the form against the one recorded in
  # member's preferences, fails immediately for non-POST requests, for guests
  # does nothing and always succeeds
  #
  # action token is wiped once used
  #
  def action_confirmed?
    return false unless 'POST' == @request.env['REQUEST_METHOD']
    return true unless @session.member

    key = action_token_key + action_location
    action_token = @request['action_token']
    if action_token and action_token == cache[key]
      cache.delete(key)
      true
    else
      false
    end
  end

  # drop-down menu to select focus from a list
  #
  def focus_select(focus = nil)
    return [] unless @request.access('vote')

    focuses = Focus.collect_focuses {|f,|
      [ f, Resource.new(@request, f).title ]
    }
    if focus and ('focus::Translation' == focus.uriref or not focuses.assoc(focus.id))
      focuses.unshift( [(focus.id or focus.uriref), focus.name] )
    end
    focuses.unshift [nil, _('SELECT FOCUS')]

    [ [:label, 'focus', _('Select focus that this resource will be related to')],
        [:select, 'focus', focuses, (focus ? (focus.id or focus.uriref) : nil)] ]
  end

  # form fields for vote on focus rating
  #
  def focus_fields(focus)
    return [] unless @request.access('vote')

    fields = focus_select(focus)

    if @request.advanced_ui?
      fields.push(
        [:label, 'focus_id', _("Enter focus id if it's not in the list")],
          [:text, 'focus_id']
      )
    end

    fields.push(
      [:label, 'rating', _('Give a rating of how strongly this resource is related to selected focus')],
        [:select, 'rating', [
          [-2, _('-2 (No)')],
          [-1, _('-1 (Not Likely)')],
          [0, _('0 (Uncertain)')],
          [1, _('1 (Likely)')],
          [2, _('2 (Yes)')] ], 0]
    )
  end

  # render link to resource with a tooltip
  #
  # _title_ should be HTML-escaped
  #
  def resource_href(id, title)
    return '' unless title.kind_of? String
    id.nil? ? limit_string(title) :
      '<a title="'+_('Click to view the resource')+%{" href="#{id}">#{limit_string(title)}</a>}
  end

  # render resource description for resource listing
  #
  # _title_ should be HTML-escaped
  #
  def resource(id, title, info)
%{<div class="resource">
<div class="title">#{resource_href(id, title)}</div>
<div class="info">#{info}</div>
</div>\n}
  end

  # wrap page title and body with heads and foots
  #
  # _body_ can be String or Array of pairs [title, body]; in latter case,
  # _title_ can be defaulted to the title of the first pair
  #
  # if _options_ includes :front_page flag, body isn't wrapped in box
  #
  def page(title, body, options={})
    if title and options.delete(:front_page)
      body = %{<div id="languages">#{language_list}</div>} + body if
        defined?(GetText) and 'yes' != @request.cookie('nostatic')
      body
    else
      body = [[title, body]] unless body.class == Array
      title = body[0][0] if title.nil?
      body.collect {|t, b, id| box(t, b, id) }.join
    end
  end
end
