/*
 * Sylpheed -- a GTK+ based, lightweight, and fast e-mail client
 * Copyright (C) 1999-2004 Hiroyuki Yamamoto
 * This file (C) 2004 Colin Leroy
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Partial download:
 * A mail which has been completely downloaded will have no special headers,
 * and its entry in the uidl file will end by 0 (POP3_TOTALLY_RECEIVED);
 *
 * A mail which has been partially downloaded will have some special headers,
 * and its entry in the uidl file will first be 1 (POP3_PARTIALLY_RECEIVED);
 * the special headers will be including "SC-Marked-For-Download" which can 
 * have three values:
 * 0 (POP3_PARTIAL_DLOAD_UNKN) meaning that the user has not yet chosen to
 *  download the mail or let it be deleted - this header is absent until the
 *  user first chooses an action
 * 1 (POP3_PARTIAL_DLOAD_DLOAD) meaning that the user wants to finish 
 *  downloading the mail
 * 2 (POP3_PARTIAL_DLOAD_DELE) meaning that the user does not want to finish
 *  downloading the mail
 * When updating this header to POP3_PARTIAL_DLOAD_DLOAD, the uidl line of
 * this mail will end with the mail's physical path, which Sylpheed will remove
 * after having downloaded the complete mail. msg->partial_recv will equal
 * 2 (POP3_MUST_COMPLETE_RECV).
 * When updating this header to POP3_PARTIAL_DLOAD_DELE, the uidl line of
 * this mail will be 0 (POP3_TOTALLY_RECEIVED), which will let Sylpheed delete
 * this mail from the server as soon as the leave_time preference specifies.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>

#include "intl.h"
#include "partial_download.h"
#include "utils.h"
#include "pop.h"
#include "folder.h"
#include "procheader.h"
#include "msgcache.h"

int partial_msg_in_uidl_list(MsgInfo *msginfo)
{
	gchar *path;
	FILE *fp;
	gchar buf[POPBUFSIZE];
	gchar uidl[POPBUFSIZE];
	time_t recv_time;
	time_t now;
	gint partial_recv;
	
	if (!msginfo->account_server
	||  !msginfo->account_login
	||  !msginfo->partial_recv)
		return FALSE;
	
	path = g_strconcat(get_rc_dir(), G_DIR_SEPARATOR_S,
			   "uidl", G_DIR_SEPARATOR_S, msginfo->account_server,
			   "-", msginfo->account_login, NULL);
	if ((fp = fopen(path, "rb")) == NULL) {
		if (ENOENT != errno) FILE_OP_ERROR(path, "fopen");
		g_free(path);
		path = g_strconcat(get_rc_dir(), G_DIR_SEPARATOR_S,
				   "uidl-", msginfo->account_server,
				   "-", msginfo->account_login, NULL);
		if ((fp = fopen(path, "rb")) == NULL) {
			if (ENOENT != errno) FILE_OP_ERROR(path, "fopen");
			g_free(path);
			return FALSE;
		}
	}
	g_free(path);

	now = time(NULL);

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		gchar tmp[POPBUFSIZE];
		strretchomp(buf);
		recv_time = RECV_TIME_NONE;
		partial_recv = POP3_TOTALLY_RECEIVED;
		
		if (sscanf(buf, "%s\t%ld\t%s", uidl, &recv_time, tmp) < 2) {
			if (sscanf(buf, "%s", uidl) != 1)
				continue;
			else {
				recv_time = now;
			}
		}
		if (!strcmp(uidl, msginfo->partial_recv)) {
			fclose(fp);
			return TRUE;
		}
	}

	fclose(fp);	
	return FALSE;
}

static int partial_uidl_mark_mail(MsgInfo *msginfo, int download)
{
	gchar *path;
	gchar *pathnew;
	FILE *fp;
	FILE *fpnew;
	gchar buf[POPBUFSIZE];
	gchar uidl[POPBUFSIZE];
	time_t recv_time;
	time_t now;
	int len;
	int start = TRUE;
	gchar partial_recv[POPBUFSIZE];
	int err = -1;

	gchar *filename;
	MsgInfo *tinfo;
	filename = procmsg_get_message_file_path(msginfo);
	if (!filename) {
		g_warning("can't get message file path.\n");
		return err;
	}
	tinfo = procheader_parse_file(filename, msginfo->flags, TRUE, TRUE);

	if (!tinfo->account_server
	||  !tinfo->account_login
	||  !tinfo->partial_recv) {
		goto bail;
	}
	path = g_strconcat(get_rc_dir(), G_DIR_SEPARATOR_S,
			   "uidl", G_DIR_SEPARATOR_S, tinfo->account_server,
			   "-", tinfo->account_login, NULL);
	if ((fp = fopen(path, "rb")) == NULL) {
		perror("fopen1");
		if (ENOENT != errno) FILE_OP_ERROR(path, "fopen");
		g_free(path);
		path = g_strconcat(get_rc_dir(), G_DIR_SEPARATOR_S,
				   "uidl-", tinfo->account_server,
				   "-", tinfo->account_login, NULL);
		if ((fp = fopen(path, "rb")) == NULL) {
			if (ENOENT != errno) FILE_OP_ERROR(path, "fopen");
			g_free(path);
		}
		goto bail;
	}

	pathnew = g_strconcat(get_rc_dir(), G_DIR_SEPARATOR_S,
			   "uidl", G_DIR_SEPARATOR_S, tinfo->account_server,
			   "-", tinfo->account_login, ".new", NULL);
	if ((fpnew = fopen(pathnew, "wb")) == NULL) {
		perror("fopen2");
		fclose(fp);
		g_free(pathnew);
		goto bail;
	}
	
	now = time(NULL);

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		strretchomp(buf);
		recv_time = RECV_TIME_NONE;
		sprintf(partial_recv,"0");
		
		if (sscanf(buf, "%s\t%ld\t%s", 
			   uidl, &recv_time, partial_recv) < 2) {
			if (sscanf(buf, "%s", uidl) != 1)
				continue;
			else {
				recv_time = now;
			}
		}
		if (strcmp(tinfo->partial_recv, uidl)) {
			fprintf(fpnew, "%s\t%ld\t%s\n", 
				uidl, recv_time, partial_recv);
		} else {
			gchar *stat = NULL;
			if (download == POP3_PARTIAL_DLOAD_DLOAD) {
				gchar *folder_id = folder_item_get_identifier(
							msginfo->folder);
				stat = g_strdup_printf("%s:%d",
					folder_id, msginfo->msgnum);
				g_free(folder_id);
			}
			else if (download == POP3_PARTIAL_DLOAD_UNKN)
				stat = g_strdup("1");
			else if (download == POP3_PARTIAL_DLOAD_DELE)
				stat = g_strdup("0");
			
			fprintf(fpnew, "%s\t%ld\t%s\n", 
				uidl, recv_time, stat);
			g_free(stat);
		}
	}
	fclose(fpnew);
	fclose(fp);

	move_file(pathnew, path, TRUE);

	g_free(path);
	g_free(pathnew);
	
	if ((fp = fopen(filename,"rb")) == NULL) {
		perror("fopen3");
		goto bail;
	}
	pathnew = g_strdup_printf("%s.new", filename);
	if ((fpnew = fopen(pathnew, "wb")) == NULL) {
		perror("fopen4");
		fclose(fp);
		g_free(pathnew);
		goto bail;
	}
	
	while ((len = fread(buf, sizeof(gchar), sizeof(buf)-1, fp)) > 0) {
		buf[len]='\0';
		if (start) {
			start = FALSE;
			fprintf(fpnew, "SC-Marked-For-Download: %d\n", 
					download);
			
			if(strlen(buf) > strlen("SC-Marked-For-Download: x\n")
			&& !strncmp(buf, "SC-Marked-For-Download:", 
			            strlen("SC-Marked-For-Download:"))) {
				fprintf(fpnew, "%s", 
				 buf+strlen("SC-Marked-For-Download: x\n"));
				continue;
			}
		}
		fprintf(fpnew, "%s", buf);
	}
	fclose(fpnew);
	fclose(fp);
	unlink(filename);
	rename(pathnew, filename);
	g_free(pathnew);
	msginfo->planned_download = download;
	msgcache_update_msg(msginfo->folder->cache, msginfo);

	err = 0;
bail:
	g_free(filename);
	procmsg_msginfo_free(tinfo);
	
	return err;
}
 
int partial_mark_for_delete(MsgInfo *msginfo)
{
	return partial_uidl_mark_mail(msginfo, POP3_PARTIAL_DLOAD_DELE);
}

int partial_mark_for_download(MsgInfo *msginfo)
{
	return partial_uidl_mark_mail(msginfo, POP3_PARTIAL_DLOAD_DLOAD);
}

int partial_unmark(MsgInfo *msginfo)
{
	return partial_uidl_mark_mail(msginfo, POP3_PARTIAL_DLOAD_UNKN);
}

void partial_delete_old(const gchar *file) 
{
	gchar *id = g_strdup(file);
	gchar *snum = strrchr(file, ':');
	int num = 0;
	FolderItem *item = NULL;

	debug_print("too big message updated,should remove %s\n", file);

	if (snum) {
		snum++;
	} else {
		g_free(id);
		return; /* not a real problem */
	}

	num = atoi(snum);

	if (strrchr(id, ':'))
		*(strrchr(id, ':'))='\0';

	item = folder_find_item_from_identifier(id);
	if (item) {
		debug_print("removing %d in %s\n", num, id);
		folder_item_remove_msg(item, num);
	} 
	g_free(id);
}

gchar *partial_get_filename(const gchar *server, const gchar *login,
				   const gchar *muidl)
{
	gchar *path;
	gchar *result = NULL;
	FILE *fp;
	gchar buf[POPBUFSIZE];
	gchar uidl[POPBUFSIZE];
	time_t recv_time;
	time_t now;
	gint partial_recv;
	
	path = g_strconcat(get_rc_dir(), G_DIR_SEPARATOR_S,
			   "uidl", G_DIR_SEPARATOR_S, 
			   server, "-", login, NULL);
	if ((fp = fopen(path, "rb")) == NULL) {
		if (ENOENT != errno) FILE_OP_ERROR(path, "fopen");
		g_free(path);
		path = g_strconcat(get_rc_dir(), G_DIR_SEPARATOR_S,
				   "uidl-", server,
				   "-", login, NULL);
		if ((fp = fopen(path, "rb")) == NULL) {
			if (ENOENT != errno) FILE_OP_ERROR(path, "fopen");
			g_free(path);
			return result;
		}
	}
	g_free(path);

	now = time(NULL);

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		gchar tmp[POPBUFSIZE];
		strretchomp(buf);
		recv_time = RECV_TIME_NONE;
		partial_recv = POP3_TOTALLY_RECEIVED;
		
		if (sscanf(buf, "%s\t%ld\t%s", uidl, &recv_time, tmp) < 2) {
			if (sscanf(buf, "%s", uidl) != 1)
				continue;
			else {
				recv_time = now;
			}
		}
		if (!strcmp(muidl, uidl)) {
			result = g_strdup(tmp);
			break;
		}
	}

	fclose(fp);
	
	return result;
}

