/*****************************************************************
**
** MathSpad 0.60
**
** Copyright 1996, Eindhoven University of Technology (EUT)
** 
** Permission to use, copy, modify and distribute this software
** and its documentation for any purpose is hereby granted
** without fee, provided that the above copyright notice appear
** in all copies and that both that copyright notice and this
** permission notice appear in supporting documentation, and
** that the name of EUT not be used in advertising or publicity
** pertaining to distribution of the software without specific,
** written prior permission.  EUT makes no representations about
** the suitability of this software for any purpose. It is provided
** "as is" without express or implied warranty.
** 
** EUT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
** SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL EUT
** BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
** DAMAGES OR ANY DAMAGE WHATSOEVER RESULTING FROM
** LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
** OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
** OF THIS SOFTWARE.
** 
** 
** Roland Backhouse & Richard Verhoeven.
** Department of Mathematics and Computing Science.
** Eindhoven University of Technology.
**
********************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <memory.h>

#include "leaftree.h"

#define False 0
#define True 1
#define Leaf 0
#define Breanch 1

typedef struct { char kind; MT val; } dummytype;
#define MINSIZE (sizeof(dummytype))

static int leafsize=MINSIZE;
static void (*free_leaf)(void*);

void set_leafsize(int n)
{
    if (n<MINSIZE) leafsize=MINSIZE; else leafsize=n;
}

void set_free_leaf(void (*flf)(void*))
{
    free_leaf = flf;
}

static int newbaltop[15] = {-4,-3,-2,-2,-2,-3,-2,-1,-1,-1,-3,-2,-1,0,0};
static int newbalright[15] = {-1,-1,-1,-2,-3,0,0,0,-1,-2,1,1,1,0,-1};

static LeafTree* rrot(LeafTree* t)
{
    LeafTree* b;
    int i;
    b=t->left;
    t->left=b->right;
    b->right=t;
    i= -t->bal*5-b->bal+2;
    b->bal = -newbaltop[i];
    t->bal = -newbalright[i];
    return b;
}

static LeafTree* lrot(LeafTree* t)
{
    LeafTree* b;
    int i;
    b=t->right;
    t->right=b->left;
    b->left=t;
    i=t->bal*5+b->bal+2;
    b->bal = newbaltop[i];
    t->bal = newbalright[i];
    return b;
}

static LeafTree* rotate(LeafTree* t)
{
    /*
    ** rotate tree if out of balance
    */
    if (t->bal < -1) {
	if (t->left->bal>0) t->left=lrot(t->left);
	t=rrot(t);
    } else if (t->bal > 1) {
	if (t->right->bal<0) t->right=rrot(t->right);
	t=lrot(t);
    }
    return t;
}

#define NewNode ((LeafTree*) malloc(sizeof(LeafTree)))
#define NewLeaf ((LeafTree*) calloc(1,leafsize))
#define EmptyLeaf(A)  memset((A),0,leafsize)

void free_leaftree(LeafTree* t)
{
    if (!t) return;
    if (t->kind==Leaf) {
	if (free_leaf) (*free_leaf)((void*) t);
	free(t);
    } else {
	free_leaftree(t->left);
	free_leaftree(t->right);
	free(t);
    }
}

static int dchange=0;
static MT rmin;
static LeafTree* lastleaf=NULL;

LeafTree* insert(LeafTree* t, MT val)
{
    if (!t) {
	t=NewLeaf;
	t->kind=Leaf;
	t->val=val;
	lastleaf=t;
	dchange=1;
    } else if (t->kind==Breanch) {
	if (val< t->val) {
	    t->left = insert(t->left, val);
	    if (dchange) {
		t->bal=t->bal-1;
		dchange= (t->bal == -1);
		if (t->bal < -1) t=rotate(t);
	    }
	} else {
	    t->right = insert(t->right, val);
	    if (dchange) {
		t->bal=t->bal+1;
		dchange = (t->bal == 1);
		if (t->bal > 1) t=rotate(t);
	    }
	}
    } else {
	if (t->val == val) {
	    lastleaf=t;
	    dchange=0;
	} else {
	    LeafTree* l = NewLeaf;
	    LeafTree* b = NewNode;
	    l->kind=Leaf;
	    b->kind=Breanch;
	    b->bal=0;
	    lastleaf=l;
	    dchange=1;
	    l->val=val;
	    if (t->val< val) {
		b->left=t;
		b->right=l;
		b->val=val;
	    } else {
		b->left=l;
		b->right=t;
		b->val=t->val;
	    }
	    t=b;
	}
    }
    return t;
}

LeafTree* delete(LeafTree* t, MT val)
{
    if (!t) return NULL;
    if (t->kind==Leaf) {
	if (val==t->val) {
	    if (free_leaf) (*free_leaf)((void*)t);
	    free(t);
	    t=NULL;
	    dchange=1;
	} else
	    dchange=0;
    } else {
	if (val<t->val) {
	    t->left = delete(t->left, val);
	    if (dchange) {
		if (!t->left) {
		    LeafTree* b;
		    b=t;
		    t=t->right;
		    rmin=t->val;
		    free(b);
		} else {
		    t->bal = t->bal+1;
		    if (t->bal>1) {
			dchange = (t->right->bal!=0);
			t=rotate(t);
		    } else
			dchange = !t->bal;
		}
	    }
	} else {
	    t->right = delete(t->right, val);
	    if (t->val==val) t->val=rmin;
	    if (dchange) {
		if (!t->right) {
		    LeafTree* b;
		    b=t;
		    t=t->left;
		    free(b);
		    rmin=t->val;
		} else {
		    t->bal = t->bal-1;
		    if (t->bal<-1) {
			dchange = (t->left->bal!=0);
			t=rotate(t);
		    } else
			dchange = !t->bal;
		}
	    }
	}
    }
    return t;
}

LeafTree *add_leaftree(LeafTree *lt, LeafTree *ltadd)
{
    if (!ltadd) return lt;
    if (ltadd->kind==Breanch) {
	lt = add_leaftree(lt, ltadd->left);
	return add_leaftree(lt, ltadd->right);
    } else
	return insert(lt, ltadd->val);
}

int member(LeafTree* t, MT val)
{
    if (!t) {
	lastleaf=NULL;
	return False;
    }
    if (t->kind==Breanch)
	if (val<t->val)
	    return member(t->left, val);
	else
	    return member(t->right,val);
    else {
	if (t->val == val) lastleaf=t; else lastleaf=NULL;
	return t->val==val;
    }
}

void* found_leaf(void)
{
    return (void*) lastleaf;
}

#if 0
#define printval(A) printf("%d ", A)

void printnode(LeafTree* t)
{
    if (t->kind==Leaf)
        printval( t->val);
    else {
        printnode(t->left);
        printnode(t->right);
    }
}

char line[70]=
      "                                                                  -- ";

void shownode(LeafTree* t, int n)
{
    if (t->kind == Leaf) {
        printf("%s", line+(66-2*n));
        printval(t->val);
        printf("\n");
    } else {
        shownode(t->right,n+1);
        printf("%s[", line+(66-2*n));
	printval(t->val);
	printf("]\n");
        shownode(t->left, n+1);
    }
}


#define toJN(A) ('N'+('J'-'N')*(A))
int shown=0;

int main(int argc, char **argv)
{
    LeafTree* t;
    int i,j;
    char buffer[1000];
    t=NULL;
    free_leaftree(t);
    printf("> ");
    while ((fgets(buffer,1000,stdin))) {
        i=0;
        switch (buffer[0]) {
        case 'm':
            while (buffer[i]) {
                while (buffer[i] && buffer[i]!=' ') i++;
                while (buffer[i] && buffer[i]==' ') i++;
                if (sscanf(buffer+i,"%i",&j)==1)
                    printf("%c ", toJN(member(t,j)));
            }
            printf("\n");
            break;
        case 'i':
            while (buffer[i]) {
                while (buffer[i] && buffer[i]!=' ') i++;
                while (buffer[i] && buffer[i]==' ') i++;
                if (sscanf(buffer+i,"%i",&j)==1) t=insert(t,j);
            }
            break;
        case 'd':
            while (buffer[i]) {
                while (buffer[i] && buffer[i]!=' ') i++;
                while (buffer[i] && buffer[i]==' ') i++;
                if (sscanf(buffer+i,"%i",&j)==1) t=delete(t,j);
            }
            break;
	case 'D':
	    shown = !shown;
	    break;
        case 'p':
            if (t) printnode(t);
            break;
        case 's':
            if (t) shownode(t, 0);
            break;
        case 'c':
            free_leaftree(t);
            t=NULL;
            break;
        case 'q':
            printf("Thank you for testing this program.\n");
            free_leaftree(t);
            return 0;
            break;
        default:
            printf("Unknown command. Use one of the following:\n"
                   "\tm x\tmembership test for x\n"
                   "\ti x\tinsert x\n"
                   "\td x\tdelete x\n"
                   "\tD\ttoggle debug information\n"
                   "\tp\tshow contents\n"
                   "\ts\tshow tree structure\n"
                   "\tc\tclear the set\n"
                   "\tq\tquit\n");
            break;
        }
        printf("> ");
    }
    return 0;
}

#endif
