(*
 * Multimedia
 *)
open Unix
open Lexing
open Url
open Http_headers
open Document
open Feed
open Www

(* The context given to a viewer *)
(* hyper functions are: "goto", "save", "gotonew" *)
type vparams = (string * string) list
type context = {
  viewer_base : Document.document_id;
  viewer_hyper: (string * Hyper.func) list;
  viewer_log : string -> unit;
  viewer_params : vparams
  }

(* The object created/returned by a viewer *)
type display_info = {
  di_widget : Widget.widget;
  di_abort : unit -> unit;		(* stop display *)
  di_destroy : unit -> unit;		(* die *)
  di_fragment : string option -> unit;	(* for # URIs *)
  di_redisplay : unit -> unit;		(* redisplay *)
  di_title : unit -> string;		(* some visible title *)
  di_source : unit -> unit;	        (* source viewer *)
  di_load_images : unit -> unit;	(* load images *)
  mutable di_last_used : int
}


let di_compare di di' = di.di_last_used > di'.di_last_used

(* Table of viewers, according to media-type (MIME)
 * Actually, this is only for internal viewers, since the rest
 * will be passed to metamail.
 *)
let viewers = Hashtbl.create 17

let add_viewer ctype viewer =
  Hashtbl.add viewers ctype viewer
and rem_viewer ctype =
  Hashtbl.remove viewers ctype
(* 
 * The default external viewer
 *)

(* Metamail options
   -b : not an RFC822 message
   -z : delete when finished 
   -x : not on a tty 
 *)
let metamail ctype file =
  Munix.system "metamail -b -z -x -c" [ctype; file] true;
  ()

(* Batch version: we transfer everything and then run metamail *)
let extern_batch dh ctype = 
  let outfile = Msys.mktemp "mmm" in
  Document.add_log dh (
    I18n.sprintf "Saving %s\n for external display with MIME type %s"
	      (Url.string_of dh.document_id.document_url) ctype)
    (fun () -> Msys.rm outfile);
  let endmsg =
    I18n.sprintf "Running metamail with MIME media-type: %s" ctype in
    Save.tofile (metamail ctype) (Decoders.insert dh) outfile endmsg

(* "interactive" version: 
 *    send data to metamail as it arrives, but allow abort
 * NOTE: There are sometimes weird errors when the child dumps core
 *     	 between fork/exec with no apparent reason (on SunOS4.1 only)
 *)
let extern dh ctype =
  let (pin, pout) = pipe() in
  match Low.fork() with
    0 ->
      close pout; dup2 pin stdin; close pin;
      Munix.execvp "metamail" [| "metamail"; "-b"; "-x"; "-c"; ctype |]
  | pid ->  
      close pin;
      let kill () = 
         try Unix.kill pid 2
      	 with Unix_error (e,_,_) ->
	     Printf.eprintf "%s\n" (Unix.error_message e);
             flush Pervasives.stderr
      	      in
      let url = Url.string_of dh.document_id.document_url in
      Document.add_log dh (
	I18n.sprintf "Retrieving %s\n for external display with MIME type %s"
		     url ctype)
      	kill;

      let red = ref 0 
      and size =   
      	try Http_headers.contentlength dh.document_headers 
      	with Not_found -> 40000 (* duh *)
      and buffer = String.create 4096 in
      dh.document_feed.feed_schedule
      	(fun () ->
	  try
	   let n = dh.document_feed.feed_read buffer 0 4096 in
	   if n = 0 then begin
	       dclose true dh;
	       close pout;
	       Document.end_log dh (I18n.sprintf "End of transmission")
	       end
	   else begin
	     write pout buffer 0 n;
	     red := !red + n;
	     Document.progress_log dh (!red * 100 / size)
	     end
	  with
	   Unix_error(e,_,_) ->
	     Printf.eprintf "%s\n" (Unix.error_message e);
      	     flush Pervasives.stderr;
	     dclose true dh;
	     kill();
	     close pout;
	     Document.destroy_log dh false;
	     Error.default#f (I18n.sprintf "Error during retrieval of %s" url)
	   )

let rec unknown frame ctx dh =
  match Frx_dialog.f frame (Mstring.gensym "error")
         (I18n.sprintf "MMM Warning")
	 (I18n.sprintf
	   "No MIME type given for the document\n%s"
	   (Url.string_of dh.document_id.document_url))
         (Tk.Predefined "question") 0
	 [I18n.sprintf "Retry with type";
	  I18n.sprintf "Save to file";
	  I18n.sprintf "Abort"] with
   0 ->
    let v = Textvariable.create_temporary frame in
     Textvariable.set v "text/html";
     if Frx_req.open_simple_synchronous (I18n.sprintf "MIME type") v then
       let ctype = Textvariable.get v in
      	 dh.document_headers <- 
      	  ("Content-Type: " ^ ctype) :: dh.document_headers;
      	 view frame ctx dh
     else begin
       Save.interactive (fun _ -> ()) dh;
       None
     end
 | 1 ->
      Save.interactive (fun _ -> ()) dh; None
 | 2 -> dclose true dh; None

(* the meat *)
and view frame ctx dh =
  (* Decode. *)
  let ddh = Decoders.insert dh in
  try 
    let ctype = Mstring.lowercase (contenttype dh.document_headers) in
    try
	(* Get the viewer *)
	let (typ,sub),pars = Lexheaders.media_type ctype in
	let viewer = Hashtbl.find viewers (typ,sub) in
	ctx.viewer_log (I18n.sprintf "Displaying...");
      	viewer pars frame ctx ddh
    with
	Failure "too late" -> (* custom for our internal viewers *)
	    dclose true ddh;
	    Document.destroy_log ddh false;
	    None
      | Not_found -> (* we don't know how to handle this *)
	  ctx.viewer_log (I18n.sprintf "Displaying externally");
	  extern ddh ctype;
	  None
      | Invalid_HTTP_header e ->
	  ctx.viewer_log (I18n.sprintf "Malformed type: %s" ctype);
          unknown frame ctx ddh
  with 
    Not_found -> 
    (* Content-type was not defined in the headers *)
    (* and could not be computed from url *)
    unknown frame ctx ddh
