/* AX.25 Utilities: Listen on an interface, and execute a program
 *	when someone calls.
 *
 * Alan Cox, 1993, 1994
 * modified by Bruce Perens, November 1994
 *
 * Copyright 1994 Alan Cox.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/ax25.h>
#include <termios.h>

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>

#include "config.h"
#include "ax25_paths.h"
#include "global.h"

#define	MAX_ARGS	64

static char * *	environment;
static char *	myname;
static int	port = 0;
static char	prototype[1024] = CONFIG_PMS_PATH;
static int	use_pseudo_terminal = 0;

static char * const *
		build_argument_vector(char * command);
static void	build_command(const char *, char *, const char *, int);
void		do_accept(int fd);
static void	execute_on_pseudo_terminal(const char *, char * const *,
		 char * const *, int, const char *, int);
static void	process_arguments(int argc, char * * argv);
static int	sockaddr_to_callsign(const struct full_sockaddr_ax25 *, char *);
static void	usage();

extern int	bridge_add_special_file(int fd, void (*function)(int fd));
extern void	bridge_remove_file(int fd);
extern int	bridge_add_file_pair(int a, int b);
extern int	bridge_run();
extern int	execute(const char * file, char * const * argv
		,char * const * envp, int ioFile);
extern int	closest_termios_speed_encoding(int speed);
extern int	get_pseudo_tty(int * slave);

int
main(int argc, char * * argv, char * * env)
{
	struct full_sockaddr_ax25	ax;
	int				s;
	int				sz;
	
	myname = argv[0];

	if(config_load_ports()==-1)
	{
		fprintf(stderr,"%s: No ports configured.\n",argv[0]);
		exit(1);
	}

	process_arguments(argc, argv);

	s=socket(AF_AX25,SOCK_SEQPACKET,PF_AX25);
	if(s==-1)
	{
		perror("socket");
		exit(1);
	}
	
	sz=convert_call(config_get_addr(port),&ax);
		
	if(bind(s,(struct sockaddr *)&ax,sz)==-1)
	{
		perror("bind");
		exit(1);
	}
	
	if(listen(s,5)==-1)
	{
		perror("connect");
		exit(1);
	}
	
	bridge_add_special_file(s, do_accept);

	while ( bridge_run() )
		;

	close(s);

	return 0;
}

static char * const *
build_argument_vector(char * command)
{
	static char *	argv[MAX_ARGS + 1];
	int		arg;

	for ( arg = 0; arg < MAX_ARGS && *command != '\0' ; ) {
		argv[arg++] = command;
		while ( *command != '\0' && !isspace(*command) )
			command++;
		if ( *command == '\0' )
			break;
		else
			*command++ = '\0';
	}
	argv[arg] = 0;
	return argv;
}

static void
build_command(
 const char *	prototype
,char *		command
,const char *	call
,int		port)
{
	char *		s = command;
	const char *	p = prototype;

	while ( *p != '\0' ) {
		if ( *p != '%' ) {
			*s++ = *p++;
			continue;
		}

		*s = '\0';

		switch ( *++p ) {
		case 'c':	/* %c = My address */
		case 'C':
			strcpy(s, config_get_addr(port));
			if ( *p == 'C' ) {
				char *	x = strrchr(s, '-');
				if ( x != 0 && x != s )
					*x = '\0';
			}
			p++;
			break;
		case 'f':	/* %f = Frequency */
			strcpy(s, config_get_freq(port));
			p++;
			break;
		case 'r':	/* %r = Remote station's address */
		case 'R':
			p++;
			strcpy(s, call);
			if ( *p == 'R' ) {
				char *	x = strrchr(s, '-');
				if ( x != 0 && x != s )
					*x = '\0';
			}
			break;
		case 's':	/* %s = Speed */
			p++;
			sprintf(s, "%d", config_get_baud(port));
			break;
		case 'w':	/* %w = Window */
			p++;
			sprintf(s, "%d", config_get_window(port));
			break;
		case '\0':
		default:
			*s++ = '%';
			*s++ = *p++;
			*s++ = '\0';
			break;
		}
		while ( *s != '\0' )	
			s++;
	}
	*s = '\0';
}

void
do_accept(int fd)
{
	struct full_sockaddr_ax25	address;
	char * const *			arguments;
	int	length = sizeof(address);
	int	new;
	char	call[10];
	char	command[2048];
	int	window = config_get_window(port);

	memset(&address, 0, sizeof(address));
	errno = 0;
	new = accept(fd, (struct sockaddr *)&address, &length);

	if(new < 0) {
		if ( errno == EAGAIN )
			return;

		perror("accept");
		return;
	}

	setsockopt(new, SOL_AX25, AX25_WINDOW, &window, sizeof(window));

	sockaddr_to_callsign(&address, call);

	build_command(prototype, command, call, port);

	arguments = build_argument_vector(command);

	if ( !use_pseudo_terminal ) {
		execute(command, arguments, environment, new);
		close(new);
	}
	else
		execute_on_pseudo_terminal(
		 command
		,arguments
		,environment
		,new
		,call
		,port);
}

static void
execute_on_pseudo_terminal(
 const char *	command
,char * const *	arguments
,char * const *	environment
,int		radio_port
,const char *	call
,int		port)
{
	int	pseudo_tty_slave;
	int	pseudo_tty_master = get_pseudo_tty(&pseudo_tty_slave);
	int	speed;
	struct termios	t;

	if ( pseudo_tty_master < 0 )
		return;

	if ( tcgetattr(pseudo_tty_slave, &t) < 0 )
		return;

	speed = config_get_baud(port);
	if ( speed <= 0 )
		speed = 1200;
	speed = closest_termios_speed_encoding(speed);
	t.c_lflag &= ~ECHO;

	cfsetispeed(&t, speed);
	cfsetospeed(&t, speed);

	if ( tcsetattr(pseudo_tty_slave, TCSANOW, &t) < 0 )
		return;

	execute(command, arguments, environment, pseudo_tty_slave);
	close(pseudo_tty_slave);
	bridge_add_file_pair(radio_port, pseudo_tty_master);
}

static void
process_arguments(int argc, char * * argv)
{
	while (argc>1 && argv[1][0] == '-')
	{
		switch (argv[1][1]) {
		case 'c':
			strncpy(prototype, argv[2], sizeof(prototype));
			argv+=2;
			argc-=2;
			break;
		case 'p':
			if(argc < 3)
			{
				fprintf(stderr,"%s: -p option requires an argument.\n",argv[0]);
				exit(1);
			}
			if(sscanf(argv[2],"%d",&port)!=1)
			{
				fprintf(stderr,"%s: -p option requires a numeric argument.\n",argv[0]);
				exit(1);
			}
			port--;
			if(port<0||port>=config_num_ports())
			{
				fprintf(stderr,"%s: Unknown port.\n",argv[0]);
				exit(1);
			}
			argv+=2;
			argc-=2;
			break;
		case 't':
			use_pseudo_terminal = 1;
			argv++;
			argc--;
			break;
		default:
			usage();
		}
	}
	if(argc!=1)
		usage();
}

static int
sockaddr_to_callsign(const struct full_sockaddr_ax25 * address, char * c)
{
	const char *	a = (char *)&(address->fsa_ax25.sax25_call);
	int		n;
	int		ssid;

	for ( n = 0; n < 6; n++ ) {
		int x = (a[n] >> 1) & 0x7F;

		if ( isupper(x) )
			c[n] = tolower(x);
		else
			c[n] = x;

		if ( a[n] & 1 || (a[n+1] >> 1) == ' ' )
			break;
	}

	ssid = ((int)(a[6] >> 1)) - '0';
	if ( ssid >= 0 && ssid <= 15 ) {
		c[++n] = '-';
		if ( ssid >= 10 )
			c[++n] = '1';
		c[++n] = '0' + (ssid % 10);
		c[++n] = '\0';
	}

	return 1;
}

static void
usage()
{
	static const char	message[] =
	"\n"
	"Usage: %s [-c <command>] [-p <port>] [-t].\n"
	"\n"
	"\t-c <command>: <command> will be executed for each connection.\n"
	"\t\tThe default command is \"%s\".\n"
	"\t\tParameters can be passed in <command> by using\n"
	"\t\tthese \"%%\" patterns:\n"
	"\t\t\t%%c\tLocal address (callsign+extension).\n"
	"\t\t\t%%C\tLocal address (callsign only).\n"
	"\t\t\t%%f\tFrequency.\n"
	"\t\t\t%%r\tRemote address (callsign+extension).\n"
	"\t\t\t%%R\tRemote address (callsign only).\n"
	"\t\t\t%%s\tSpeed.\n"
	"\t\t\t%%w\tWindow.\n"
	"\t\t\t%%%%\tA single %% character.\n"
	"\n"
	"\t-p <port>: Listen on the specified port.\n"
	"\n"
	"\t-t:\tRun <command> on a pseudo-terminal. Required for\n"
	"\t\tterminal-oriented commands such as \"/bin/login\".\n"
	"\n";

	fprintf(stderr, message, myname, CONFIG_PMS_PATH);
	exit(1);
}
