// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
extern "C" {
#include <ctype.h>
}
#include <qcombo.h>
#include <qlabel.h>
#include <qlined.h>
#include <qlistbox.h>
#include <qmsgbox.h>
#include <qpushbt.h>
#include "Options.h"
#include "SgmlCatalog.h"

SgmlCatalog::SgmlCatalog()
        : QDialog( 0, 0, TRUE )
{
    _entries.setAutoDelete( TRUE );

    // Check for a NULL catalog file name.
    if ( options->catalogFile().length() == 0 ) {
        QString error;
        error.sprintf( "No SGML Catalog file has been specified.\nPlease correct this in Preferences, and restart QWeb." );
        QMessageBox::message( "QWeb: Error", error );
        return;
    }
    
    // Read in the SGML catalog.
    FILE* fptr = fopen( options->catalogFile(), "r" );
    QString token;
	
    if ( fptr ) {
        while ( !feof( fptr ) ) {
            token = getToken( fptr );
            if ( token == "DOCTYPE" ) {
                _entries.append( new CatalogEntry( CatalogEntry::Doctype, getToken( fptr ), getToken( fptr ) ) );
            } else if ( token == "ENTITY" ) {
                _entries.append( new CatalogEntry( CatalogEntry::Entity, getToken( fptr ), getToken( fptr ) ) );
            } else if ( token == "STYLE" ) {
                _entries.append( new CatalogEntry( CatalogEntry::Style, getToken( fptr ), getToken( fptr ) ) );
            }
        }
        fclose( fptr );
    } else {
        QString error;
        error.sprintf( "Cannot open SGML Catalog file '%s'.\nPlease correct this in Preferences, and restart QWeb.", options->catalogFile().data() );
        QMessageBox::message( "QWeb: Error", error );
    }

    // Create the dialog.
    setCaption( "QWeb: SGML Catalog" );
    setIconText( "QWeb: SGML Catalog" );

    _list = new QListBox( this );
    connect( _list, SIGNAL( highlighted( int ) ), this, SLOT( doHighlighted( int ) ) );
    
    CatalogEntry* e;
    for ( e = _entries.first(); e; e = _entries.next() ) {
        _list->inSort( entryToString( e ) );
    }

    _typeLabel = new QLabel( "Type:", this );
    _typeLabel->setAlignment( AlignRight | AlignVCenter );

    _typeCombo = new QComboBox( this );
    _typeCombo->insertItem( "Doctype" );
    _typeCombo->insertItem( "Entity" );
    _typeCombo->insertItem( "Style" );

    _publicIdLabel = new QLabel( "Public ID:", this );
    _publicIdLabel->setAlignment( AlignRight | AlignVCenter );

    _publicIdEntry = new QLineEdit( this );

    _storageIdLabel = new QLabel( "Storage Object ID:", this );
    _storageIdLabel->setAlignment( AlignRight | AlignVCenter );

    _storageIdEntry = new QLineEdit( this );

    _separator = new QFrame( this );
    _separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );

    _addButton = new QPushButton( "Add", this );
    connect( _addButton, SIGNAL( clicked() ), this, SLOT( doAdd() ) );
    
    _updateButton = new QPushButton( "Update", this );
    connect( _updateButton, SIGNAL( clicked() ), this, SLOT( doUpdate() ) );

    _deleteButton = new QPushButton( "Delete", this );
    connect( _deleteButton, SIGNAL( clicked() ), this, SLOT( doDelete() ) );

    _closeButton = new QPushButton( "Close", this );
    connect( _closeButton, SIGNAL( clicked() ), this, SLOT( accept() ) );

    resize( 500, 400 );
}

SgmlCatalog::~SgmlCatalog()
{
    // Make g++ happy.
}

CatalogEntry* SgmlCatalog::find( CatalogEntry::Type type, QString id )
{
    CatalogEntry* e;

    for ( e = _entries.first(); e; e = _entries.next() ) {
        if ( ( e->type() == type ) && ( id == e->id() ) ) {
            return e;
        }
    }
    return 0;
}

QString SgmlCatalog::getToken( FILE* fp ) const
{
    QString tok;
    char ch = 0;
    char delim = 0;
	
    while ( !feof( fp ) && isspace( ch = getc( fp ) ) );
	
    switch ( ch ) {
        case '"':
            delim = ch;
            break;
		
        case '\'':
            delim = ch;
            break;
		
        default:
            tok += ch;
    }

    while ( !feof( fp ) ) {
        ch = getc( fp );
        if ( delim ) {
            if ( ch == delim ) {
                return tok;
            } else {
                tok += ch;
            }
        } else {
            if ( isspace( ch ) ) {
                return tok;
            } else {
                tok += ch;
            }
        }
    }
    return tok;
}

void SgmlCatalog::resizeEvent( QResizeEvent* )
{
    const int pad         = 4;
    const int buttonWidth = 50;
    const int comboWidth  = 100;
    const int labelWidth  = _storageIdLabel->sizeHint().width();
    const int lineHeight  = 24;
    const int listHeight  = height() - 8 * pad - 4 * lineHeight;
    const int entryWidth  = width() - 3 * pad - labelWidth;

    _list->setGeometry( pad, pad, width() - 2 * pad, listHeight );

    _typeLabel->setGeometry     ( pad, 2 * pad + listHeight                 , labelWidth, lineHeight );
    _publicIdLabel->setGeometry ( pad, 3 * pad + listHeight +     lineHeight, labelWidth, lineHeight );
    _storageIdLabel->setGeometry( pad, 4 * pad + listHeight + 2 * lineHeight, labelWidth, lineHeight );

    _typeCombo->setGeometry     ( 2 * pad + labelWidth, 2 * pad + listHeight                 , comboWidth, lineHeight );
    _publicIdEntry->setGeometry ( 2 * pad + labelWidth, 3 * pad + listHeight +     lineHeight, entryWidth, lineHeight );
    _storageIdEntry->setGeometry( 2 * pad + labelWidth, 4 * pad + listHeight + 2 * lineHeight, entryWidth, lineHeight );

    _separator->setGeometry( 0, 5 * pad + listHeight + 3 * lineHeight, width(), pad );

    _addButton->setGeometry   ( pad                        , 7 * pad + listHeight + 3 * lineHeight, buttonWidth, lineHeight );
    _updateButton->setGeometry( 2 * pad +     buttonWidth  , 7 * pad + listHeight + 3 * lineHeight, buttonWidth, lineHeight );
    _deleteButton->setGeometry( 3 * pad + 2 * buttonWidth  , 7 * pad + listHeight + 3 * lineHeight, buttonWidth, lineHeight );
    _closeButton->setGeometry ( width() - pad - buttonWidth, 7 * pad + listHeight + 3 * lineHeight, buttonWidth, lineHeight );
}

void SgmlCatalog::doAdd()
{
    QString id  = _publicIdEntry->text();
    QString soi = _storageIdEntry->text();
    
    if ( ( id.length() == 0 ) || ( soi.length() == 0 ) ) {
        QMessageBox::message( "QWeb: Error", "You must supply a Public ID and a Storage Object ID." );
        return;
    }

    // Search for an existing entry with the same public ID and type.
    int                val = _typeCombo->currentItem();
    CatalogEntry::Type t = CatalogEntry::Doctype;
    switch ( val ) {
        case 0:
            t = CatalogEntry::Doctype;
            break;
        case 1:
            t = CatalogEntry::Entity;
            break;
        case 2:
            t = CatalogEntry::Style;
            break;
    }
    
    CatalogEntry* e;
    for ( e = _entries.first(); e; e = _entries.next() ) {
        if ( ( e->type() == t ) && ( e->id() == id ) ) {
            break;
        }
    }

    if ( e ) {
        // We've got a duplicate.
        QString q;
        q.sprintf( "There is already an entry in the catalog with a public identifier of '%s'.\nWould you like to update it instead?", id.data() );
        bool ok = QMessageBox::query( "QWeb: Duplicate Entry", q );
        if ( ok ) {
            doUpdate();
        }
        return;
    }

    // Add the entry to the list.
    e = new CatalogEntry( t, id, soi );
    _entries.append( e );

    _list->inSort( entryToString( e ) );
}

void SgmlCatalog::doDelete()
{
    // Check for a selected item.
    int index = _list->currentItem();
    if ( index == -1 ) {
        QMessageBox::message( "QWeb: Error", "You must select an entry to delete." );
        return;
    }
    
    // Find the entry in the catalog.
    int     type;
    QString id, soi;
    stringToEntry( _list->text( index ), type, id, soi );

    CatalogEntry* e;
    for ( e = _entries.first(); e; e = _entries.next() ) {
        if ( ( e->type() == type ) && ( e->id() == id ) ) {
            break;
        }
    }

    if ( e ) {
        _entries.remove( e );
        _list->removeItem( index );
    }
}

void SgmlCatalog::doUpdate()
{
    QString id  = _publicIdEntry->text();
    QString soi = _storageIdEntry->text();
    
    if ( ( id.length() == 0 ) || ( soi.length() == 0 ) ) {
        QMessageBox::message( "QWeb: Error", "You must supply a Public ID and a Storage Object ID." );
        return;
    }

    // Search for an existing entry with the same public ID and type.
    int                val = _typeCombo->currentItem();
    CatalogEntry::Type t = CatalogEntry::Doctype;
    switch ( val ) {
        case 0:
            t = CatalogEntry::Doctype;
            break;
        case 1:
            t = CatalogEntry::Entity;
            break;
        case 2:
            t = CatalogEntry::Style;
            break;
    }
    
    CatalogEntry* e;
    for ( e = _entries.first(); e; e = _entries.next() ) {
        if ( ( e->type() == t ) && ( e->id() == id ) ) {
            break;
        }
    }

    if ( !e ) {
        // No existing entry.
        QString q;
        q.sprintf( "There is no entry in the catalog with a public identifier of '%s'.\nWould you like to add it instead?", id.data() );
        bool ok = QMessageBox::query( "QWeb: Nonexistant Entry", q );
        if ( ok ) {
            doAdd();
        }
        return;
    }

    // Replace the entry in the list.
    e->soi( soi );

    // Check for a selected item.
    QString item = entryToString( e );
    uint    index;
    for ( index = 0; index < _list->count(); index++ ) {
        if ( !strncmp( _list->text( index ), item, 48 ) ) {
            break;
        }
    }

    if ( index < _list->count() ) {
        _list->removeItem( index );
        _list->inSort( item );
    } else {
        QMessageBox::message( "QWeb: Confused", "Cannot find row in table to update?!?." );
        return;
    }
}

void SgmlCatalog::doHighlighted( int index )
{
    int     type;
    QString id, soi;
    
    stringToEntry( _list->text( index ), type, id, soi );

    switch ( type ) {
        case CatalogEntry::Doctype:
            _typeCombo->setCurrentItem( 0 );
            break;
        case CatalogEntry::Entity:
            _typeCombo->setCurrentItem( 1 );
            break;
        case CatalogEntry::Style:
            _typeCombo->setCurrentItem( 2 );
            break;
    }

    _publicIdEntry->setText( id );

    _storageIdEntry->setText( soi );
}

QString SgmlCatalog::entryToString( const CatalogEntry* e )
{
    const char* type = 0;
    QString     item;
    switch ( e->type() ) {
        case CatalogEntry::Doctype:
            type = "DOCTYPE";
            break;
        case CatalogEntry::Style:
            type = "STYLE";
            break;
        case CatalogEntry::Entity:
            type = "ENTITY";
            break;
    }
    QString id = e->id().copy();
    id.prepend( "\"" );
    id.append( "\"" );
    item.sprintf( "%-7s %-40s %s", type, id.data(), e->soi().data() );

    return item;
}

void SgmlCatalog::stringToEntry( const QString str,
                                 int&          type,
                                 QString&      id,
                                 QString&      soi )
{
    QString tstr = str.left( 7 );

    id  = str.mid( 8, 40 );
    soi = str.right( str.length() - 47 );

    tstr = tstr.stripWhiteSpace();
    if ( tstr == "DOCTYPE" ) {
        type = CatalogEntry::Doctype;
    } else if ( tstr == "ENTITY" ) {
        type = CatalogEntry::Entity;
    } else if ( tstr == "STYLE" ) {
        type = CatalogEntry::Style;
    }

    id = id.stripWhiteSpace();
    if ( id.left( 1 )  == "\"" ) id.remove( 0, 1 );
    if ( id.right( 1 ) == "\"" ) id.remove( id.length()-1, 1 );
    id = id.stripWhiteSpace();
    
    soi = soi.stripWhiteSpace();
}

void SgmlCatalog::save()
{
    FILE* fptr = fopen( options->catalogFile(), "w" );

    if ( !fptr ) {
        QString err;
        err.sprintf( "Cannot open SGML catalog file '%s' for writing.", options->catalogFile().data() );
        QMessageBox::message( "QWeb: Error", err );
        return;
    }

    CatalogEntry* e;
    for ( e = _entries.first(); e; e = _entries.next() ) {
        fprintf( fptr, entryToString( e ).data() );
        fprintf( fptr, "\n" );
    }

    fclose( fptr );
}
