// Copyright (c) 1997-1998 Karl M. Hegbloom <karlheg@debian.org>
//
//  Originally Based on `tmpwatch-1.2/1.4' RHS, Erik Troan <ewt@redhat.com>
//
// This program may be freely redistributed under the terms of the GNU
// Public License.  You should be able to find a copy of the GPL in
// your "/usr/doc/copyright" directory on most GNU/Linux installations.

#include <dirent.h>
#include <errno.h>
#include <getopt.h>
#include <glob.h>
#include <linux/limits.h>
#include <malloc.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

// tmpreaper.c -- remove files in a directory, but do it carefully

#define FLAGS_FORCE		(1 << 0) // `--force': also rm files that are mode `-w EUID $$' 
#define FLAGS_FORCE_P(fl)	((fl) & FLAGS_FORCE)

#define FLAGS_MTIME		(1 << 1) // '--mtime': Use `mtime' rather than `atime'.
#define FLAGS_MTIME_P(fl)	((fl) & FLAGS_MTIME)

#define FLAGS_MTIME_MT_DIR	(1 << 2) // `--mtime-mt-dir': rm empty directories based on mtime
#define FLAGS_MTIME_MT_DIR_P(fl) ((fl) & FLAGS_MTIME_MT)

#define FLAGS_SYMLINKS		(1 << 3) // `--symlinks': also rm symlinks like files and directories
#define FLAGS_SYMLINKS_P(fl)	((fl) & FLAGS_SYMLINKS)

#define FLAGS_ALLFILES		(1 << 4) // `--all': also rm symlinks, fifos, devices, and sockets...
#define FLAGS_ALLFILES_P(fl)	((fl) & FLAGS_ALLFILES)

#define FLAGS_PROTECT		(1 << 5) // the `--protect' option was given
#define FLAGS_PROTECT_P(fl)	((fl) & FLAGS_PROTECT)

#define FLAGS_TEST		(1 << 6) // the `--test' option was given
#define FLAGS_TEST_P(fl)	((fl) & FLAGS_TEST)


#define TIME_SECONDS	1
#define TIME_MINUTES	60
#define TIME_HOURS	60 * 60
#define TIME_DAYS	24 * 60 * 60



glob_t	protect_glob; // list of files to not rm eg:

typedef struct protect_entry
{
    ino_t  inode;
    char * name;
} protect_entry;

protect_entry * protect_table;	// Global


#define LOG_REALDEBUG	1
#define LOG_DEBUG	2
#define LOG_VERBOSE	3
#define LOG_NORMAL	4
#define LOG_ERROR	5
#define LOG_FATAL	6

int	logLevel = LOG_NORMAL;

void
message (const int    level,
	 const char * format,
	 ...) // varargs
{
    va_list args;
    FILE  * where = stderr;
    
    if (level >= logLevel) {
	va_start (args, format);

	switch (level) {
	case LOG_DEBUG:
	    where = stdout;
	case LOG_REALDEBUG:
	    fprintf (where, "debug: ");
	    break;

	case LOG_ERROR:
	case LOG_FATAL:
	    fprintf (where, "error: ");
	    break;

	case LOG_NORMAL:
	case LOG_VERBOSE:
	    where = stdout;
	    break;

	default:
	    break;
	}

	vfprintf (where, format, args);

	if (level == LOG_FATAL) exit (1);
    }
}

int
safe_chdir (const char * dirname)
{
    struct stat sb1, sb2;

    if (lstat (dirname, &sb1)) {
	message (LOG_ERROR,
		 "lstat() of directory `%s' failed. : %s\n",
		 dirname, strerror (errno));
	return 1;
    }
    if (! S_ISDIR (sb1.st_mode)) {
	if (S_ISLNK (sb1.st_mode)) { // gnu/types.h:47:typedef __u_long __ino_t;
	    message (LOG_ERROR,
		     "safe_chdir(): Will not chdir across symlink `%s' (inode %lu).\n",
		     dirname, (u_long) sb1.st_ino);
	    return 1;
	} 
	message (LOG_ERROR,
		 "Not a directory: `%s' (inode %lu).\n",
		 dirname, (u_long) sb1.st_ino);
	return 1;
    }

    message (LOG_DEBUG,
	     "safe_chdir() : Before chdir(), `dirname' is inode (%lu).\n",
	     (u_long) sb1.st_ino);
    message (LOG_DEBUG,
	     "safe_chdir() : Before chdir(), `dirname' on device `%lu'.\n",
	     (u_long) sb1.st_dev);

    if (chdir (dirname)) {
	message (LOG_ERROR,
		 "chdir() to directory `%s' (inode %lu), failed. : %s\n",
		 dirname, (u_long) sb1.st_ino, strerror (errno));
	return 1;
    }
    if (lstat (".", &sb2)) {
	message (LOG_ERROR,
		 "lstat() of directory `%s' after chdir() failed. : %s\n",
		 dirname, strerror (errno));
	return 1;
    }

    message (LOG_DEBUG,
	     "safe_chdir() : After chdir(), `dirname' is inode (%lu).\n",
	     (u_long) sb2.st_ino);
    message (LOG_DEBUG,
	     "safe_chdir() : After chdir(), `dirname' on device `%lu'.\n",
	     (u_long) sb2.st_dev);

    if (sb1.st_ino != sb2.st_ino) {
	message (LOG_ERROR,
		 "Inode mismatch!!! : `%s' (inode %lu) != `.' (inode %lu)\n",
		 dirname,
		 (u_long) sb1.st_ino, (u_long) sb2.st_ino);
	message (LOG_FATAL,
		 "This indicates a possible subversion attempt.\n");
	// NOT_REACHED
    }
    else if (sb1.st_dev != sb2.st_dev) {
	message (LOG_ERROR,
		 "Device mismatch!!!\n");
	message (LOG_ERROR,
		 "`%s' (inode %lu), device `%lu' != `.' (inode %lu), device `%lu'\n",
		 dirname,
		 (u_long) sb1.st_ino, (u_long) sb1.st_dev,
		 (u_long) sb2.st_ino, (u_long) sb2.st_dev);
	message (LOG_FATAL,
		 "This indicates a possible subversion attempt.\n");
	// NOT_REACHED
    }
    return 0;
}

int
dir_empty_p (const char * dirname)
{
    DIR * dir;
    struct dirent * ent;
    int count = 0;

    if (! (dir = opendir (dirname))) {
	message (LOG_ERROR,
		 "Cannot opendir(%s).\n",
		 dirname);
	return 0;
    }
    while ((ent = readdir (dir))) {
	if (((ent->d_name[0] == '.') && (ent->d_name[1] == '\0')) ||
	    ((ent->d_name[1] == '.') && (ent->d_name[2] == '\0')))
	    continue;
	
	count++;
    }
    return count != 0 ? 0 : 1;	/* #t when empty. */
}

int
cleanupDirectory (const char	   * dirname,
		  const unsigned int killTime,
		  const int	     flags)
{
    DIR		  * dir;
    struct dirent * ent;
    struct stat	    sb, here;
    int		    status, pid, skip = 0;
    int		    i;

    if (FLAGS_TEST_P (flags)) {
	message (LOG_VERBOSE,
		 "Pretending to clean up directory `%s'.\n",
		 dirname);
    }
    else {
	message(LOG_DEBUG,
		"Cleaning up directory `%s'.\n",
		dirname);
    }

    // Do everything in a child process so we don't have to chdir(".."),
    // which would lead to a race condition. fork() on Linux is very efficient
    // so this shouldn't be a big deal (probably just a exception on one page
    // of stack, not bad). I should probably just keep a directory stack
    // and fchdir() back up it, but it's not worth changing now. - ewt
    //
    // After reading "http://www.geek-girl.com/bugtraq/1996_2/0054.html" (look for
    // "cd .."), I think it is best to use this method rather than the contrasting
    // algorithm mentioned in Eric's comment. IMHO he has made the right choice by
    // having chosen the `recursive forking' algorithm.  It is much easier to
    // implement and understand as well as being more secure through avoiding
    // `cd ..'. - karlheg

    if (! (pid = fork ()))
	{
	    if (safe_chdir (dirname)) return 1;

	    if (! (dir = opendir ("."))) {
		message (LOG_FATAL,
			 "Opening directory, `%s' as `.' : %s\n",
			 dirname, strerror (errno));
	    }

	    if (lstat (".", &here)) {
		message (LOG_FATAL,
			 "Statting current directory, `%s' as `.' : %s\n",
			 dirname, strerror (errno));
	    }

	    do {
		errno = 0;
		ent = readdir (dir);
		if (errno) {
		    message (LOG_ERROR,
			     "Reading directory entry. : %s\n",
			     strerror (errno));
		}
		if (!ent) break;

		if (((ent->d_name[0] == '.') && (ent->d_name[1] == '\0')) ||
		    ((ent->d_name[1] == '.') && (ent->d_name[2] == '\0')))
		    continue;

		message (LOG_DEBUG,
			 "(PID %u) Found directory entry `%s'.\n",
			 (u_int) getpid(), ent->d_name);

		if (lstat (ent->d_name, &sb)) {
		    message (LOG_ERROR,
			     "Failed to lstat() `%s/%s'. : %s\n",
			     dirname, ent->d_name, strerror (errno));
		    continue;
		}
		if (((!getuid() && !sb.st_uid) || (sb.st_uid == geteuid())) &&
		    !FLAGS_FORCE_P (flags) && !(sb.st_mode & S_IWUSR)) {
		    message (LOG_VERBOSE,
			     "Non-writeable file, owned by UID (%u) skipped. `%s'\n",
			     (u_int) sb.st_uid, ent->d_name);
		    continue;
		}
		if (sb.st_dev != here.st_dev) {
		    message (LOG_VERBOSE,
			     "File on different device skipped. `%s/%s'\n",
			     dirname, ent->d_name);
		    continue;
		}

		if (S_ISDIR (sb.st_mode)) {
		    cleanupDirectory (ent->d_name, killTime, flags);
		    message (LOG_VERBOSE,
			     "Back from recursing down `%s'.\n",
			     ent->d_name);
		}

		if (FLAGS_PROTECT_P (flags)) {
		    skip = i = 0;
		    do {
			if (sb.st_ino == protect_table[i].inode) {
			    message (LOG_VERBOSE,
				     "Entry matching `--protect' pattern skipped. `%s'\n",
				     protect_table[i].name);
			    skip = 1;
			    break;
			}
		    } while (protect_table[i++].name);
		    if (skip)
			continue;
		}

		// Decide whether to remove the file or not.
		if (FLAGS_MTIME_P (flags)) {
		    if (sb.st_mtime >= killTime) continue;
		}
		else {
		    if (sb.st_atime >= killTime) continue;
		}

		if (S_ISDIR (sb.st_mode)) {
		    if (FLAGS_TEST_P (flags)) {
			message (LOG_VERBOSE,
				 "Pretending to maybe remove possibly empty directory `%s'.\n",
				 ent->d_name);
		    }
		    else {
			if (dir_empty_p (ent->d_name)) {
			    message (LOG_VERBOSE,
				     "Removing directory `%s'.\n",
				     ent->d_name);
			    if (rmdir (ent->d_name))
				message (LOG_ERROR,
					 "Failed to rmdir `%s'. : %s\n",
					 ent->d_name, strerror (errno));
			}
		    }
		}
		else if (FLAGS_ALLFILES_P (flags) || S_ISREG (sb.st_mode)
			 || (FLAGS_SYMLINKS_P (flags) && S_ISLNK (sb.st_mode))) {
		    if (FLAGS_TEST_P (flags)) {
			message (LOG_VERBOSE,
				 "Pretending to remove file `%s/%s'.\n",
				 dirname, ent->d_name);
		    }
		    else {
			message (LOG_VERBOSE,
				 "Removing file `%s/%s'.\n",
				 dirname, ent->d_name);
			if (unlink (ent->d_name))
			    message (LOG_ERROR,
				     "Failed to unlink `%s'. : %s\n",
				     ent->d_name, strerror (errno));
		    }
		}
		else {
		    if (FLAGS_SYMLINKS_P (flags)) {
			message (LOG_VERBOSE,
				 "Not a regular file, symlink, or directory `%s' -- skipping.\n",
				 ent->d_name);
		    }
		    else {		    
			message (LOG_VERBOSE,
				 "Not a regular file or directory `%s' -- skipping.\n",
				 ent->d_name);
		    }
		}
	    } while (ent);
	    
	    closedir (dir);
	    
	    exit (0);
	}

    waitpid (pid, &status, 0);

    if (WIFEXITED (status))
	return WEXITSTATUS (status);

    return 0;
}

void
printCopyright (void)
{
    fprintf (stderr,
	     "tmpreaper -- Version: " VERSION "\n"
	     "(c) 1997 Karl M. Hegbloom <karlheg@debian.org>\n"
	     "This may be freely redistributed under the terms of the GNU Public License.\n");
}
//	               1         2         3         4         5         6         7         8
//	      02345678901234567890123456789012345678901234567890123456789012345678901234567890
void
usage(void)
{
    printCopyright ();
    fprintf (stderr, "\n"
	     "tmpreaper [-htvfmsa] [--help] [--test] [--verbose] [--force] [--mtime]\n"
	     "[--symlinks] [--all] [[--protect '<shell_pattern>']...] <time> <dirs>...\n"
	     "<time> is time since a file in a <dir> was last accessed.\n"
	     "It defaults to hours, or you may suffix with `s', `m', `h', or `d'.\n\n");
    exit (1);
}

int
main (int     argc,
      char ** argv)
{
    unsigned int  grace = 0;
    unsigned int  killTime = 0, long_index = 0;
    int		  ret = 0, arg = 0, i = 0, j = 0, k = 0, adir = 0;
    int		  flags = 0;
    char	* pwd = NULL;
    char        **pp;

    struct stat	  sb;

    // Utilize GNU C dynamic array allocation and statement
    // expressions to allocate an array to hold the --protect argument
    // strings.
    char *protect_argv[ ({ int i;
			   int count = 0;
			   for (i = 1; i < argc; i++)
			       if (! strncmp (argv[i], "--protect", 9))
				   count++;
			   ++count; }) ];

    char **p = protect_argv;

    struct option const options[] = {
	{ "help",    no_argument,	NULL, 'h' },
	{ "test",    no_argument,	NULL, 't' },
	{ "verbose", no_argument,	NULL, 'v' },
	{ "force",   no_argument,	NULL, 'f' },
	{ "mtime",   no_argument,	NULL, 'm' },
	{ "symlinks",no_argument,	NULL, 's' },
	{ "all",     no_argument,	NULL, 'a' },
	{ "protect", required_argument, NULL, 'p' },
	{ 0, 0, 0, 0 }
    };

    if (argc == 1) usage ();

    while (1) {
	long_index = 0;

	arg = getopt_long (argc, argv, "htvfmsa:", options, &long_index);
	if (arg == -1) break;

	switch (arg) {
	case '?':
	case 'h':
	    usage ();
	    break;

	case 't':
	    flags |= FLAGS_TEST;
	    // FALL_THROUGH

	case 'v':
	    logLevel ? logLevel -= 1 : 0;
	    break;

	case 'f':
	    flags |= FLAGS_FORCE;
	    break;

	case 'm':
	    flags |= FLAGS_MTIME;
	    break;

	case 's':
	    flags |= FLAGS_SYMLINKS;
	    break;

	case 'a':
	    flags |= FLAGS_ALLFILES;
	    break;

	case 'p':
	    protect_argv[k++] = optarg;
	    protect_argv[k] = NULL;
	    flags |= FLAGS_PROTECT;
	    break;
	}
    }

    if (optind == argc) {
	message (LOG_FATAL,
		 "Time must be given.\n");
	// NOT_REACHED
    }

#define multiplier j

    i = strlen (argv[optind]) - 1;
    switch (argv[optind][i]) {
    case 's':
    case 'S':
	multiplier = TIME_SECONDS;
	argv[optind][i] = '\0';
	break;
    case 'm':
    case 'M':
	multiplier = TIME_MINUTES;
	argv[optind][i] = '\0';
	break;
    case 'd':
    case 'D':
	multiplier = TIME_DAYS;
	argv[optind][i] = '\0';
	break;
    case 'h':
    case 'H':
	argv[optind][i] = '\0';
    default:
	multiplier = TIME_HOURS;
    }

    if (sscanf (argv[optind], "%d", &grace) != 1) {
	message (LOG_FATAL,
		 "Bad time argument `%s'.\n",
		 argv[optind]);
	// NOT_REACHED
    }

    optind++;
    if (optind == argc) {
	message (LOG_FATAL,
		 "Directory name(s) expected.\n");
	// NOT_REACHED
    }

    // Use line-buffered I/O for stdout, so that even when the output
    // is being piped to a file, it gets flushed after a newline.  If
    // we don't do this, when it forks, the child inherits an
    // unflushed buffer, so when it flushes and the parent does too,
    // we get doubled output.  Note that `stderr' is unbufferd
    // already.
    // See: `APUE', W. Richard Stevens,  pp. 189-190
    // Bug pointed out by Joey Hess <joey@kitenet.net>
    if (setvbuf (stdout, (char*)NULL, _IOLBF, BUFSIZ) != 0) {
	message (LOG_FATAL,
		 "setvbuf() failed for `stdout'.\n");
	// NOT_REACHED
    }

    grace = grace * multiplier;
    message (LOG_DEBUG,
	     "Grace period is `%u' seconds.\n",
	     (u_int)grace);
    
    message (LOG_DEBUG,
	     "This is PID (%u) being run by UID (%u), EUID (%u).\n",
	     getpid(), getuid(), geteuid());

    killTime = time (NULL) - grace;

    if ((pwd = getcwd (NULL, 0)) == NULL) {
	message (LOG_FATAL,
		 "Cannot getcwd().");
	// NOT_REACHED
    }
    // For each directory in the cleanup list, cd there, glob, then delete
    for (adir = optind; adir < argc; adir++) {
	if ((ret = chdir (argv[adir])) != 0) {
	    message (LOG_ERROR,
		     "Cannot chdir() to `%s' for --protect glob.\n",
		     argv[adir]);
	    continue;
	}
	message (LOG_DEBUG,
		 "getcwd() is: %s\n",
		 getcwd (NULL, 0));
	//  ... for each protect_argv, glob, using GLOB_APPEND
	if (FLAGS_PROTECT_P (flags)) {
	    pp=p;
	    while (*pp != NULL) {
		if (p == pp) {
		    ret = glob (*pp, (GLOB_NOSORT | GLOB_BRACE), NULL, &protect_glob);
		} else {
		    ret = glob (*pp, (GLOB_APPEND | GLOB_NOSORT | GLOB_BRACE), NULL, &protect_glob);
		}
		if (ret == GLOB_NOSPACE) {
		    message (LOG_FATAL,
			     "glob() returned GLOB_NOSPACE : Insufficient memory to hold pattern expansion.\n");
		    // NOT_REACHED
		}
		if (ret == GLOB_NOMATCH) {
		    message (LOG_VERBOSE,
			     "Nothing in `%s' matches `%s'.\n",
			     getcwd (NULL, 0), *pp);
		}
		pp++;
	    }
	}

	protect_table = malloc ((1 + protect_glob.gl_pathc) * sizeof (protect_entry));
	if (! protect_table) {
	    message (LOG_FATAL, "malloc() failed. : %s\n", strerror (errno));
	    // NOT_REACHED
	}
	j = 0;
	for (i = 0; i < protect_glob.gl_pathc; i++) {
	    if (lstat (protect_glob.gl_pathv[i], &sb)) {
		message (LOG_ERROR,
			 "lstat() of `%s' failed (`--protect'). : %s\n",
			 protect_glob.gl_pathv[i], strerror (errno));
		continue;
	    }
	    protect_table[j].inode = sb.st_ino;
	    protect_table[j++].name = protect_glob.gl_pathv[j];
	}
	protect_table[j].name = NULL;

	// Now go to work.
	if (lstat (argv[adir], &sb)) {
	    message (LOG_ERROR,
		     "lstat() of directory `%s' failed (argv). : %s\n",
		     argv[adir], strerror (errno));
	}
	if (S_ISLNK (sb.st_mode)) {
	    message (LOG_DEBUG,
		     "Initial directory `%s' is a symlink. -- skipping\n",
		     argv[adir]);
	}
	else {
	    cleanupDirectory (argv[adir], killTime, flags);
	}
	free(protect_table);
	globfree(&protect_glob);
    }
    return 0;
}


// Local Variables:
// eval: (c-set-style "cc-mode")
// End:
