/*
 * Worldvisions Weaver Software:
 *   Copyright (C) 1997, 1998 Worldvisions Computer Technology, Inc.
 *
 * Implementation of the WvDialer smart-dialer class.  
 *
 */
#include "dialer.h"
#include "wvdialver.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <termios.h>
#include <time.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <assert.h>


static char *	init_responses[] = {
	"ok",
	"error",
	NULL
};

static char *	dial_responses[] = {
	"connect",
	"no carrier",
	"no dialtone",
	"no dial tone",
	"busy",
	"error",
	"voice",
	"fclass",
	NULL
};

static char *	prompt_strings[] = {
	"}!}",
	"!}!",
	NULL
};




//**************************************************
//       WvDialer Public Functions
//**************************************************

WvDialer::WvDialer( WvConf& cfg, WvStringList& sect_list )
/********************************************************/
: WvStreamClone( (WvStream **)&modem ),
    log( "dialer", WvLog::Debug ), err( log.split( WvLog::Error ) ),
    modemrx( "modem", WvLog::Debug )
{
    modem 		= NULL;
    ppp_pipe 		= NULL;
    been_online 	= false;
    stat 		= Ready;
    offset 		= 0;
    prompt_tries 	= 0;
    last_rx 		= last_execute = 0;
    prompt_response 	= "";
    
    log( "WvDial: internet dialer version " WVDIAL_VER_STRING "\n" );

    // Ensure all sections in sect_list actually exist, warning if not.
    WvStringList::Iter	iter( sect_list );
    for( iter.rewind(); iter.next(); ) {
    	if( cfg.get_section( iter.data()->str ) == NULL ) {
    	    err( "Warning: section [%s] does not exist in /etc/wvdial.conf.\n",
    	    	 iter.data()->str );
    	}
    }

    setcallback( execute_callback, this );

    // Activate the brain and read configuration.
    brain = new WvDialBrain( this );
    load_options( cfg, sect_list );

    if( !options.modem.str[0] ) {
	err( "Configuration does not specify a valid modem device.\n" );
    	stat = ModemError;
	return; // if we get this error, we already have a problem.
    }
    
    if( !init_modem() )
    {
	// init_modem() printed an error
	stat = ModemError;
	return;
    }
}

WvDialer::~WvDialer()
/*******************/
{
    if( ppp_pipe )
	delete ppp_pipe;
    if( modem )
	delete modem;
    if( brain )
    	delete brain;
}

bool WvDialer::dial()
/*******************/
// Returns false on error, or true to go asynchronous while dialing.
{
    if( stat == Online ) {
    	err( "Trying to dial while already online.\n" );
    	return( false );
    } else if( stat != Ready ) {
	// (error message has already been printed elsewhere)
    	// err( "Modem is not ready to dial.\n" );
    	return( false );
    }

    if( !options.phnum.str[0] ) {
	err( "Configuration does not specify a valid phone number.\n" );
    	stat = OtherError;
    }

    if( !options.login.str[0] ) {
	err( "Configuration does not specify a valid login name.\n" );
    	stat = OtherError;
    }

    if( !options.password.str[0] ) {
	err( "Configuration does not specify a valid password.\n" );
    	stat = OtherError;
    }
    
    if( stat != Ready )
	return( false );

    stat = Dial;
    brain->reset();
    
    // we need to re-init the modem if we were online before.
    if( been_online && !init_modem() )
	stat = ModemError;

    return( true );
}

void WvDialer::hangup()
/*********************/
{
    if( ppp_pipe ) {
    	delete ppp_pipe;
	ppp_pipe = NULL;
    }

    if( stat != Ready ) {
	if( modem && !modem->isok() ) {
    	    stat = ModemError;
	    return;
	}

	time_t 	now;
	time( &now );
	log( "Disconnecting at %s", ctime( &now ) );
	if( modem )
	    modem->hangup();
	stat = Ready;
    }
}

bool WvDialer::select_setup( fd_set &r, fd_set &w, fd_set &x, int &max_fd,
			    bool readable, bool writable, bool isexception )
/**************************************************************************/
{
    if( isok() && stat != Online && stat != Ready
	       && time( NULL ) - last_execute > 1 )
    {
	// Pretend we have "data ready," so execute() gets called.
	// select() already returns true whenever the modem is readable,
	// but when we are doing a timeout (eg. PreDial1/2) for example,
	// we need to execute() even if no modem data is incoming.
	return( true );
    } else {
	return WvStreamClone::select_setup(r, w, x, max_fd,
					   readable, writable, isexception);
    }
}

bool WvDialer::isok() const
/*************************/
{
    return( modem && modem->isok()
    	&& stat != ModemError && stat != OtherError );
}

int WvDialer::execute_callback( WvStream &, void *userdata )
/**********************************************************/
{
    WvDialer &d = *(WvDialer *) userdata;
    d.execute();
    return( 0 ); 
}

void WvDialer::execute()
/**********************/
{
    // the modem object might not exist, if we just disconnected and are
    // redialing.
    if( !modem && !init_modem() )
    	return;

    last_execute = time( NULL );
    
    switch( stat ) {
    case Dial:
    case WaitDial:
    case PreDial1:
    case PreDial2:
    	async_dial();
    	break;
    case WaitAnything:
	// we allow some time after connection for silly servers/modems.
	if( modem->select( 500 ) ) {
	    // if any data comes in at all, switch to impatient mode.
	    stat = WaitPrompt;
	    last_rx = time( NULL );
	} else if( time( NULL ) - last_rx >= 30 ) {
	    // timed out - do what WaitPrompt would do on a timeout.
	    stat = WaitPrompt;
	} else {
	    // We prod the server with a CR character every once in a while.
	    // FIXME: Does this cause problems with login prompts?
	    modem->write( "\r", 1 );
	}
	break;
    case WaitPrompt:
    	async_waitprompt();
    	break;
    case Online:
    	// If already online, we only need to make sure pppd is still there.
	if( ppp_pipe && ppp_pipe->child_exited() ) {
	    if( ppp_pipe->child_killed() ) {
		log( WvLog::Error, "PPP was killed! (signal = %s)\n",
		      ppp_pipe->exit_status() );
	    } else {
		log( WvLog::Error, "PPP daemon has died! (exit code = %s)\n",
		      ppp_pipe->exit_status() );
	    }
	    
	    // set up to dial again, if it is requested.
	    // we must delete the WvModem object so it can be recreated
	    // later; starting pppd seems to screw up the file descriptor.
	    hangup();
	    delete( modem );
	    modem = NULL;
	}
	break;
    case Ready:
    case ModemError:
    case OtherError:
    default:
	drain();
    	break;
    }
}


//**************************************************
//       WvDialer Private Functions
//**************************************************

void WvDialer::load_options( WvConf& cfg, WvStringList& sect_list )
/*****************************************************************/
{
    OptInfo opts[] = {
    // string options:
    	{ "Modem",           &options.modem,        NULL, "/dev/modem",     0 },
    	{ "Init1",           &options.init1,        NULL, "ATZ",            0 },
    	{ "Init2",           &options.init2,        NULL, "",               0 },
    	{ "Init3",           &options.init3,        NULL, "",               0 },
    	{ "Init4",           &options.init4,        NULL, "",               0 },
    	{ "Init5",           &options.init5,        NULL, "",               0 },
    	{ "Init6",           &options.init6,        NULL, "",               0 },
    	{ "Init7",           &options.init7,        NULL, "",               0 },
    	{ "Init8",           &options.init8,        NULL, "",               0 },
    	{ "Init9",           &options.init9,        NULL, "",               0 },
    	{ "Phone",           &options.phnum,        NULL, "",               0 },
    	{ "Dial Prefix",     &options.dial_prefix,  NULL, "",               0 },
    	{ "Dial Command",    &options.dial_cmd,     NULL, "ATDT",           0 },
    	{ "Username",        &options.login,        NULL, "",               0 },
    	{ "Login Prompt",    &options.login_prompt, NULL, "",               0 },
    	{ "Password",        &options.password,     NULL, "",               0 },
    	{ "Password Prompt", &options.pass_prompt,  NULL, "",               0 },
    	{ "PPPD Path",       &options.where_pppd,   NULL, "/usr/sbin/pppd", 0 },

    // int/bool options
    	{ "Baud",            NULL, &options.baud,          "", DEFAULT_BAUD },
    	{ "Carrier Check",   NULL, &options.carrier_check, "", true         },
    	{ "Stupid Mode",     NULL, &options.stupid_mode,   "", false        },
    	{ "New PPPD",	     NULL, &options.new_pppd, 	   "", false	    },
    	{ NULL,		     NULL, NULL,                   "", 0            }
    };

    char *	d = "Dialer Defaults";

    for( int i=0; opts[i].name != NULL; i++ ) {
    	if( opts[i].str_member == NULL ) {
    	    // it's an int/bool option.
    	    *( opts[i].int_member ) =
    	    		cfg.fuzzy_get( sect_list, opts[i].name,
    	    		    cfg.get( d, opts[i].name, opts[i].int_default ) );
    	} else {
    	    // it's a string option.
    	    *( opts[i].str_member ) = 
    	    		cfg.fuzzy_get( sect_list, opts[i].name, 
    	    		    cfg.get( d, opts[i].name, opts[i].str_default ) );
    	}
    }
}

bool WvDialer::init_modem()
/*************************/
{
    int	received;

    // Open the modem...
    if( !modem ) {
	modem = new WvModem( options.modem.str, options.baud );
	if( !modem->isok() ) {
	    err( "Cannot open %s: %s\n", options.modem, modem->errstr() );
    	    return( false );
	}
    }
    log( "Initializing modem.\n" );

    // Send up to nine init strings, in order.
    int	init_count;
    for( init_count=1; init_count<=9; init_count++ ) {
    	WvString *	this_str;
    	switch( init_count ) {
	    case 1:    this_str = &options.init1;	break;
	    case 2:    this_str = &options.init2;	break;
	    case 3:    this_str = &options.init3;	break;
	    case 4:    this_str = &options.init4;	break;
    	    case 5:    this_str = &options.init5;	break;
    	    case 6:    this_str = &options.init6;	break;
    	    case 7:    this_str = &options.init7;	break;
    	    case 8:    this_str = &options.init8;	break;
    	    case 9:
            default:
	               this_str = &options.init9;	break;
    	}
    	if( this_str->str[0] ) {
    	    modem->print( "%s\r", *this_str );
    	    log( "Sending: %s\n", *this_str );

    	    received = wait_for_modem( init_responses, 5000, true );
    	    switch( received ) {
    	    case -1:
    	    	err( "Modem not responding.\n" );
    	    	return( false );
    	    case 1:
    	    	err( "Bad init string.\n" );
    	    	return( false );
    	    }
    	}
    }
    
    // Everything worked fine.
    log( "Modem initialized.\n" );
    return( true );
}

void WvDialer::async_dial()
/*************************/
{
    int	received;

    if( stat == PreDial2 ) {
    	// Wait for three seconds and then go to PreDial1.
    	usleep( 3 * 1000 * 1000 );
    	stat = PreDial1;
    	return;
    }
    
    if( stat == PreDial1 ) {
	// Hit enter a few times.
	for( int i=0; i<3; i++ ) {
	    modem->write( "\r", 1 );
	    usleep( 500 * 1000 );
	}
	stat = Dial;
	return;
    }
	
    if( stat == Dial ) {
    	// Construct the dial string.  We use the dial command, prefix, and
    	// phone number as specified in the config file.
	WvString s( "%s %s%s\r", options.dial_cmd,
				 options.dial_prefix,
				 options.phnum );
	modem->print( s );
	log( "Sending: %s\n", s );
	log( "Waiting for carrier.\n" );

	stat = WaitDial;
    }

    received = async_wait_for_modem( dial_responses, true );
    switch( received ) {
    case -1:	// nothing -- return control.
	if( last_rx - time( NULL ) >= 60 ) {
	    log( WvLog::Warning, "Timed out while dialing.  Trying again.\n" );
	    stat = PreDial1;
	}
	return;
    case 0:	// CONNECT
	if( options.stupid_mode == true ) {
	    log( "Carrier detected.  Starting ppp.\n" );
	    start_ppp();
	} else {
	    log( "Carrier detected.  Waiting for prompt.\n" );
	    stat = WaitAnything;
	}
	return;
    case 1:	// NO CARRIER
	log( WvLog::Warning, "No Carrier!  Trying again.\n" );
	stat = PreDial1;
	sleep( 2 );
	return;
    case 2:	// NO DIALTONE
    case 3:	// NO DIAL TONE
	err( "No dial tone.  Trying again in 5 seconds.\n" );
	stat = PreDial2;
	return;
    case 4:	// BUSY
	log( WvLog::Warning, "The line is busy.  Trying again.\n" );
	stat = PreDial1;
	sleep( 2 );
	return;
    case 5:	// ERROR
	err( "Invalid dial command.\n" );
	return;
    case 6:	// VOICE
    	log( "Voice line detected.  Trying again.\n" );
    	stat = PreDial2;
    	return;
    case 7:	// FCLASS
    	log( "Fax line detected.  Trying again.\n" );
    	stat = PreDial2;
    	return;
    default:
	err( "Unknown dial response string.\n" );
	stat = ModemError;
	return;
    }
}

void WvDialer::start_ppp()
/************************/
{
    a_pap_secret *	secret;
    
    char *argv[] = {
	options.where_pppd.str,
	"modem",
	"crtscts",
	"defaultroute",
	"-detach",
	"user", options.login.str,
	options.new_pppd ? "call" : NULL, 
	options.new_pppd ? "wvdial" : NULL,
	NULL
    };

    if( access( options.where_pppd.str, X_OK ) != 0 ) {
        err( "Unable to run %s.\n", options.where_pppd );
        err( "Check permissions, or specify a \"PPPD Path\" option"
             "in wvdial.conf.\n" );
    	return;
    }

    pap_secrets = new WvPapSecrets();
    if( pap_secrets == NULL || pap_secrets->isok() == false ) {
        err( "Warning: can't modify %s: %s\n",
	     PAP_SECRETS, strerror( errno ) );
    } else {
    	secret = pap_secrets->get_secret( options.login );
    	if( secret == NULL ) {
    	    // pap secret not there!  Add it.
    	    if( ! pap_secrets->put_secret( options.login, 
    	    			           options.password ) ) {
    	    	err( "Can't add secret to %s\n"
    	    	 "--> PAP (PPP Password Authentication Protocol) disabled.\n", 
    	    	     PAP_SECRETS );
    	    }
    	} else if( secret->password != options.password ) {
    	    // uh oh.  Passwords don't match!  Fix secrets file...
    	    log( "Password in %s does not match %s\n"
    	         "--> Correcting the %s version.\n",
    	         PAP_SECRETS, "wvdial.conf", PAP_SECRETS );
    	    pap_secrets->del_secret( options.login );
    	    if( ! pap_secrets->put_secret( options.login, 
    	    				   options.password ) ) {
    	    	err( "Can't add secret to %s\n"
    	    	 "--> PAP (PPP Password Authentication Protocol) disabled.\n", 
    	    	     PAP_SECRETS );
    	    }
    	}
    	// DLC_assert( the secret is fine );
    }
    // Done with the secrets file now.
    if( pap_secrets ) {
    	delete pap_secrets;
    }

    time_t	now;
    time( &now );
    log( WvLog::Notice, "Starting pppd at %s", ctime( &now ) );

    ppp_pipe = new WvPipe( argv[0], argv, false, false, false,
			   modem, modem, modem );

    stat 	= Online;
    been_online = true;
}

void WvDialer::async_waitprompt()
/*******************************/
{
    int		received;
    char *	prompt_response;

    if( options.carrier_check == true ) {
	if( !modem || !modem->carrier() ) {
	    stat = ModemError;
    	    return;
	}
    }

    received = async_wait_for_modem( prompt_strings, false, true );
    if( received >= 0 ) {
    	// We have a PPP sequence!
    	log( "PPP negotiation detected.\n" );
    	start_ppp();
    } else if( received == -1 ) {
    	// some milliseconds must have passed without receiving anything,
	// or async_wait_for_modem() would not have returned yet.
	
    	// check to see if we are at a prompt.
        // Note: the buffer has been lowered by strlwr() already.

	prompt_response = brain->check_prompt( buffer );
	if( prompt_response != NULL )
	    modem->print( "%s\r", prompt_response );
    }
}

int WvDialer::wait_for_modem( char * 	strs[], 
			      int	timeout, 
			      bool	neednewline,
			      bool 	verbose )
/***********************************************/
{
    off_t	onset;
    char *	soff;
    int		result;
    int		len;

    while( modem->select( timeout ) ) {
	last_rx = time( NULL );
    	onset = offset;
	offset += modem->read( buffer + offset, INBUF_SIZE - offset );
	
	// make sure we do not split lines TOO arbitrarily, or the
	// logs will look bad.
	while( offset < INBUF_SIZE && modem->select( 100 ) )
	    offset += modem->read( buffer + offset, INBUF_SIZE - offset );

	// Make sure there is a NULL on the end of the buffer.
	buffer[ offset ] = 0;

	// Now turn all the NULLs in the middle of the buffer to spaces, for
	// easier parsing.
	replace_char( buffer + onset, '\0', ' ', offset - onset );

	if( verbose )
	    modemrx.write( buffer + onset, offset - onset );

	strlwr( buffer + onset );

	// Search the buffer for a valid menu option...
	// If guess_menu returns an offset, we zap everything before it in
	// the buffer.  This prevents finding the same menu option twice.
	char *	ppp_marker = brain->guess_menu( buffer );
	if( ppp_marker != NULL )
	    memset( buffer, ' ', ppp_marker-buffer );
	
	// Now we can search using strstr.
	for( result = 0; strs[ result ] != NULL; result++ ) {
	    len = strlen( strs[ result ] );
	    soff = strstr( buffer, strs[ result ] );
	    
	    if( soff && ( !neednewline 
			 || strchr( soff, '\n' ) || strchr( soff, '\r' ) ) )
	    {
		memmove( buffer, soff + len,
			 offset - (int)( soff+len - buffer ) );
		offset -= (int)( soff+len - buffer );
		return( result );
	    }
	}

	// Looks like we did not find anything.  Is the buffer full yet?
	if( offset == INBUF_SIZE ) {
	    // yes, move the second half to the first half for next time.
	    memmove( buffer, buffer + INBUF_SIZE/2,
		     INBUF_SIZE - INBUF_SIZE/2 );
	    offset = INBUF_SIZE/2;
	}
    }
    
    buffer[ offset ] = 0;
    return( -1 ); // timeout
}

int WvDialer::async_wait_for_modem( char * strs[], bool neednl, bool verbose )
/****************************************************************************/
{
    return( wait_for_modem( strs, 10, neednl, verbose ) );
}

