
/*
 * MAIN.C
 *
 * xmake [-d[#]] [-f XMakefile] dependancies...
 *
 */

#include "defs.h"

Prototype List	DependList;
Prototype List	VarList;
Prototype List  DepList;
Prototype int	DebugOpt;
Prototype int	QuietOpt;
Prototype VNode	*ExtDefs;

int run_shell(const char *str);
int RunDependancy(DNode *dn);
int DNodeHasRhsDependancies(DNode *dnode);

List	DependList = INITLIST(DependList);	/* Depend structures	*/
List	VarList = INITLIST(VarList);
List	DepList = INITLIST(DepList);		/* DNode structures	*/
VNode	*ExtDefs;

int	DebugOpt;
int	QuietOpt;
int	ForRealOpt = 1;

int
main(int ac, char **av)
{
    int i;
    int r = DEPEND_UPTODATE;
    const char *xmFile = "XMakefile";

    ExtDefs = MakeVNode(&VarList, "EXTDEFS");

    /*
     * Process options
     */

    for (i = 1; i < ac; ++i) {
	char *ptr = av[i];

	if (*ptr != '-') {
	    DNode *node = MakeDNode(ptr);
	    AddTail(&DepList, &node->dn_Node);
	    continue;
	}
	ptr += 2;
	switch(ptr[-1]) {
	case 'q':
	    QuietOpt = 1;
	    break;
	case 'd':
	    DebugOpt = (*ptr) ? strtol(ptr, NULL, 0) : 1;
	    break;
	case 'n':
	    ForRealOpt = 0;
	    break;
	case 'f':
	    xmFile = av[++i];
	    break;
	case 'D':
	    {
		Node *node;
		char *tmp;

		ptr = (*ptr) ? ptr : av[++i];

		tmp = malloc(strlen(ptr) + 4);
		sprintf(tmp, "-D%s", ptr);
		node = MakeNode(tmp, sizeof(Node), 0);
		free(tmp);

		AddTail(&ExtDefs->vn_List, node);
	    }
	    break;
	default:
	    fatal(NULL, "Illegal option: %s\n", ptr - 2);
	    /* not reached */
	    break;
	}
    }
    if (i > ac) {
	fatal(NULL, "Expected argument to last option\n");
	/* not reached */
    }

    CommitVNode(ExtDefs);

    /*
     * Process XMakefile
     */

    IncludeFile(xmFile);

    /*
     * Execute Dependancies
     */

    if (GetHead(&DepList) == NULL) {
	DNode *node = MakeDNode("all");
	AddTail(&DepList, &node->dn_Node);
    }

    {
	DNode *dnode;

	for (dnode = GetHead(&DepList); dnode; dnode = GetNext(&dnode->dn_Node)) {
	    int rv = RunDependancy(dnode);

	    if (r < rv)
		r = rv;
	    if (r > EXITSTATUS) {
		if (!QuietOpt)
		    printf("%s Exit %d\n", av[0], r);
		break;
	    }
	}
    }
    if (r < 0) {
	if (!QuietOpt)
	    printf("All targets up to date.\n");
	r = 0;
    } else if (r > 0) {
	if (!QuietOpt && r <= EXITSTATUS)
	    printf("%s exit %d\n", av[0], r);
    }
    exit(r);
}

/*
 * RunDependancy(dep)
 *
 */

int
RunDependancy(DNode *dn)
{
    Depend *dep;
    int xr = DEPEND_UPTODATE;
    int executeMe = 0;
    int hasRhs = 0;

    for (dep = dn->dn_Depend; dep; dep = dep->de_Chain) {
	if (DebugOpt)
	    printf("Run %s (@ %d)\n", dep->de_Node.no_Name, dep->de_Node.no_Value);

	if (dep->de_Node.no_Value <= DEPEND_NOTRUN) {
	    /*
	     * If there is no rhs, we always run the execution list
	     * and we are never uptodate.
	     */
	    dep->de_Node.no_Value = DEPEND_UPTODATE;

	    if (hasRhs == 0)
		hasRhs = -1;

	    if (dep->de_Rhs != NULL && GetHead(dep->de_Rhs) != NULL) {
		DNode *scan;
		int overrideOk = 1;

		hasRhs = 1;

		/*
		 * Dependancies
		 */

		if (DebugOpt)
		    printf("SCAN\n");

		for (scan = GetHead(dep->de_Rhs); scan; scan = GetNext(&scan->dn_Node)) {
		    int r = DEPEND_UPTODATE;
		    int rhsHasRhsDep = DNodeHasRhsDependancies(scan);

		    if (DebugOpt)
			printf("SCAN %s rhsHasRhsDep %d\n", scan->dn_Node.no_Name, rhsHasRhsDep);

		    if (rhsHasRhsDep) {
			/*
			 * The rhs of the right hand dependancy has dependancies of its own
			 */
			r = RunDependancy(scan);
			if (r != 0)
			    overrideOk = 0;
		    } /* else */ { 
			/*
			 * The rhs has no dependancies of its own, we
			 * test the files.  If lhs == rhs, we test for
			 * simple existance.
			 *
			 * If the rhs has dependancies, we allow the 
			 * right-hand side 'name' to not exist.
			 */

			if (strcmp(dep->de_Node.no_Name, scan->dn_Node.no_Name) == 0) {
			    struct stat st;

			    if (stat(dep->de_Node.no_Name, &st) < 0) {
				overrideOk = 0;
				if (r < 0)
				    r = 0;
			    }
			} else {
			    struct stat st1;
			    struct stat st2;

			    if (stat(dep->de_Node.no_Name, &st1) == 0) {		/* LHS, e.g. .o */
				if (stat(scan->dn_Node.no_Name, &st2) == 0) {	/* RHS, e.g. .c */
				    long dt = (long)(st1.st_mtime - st2.st_mtime);
				    if (dt < 0) {
					overrideOk = 0;
					if (r < 0)
					    r = 0;
				    } else if (overrideOk) {
					/*
					 * We override the return code from any
					 * rhs SUB-dependancies that are run, allowing
					 * us to do something like OBJS :: PROTOS and
					 * PROTOS : SRCS, yet not touch PROTOS if the
					 * change to the source would not otherwise effect
					 * it.
					 */
					r = DEPEND_UPTODATE;
				    }
				} else if (rhsHasRhsDep == 0) {
				    overrideOk = 0;
				    r = 1;
				    fprintf(stderr, "Could not stat %s\n", scan->dn_Node.no_Name);
				}
			    } else {
				overrideOk = 0;
				if (r < 0)
				    r = 0;
			    }
			}
		    }

		    if (dep->de_Node.no_Value < r)
			dep->de_Node.no_Value = r;

		    if (dep->de_Node.no_Value > EXITSTATUS) {
			fprintf(stderr, "(%s : %s) failed code %d.\n", 
			    dep->de_Node.no_Name, 
			    scan->dn_Node.no_Name, 
			    dep->de_Node.no_Value
			);
			break;
		    }
		}
	    }

	    /*
	     * Execution list
	     */

	    if (dep->de_Node.no_Value > DEPEND_UPTODATE && 
		dep->de_Node.no_Value <= EXITSTATUS &&
		dep->de_XList != NULL
	    ) {
		executeMe = 1;
	    }
	}
	if (xr < dep->de_Node.no_Value)
	    xr = dep->de_Node.no_Value;
	if (xr > EXITSTATUS)
	    break;
    }

    /*
     * If there is no right-hand side and we are
     * (degenerate case) all up to date, execute
     * the command list anyway.  e.g. clean: (nothing)
     */

    if (xr <= DEPEND_UPTODATE && hasRhs < 0) {
	xr = 0;
	executeMe = 1;
    }

    if (executeMe && xr <= EXITSTATUS) {
	for (dep = dn->dn_Depend; dep; dep = dep->de_Chain) {
	    Node *node;

	    if (dep->de_XList == NULL)
		continue;

	    for (node = GetHead(dep->de_XList); node; node = GetNext(node)) {
		const char *s1 = expToken(dep, NULL, node->no_Name, EXP_FORCE);
		const char *s2 = s1;
		int r = DEPEND_UPTODATE;
		int ignoreRet = 0;
		int noEcho = 0;

		if (ForRealOpt) {
		    while (*s2 == '-' || *s2 == '@') {
			if (*s2 == '-')
			    ignoreRet = 1;
			if (*s2 == '@')
			    noEcho = 1;
			++s2;
		    }
		}

		switch(node->no_Type) {
		case TYPE_VAR:
		    {
			VNode *vnode = MakeVNode(&VarList, node->no_Name);
			Node *t;

			while ((t = RemHead(&((VNode *)node)->vn_List)) != NULL)
			    AddTail(&vnode->vn_List, &MakeDNode(expToken(dep, NULL, t->no_Name, EXP_FORCE))->dn_Node);
			CommitVNode(vnode);
		    }
		    break;
		case TYPE_EXEC:
		    if (noEcho == 0)
			printf("    %s\n", s2);
		    if (ForRealOpt)
			r = run_shell(s2);
		    else
			r = 0;
		    break;
		}

		if (ignoreRet && r > 0)
		    r = 0;

		if (dep->de_Node.no_Value < r)
		    dep->de_Node.no_Value = r;

		if (dep->de_Node.no_Value > EXITSTATUS) {
		    if (noEcho) {
			fprintf(stderr, "%s command failed code %d (%s)\n",
			    dep->de_Node.no_Name, 
			    dep->de_Node.no_Value,
			    s2
			);
		    }
		    break;
		}
	    }

	    if (xr < dep->de_Node.no_Value)
		xr = dep->de_Node.no_Value;

	    if (xr > EXITSTATUS)
		break;
	}
    }
    /*
    if (xr > EXITSTATUS)
	fprintf(stderr, "%s failed code %d.\n", dn->dn_Node.no_Name, xr);
    */
    return(xr);
}

int
DNodeHasRhsDependancies(DNode *dnode)
{
    Depend *dep;

    for (dep = dnode->dn_Depend; dep; dep = dep->de_Chain) {
	if (dep->de_XList && GetHead(dep->de_XList) != NULL)
	    return(1);

	if (dep->de_Rhs && GetHead(dep->de_Rhs) != NULL)
	    return(1);
    }
    return(0);
}

int
run_shell(const char *str)
{
    pid_t pid;
    int r = 0;

    if ((pid = fork()) == 0) {
	execl("/bin/sh", "sh", "-c", str, NULL);
	_exit(1);
    }
    if (pid < 0) {
	perror("fork");
	r = 1;
    } else {
	int status = 0;
	waitpid(pid, &status, 0);
	r = WEXITSTATUS(status);
    }
    return(r);
}

