/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <sys/types.h>	/* readdir, dirent */
#include <dirent.h>	/* readdir, dirent */
#include <string.h>	/* strlen, strcpy, strcmp, memset */
#include <stdlib.h>	/* qsort */
#include <gtk/gtk.h>

#include "pref.h"
#include "utils.h"
#include "skin.h"
#include "skin_pcx.h"


static guchar quake_pallete[768] = {
#include "quake_pal.h"
};


static GdkColor pcolors[14];
static int pcolors_allocated = FALSE;


static inline int is_pcx (char *str) {
  return 
    str &&
    (strlen (str) > 4) &&
    (strcmp (str + strlen (str) - 4, ".pcx") == 0);
}


static int qsort_helper (const void *a, const void *b) {
  return strcmp (* (char**) a, * (char**) b);
}


GList *build_skin_list (char *dirname) {
  DIR *directory;
  struct dirent *dirent_ptr;
  int counter = 0;
  int name_buffer_size = 0;
  char **entries;
  char *name_buffer;
  char *buffer_ptr;
  GList *list = NULL;
  GtkWidget *list_item;
  int i;

  directory = opendir (dirname);
  if (directory == NULL)
    return NULL;

  while ((dirent_ptr = readdir (directory)) != NULL) {
    if (is_pcx (dirent_ptr->d_name)) {
      name_buffer_size += (strlen (dirent_ptr->d_name) + 1) - 4;
      counter++;
    }
  }

  entries = g_malloc (sizeof (char*) * counter);
  name_buffer = g_malloc (name_buffer_size);
  buffer_ptr = name_buffer;

  rewinddir (directory);

  for (i = 0; i < counter; ) {
    if ((dirent_ptr = readdir (directory)) == NULL) {
      g_free (name_buffer);
      g_free (entries);
      return NULL;
    }

    if (is_pcx (dirent_ptr->d_name)) {
      entries[i] = buffer_ptr;
      strncpy (buffer_ptr, dirent_ptr->d_name, strlen (dirent_ptr->d_name) - 4);
      buffer_ptr += strlen (dirent_ptr->d_name) - 4;
      *buffer_ptr = '\0';
      buffer_ptr++;
      i++;
    }
  }

  closedir (directory);

  qsort (entries, counter, sizeof (char*), qsort_helper);

  for (i = counter-1; i >=0; i--) {
    list_item = gtk_list_item_new_with_label (entries[i]);
    list = g_list_prepend (list, list_item);
    gtk_widget_show (list_item);
  }

  g_free (entries);
  g_free (name_buffer);

  return list;
}


void draw_skin (GtkWidget *preview, guchar *data) {
  guchar buf[320*3];
  guchar colormap[256];
  guchar *ptr;
  int i, j, k;

  if (data) {
    for (i = 0; i < 256; i++)
      colormap[i] = i;

    if (default_top_color != 1) {
      for (i = 0; i<16; i++)
	colormap[16*1 + i] = default_top_color*16 + 
                                     ((default_top_color <= 7)? i : 15 - i);
    }

    if (default_bottom_color != 6) {
      for (i = 0; i<16; i++)
	colormap[16*6 + i] = default_bottom_color*16 +
	                             ((default_bottom_color <= 7)? i : 15 - i);
    }

    for (i = 0; i < 200; i++) {
      ptr = buf;
      for (j = 0; j < 320; j++) {
	*ptr++ = quake_pallete[colormap[*data]*3 + 0];
	*ptr++ = quake_pallete[colormap[*data]*3 + 1];
	*ptr++ = quake_pallete[colormap[*data]*3 + 2];
	data++;
      }
      gtk_preview_draw_row (GTK_PREVIEW (preview), buf, 0, i, 320);
    }
  }
  else {

#define BCOLOR	0x80
#define WCOLOR	0xC0

    for (j = 0, k = WCOLOR; j < 200; j += 40) {
      i = k;
      for (ptr = buf; ptr < buf+320*3; ptr += 40*3) {
	memset (ptr, i, 40*3);
	i = (WCOLOR+BCOLOR) - i;
      }
      for (i = 0; i < 40; i++) {
	gtk_preview_draw_row (GTK_PREVIEW (preview), buf, 0, j+i, 320);
      }
      k = (WCOLOR+BCOLOR) - k;
    }
  }

  gtk_widget_draw (preview, NULL);
}


guchar *get_skin (char *skin) {
  char pcx_ext[] = ".pcx";
  char *skins_subdir;
  char *filename;
  char *ptr;
  guchar *pcx;

  if (skin && skin[0] != '\0') {
    skins_subdir = file_in_dir (default_quake_dir, "qw/skins/");
    filename = g_malloc (strlen (skins_subdir) + strlen (skin) + 
                         (sizeof (pcx_ext) - 1) + 1);
    ptr = filename;
    strcpy (ptr, skins_subdir);  ptr += strlen (skins_subdir);
    strcpy (ptr, skin);  ptr += strlen (skin);
    strcpy (ptr, pcx_ext);

    pcx = read_skin_pcx (filename);

    g_free (filename);
    g_free (skins_subdir);
    return pcx;
  }
  
  return NULL;
}


static gushort convert_color (unsigned c) {
  c *= 257;			/* scale from char to short */
  c = c + c/3 + 0x2800;		/* add brightness */
  return (c > 0xffff)? 0xffff : c;
}


void allocate_quake_player_colors (GdkWindow *window) {
  GdkColormap *colormap;
  int i, j;

  if (!pcolors_allocated) {
    colormap = gdk_window_get_colormap (window);

    for (i = 0; i < 14; i++) {
      j = (i<8)? 11 : 15 - 11;
      pcolors[i].pixel = 0;
      pcolors[i].red   = convert_color (quake_pallete [(i*16 + j)*3 + 0]);
      pcolors[i].green = convert_color (quake_pallete [(i*16 + j)*3 + 1]);
      pcolors[i].blue  = convert_color (quake_pallete [(i*16 + j)*3 + 2]);
      if (!gdk_color_alloc (colormap, &pcolors[i])) {
	g_warning ("unable to allocate color: ( %d %d %d )", 
		   pcolors[i].red, pcolors[i].green, pcolors[i].blue);
      }
    }
    pcolors_allocated = TRUE;
  }
}


void set_bg_color (GtkWidget *widget, int color) {
  GtkStyle *style;

  style = gtk_style_new ();
  style->bg [GTK_STATE_NORMAL]   = pcolors [color];
  style->bg [GTK_STATE_ACTIVE]   = pcolors [color];
  style->bg [GTK_STATE_PRELIGHT] = pcolors [color];

  gtk_widget_set_style (widget, style);
}


GtkWidget *create_color_menu (void (*callback) (GtkWidget*, int)) {
  GtkWidget *menu;
  GtkWidget *menu_item;
  int i;

  menu = gtk_menu_new ();
  
  for (i = 0; i < 14; i++) {
    menu_item = gtk_menu_item_new_with_label (" ");
    gtk_widget_set_usize (GTK_WIDGET (menu_item), 40, -1);
    gtk_menu_append (GTK_MENU (menu), menu_item);
    gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                          GTK_SIGNAL_FUNC (callback), (gpointer) i);
    set_bg_color (menu_item, i);
    gtk_widget_show (menu_item);
  }

  return menu;
}


GdkPixmap *create_pm (GtkWidget *window, int top, int bottom, 
		                                           GSList **pm_cache) {
  GdkPixmap *pm;
  GdkGC *gc;
  GSList *tmp;
  struct cached_pixmap *cp;

  for (tmp = *pm_cache; tmp; tmp = tmp->next) {
    cp = (struct cached_pixmap *) tmp->data;
    if (cp->top == top && cp->bottom == bottom)
      return cp->pixmap;
  }

  pm = gdk_pixmap_new (window->window, PCOL_WIDTH, PCOL_HEIGHT, -1);

  gc = gdk_gc_new (pm);

  gdk_gc_set_foreground (gc, &pcolors[top]);
  gdk_draw_rectangle (pm, gc, TRUE, 0, 0, PCOL_WIDTH, PCOL_HEIGHT/2);

  gdk_gc_set_foreground (gc, &pcolors[bottom]);
  gdk_draw_rectangle (pm, gc, TRUE, 0, PCOL_HEIGHT/2, PCOL_WIDTH, PCOL_HEIGHT);

  gdk_gc_destroy (gc);

  cp = g_malloc (sizeof (struct cached_pixmap));
  cp->top = top;
  cp->bottom = bottom;
  cp->pixmap = pm;
  *pm_cache = g_slist_prepend (*pm_cache, cp);

  return pm;
}


static void free_cached_pixmap (struct cached_pixmap *cp) {
  gdk_pixmap_unref (cp->pixmap);
  g_free (cp);
}


void clear_pm_cache (GSList *pm_cache) {
  g_slist_foreach ((GSList *) pm_cache, (GFunc) free_cached_pixmap, NULL);
  g_slist_free ((GSList *) pm_cache);
}


GdkBitmap *create_dummy_mask (GtkWidget *window) {
  GdkBitmap *bm;
  GdkGC *gc;
  GdkColor mask_pattern;

  bm = gdk_pixmap_new (window->window, PCOL_WIDTH, PCOL_HEIGHT, 1);

  gc = gdk_gc_new (bm);
  mask_pattern.pixel = 1;
  gdk_gc_set_foreground (gc, &mask_pattern);

  gdk_draw_rectangle (bm, gc, TRUE, 0, 0, -1, -1);

  gdk_gc_destroy (gc);

  return bm;
}
