// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  ModelModule.C

  Stephane Rehel

  November 15 1996
*/

#include <stdio.h>

#include "tools/Command.h"
#include "tools/thread/MLThread.h"

#include "graphics/SystemWindow.h"

#include "interface/MLAllocator.h"
#include "interface/MLInterfaceCommand.h"
#include "interface/MLScrollingDialog.h"
#include "interface/MLScrollingDialogs.h"
#include "interface/MLMode.h"

#include "ModelCanvas.h"

#include "scene/MLCamera.h"
#include "scene/MLObject.h"
#include "scene/MLScene.h"
#include "scene/file/MLDatabasesList.h"

#include "ModelColors.h"
#include "ModelModes.h"
#include "ModelFunctions.h"
#include "ModelCommand.h"
#include "ModelScene.h"

#include "MLLightingProcess.h"

#include "MLStatusLightingSwitch.h"
#include "ModelStatusBars.h"

#include "ModelFlags.h"
#include "ModelPopups.h"

#include "ModelModule.h"

#include "render/ModelRender.h"
#include "dialogs/MLDModesList.h"

/////////////////////////////////////////////////////////////////////////////

ModelModule::ModelModule( Interface* _interface ): MLModule(_interface)
{
  modelRender= 0;

  flags= new ModelFlags(this);
  popups= new ModelPopups(this);

  resetFlags();

  mcanvas[0]= mcanvas[1]= mcanvas[2]= mcanvas[3]= mcanvas[4]= 0;
  scene= new MLScene;
  modelScene= new ModelScene(this,scene);

  lightingThread= new MLThread;
  lightingProcess= new MLLightingProcess(this,scene);

  status= 0;

  MLModule::name= OString("Scene");
  MLModule::color= Color(0.10,0.10,0.25);

  ModelColors::init(this);
  ModelModes::init(this,modes);
  ModelFunctions::init(this);

  ModelCommand::initModelCommands();
}

/////////////////////////////////////////////////////////////////////////////

ModelModule::~ModelModule()
{
  lightingProcess->quitThread();
  delete lightingProcess;
  lightingProcess= 0;
  delete lightingThread;
  lightingThread= 0;

  delete mcanvas[1]; mcanvas[1]= 0;
  delete mcanvas[2]; mcanvas[2]= 0;
  delete mcanvas[3]; mcanvas[3]= 0;
  delete mcanvas[4]; mcanvas[4]= 0;

  delete modelScene;
  modelScene= 0;

  delete scene;
  scene= 0;

  delete status;
  status= 0;

  delete flags;
  flags= 0;

  delete popups;
  popups= 0;
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::init( Command& command )
{
  MLModule::init(command);

  if( command.findOption( "-mlrc" ) )
    {
    OString resourceFilename= command.getString();
    if( resourceFilename.length() > 0 )
      MLDatabasesList::setDatabasesResourceFilename(resourceFilename);
    }

  status= new ModelStatusBars(this,statusBars);
  status->init();

  mcanvas[1]= new ModelCanvas;
  mcanvas[1]->create( this, CANVAS_1 );

  mcanvas[2]= new ModelCanvas;
  mcanvas[2]->create( this, CANVAS_2 );

  mcanvas[3]= new ModelCanvas;
  mcanvas[3]->create( this, CANVAS_3 );

  mcanvas[4]= new ModelCanvas;
  mcanvas[4]->create( this, CANVAS_4 );

//  popups->init(MLModule::buttonsPanel);
  popups->init(buttonsPanel);
  updatePopupsFlags();

  lightingThread->create(lightingProcess);

  createDialog("Welcome");
  modelScene->resetAll();
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::updatePopupsFlags()
{
  if( popups != 0 )
    popups->updateFlags();
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::updateStatusFlags()
{
  if( status != 0 )
    status->update();
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::updateFlags()
{
  updateStatusFlags();
  updatePopupsFlags();
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::idle()
{
  if( scene == 0 )
    return;

  if( modelRender != 0 )
    {
    if( modelRender->isRendering() )
      return; // don't disturb it
    }

  if( lightingProcess == 0 )
    {
    status->dlt->setStatus(LS_DISABLED);
    status->ilt->setStatus(LS_DISABLED);
    return;
    }

  IBOOL run_it= IFALSE;

  if( scene->dirtyPoolsLighting ||
      scene->dirtyObjectsLighting )
    {
    scene->dirtyFinalLighting= IFALSE;

    if( flags->computeDirectLighting )
      run_it= ITRUE;

    goto update_status;
    }

  if( scene->dirtyFinalLighting )
    {
    scene->dirtyFinalLighting= IFALSE;
    refreshScreen(REFRESH_ILLUM);
    }

  if( flags->computeIndirectLighting )
    {
    // could be in the diffusing process, no matter.
    run_it= ITRUE;
    }

update_status:

  if( run_it )
    {
    // Check for boundings/hashboxes to update
    modelScene->updateBoundingHashBox();
    runLightingProcess();
    }

  if( flags->computeDirectLighting )
    {
    status->dlt->setStatus( lightingProcess->inDirectDiffusing()
                            ? (LS_WORKING1 + (lightingProcess->getStep() & 1))
                            : LS_ASLEEP );
    }
   else
    status->dlt->setStatus(LS_DISABLED);

  if( flags->computeIndirectLighting )
    {
    status->ilt->setStatus( lightingProcess->inIndirectDiffusing()
                            ? (LS_WORKING1 + (lightingProcess->getStep() & 1))
                            : LS_ASLEEP );
    }
   else
    status->ilt->setStatus(LS_DISABLED);
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::map()
{
  if( _mapped )
    return;

  MLModule::map();

//*** error when a canvas if maximized...
  if( mcanvas[1] != 0 ) mcanvas[1]->map();
  if( mcanvas[2] != 0 ) mcanvas[2]->map();
  if( mcanvas[3] != 0 ) mcanvas[3]->map();
  if( mcanvas[4] != 0 ) mcanvas[4]->map();
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::draw()
{
  if( ! _mapped )
    return;

  if( mcanvas[1] != 0 ) mcanvas[1]->titleDraw();
  if( mcanvas[2] != 0 ) mcanvas[2]->titleDraw();
  if( mcanvas[3] != 0 ) mcanvas[3]->titleDraw();
  if( mcanvas[4] != 0 ) mcanvas[4]->titleDraw();

  MLModule::draw();
}

/////////////////////////////////////////////////////////////////////////////

// color: 0 = gray
//        1 = white
//        2 = yellow
void ModelModule::printMessage( const OString& message, int color /* = 1 */ )
{
  if( status->message == 0 )
    return;

  status->message->printMessage(message,color);
}

/////////////////////////////////////////////////////////////////////////////

// color: 0 = gray
//        1 = white
//        2 = yellow
void ModelModule::pushMessage( const OString& message, int color /* = 1 */ )
{
  if( status->message == 0 )
    return;

  status->message->pushMessage(message,color);
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::popMessage()
{
  if( status->message == 0 )
    return;

  status->message->popMessage();
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::handleCommand( MLCommand* cmd )
{
  if( cmd == 0 )
    return;

  if( cmd->getBase() == MLInterfaceCommand::BASE )
    {
    MLInterfaceCommand* icmd= (MLInterfaceCommand*) cmd;
    switch( icmd->getType() )
      {
      case MLInterfaceCommand::MINIMIZE_CANVAS:
      case MLInterfaceCommand::MAXIMIZE_CANVAS:
        {
        MLCanvas* c= MLAllocator::getCanvas(icmd->canvas);
        if( c != 0 )
          {
          for( int i= 1; i <= 4; ++i )
            {
            if( mcanvas[i] == 0 || mcanvas[i] == c )
              continue;
            if( icmd->getType() == MLInterfaceCommand::MAXIMIZE_CANVAS )
              mcanvas[i]->unmap();
             else
              mcanvas[i]->map();
            }
          }
        MLModule::handleCommand(icmd);
        return;
        }

      case MLInterfaceCommand::POPUP:
        {
        handlePopupCommand(icmd->index);
        return;
        }

      default:
        break;
      }
    }

  if( cmd->getBase() == ModelCommand::BASE )
    {
    handleModelCommand( (ModelCommand*) cmd );
    return;
    }

  MLModule::handleCommand(cmd);
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::enterMode( MLMode* mm )
{
  MLModule::enterMode(mm);

  if( mm != 0 && status->modeInfo != 0 )
    status->modeInfo->pushMessage(mm->getName());
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::leaveMode( MLMode* mm /* = 0 */ )
{
  if( status->modeInfo != 0 )
    status->modeInfo->popMessage();

  MLModule::leaveMode(mm);
}

/////////////////////////////////////////////////////////////////////////////

int ModelModule::getCanvasIndex( MLCanvas* c ) const
{
  if( c == mcanvas[1] ) return 1;
  if( c == mcanvas[2] ) return 2;
  if( c == mcanvas[3] ) return 3;
  if( c == mcanvas[4] ) return 4;

  return 0;
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::refreshScreen( unsigned int what )
{
  for( int i= 1; i <= 4; ++i )
    {
    if( mcanvas[i] == 0 )
      continue;
    mcanvas[i]->refreshScreen(what);
    }

  MLModule::refreshScreen(what);
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::refreshMainCamera( const MLCamera& camera )
{
  for( int i= 1; i <= 4; ++i )
    {
    if( mcanvas[i] == 0 )
      continue;
    mcanvas[i]->refreshMainCamera(camera);
    }
}

/////////////////////////////////////////////////////////////////////////////

const MLCamera& ModelModule::getMainCamera() const
{
  if( mcanvas[2] == 0 )
    {
    static MLCamera tmp;
    MLCamera c;
    tmp= c; // force initialization
    return tmp;
    }
  return mcanvas[2]->getPerspectiveCamera();
}

/////////////////////////////////////////////////////////////////////////////

// f==0 if leaving current
void ModelModule::enteringFunction( MLFunction* f )
{
  MLScrollingDialog* sd= MLModule::dialogs->getDialog("ModesList");
  if( sd == 0 )
    return;

  MLDModesList* modesList= (MLDModesList*) sd;
  modesList->update(f);
}

/////////////////////////////////////////////////////////////////////////////

// return 0 if none selected or more than a single one
int ModelModule::getSingleSelected()
{
  return modelScene->getSingleSelected();
}

/////////////////////////////////////////////////////////////////////////////

IBOOL ModelModule::isProjectEmpty()
{
  if( scene->getRootObject()->getNChildren() > 0 )
    return IFALSE;

  if( scene->getNMaterials() != 0 )
    return IFALSE;

  return ITRUE;
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::stopLightingProcess()
{
  if( lightingProcess == 0 )
    return;

  lightingProcess->stop();
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::runLightingProcess()
{
  if( lightingProcess == 0 )
    return;

  lightingProcess->compute();
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::disableLighting()
{
  if( !flags->computeDirectLighting && !flags->computeIndirectLighting )
    return;

  stopLightingProcess();

  flags->computeDirectLighting= IFALSE;
  flags->computeIndirectLighting= IFALSE;

  updateFlags();
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::enableLighting()
{
  if( flags->computeDirectLighting )
    return;

  stopLightingProcess();

  flags->computeDirectLighting= ITRUE;

  updateFlags();
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::resetFlags()
{
  flags->reset();
}

/////////////////////////////////////////////////////////////////////////////

void ModelModule::resetAll()
{
  stopLightingProcess();

  resetFlags();

  delete modelRender;
  modelRender= 0;

  updatePopupsFlags();
  updateStatusFlags();

  modelScene->resetAll();

  MLModule::resetAll();

  refreshScreen(REFRESH_ALL);
}

/////////////////////////////////////////////////////////////////////////////
