// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  PhotoModule.C

  Stephane Rehel

  October 25 1997
*/

#include <stdio.h>

#include "tools/Command.h"

#include "graphics/SystemWindow.h"

#include "interface/MLAllocator.h"
#include "interface/MLInterfaceCommand.h"
#include "interface/MLScrollingDialog.h"
#include "interface/MLMode.h"
#include "interface/MLStatusLineText.h"

#include "PhotoCanvas.h"
#include "PhotoModes.h"
#include "PhotoFunctions.h"
#include "PhotoCommand.h"
#include "PhotoStatusBars.h"
#include "PhotoPopups.h"

#include "PhotoModule.h"

#include "photo/photo/Photo.h"

/////////////////////////////////////////////////////////////////////////////

PhotoModule::PhotoModule( Interface* _interface ): MLModule(_interface)
{
  mcanvas[0]= mcanvas[1]= mcanvas[2]= mcanvas[3]= mcanvas[4]= 0;

  status= 0;

  the_photo= new Photo;

  popups= new PhotoPopups(this);

  MLModule::name= OString("Photo");
  MLModule::color= Color(0.05,0.20,0.15);

  PhotoModes::init(this,modes);
  PhotoFunctions::init(this);

  PhotoCommand::initPhotoCommands();
}

/////////////////////////////////////////////////////////////////////////////

PhotoModule::~PhotoModule()
{
  delete mcanvas[1]; mcanvas[1]= 0;
  delete mcanvas[2]; mcanvas[2]= 0;
  delete mcanvas[3]; mcanvas[3]= 0;
  delete mcanvas[4]; mcanvas[4]= 0;

  delete status;
  status= 0;

  delete popups;
  popups= 0;

  delete the_photo;
  the_photo= 0;
}

/////////////////////////////////////////////////////////////////////////////

void PhotoModule::init( Command& command )
{
  MLModule::init(command);

  status= new PhotoStatusBars(this,statusBars);
  status->init();

  mcanvas[1]= new PhotoCanvas;
  mcanvas[1]->create( this, CANVAS_1 );

  mcanvas[2]= new PhotoCanvas;
  mcanvas[2]->create( this, CANVAS_2 );

  mcanvas[3]= new PhotoCanvas;
  mcanvas[3]->create( this, CANVAS_3 );

  mcanvas[4]= new PhotoCanvas;
  mcanvas[4]->create( this, CANVAS_4 );

  popups->init(MLModule::buttonsPanel);
  updatePopupsFlags();

  createDialog("Welcome");
}

/////////////////////////////////////////////////////////////////////////////

void PhotoModule::updatePopupsFlags()
{
  if( popups != 0 )
    popups->updateFlags();
}

/////////////////////////////////////////////////////////////////////////////

void PhotoModule::updateStatusFlags()
{
  if( status != 0 )
    status->update();
}

/////////////////////////////////////////////////////////////////////////////

void PhotoModule::updateFlags()
{
  updateStatusFlags();
  updatePopupsFlags();
}

/////////////////////////////////////////////////////////////////////////////

void PhotoModule::idle()
{}

/////////////////////////////////////////////////////////////////////////////

void PhotoModule::map()
{
  if( _mapped )
    return;

  MLModule::map();

  if( mcanvas[1] != 0 ) mcanvas[1]->map();
  if( mcanvas[2] != 0 ) mcanvas[2]->map();
  if( mcanvas[3] != 0 ) mcanvas[3]->map();
  if( mcanvas[4] != 0 ) mcanvas[4]->map();
}

/////////////////////////////////////////////////////////////////////////////

void PhotoModule::draw()
{
  if( ! _mapped )
    return;

  if( mcanvas[1] != 0 ) mcanvas[1]->titleDraw();
  if( mcanvas[2] != 0 ) mcanvas[2]->titleDraw();
  if( mcanvas[3] != 0 ) mcanvas[3]->titleDraw();
  if( mcanvas[4] != 0 ) mcanvas[4]->titleDraw();

  MLModule::draw();
}

/////////////////////////////////////////////////////////////////////////////

// color: 0 = gray
//        1 = white
//        2 = yellow
void PhotoModule::printMessage( const OString& message, int color /* = 1 */ )
{
  if( status->message == 0 )
    return;

  status->message->printMessage(message,color);
}

/////////////////////////////////////////////////////////////////////////////

// color: 0 = gray
//        1 = white
//        2 = yellow
void PhotoModule::pushMessage( const OString& message, int color /* = 1 */ )
{
  if( status->message == 0 )
    return;

  status->message->pushMessage(message,color);
}

/////////////////////////////////////////////////////////////////////////////

void PhotoModule::popMessage()
{
  if( status->message == 0 )
    return;

  status->message->popMessage();
}

/////////////////////////////////////////////////////////////////////////////

void PhotoModule::handleCommand( MLCommand* cmd )
{
  if( cmd == 0 )
    return;

  if( cmd->getBase() == MLInterfaceCommand::BASE )
    {
    MLInterfaceCommand* icmd= (MLInterfaceCommand*) cmd;
    switch( icmd->getType() )
      {
      case MLInterfaceCommand::MINIMIZE_CANVAS:
      case MLInterfaceCommand::MAXIMIZE_CANVAS:
        {
        MLCanvas* c= MLAllocator::getCanvas(icmd->canvas);
        if( c != 0 )
          {
          for( int i= 1; i <= 4; ++i )
            {
            if( mcanvas[i] == 0 || mcanvas[i] == c )
              continue;
            if( icmd->getType() == MLInterfaceCommand::MAXIMIZE_CANVAS )
              mcanvas[i]->unmap();
             else
              mcanvas[i]->map();
            }
          }
        MLModule::handleCommand(icmd);
        return;
        }

      case MLInterfaceCommand::POPUP:
        {
        handlePopupCommand(icmd->index);
        return;
        }

      default:
        break;
      }
    }

  if( cmd->getBase() == PhotoCommand::BASE )
    {
    handlePhotoCommand( (PhotoCommand*) cmd );
    return;
    }

  MLModule::handleCommand(cmd);
}

/////////////////////////////////////////////////////////////////////////////

void PhotoModule::enterMode( MLMode* mm )
{
  MLModule::enterMode(mm);

  if( mm != 0 && status->modeInfo != 0 )
    status->modeInfo->pushMessage(mm->getName());
}

/////////////////////////////////////////////////////////////////////////////

void PhotoModule::leaveMode( MLMode* mm /* = 0 */ )
{
  if( status->modeInfo != 0 )
    status->modeInfo->popMessage();

  MLModule::leaveMode(mm);
}

/////////////////////////////////////////////////////////////////////////////

int PhotoModule::getCanvasIndex( MLCanvas* c ) const
{
  if( c == mcanvas[1] ) return 1;
  if( c == mcanvas[2] ) return 2;
  if( c == mcanvas[3] ) return 3;
  if( c == mcanvas[4] ) return 4;

  return 0;
}

/////////////////////////////////////////////////////////////////////////////

void PhotoModule::refreshScreen( unsigned int what )
{
  for( int i= 1; i <= 4; ++i )
    {
    if( mcanvas[i] == 0 )
      continue;
    mcanvas[i]->refreshScreen(what);
    }

  MLModule::refreshScreen(what);
}

/////////////////////////////////////////////////////////////////////////////

void PhotoModule::resetAll()
{
  updatePopupsFlags();
  updateStatusFlags();

  MLModule::resetAll();

  refreshScreen(REFRESH_ALL);
}

/////////////////////////////////////////////////////////////////////////////
