/*
 *  bltDragDrop.c --
 *
 *	This module implements a drag-and-drop mechanism for the Tk
 *	Toolkit.  Allows widgets to be registered as drag&drop sources
 *	and targets for handling "drag-and-drop" operations between
 *	Tcl/Tk applications.
 *
 * Copyright 1993-1997 Bell Labs Innovations for Lucent Technologies.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that the
 * copyright notice and warranty disclaimer appear in supporting documentation,
 * and that the names of Lucent Technologies any of their entities not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this software,
 * including all implied warranties of merchantability and fitness.  In no event
 * shall Lucent Technologies be liable for any special, indirect or
 * consequential damages or any damages whatsoever resulting from loss of use,
 * data or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or performance
 * of this software.
 *
 * The "drag&drop" command was created by Michael J. McLennan.
 */
#include "bltInt.h"
#include <X11/Xatom.h>

#define DRAGDROP_VERSION "2.1"

#if HAVE_NAMESPACES
static char dragDropCmd[] = "blt::drag&drop";
#else
static char dragDropCmd[] = "drag&drop";
#endif

/* Error Proc used to report drag&drop background errors */
#define DEF_ERROR_PROC              "tkerror"

static char className[] = "DragDrop";		/* CLASS NAME for token window */
static char propName[] = "DragDropInfo";	/* Property name */

/*
 *  DRAG&DROP ROOT WINDOW HIERARCHY (cached during "drag" operations)
 */

typedef struct DD_WinRep {
    Window win;			/* X window for this record */
    int initialized;		/* non-zero => rest of info is valid */
    int x0, y0;			/* upper-left corner of window */
    int x1, y1;			/* lower-right corner of window */
    char *ddprop;		/* drag&drop property info */
    char *ddinterp;		/* interp name within ddprop */
    char *ddwin;		/* target window name within ddprop */
    char *ddhandlers;		/* list of handlers within ddprop */
    struct DD_WinRep *parent;	/* window containing this as a child */
    struct DD_WinRep *kids;	/* list of child windows */
    struct DD_WinRep *next;	/* next sibling */
} DD_WinRep;

/*
 *  DRAG&DROP REGISTRATION DATA
 */
typedef struct {

    Tcl_Interp *interp;		/* Interpreter managing this drag&drop command */
    Tk_Window root;		/* Main window for application */
    Tcl_HashTable sourceTable;	/* List of source widgets */
    Tcl_HashTable targetTable;	/* List of target widgets */
    char *errorProc;		/* Proc invoked for drag&drop errors */
    int numActive;		/* Number of active drag&drop operations */
    int locx, locy;		/* Last location point */
    DD_WinRep *pool;		/* Pool of available DD_WinRep records */

} Registry;

typedef struct {
    Registry *registry;		/* parent registration list */
    Tk_Window tkwin;		/* registered window */
} DD_RegEntry;

/*
 *  DRAG&DROP SOURCE REGISTRATION RECORD
 */
typedef struct DragSourceHndl {
    char *dataType;		/* name of data type */
    char *cmd;			/* command used to send data */
    struct DragSourceHndl *next;	/* next handler in linked list */
} DragSourceHndl;

typedef struct Token {
    Tk_Window tkwin;		/* Window representing drag item */
    int lastX, lastY;		/* last position of token window */
    int overTarget;		/* non-zero => over target window */
    Tk_TimerToken timerToken;	/* token for routine to hide tokenwin */
    GC rejectFgGC;		/* GC used to draw rejection fg: (\) */
    GC rejectBgGC;		/* GC used to draw rejection bg: (\) */
    Window Xid;			/* X Window identifier for token */

    /* User-configurable fields */

    Tk_Anchor anchor;		/* Position of token win relative to mouse */
    Tk_Cursor cursor;		/* Cursor used when dragging token */
    Tk_3DBorder outline;	/* Outline border around token window */
    Tk_3DBorder normalBorder;	/* Border/background for token window */
    Tk_3DBorder activeBorder;	/* Border/background for token window */
    int relief;
    int activeRelief;
    int borderWidth;		/* Border width in pixels */
    int activeBorderWidth;	/* Border width in pixels */
    XColor *rejectFg;		/* Color used to draw rejection fg: (\) */
    XColor *rejectBg;		/* Color used to draw rejection bg: (\) */
    Pixmap rejectStipple;	/* Stipple used to draw rejection: (\) */

} Token;

typedef struct {
    Registry *registry;		/* registration list containing this */

    Display *display;		/* drag&drop source window display */
    Tk_Window tkwin;		/* drag&drop source window */
    Atom ddAtom;		/* X atom referring to "DragDropInfo" */
    int button;			/* button used to invoke drag for sources */

    Token token;		/* Information about drag&drop token window */

    int pkgcmdInProg;		/* non-zero => executing pkgcmd */
    char *pkgcmd;		/* cmd executed on start of drag op */
    char *pkgcmdResult;		/* result returned by recent pkgcmd */
    char *sitecmd;		/* cmd executed to update token win */

    DD_WinRep *allwins;		/* window info (used during "drag") */
    int selfTarget;		/* non-zero => source can drop onto itself */
    Tk_Cursor normalCursor;	/* cursor restored after dragging */

    char *send;			/* list of data handler names or "all" */
    DragSourceHndl *handlers;	/* list of data handlers */
} DragSource;

/*
 *  STACK INFO
 */
typedef struct DD_Stack {
    ClientData *values;		/* values on stack */
    int len;			/* number of values on stack */
    int max;			/* maximum size of stack */
    ClientData space[5];	/* initial space for stack data */
} DD_Stack;

/*
 *  PERCENT SUBSTITUTIONS
 */
typedef struct DD_PercentSubst {
    char letter;		/* character like 'x' in "%x" */
    char *value;		/* value to be substituted in place of "%x" */
} DD_PercentSubst;


/*
 *  DRAG&DROP TARGET REGISTRATION RECORD
 */
typedef struct DropTargetHndl {
    char *dataType;		/* Name of data type (malloc-ed) */
    char *command;		/* command to handle data type (malloc-ed) */
    struct DropTargetHndl *next;	/* next handler in linked list */
} DropTargetHndl;

typedef struct {
    Registry *registry;		/* registration list containing this */

    Display *display;		/* drag&drop target window display */
    Tk_Window tkwin;		/* drag&drop target window */
    DropTargetHndl *handlers;	/* list of data handlers */

} DropTarget;

/*
 * Each "drag&drop" widget window is tagged with a "DragDropInfo"
 * property in XA_STRING format.  This property identifies the
 * window as a "drag&drop" widget, and contains the following:
 *
 *     "<interp-name>]<drag&drop-path>]<handler-list>"
 *
 * The <drag&drop-path> is the window path name of the drag&drop
 * widget, <interp-name> is the name of the interpreter controlling
 * the widget (useful for the "send" command), and <handler-list>
 * is the list of handler types recognized by the widget.
 *
 * When the user invokes the "drag" operation, a snapshot of the
 * entire window hierarchy is made, and windows carrying a
 * "DragDropInfo" property are identified.  As the token window is
 * dragged around, * this snapshot can be queried to determine when
 * the token is over a valid target window.  When the token is
 * dropped over a valid site, the drop information is sent to the
 * application via the usual "send" command.  If communication fails,
 * the drag&drop facility automatically posts a rejection symbol on
 * the token window.
 */

/*
 *  Maximum size property that can be read at one time:
 */
#define MAX_PROP_SIZE 1000

/*
 *  CONFIG PARAMETERS
 */

#define DEF_DD_BUTTON_BG_COLOR		"yellow"
#define DEF_DD_BUTTON_BG_MONO		STD_NORMAL_BG_MONO
#define DEF_DD_BUTTON_NUMBER		"3"
#define DEF_DD_PACKAGE_COMMAND		(char *)NULL
#define DEF_DD_REJECT_BG_COLOR		STD_NORMAL_BG_COLOR
#define DEF_DD_REJECT_BG_MONO		WHITE
#define DEF_DD_REJECT_FG_COLOR		"red"
#define DEF_DD_REJECT_FG_MONO		BLACK
#define DEF_DD_REJECT_STIPPLE_COLOR	(char *)NULL
#define DEF_DD_REJECT_STIPPLE_MONO	"gray50"
#define DEF_DD_SELF_TARGET		"no"
#define DEF_DD_SEND			"all"
#define DEF_DD_SITE_COMMAND		(char *)NULL
#define DEF_DD_TOKEN_ACTIVE_BG_COLOR	STD_ACTIVE_BG_COLOR
#define DEF_DD_TOKEN_ACTIVE_BG_MONO	STD_ACTIVE_BG_MONO
#define DEF_DD_TOKEN_ACTIVE_BORDERWIDTH	"3"
#define DEF_DD_TOKEN_ACTIVE_RELIEF	"sunken"
#define DEF_DD_TOKEN_ANCHOR		"se"
#define DEF_DD_TOKEN_BG_COLOR		STD_NORMAL_BG_COLOR
#define DEF_DD_TOKEN_BG_MONO		STD_NORMAL_BG_MONO
#define DEF_DD_TOKEN_BORDERWIDTH	"3"
#define DEF_DD_TOKEN_CURSOR		"top_left_arrow"
#define DEF_DD_TOKEN_OUTLINE_COLOR	BLACK
#define DEF_DD_TOKEN_OUTLINE_MONO	BLACK
#define DEF_DD_TOKEN_RELIEF		"raised"

static Tk_ConfigSpec configSpecs[] =
{
    {TK_CONFIG_INT, "-button", "buttonBinding", "ButtonBinding",
	DEF_DD_BUTTON_NUMBER, Tk_Offset(DragSource, button), 0},
    {TK_CONFIG_STRING, "-packagecmd", "packageCommand", "Command",
	DEF_DD_PACKAGE_COMMAND, Tk_Offset(DragSource, pkgcmd), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-rejectbg", "rejectBackground", "Background",
	DEF_DD_REJECT_BG_COLOR, Tk_Offset(DragSource, token.rejectBg), 
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-rejectbg", "rejectBackground", "Background",
	DEF_DD_REJECT_BG_MONO, Tk_Offset(DragSource, token.rejectBg), 
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_COLOR, "-rejectfg", "rejectForeground", "Foreground",
	DEF_DD_REJECT_FG_COLOR, Tk_Offset(DragSource, token.rejectFg), 
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-rejectfg", "rejectForeground", "Foreground",
	DEF_DD_REJECT_BG_COLOR, Tk_Offset(DragSource, token.rejectFg), 
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_BITMAP, "-rejectstipple", "rejectStipple", "Stipple",
	DEF_DD_REJECT_STIPPLE_COLOR, Tk_Offset(DragSource, token.rejectStipple), 
	TK_CONFIG_COLOR_ONLY | TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-rejectstipple", "rejectStipple", "Stipple",
	DEF_DD_REJECT_STIPPLE_MONO, Tk_Offset(DragSource, token.rejectStipple), 
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_BOOLEAN, "-selftarget", "selfTarget", "SelfTarget",
	DEF_DD_SELF_TARGET, Tk_Offset(DragSource, selfTarget), 0},
    {TK_CONFIG_STRING, "-send", "send", "Send",
	DEF_DD_SEND, Tk_Offset(DragSource, send), TK_CONFIG_NULL_OK},
    {TK_CONFIG_STRING, "-sitecmd", "siteCommand", "Command",
	DEF_DD_SITE_COMMAND, Tk_Offset(DragSource, sitecmd), TK_CONFIG_NULL_OK},
    {TK_CONFIG_ANCHOR, "-tokenanchor", "tokenAnchor", "Anchor",
	DEF_DD_TOKEN_ANCHOR, Tk_Offset(DragSource, token.anchor), 0},
    {TK_CONFIG_BORDER, "-tokenactivebackground", 
	"tokenActiveBackground", "ActiveBackground",
	DEF_DD_TOKEN_ACTIVE_BG_COLOR, Tk_Offset(DragSource, token.activeBorder), 
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-tokenactivebackground", 
        "tokenActiveBackground", "ActiveBackground",
	DEF_DD_TOKEN_ACTIVE_BG_MONO, Tk_Offset(DragSource, token.activeBorder), 
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_BORDER, "-tokenbg", "tokenBackground", "Background",
	DEF_DD_TOKEN_BG_COLOR, Tk_Offset(DragSource, token.normalBorder), 
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-tokenbg", "tokenBackground", "Background",
	DEF_DD_TOKEN_BG_MONO, Tk_Offset(DragSource, token.normalBorder), 
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_BORDER, "-tokenoutline", "tokenOutline", "Outline",
	DEF_DD_TOKEN_OUTLINE_COLOR, Tk_Offset(DragSource, token.outline),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-tokenoutline", "tokenOutline", "Outline",
	DEF_DD_TOKEN_OUTLINE_MONO, Tk_Offset(DragSource, token.outline),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_PIXELS, "-tokenborderwidth", "tokenBorderWidth", "BorderWidth",
	DEF_DD_TOKEN_BORDERWIDTH, Tk_Offset(DragSource, token.borderWidth), 0},
    {TK_CONFIG_CURSOR, "-tokencursor", "tokenCursor", "Cursor",
	DEF_DD_TOKEN_CURSOR, Tk_Offset(DragSource, token.cursor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_END, (char *)NULL, (char *)NULL, (char *)NULL, (char *)NULL, 0, 0},
};

static Tk_ConfigSpec tokenConfigSpecs[] =
{
    {TK_CONFIG_BORDER, "-activebackground", "activeBackground", "ActiveBackground",
	DEF_DD_TOKEN_ACTIVE_BG_COLOR, Tk_Offset(Token, activeBorder), 
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-activebackground", "activeBackground", "ActiveBackground",
	DEF_DD_TOKEN_ACTIVE_BG_MONO, Tk_Offset(Token, activeBorder), 
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_RELIEF, "-activerelief", "activeRelief", "activeRelief",
	DEF_DD_TOKEN_ACTIVE_RELIEF, Tk_Offset(Token, activeRelief), 0},
    {TK_CONFIG_ANCHOR, "-anchor", "anchor", "Anchor",
	DEF_DD_TOKEN_ANCHOR, Tk_Offset(Token, anchor), 0},
    {TK_CONFIG_PIXELS, "-activeborderwidth", "activeBorderWidth", 
	"ActiveBorderWidth",
	DEF_DD_TOKEN_ACTIVE_BORDERWIDTH, Tk_Offset(Token, activeBorderWidth), 0},
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	DEF_DD_TOKEN_BG_COLOR, Tk_Offset(Token, normalBorder), 
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	DEF_DD_TOKEN_BG_MONO, Tk_Offset(Token, normalBorder), 
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	DEF_DD_TOKEN_BORDERWIDTH, Tk_Offset(Token, borderWidth), 0},
    {TK_CONFIG_CURSOR, "-cursor", "cursor", "Cursor",
	DEF_DD_TOKEN_CURSOR, Tk_Offset(Token, cursor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BORDER, "-outline", "outline", "Outline",
	DEF_DD_TOKEN_OUTLINE_COLOR, Tk_Offset(Token, outline), 
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-outline", "outline", "Outline",
	DEF_DD_TOKEN_OUTLINE_MONO, Tk_Offset(Token, outline), TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_COLOR, "-rejectbg", "rejectBackground", "Background",
	DEF_DD_REJECT_BG_COLOR, Tk_Offset(Token, rejectBg), TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-rejectbg", "rejectBackground", "Background",
	DEF_DD_REJECT_BG_MONO, Tk_Offset(Token, rejectBg), TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_COLOR, "-rejectfg", "rejectForeground", "Foreground",
	DEF_DD_REJECT_FG_COLOR, Tk_Offset(Token, rejectFg), TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-rejectfg", "rejectForeground", "Foreground",
	DEF_DD_REJECT_BG_COLOR, Tk_Offset(Token, rejectFg), TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_BITMAP, "-rejectstipple", "rejectStipple", "Stipple",
	DEF_DD_REJECT_STIPPLE_COLOR, Tk_Offset(Token, rejectStipple), 
	TK_CONFIG_COLOR_ONLY | TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-rejectstipple", "rejectStipple", "Stipple",
	DEF_DD_REJECT_STIPPLE_MONO, Tk_Offset(Token, rejectStipple), 
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
	DEF_DD_TOKEN_RELIEF, Tk_Offset(Token, relief), 0},
    {TK_CONFIG_END, (char *)NULL, (char *)NULL, (char *)NULL, (char *)NULL, 0, 0},
};

/*
 *  FORWARD DECLARATIONS
 */
int Blt_DragDropInit _ANSI_ARGS_((Tcl_Interp *interp));

static void DragDrop_Delete _ANSI_ARGS_((ClientData clientData));
static int DragDrop_Cmd _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, int argc, char **argv));

static DragSource *CreateSource _ANSI_ARGS_((Registry * registry, char *pathname, 
	int *newEntry));
static DragSource *FindSource _ANSI_ARGS_((Registry * registry, char *pathname));
static void DestroySource _ANSI_ARGS_((Registry * registry, char *pathName));
static int ConfigureSource _ANSI_ARGS_((Tcl_Interp *interp, DragSource * srcPtr,
	int argc, char **argv, int flags));
static int ConfigureToken _ANSI_ARGS_((Tcl_Interp *interp, DragSource *srcPtr,
	int argc, char **argv));
static char *FindSourceHandler _ANSI_ARGS_((DragSource * srcPtr, char *dtname));
static void PutSourceHandler _ANSI_ARGS_((DragSource * srcPtr, char *dtname,
	char *cmd));
static DragSourceHndl *CreateSourceHandler _ANSI_ARGS_((char *dtname, char *cmd));
static void DestroySourceHandler _ANSI_ARGS_((DragSourceHndl * dsHndl));
static void UnregSource _ANSI_ARGS_((ClientData clientData, XEvent *eventPtr));

static DropTarget *CreateTargetInfo _ANSI_ARGS_((Registry * registry,
	char *pathname, int *newEntry));
static void DestroyTargetInfo _ANSI_ARGS_((Registry * registry,
	char *pathName));
static char *FindTargetHandler _ANSI_ARGS_((DropTarget * targetPtr, char *dtname));
static void PutTargetHandler _ANSI_ARGS_((DropTarget * targetPtr, char *dtname,
	char *cmd));
static DropTargetHndl *CreateTargetHandler _ANSI_ARGS_((char *dtname,
	char *cmd));
static void DestroyTargetHandler _ANSI_ARGS_((DropTargetHndl * dtHndl));
static void UnregTarget _ANSI_ARGS_((ClientData clientData, XEvent *eventPtr));

static void DragDropSend _ANSI_ARGS_((DragSource * srcPtr));
static char *DragDropSendHndlr _ANSI_ARGS_((DragSource * srcPtr,
	char *interpName, char *ddName));

static DD_WinRep *GetWinRepInfo _ANSI_ARGS_((DragSource * srcPtr, 
	Registry * registry));
static DD_WinRep *FindTargetWin _ANSI_ARGS_((DragSource * srcPtr, int x, int y));
static DD_WinRep *WinRepAlloc _ANSI_ARGS_((Registry * registry));
static void WinRepRelease _ANSI_ARGS_((DD_WinRep * wr, Registry * registry));
static void WinRepInit _ANSI_ARGS_((DD_WinRep * wr, DragSource * srcPtr));

static void AddPropertyToTarget _ANSI_ARGS_((DropTarget * targetPtr));
static void TokenEventProc _ANSI_ARGS_((ClientData clientData, XEvent *eventPtr));
static void MoveToken _ANSI_ARGS_((DragSource * srcPtr, Token *tokenPtr));
static void UpdateToken _ANSI_ARGS_((ClientData clientData));
static void HideToken _ANSI_ARGS_((Token *tokenPtr));
static void RejectToken _ANSI_ARGS_((Token *tokenPtr));

static void StackInit _ANSI_ARGS_((DD_Stack * stack));
static void StackDelete _ANSI_ARGS_((DD_Stack * stack));
static void StackPush _ANSI_ARGS_((ClientData cdata, DD_Stack * stack));
static ClientData StackPop _ANSI_ARGS_((DD_Stack * stack));

static char *ExpandPercents _ANSI_ARGS_((char *str, DD_PercentSubst * subs, 
	int nsubs, Tcl_DString *dsPtr));


/*
 * ------------------------------------------------------------------------
 *  Blt_DragDropInit()
 *
 *  Adds the drag&drop command to the given interpreter.  Should be
 *  invoked to properly install the command whenever a new interpreter
 *  is created.
 * ------------------------------------------------------------------------
 */
int
Blt_DragDropInit(interp)
    Tcl_Interp *interp;		/* interpreter to be updated */
{
    Registry *registry;
    static Blt_CmdSpec cmdSpec = {"drag&drop", DragDrop_Cmd, DragDrop_Delete,};

    /*
     *  Install drag&drop facilities.
     */
    registry = (Registry *) malloc(sizeof(Registry));
    registry->interp = interp;
    registry->root = Tk_MainWindow(interp);
    Tcl_InitHashTable(&(registry->sourceTable), TCL_STRING_KEYS);
    Tcl_InitHashTable(&(registry->targetTable), TCL_STRING_KEYS);
    registry->errorProc = strdup(DEF_ERROR_PROC);
    registry->numActive = 0;
    registry->locx = registry->locy = 0;
    registry->pool = NULL;

    cmdSpec.clientData = (ClientData)registry;
    if (Blt_InitCmd(interp, "blt", &cmdSpec) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * ------------------------------------------------------------------------
 *  DragDrop_Delete()
 *
 *  Invoked when the drag&drop command is removed from an interpreter
 *  to free up allocated memory.
 * ------------------------------------------------------------------------
 */
static void
DragDrop_Delete(cdata)
    ClientData cdata;		/* client data for drag&drop command */
{
    Registry *registry = (Registry *) cdata;
    DD_WinRep *wrpool, *wrnext;

    Tcl_DeleteHashTable(&(registry->sourceTable));
    Tcl_DeleteHashTable(&(registry->targetTable));

    if (registry->errorProc != NULL) {
	free((char *)registry->errorProc);
    }
    wrnext = NULL;		/* Suppress compiler warning */
    for (wrpool = registry->pool; wrpool != NULL; wrpool = wrnext) {
	wrnext = wrpool->next;
	free((char *)wrpool);
    }
    free((char *)registry);
}

static void
RaiseToken(srcPtr, tokenPtr)
    DragSource *srcPtr;
    Token *tokenPtr;
{
    XWindowChanges changes;
    unsigned int mask;

    if (tokenPtr->Xid == 0) {
	tokenPtr->Xid = Blt_XWindowId(tokenPtr->tkwin);
    }
    changes.stack_mode = Above;
    changes.sibling = None;
    mask = CWStackMode;
    XReconfigureWMWindow(srcPtr->display, tokenPtr->Xid, 
	 Tk_ScreenNumber(tokenPtr->tkwin), mask,  &changes);
}

/*
 *  Create the window for the drag&drop token...
 */
static int
CreateToken(interp, srcPtr, registry)
    Tcl_Interp *interp;
    DragSource *srcPtr; 
    Registry *registry;
{
    DD_RegEntry *ddentry;
    XSetWindowAttributes atts;
    Tk_Window tkwin;
    char tokenName[200];
    static nextTokenId = 0;
    Token *tokenPtr = &(srcPtr->token);

    sprintf(tokenName, "dd-token%d", ++nextTokenId);
    tkwin = Tk_CreateWindow(interp, srcPtr->tkwin, tokenName, "");
    if (tkwin == NULL) {
	Tcl_AppendResult(interp, "could not create token window", (char *)NULL);
	return TCL_ERROR;
    }
    Tk_SetClass(tkwin, className);
    Tk_CreateEventHandler(tkwin, ExposureMask | StructureNotifyMask,
	  TokenEventProc, (ClientData)tokenPtr);
    
    atts.override_redirect = True;
    atts.save_under = True;
    Tk_ChangeWindowAttributes(tkwin, CWOverrideRedirect | CWSaveUnder, &atts);
    Tk_SetInternalBorder(tkwin, tokenPtr->borderWidth + 2);
    tokenPtr->tkwin = tkwin;
    
    if (srcPtr->button > 0) {
	char eventSpec[200];

	sprintf(eventSpec, " <ButtonPress-%d> ", srcPtr->button);
	if (Tcl_VarEval(interp, "bind ", Tk_PathName(srcPtr->tkwin), eventSpec, 
	" { ", dragDropCmd, " drag ", Tk_PathName(srcPtr->tkwin), " %X %Y }",
		(char *)NULL) != TCL_OK) {
	    goto error;
	}
	sprintf(eventSpec, " <B%d-Motion> ", srcPtr->button);
	if (Tcl_VarEval(interp, "bind ", Tk_PathName(srcPtr->tkwin), eventSpec, 
	" { ", dragDropCmd, " drag ", Tk_PathName(srcPtr->tkwin), " %X %Y }",
		(char *)NULL) != TCL_OK) {
	    goto error;
	}
	sprintf(eventSpec, " <ButtonRelease-%d> ", srcPtr->button);
	if (Tcl_VarEval(interp, "bind ", Tk_PathName(srcPtr->tkwin), eventSpec, 
	" { ", dragDropCmd, " drop ", Tk_PathName(srcPtr->tkwin), " %X %Y }",
		(char *)NULL) != TCL_OK) {
	    goto error;
	}
    }
    /*
     *  Arrange for the window to unregister itself when it
     *  is destroyed.
     */
    ddentry = (DD_RegEntry *) malloc(sizeof(DD_RegEntry));
    ddentry->registry = registry;
    ddentry->tkwin = srcPtr->tkwin;
    Tk_CreateEventHandler(srcPtr->tkwin, StructureNotifyMask, UnregSource, 
	(ClientData)ddentry);
    return TCL_OK;
 error:
    Tk_DestroyWindow(tkwin);
    return TCL_ERROR;
}


/*
 * ------------------------------------------------------------------------
 *  DragDrop_Cmd()
 *
 *  Invoked by TCL whenever the user issues a drag&drop command.
 *  Handles the following syntax:
 *
 *    drag&drop source
 *    drag&drop source <pathName> ?options...?
 *    drag&drop source <pathName> handler ?<dataType>? ?<cmd> <arg>...?
 *
 *    drag&drop target
 *    drag&drop target <pathName> handler ?<dataType> <cmd> <arg>...?
 *    drag&drop target <pathName> handle <dataType> ?<value>?
 *
 *    drag&drop token <pathName>
 *    drag&drop drag <pathName> <x> <y>
 *    drag&drop drop <pathName> <x> <y>
 *
 *    drag&drop errors ?<proc>?
 *    drag&drop active
 *    drag&drop location ?<x> <y>?
 *
 * ------------------------------------------------------------------------
 */
static int
DragDrop_Cmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Registration information */
    Tcl_Interp *interp;		/* current interpreter */
    int argc;			/* number of arguments */
    char **argv;		/* argument strings */
{
    Registry *registry = (Registry *) clientData;
    DD_RegEntry *ddentry;
    register DragSource *srcPtr;
    register DropTarget *targetPtr;
    Token *tokenPtr;
    int status, length, x, y, isNew;
    char c, *cmd;
    DD_PercentSubst subst[2];
    Tk_Window tkwin;
    char buffer[1024];

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
	    " oper ?args?\"", (char *)NULL);
	return TCL_ERROR;
    }
    c = argv[1][0];
    length = strlen(argv[1]);

    /*
     *  HANDLE:  drag&drop source
     *           drag&drop source <pathName> ?options...?
     *           drag&drop source <pathName> handler ?<data>? ?<scmd> <arg>...?
     */
    if ((c == 's') && strncmp(argv[1], "source", length) == 0) {
	/*
         *  HANDLE:  drag&drop source
         */
	if (argc == 2) {
	    Tcl_HashSearch cursor;
	    Tcl_HashEntry *hPtr;
	    char *name;

	    for (hPtr = Tcl_FirstHashEntry(&(registry->sourceTable), &cursor);
		 hPtr != NULL; hPtr = Tcl_NextHashEntry(&cursor)) {
		name = Tcl_GetHashKey(&(registry->sourceTable), hPtr);
		Tcl_AppendElement(interp, name);
	    }
	    return TCL_OK;
	}
	/*
         *  Find or create source info...
         */
	srcPtr = CreateSource(registry, argv[2], &isNew);
	if (srcPtr == NULL) {
	    return TCL_ERROR;
	}
	tokenPtr = &(srcPtr->token);
	if (argc > 3) {
	    /*
	     *  HANDLE:  drag&drop source <pathName> ?options...?
	     */
	    c = argv[3][0];
	    length = strlen(argv[3]);

	    if (c == '-') {
		if (argc == 3) {
		    status = Tk_ConfigureInfo(interp, tokenPtr->tkwin, configSpecs,
			(char *)srcPtr, (char *)NULL, 0);
		} else if (argc == 4) {
		    status = Tk_ConfigureInfo(interp, tokenPtr->tkwin, configSpecs,
			 (char *)srcPtr, argv[3], 0);
		} else {
		    status = ConfigureSource(interp, srcPtr, argc - 3, argv + 3,
			TK_CONFIG_ARGV_ONLY);
		}
		if (status != TCL_OK) {
		    return TCL_ERROR;
		}
	    } else if ((c == 'h') && strncmp(argv[3], "handler", length) == 0) { 
		/*
		 *  HANDLE:  drag&drop source <pathName> handler \
		 *             ?<data>? ?<scmd>...?
		 */
		if (argc == 4) {
		    DragSourceHndl *dsHndl = srcPtr->handlers;
		    while (dsHndl) {
			Tcl_AppendElement(interp, dsHndl->dataType);
			dsHndl = dsHndl->next;
		    }
		    return TCL_OK;
		} else if (argc == 5) {
		    /*
		     *  HANDLE:  drag&drop source <pathName> handler <data>
		     *
		     *    Create the new <data> type if it doesn't already
		     *    exist, and return the code associated with it.
		     */
		    cmd = FindSourceHandler(srcPtr, argv[4]);
		    if (cmd) {
			Tcl_SetResult(interp, cmd, TCL_STATIC);
		    } else if (strstr(argv[4], " ")) {
			Tcl_AppendResult(interp,
			    "bad source handler name \"",
			    argv[4], "\": should not contain spaces",
			    (char *)NULL);
			return TCL_ERROR;
		    } else {
			PutSourceHandler(srcPtr, argv[4], "");
		    }
		    return TCL_OK;
		} else {
		    /*
		     *  HANDLE:  drag&drop source <pathName> handler \
		     *               <data> <cmd> ?<arg>...?
		     *
		     *    Create the new <data> type and set its command
		     */
		    if (strstr(argv[4], " ")) {
			Tcl_AppendResult(interp,
			    "bad source handler name \"",
			    argv[4], "\": should not contain spaces",
			    (char *)NULL);
			return TCL_ERROR;
		    }
		    cmd = Tcl_Concat(argc - 5, argv + 5);
		    PutSourceHandler(srcPtr, argv[4], cmd);
		    ckfree(cmd);
		    return TCL_OK;
		}
	    } else {
		Tcl_AppendResult(interp, "bad operation \"", argv[3],
		    "\": must be \"handler\" or a configuration option",
		    (char *)NULL);
		return TCL_ERROR;
	    }
	}

	if (isNew) {
	    /*
             *  Create the window for the drag&drop token...
             */
	    if (CreateToken(interp, srcPtr, registry) != TCL_OK) {
		DestroySource(registry, argv[2]);
		return TCL_ERROR;
	    }
	}
    } else if ((c == 't') && (length >= 2) && 
	       (strncmp(argv[1], "target", length) == 0)) {
	/*
	 *  HANDLE:  drag&drop target ?<pathName>? ?handling info...?
	 */
	if (argc == 2) {
	    Tcl_HashSearch cursor;
	    Tcl_HashEntry *hPtr;

	    for(hPtr = Tcl_FirstHashEntry(&(registry->targetTable), &cursor); 
		hPtr != NULL; hPtr = Tcl_NextHashEntry(&cursor)) {
		Tcl_AppendElement(interp, 
				  Tcl_GetHashKey(&(registry->targetTable), hPtr));
	    }
	    return TCL_OK;
	}
	tkwin = Tk_NameToWindow(interp, argv[2], registry->root);
	if (tkwin == NULL) {
	    Tcl_AppendResult(interp, "window does not exist: ", argv[2],
		(char *)NULL);
	    return TCL_ERROR;
	}
	targetPtr = CreateTargetInfo(registry, argv[2], &isNew);
	targetPtr->tkwin = tkwin;
	targetPtr->display = Tk_Display(tkwin);

	/*
         *  If this is a new target, attach a property to identify
         *  window as "drag&drop" target, and arrange for the window
         *  to un-register itself when it is destroyed.
         */
	if (isNew) {
	    Tk_MakeWindowExist(targetPtr->tkwin);
	    AddPropertyToTarget(targetPtr);

	    /*
             *  Arrange for the window to unregister itself when it
             *  is destroyed.
             */
	    ddentry = (DD_RegEntry *) malloc(sizeof(DD_RegEntry));
	    ddentry->registry = registry;
	    ddentry->tkwin = targetPtr->tkwin;
	    Tk_CreateEventHandler(targetPtr->tkwin, StructureNotifyMask,
		UnregTarget, (ClientData)ddentry);
	}

	if ((argc >= 4) && (strcmp(argv[3], "handler") == 0)) {
	    /*
	     *  HANDLE:  drag&drop target <pathName> handler
	     *           drag&drop target <pathName> handler ?<data> <cmd> <arg>...?
	     */
	    if (argc == 4) {
		DropTargetHndl *dtHndl = targetPtr->handlers;
		while (dtHndl) {
		    Tcl_AppendElement(interp, dtHndl->dataType);
		    dtHndl = dtHndl->next;
		}
		return TCL_OK;
	    } else if (argc >= 6) {
		/*
		 *  Process handler definition
		 */
		if (strstr(argv[4], " ")) {
		    Tcl_AppendResult(interp,
			"bad source handler name \"",
			argv[4], "\": should not contain spaces",
			(char *)NULL);
		    return TCL_ERROR;
		}
		cmd = Tcl_Concat(argc - 5, argv + 5);
		PutTargetHandler(targetPtr, argv[4], cmd);
		ckfree(cmd);
		return TCL_OK;
	    }
	    Tcl_AppendResult(interp,
		"wrong # args: should be \"", argv[0], " ", argv[1],
		" ", argv[2], " ", argv[3], " data command ?arg arg...?",
		(char *)NULL);
	    return TCL_ERROR;
	} else if ((argc >= 4) && (strcmp(argv[3], "handle") == 0)) {
	    /*
	     *  HANDLE:  drag&drop target <pathName> handle <data> ?<value>?
	     */
	    Tcl_DString cmdStr;
	    int result;

	    if (argc < 5 || argc > 6) {
		Tcl_AppendResult(interp,
		    "wrong # args: should be \"", argv[0], " ", argv[1],
		    " ", argv[2], " handle data ?value?",
		    (char *)NULL);
		return TCL_ERROR;
	    }
	    cmd = FindTargetHandler(targetPtr, argv[4]);
	    if (cmd == NULL) {
		Tcl_AppendResult(interp, "target cannot handle datatype: ",
		    argv[4], (char *)NULL);
		return TCL_ERROR;	/* no handler found */
	    }
	    subst[0].letter = 'W';
	    subst[0].value = Tk_PathName(targetPtr->tkwin);
	    subst[1].letter = 'v';
	    if (argc > 5) {
		subst[1].value = argv[5];
	    } else {
		subst[1].value = "";
	    }
	    Tcl_DStringInit(&cmdStr);
	    result = Tcl_Eval(interp, ExpandPercents(cmd, subst, 2, &cmdStr));
	    Tcl_DStringFree(&cmdStr);
	    return result;
	} else {
	    Tcl_AppendResult(interp, "usage: ", argv[0], " target ", argv[2],
		" handler ?data command arg arg...?\n   or: ",
		argv[0], " target ", argv[2], " handle <data>",
		(char *)NULL);
	    return TCL_ERROR;
	}
    } else if ((c == 't') && (length >= 2) && 
	(strncmp(argv[1], "token", length) == 0)) {
	/*
	 *  HANDLE:  drag&drop token <pathName>
	 */
	srcPtr = FindSource(registry, argv[2]);
	if (srcPtr == NULL) {
	    Tcl_AppendResult(interp, "window \"", argv[2],
		"\" has not been initialized as a drag&drop source",
		(char *)NULL);
	    return TCL_ERROR;
	}
	if (argc > 3) {
	    if (ConfigureToken(interp, srcPtr, argc - 3, argv + 3) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
	Tcl_SetResult(interp, Tk_PathName(srcPtr->token.tkwin), TCL_STATIC);
	return TCL_OK;
    } else if ((c == 'd') && strncmp(argv[1], "drag", length) == 0) {
	/*
	 *  HANDLE:  drag&drop drag <path> <x> <y>
	 */
	if (argc < 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " drag pathname x y\"",
		(char *)NULL);
	    return TCL_ERROR;
	}
	srcPtr = FindSource(registry, argv[2]);
	if (srcPtr == NULL) {
	    Tcl_AppendResult(interp, "not a drag&drop source: ", argv[2],
		(char *)NULL);
	    return TCL_ERROR;
	}
	if ((Tcl_GetInt(interp, argv[3], &x) != TCL_OK) ||
	    (Tcl_GetInt(interp, argv[4], &y) != TCL_OK))
	    return TCL_ERROR;

	tokenPtr = &(srcPtr->token);
	registry->locx = x;	/* save drag&drop location */
	registry->locy = y;
	tokenPtr->lastX = x;
	tokenPtr->lastY = y;

	/*
         *  If HideToken() is pending, then do it now!
         */
	if (tokenPtr->timerToken != 0) {
	    Tk_DeleteTimerHandler(tokenPtr->timerToken);
	    HideToken(tokenPtr);
	}
	/*
         *  If pkgcmd is in progress, then ignore subsequent calls
         *  until it completes.  Only perform drag if pkgcmd
         *  completed successfully and token window is mapped.
         */
	if ((!Tk_IsMapped(tokenPtr->tkwin)) && (!srcPtr->pkgcmdInProg)) {
	    Tcl_DString cmdStr;

	    /*
             *  No list of send handlers?  Then source is disabled.
             *  Abort drag quietly.
             */
	    if (srcPtr->send == NULL)
		return TCL_OK;

	    /*
             *  No token command?  Then cannot build token.
             *  Signal error.
             */
	    if (srcPtr->pkgcmd == NULL) {
		Tcl_AppendResult(interp, "missing -packagecmd: ", argv[2],
		    (char *)NULL);
		return TCL_ERROR;
	    }
	    /*
             *  Execute token command to initialize token window.
             */
	    srcPtr->pkgcmdInProg = ~0;
	    subst[0].letter = 'W';
	    subst[0].value = Tk_PathName(srcPtr->tkwin);
	    subst[1].letter = 't';
	    subst[1].value = Tk_PathName(tokenPtr->tkwin);

	    Tcl_DStringInit(&cmdStr);
	    status = Tcl_Eval(srcPtr->registry->interp,
		ExpandPercents(srcPtr->pkgcmd, subst, 2, &cmdStr));
	    Tcl_DStringFree(&cmdStr);

	    srcPtr->pkgcmdInProg = 0;

	    /*
             *  Null string from the package command?
             *  Then quietly abort the drag&drop operation.
             */
	    if (*interp->result == '\0') {
		return TCL_OK;
	    }
	    /*
             *  Save result of token command for send command.
             */
	    if (srcPtr->pkgcmdResult != NULL) {
		free(srcPtr->pkgcmdResult);
	    }
	    srcPtr->pkgcmdResult = strdup(interp->result);

	    /*
             *  Token building failed?  If an error handler is defined,
             *  then signal the error.  Otherwise, abort quietly.
             */
	    if (status != TCL_OK) {
		if (registry->errorProc && *registry->errorProc) {
		    return Tcl_VarEval(registry->interp,
			registry->errorProc, " {", registry->interp->result, "}",
			(char *)NULL);
		} else
		    return TCL_OK;
	    }
	    /*
             *  Install token cursor...
             */
	    if (tokenPtr->cursor != None) {
		status = Tcl_VarEval(srcPtr->registry->interp,
		    Tk_PathName(srcPtr->tkwin), " configure -cursor",
		    (char *)NULL);

		if (status == TCL_OK) {
		    char *cname = interp->result;
		    while (*cname != '\0') {
			cname++;
		    }
		    while ((cname > interp->result) && (*(cname - 1) != ' ')) {
			cname--;
		    }
		    if (srcPtr->normalCursor != None) {
			Tk_FreeCursor(srcPtr->display, srcPtr->normalCursor);
			srcPtr->normalCursor = None;
		    }
		    if (strcmp(cname, "{}") != 0) {
			srcPtr->normalCursor = Tk_GetCursor(interp,
			    srcPtr->tkwin, Tk_GetUid(cname));
		    }
		}
		Tk_DefineCursor(srcPtr->tkwin, tokenPtr->cursor);
	    }
	    /*
             *  Get ready to drag token window...
             *  1) Cache info for all windows on root
             *  2) Map token window to begin drag operation
             */
	    if (srcPtr->allwins) {
		WinRepRelease(srcPtr->allwins, registry);
	    }
	    srcPtr->allwins = GetWinRepInfo(srcPtr, registry);

	    registry->numActive++;	/* one more drag&drop window active */
	    
	    if (!Tk_IsMapped(tokenPtr->tkwin)) {
		Tk_MapWindow(tokenPtr->tkwin);
	    }
	    if (Tk_WindowId(tokenPtr->tkwin) == None) {
		Tk_MakeWindowExist(tokenPtr->tkwin);
	    }
	    RaiseToken(srcPtr, tokenPtr);
	}
	/*
         *  Arrange to update status of token window...
         */
	Tk_CancelIdleCall(UpdateToken, (ClientData)srcPtr);
	Tk_DoWhenIdle(UpdateToken, (ClientData)srcPtr);

	/*
         *  Move the token window to the current drag point...
         */
	MoveToken(srcPtr, tokenPtr);
    } else if ((c == 'd') && strncmp(argv[1], "drop", length) == 0) {
	/*
	 *  HANDLE:  drag&drop drop <path> <x> <y>
	 */
	if (argc < 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " drop pathname x y\"", (char *)NULL);
	    return TCL_ERROR;
	}
	srcPtr = FindSource(registry, argv[2]);
	if (srcPtr == NULL) {
	    Tcl_AppendResult(interp, "not a drag&drop source: ", argv[2],
		(char *)NULL);
	    return TCL_ERROR;
	}
	if ((Tcl_GetInt(interp, argv[3], &x) != TCL_OK) ||
	    (Tcl_GetInt(interp, argv[4], &y) != TCL_OK))
	    return TCL_ERROR;

	registry->locx = x;	/* save drag&drop location */
	registry->locy = y;
	srcPtr->token.lastX = x;
	srcPtr->token.lastY = y;

	/*
         *  Put the cursor back to its usual state.
         */
	if (srcPtr->normalCursor == None) {
	    Tk_UndefineCursor(srcPtr->tkwin);
	} else {
	    Tk_DefineCursor(srcPtr->tkwin, srcPtr->normalCursor);
	}
	Tk_CancelIdleCall(UpdateToken, (ClientData)srcPtr);

	/*
         *  Make sure that token window was not dropped before it
         *  was either mapped or packed with info.
         */
	if (Tk_IsMapped(srcPtr->token.tkwin) && !srcPtr->pkgcmdInProg) {
	    UpdateToken((ClientData)srcPtr);

	    if (srcPtr->send) {
		if (srcPtr->token.overTarget) {
		    DragDropSend(srcPtr);
		} else {
		    HideToken(&(srcPtr->token));
		}
	    }
	    registry->numActive--;	/* one fewer active token window */
	}
    } else if ((c == 'e') && strncmp(argv[1], "errors", length) == 0) {
	/*
	 *  HANDLE:  drag&drop errors ?<proc>?
	 */
	if (argc == 3) {
	    if (registry->errorProc) {
		free(registry->errorProc);
	    }
	    registry->errorProc = strdup(argv[2]);
	} else if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " errors ?proc?\"",
		(char *)NULL);
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, registry->errorProc, TCL_VOLATILE);
	return TCL_OK;
    } else if ((c == 'a') && strncmp(argv[1], "active", length) == 0) {
	/*
	 *  HANDLE:  drag&drop active
	 */
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " active\"",
		(char *)NULL);
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, (registry->numActive > 0) ? "1" : "0", TCL_STATIC);
	return TCL_OK;
    } else if ((c == 'l') && strncmp(argv[1], "location", length) == 0) {
	/*
	 *  HANDLE:  drag&drop location ?<x> <y>?
	 */
	if (argc == 2) {
	    sprintf(buffer, "%d %d", registry->locx, registry->locy);
	    Tcl_SetResult(interp, buffer, TCL_VOLATILE);
	    return TCL_OK;
	} else if ((argc == 4) && (Tcl_GetInt(interp, argv[2], &x) == TCL_OK) &&
	    (Tcl_GetInt(interp, argv[3], &y) == TCL_OK)) {
	    registry->locx = x;
	    registry->locy = y;
	    sprintf(buffer, "%d %d", registry->locx, registry->locy);
	    Tcl_SetResult(interp, buffer, TCL_VOLATILE);
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " location ?x y?\"",
		(char *)NULL);
	    return TCL_ERROR;
	}
    } else {
	/*
	 *  Report improper command arguments
	 */
	Tcl_AppendResult(interp, "bad operation \"", argv[1],
	    "\": must be active, drag, drop, errors, location, ",
	    "source, target or token",
	    (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}


/*
 * ------------------------------------------------------------------------
 *
 *  CreateSource --
 *
 *	Looks for a DragSource record in the hash table for drag&drop source
 *	widgets.  Creates a new record if the widget name is not already
 *	registered.  Returns a pointer to the desired record.
 *
 * ------------------------------------------------------------------------
 */
static DragSource *
CreateSource(registry, pathName, newPtr)
    Registry *registry;		/* drag&drop records for all registered widgets */
    char *pathName;		/* widget pathname for desired record */
    int *newPtr;		/* returns non-zero => new record created */
{
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_CreateHashEntry(&(registry->sourceTable), pathName, newPtr);
    if (*newPtr) {
	DragSource *srcPtr;
	Tk_Window tkwin;

	tkwin = Tk_NameToWindow(registry->interp, pathName, registry->root);
	if (tkwin == NULL) {
	    Tcl_AppendResult(registry->interp, "window does not exist: ", pathName, 
		(char *)NULL);
	    return NULL;
	}
	srcPtr = (DragSource *) calloc(1, sizeof(DragSource));
	assert(srcPtr);
	srcPtr->tkwin = tkwin;
	srcPtr->display = Tk_Display(tkwin);
	srcPtr->ddAtom = XInternAtom(srcPtr->display, propName, False);
	srcPtr->registry = registry;
	srcPtr->token.anchor = TK_ANCHOR_SE;	/* Used to be "center". */
	srcPtr->token.relief = TK_RELIEF_RAISED;
	srcPtr->token.activeRelief = TK_RELIEF_SUNKEN;
	srcPtr->token.borderWidth = srcPtr->token.activeBorderWidth = 3;
	if (ConfigureSource(registry->interp, srcPtr, 0, 
			    (char **)NULL, 0) != TCL_OK) {
	    DestroySource(registry, pathName);
	    return NULL;
	}
	Tcl_SetHashValue(hPtr, (ClientData)srcPtr);
    }
    return (DragSource *)Tcl_GetHashValue(hPtr);
}


/*
 * ------------------------------------------------------------------------
 *  DestroySource --
 *
 *  Looks for a DragSource record in the hash table for drag&drop source
 *  widgets.  Destroys the record if found.
 * ------------------------------------------------------------------------
 */
static void
DestroySource(registry, pathname)
    Registry *registry;		/* drag&drop records for all registered widgets */
    char *pathname;		/* widget pathname for desired record */
{
    DragSource *srcPtr;
    DragSourceHndl *dsHndl, *next;
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_FindHashEntry(&(registry->sourceTable), pathname);
    if (hPtr == NULL) {
	return;
    }
    srcPtr = (DragSource *) Tcl_GetHashValue(hPtr);
    if (srcPtr) {
	Tk_CancelIdleCall(UpdateToken, (ClientData)srcPtr);
	if (srcPtr->token.timerToken) {
	    Tk_DeleteTimerHandler(srcPtr->token.timerToken);
	}
	Tk_FreeOptions(configSpecs, (char *)srcPtr, srcPtr->display, 0);

	if (srcPtr->token.rejectFgGC != NULL) {
	    Tk_FreeGC(srcPtr->display, srcPtr->token.rejectFgGC);
	}
	if (srcPtr->token.rejectBgGC != NULL) {
	    Tk_FreeGC(srcPtr->display, srcPtr->token.rejectBgGC);
	}
	if (srcPtr->pkgcmdResult) {
	    free(srcPtr->pkgcmdResult);
	}
	if (srcPtr->allwins) {
	    WinRepRelease(srcPtr->allwins, registry);
	}
	if (srcPtr->normalCursor != None) {
	    Tk_FreeCursor(srcPtr->display, srcPtr->normalCursor);
	}
	dsHndl = srcPtr->handlers;
	while (dsHndl) {
	    next = dsHndl->next;
	    DestroySourceHandler(dsHndl);
	    dsHndl = next;
	}
	free((char *)srcPtr);
    }
    Tcl_DeleteHashEntry(hPtr);
}
/*
 * ------------------------------------------------------------------------
 *
 *  FindSource --
 *
 *	Looks for a DragSource record in the hash table for drag&drop source
 *	widgets.  Returns a pointer to the desired record.
 *
 * ------------------------------------------------------------------------
 */
static DragSource *
FindSource(registry, pathname)
    Registry *registry;		/* drag&drop records for all registered widgets */
    char *pathname;		/* widget pathname for desired record */
{
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_FindHashEntry(&(registry->sourceTable), pathname);
    if (hPtr == NULL) {
	return NULL;
    }
    return (DragSource *)Tcl_GetHashValue(hPtr);
}


/*
 * ------------------------------------------------------------------------
 *
 *  ConfigureSource --
 *
 *	Called to process an (argc,argv) list to configure (or
 *	reconfigure) a drag&drop source widget.
 *
 * ------------------------------------------------------------------------ 
 */
static int
ConfigureSource(interp, srcPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* current interpreter */
    register DragSource *srcPtr;	/* drag&drop source widget record */
    int argc;			/* number of arguments */
    char **argv;		/* argument strings */
    int flags;			/* flags controlling interpretation */
{
    unsigned long gcMask;
    XGCValues gcValues;
    GC newGC;

    /*
     *  Handle the bulk of the options...
     */
    if (Tk_ConfigureWidget(interp, srcPtr->tkwin, configSpecs, argc, argv, 
		(char *)srcPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    /*
     *  Check the button binding for valid range (0 or 1-5)
     */
    if ((srcPtr->button < 0) || (srcPtr->button > 5)) {
	Tcl_SetResult(interp, "button number must be 1-5, or 0 for no bindings",
	    TCL_STATIC);
	return TCL_ERROR;
    }

    /*
     *  Set up the rejection foreground GC for the token window...
     */
    gcValues.foreground = srcPtr->token.rejectFg->pixel;
    gcValues.subwindow_mode = IncludeInferiors;
    gcValues.graphics_exposures = False;
    gcMask = GCForeground | GCSubwindowMode | GCGraphicsExposures;

    if (srcPtr->token.rejectStipple != None) {
	gcValues.stipple = srcPtr->token.rejectStipple;
	gcValues.fill_style = FillStippled;
	gcMask |= GCForeground | GCStipple | GCFillStyle;
    }
    newGC = Tk_GetGC(srcPtr->tkwin, gcMask, &gcValues);

    if (srcPtr->token.rejectFgGC != NULL) {
	Tk_FreeGC(srcPtr->display, srcPtr->token.rejectFgGC);
    }
    srcPtr->token.rejectFgGC = newGC;

    /*
     *  Set up the rejection background GC for the token window...
     */
    gcValues.foreground = srcPtr->token.rejectBg->pixel;
    gcValues.subwindow_mode = IncludeInferiors;
    gcValues.graphics_exposures = False;
    gcMask = GCForeground | GCSubwindowMode | GCGraphicsExposures;

    newGC = Tk_GetGC(srcPtr->tkwin, gcMask, &gcValues);

    if (srcPtr->token.rejectBgGC != NULL) {
	Tk_FreeGC(srcPtr->display, srcPtr->token.rejectBgGC);
    }
    srcPtr->token.rejectBgGC = newGC;

    /*
     *  Reset the border width in case it has changed...
     */
    if (srcPtr->token.tkwin) {
	Tk_SetInternalBorder(srcPtr->token.tkwin, srcPtr->token.borderWidth + 2);
    }
    return TCL_OK;
}

/*
 * ------------------------------------------------------------------------
 *  FindSourceHandler()
 *
 *  Looks for the requested data type of the list of handlers for the
 *  given drag&drop source.
 * ------------------------------------------------------------------------
 */
static char *
FindSourceHandler(srcPtr, dtname)
    register DragSource *srcPtr;	/* drag&drop source widget record */
    char *dtname;		/* name of requested data type */
{
    DragSourceHndl *dsHndl;

    for (dsHndl = srcPtr->handlers; dsHndl; dsHndl = dsHndl->next) {
	if (strcmp(dsHndl->dataType, dtname) == 0) {
	    return dsHndl->cmd;
	}
    }
    return NULL;
}

/*
 * ------------------------------------------------------------------------
 *  PutSourceHandler()
 *
 *  Looks for the requested data type of the list of handlers for the
 *  given drag&drop source.  If found, then its associated commands are
 *  changed to the given commands.  If not found, then a new handler
 *  is created.
 * ------------------------------------------------------------------------
 */
static void
PutSourceHandler(srcPtr, dtname, cmd)
    register DragSource *srcPtr;	/* drag&drop target widget record */
    char *dtname;		/* name of data type */
    char *cmd;			/* command used to send data */
{
    DragSourceHndl *tail = NULL;
    DragSourceHndl *dsHndl;

    for (dsHndl = srcPtr->handlers; dsHndl; tail = dsHndl, dsHndl = dsHndl->next)
	if (strcmp(dsHndl->dataType, dtname) == 0) {
	    if (*cmd == '\0') {
		if (tail) {
		    tail->next = dsHndl->next;
		} else {
		    srcPtr->handlers = dsHndl->next;
		}
		DestroySourceHandler(dsHndl);
		return;
	    } else {
		if (dsHndl->cmd != NULL) {
		    free(dsHndl->cmd);
		}
		dsHndl->cmd = strdup(cmd);
		return;
	    }
	}
    if (tail) {
	tail->next = CreateSourceHandler(dtname, cmd);
    } else {
	srcPtr->handlers = CreateSourceHandler(dtname, cmd);
    }
}

/*
 * ------------------------------------------------------------------------
 *  CreateSourceHandler()
 *
 *  Creates a new source handler record and returns a pointer to it.
 * ------------------------------------------------------------------------
 */
static DragSourceHndl *
CreateSourceHandler(dtname, cmd)
    char *dtname;		/* name of data type */
    char *cmd;			/* command used to send data */
{
    DragSourceHndl *retn;
    retn = (DragSourceHndl *) malloc(sizeof(DragSourceHndl));

    retn->dataType = strdup(dtname);
    retn->cmd = strdup(cmd);
    retn->next = NULL;
    return retn;
}

/*
 * ------------------------------------------------------------------------
 *  DestroySourceHandler()
 *
 *  Destroys a source handler record.
 * ------------------------------------------------------------------------
 */
static void
DestroySourceHandler(dsHndl)
    DragSourceHndl *dsHndl;
{
    if (dsHndl->dataType != NULL) {
	free(dsHndl->dataType);
    }
    if (dsHndl->cmd != NULL) {
	free(dsHndl->cmd);
    }
    free((char *)dsHndl);
}

/*
 * ------------------------------------------------------------------------
 *  UnregSource()
 *
 *  Invoked by Tk_HandleEvent whenever a DestroyNotify event is received
 *  on a registered drag&drop source widget.
 * ------------------------------------------------------------------------
 */
static void
UnregSource(cdata, eventPtr)
    ClientData cdata;		/* drag&drop registration list */
    XEvent *eventPtr;		/* event description */
{
    DD_RegEntry *ddentry = (DD_RegEntry *) cdata;
    Registry *registry = ddentry->registry;
    char *ddname = Tk_PathName(ddentry->tkwin);

    if (eventPtr->type == DestroyNotify) {
	DestroySource(registry, ddname);
	free((char *)ddentry);
    }
}

#ifdef notdef
/*
 * ------------------------------------------------------------------------
 *
 *  FindTarget --
 *
 *	Looks for a DropTarget record in the hash table for drag&drop
 *	target widgets.  Creates a new record if the widget name is
 *	not already registered.  Returns a pointer to the desired
 *	record.
 *
 * ------------------------------------------------------------------------ 
 */
static DropTarget *
FindTarget(registry, pathname)
    Registry *registry;		/* drag&drop records for all registered widgets */
    char *pathname;		/* widget pathname for desired record */
{
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_FindHashEntry(&registry->targetTable, pathname);
    if (hPtr == NULL) {
	return NULL;
    } 
    return (DropTarget *) Tcl_GetHashValue(hPtr);
}

#endif
/*
 * ------------------------------------------------------------------------
 *
 *  CreateTargetInfo --
 *
 *	Looks for a DropTarget record in the hash table for drag&drop
 *	target widgets.  Creates a new record if the widget name is
 *	not already registered.  Returns a pointer to the desired
 *	record.
 *
 * ------------------------------------------------------------------------ 
 */
static DropTarget *
CreateTargetInfo(registry, pathname, newPtr)
    Registry *registry;		/* drag&drop records for all registered widgets */
    char *pathname;		/* widget pathname for desired record */
    int *newPtr;		/* returns non-zero => new record created */
{
    DropTarget *targetPtr;
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_CreateHashEntry(&registry->targetTable, pathname, newPtr);
    if (*newPtr) {
	/*
         *  Initialize a data structure for the widget...
         */
	targetPtr = (DropTarget *) calloc(1, sizeof(DropTarget));
	targetPtr->registry = registry;
	Tcl_SetHashValue(hPtr, (ClientData)targetPtr);
    }
    return (DropTarget *) Tcl_GetHashValue(hPtr);
}

/*
 * ------------------------------------------------------------------------
 *  DestroyTargetInfo()
 *
 *  Looks for a DropTarget record in the hash table for drag&drop target
 *  widgets.  Destroys the record if found.
 * ------------------------------------------------------------------------
 */
static void
DestroyTargetInfo(registry, pathname)
    Registry *registry;		/* drag&drop records for all registered widgets */
    char *pathname;		/* widget pathname for desired record */
{
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_FindHashEntry(&(registry->targetTable), pathname);
    if (hPtr != NULL) {
	DropTarget *targetPtr;
	DropTargetHndl *dtHndl, *next;

	targetPtr = (DropTarget *)Tcl_GetHashValue(hPtr);
	dtHndl = targetPtr->handlers;
	while (dtHndl) {
	    next = dtHndl->next;
	    DestroyTargetHandler(dtHndl);
	    dtHndl = next;
	}
	free((char *)targetPtr);
	Tcl_DeleteHashEntry(hPtr);
    }
}

/*
 * ------------------------------------------------------------------------
 *  FindTargetHandler()
 *
 *  Looks for the requested data type of the list of handlers for the
 *  given drag&drop target.
 * ------------------------------------------------------------------------
 */
static char *
FindTargetHandler(targetPtr, dtname)
    register DropTarget *targetPtr;	/* drag&drop target widget record */
    char *dtname;		/* name of requested data type */
{
    DropTargetHndl *dtHndl;

    for (dtHndl = targetPtr->handlers; dtHndl; dtHndl = dtHndl->next) {
	if (strcmp(dtHndl->dataType, dtname) == 0) {
	    return dtHndl->command;
	}
    }
    return NULL;
}

/*
 * ------------------------------------------------------------------------
 *  PutTargetHandler()
 *
 *  Looks for the requested data type of the list of handlers for the
 *  given drag&drop target.  If found, then its associated command is
 *  changed to the given command.  If not found, then a new handler
 *  is created.
 * ------------------------------------------------------------------------
 */
static void
PutTargetHandler(targetPtr, dtname, cmd)
    register DropTarget *targetPtr;	/* drag&drop target widget record */
    char *dtname;		/* name of data type */
    char *cmd;			/* command string for data type */
{
    DropTargetHndl *tail = NULL;
    DropTargetHndl *dtHndl;

    for (dtHndl = targetPtr->handlers; dtHndl; tail = dtHndl, dtHndl = dtHndl->next)
	if (strcmp(dtHndl->dataType, dtname) == 0) {
	    if (*cmd == '\0') {
		if (tail) {
		    tail->next = dtHndl->next;
		} else {
		    targetPtr->handlers = dtHndl->next;
		}
		DestroyTargetHandler(dtHndl);
		return;
	    } else {
		if (dtHndl->command != NULL) {
		    free(dtHndl->command);
		}
		dtHndl->command = strdup(cmd);
		return;
	    }
	}
    if (tail) {
	tail->next = CreateTargetHandler(dtname, cmd);
    } else {
	targetPtr->handlers = CreateTargetHandler(dtname, cmd);
    }
    /*
     *  Update handler list stored in target window property.
     */
    AddPropertyToTarget(targetPtr);
}

/*
 * ------------------------------------------------------------------------
 *  CreateTargetHandler()
 *
 *  Creates a new target handler record and returns a pointer to it.
 * ------------------------------------------------------------------------
 */
static DropTargetHndl *
CreateTargetHandler(dtname, cmd)
    char *dtname;		/* name of data type */
    char *cmd;			/* command string for data type */
{
    DropTargetHndl *retn;
    retn = (DropTargetHndl *) malloc(sizeof(DropTargetHndl));

    retn->dataType = strdup(dtname);
    retn->command = strdup(cmd);

    retn->next = NULL;
    return retn;
}

/*
 * ------------------------------------------------------------------------
 *  DestroyTargetHandler()
 *
 *  Destroys a target handler record.
 * ------------------------------------------------------------------------
 */
static void
DestroyTargetHandler(dtHndl)
    DropTargetHndl *dtHndl;
{
    if (dtHndl->dataType != NULL) {
	free(dtHndl->dataType);
    }
    if (dtHndl->command != NULL) {
	free(dtHndl->command);
    }
    free((char *)dtHndl);
}

/*
 * ------------------------------------------------------------------------
 *  UnregTarget()
 *
 *  Invoked by Tk_HandleEvent whenever a DestroyNotify event is received
 *  on a registered drag&drop target widget.
 * ------------------------------------------------------------------------
 */
static void
UnregTarget(cdata, eventPtr)
    ClientData cdata;		/* drag&drop registration list */
    XEvent *eventPtr;		/* event description */
{
    DD_RegEntry *ddentry = (DD_RegEntry *) cdata;
    Registry *registry = ddentry->registry;
    char *ddname = Tk_PathName(ddentry->tkwin);

    if (eventPtr->type == DestroyNotify) {
	DestroyTargetInfo(registry, ddname);
	free((char *)ddentry);
    }
}

/*
 * ------------------------------------------------------------------------
 *  DragDropSend()
 *
 *  Invoked after a drop operation to send data to the drop application.
 * ------------------------------------------------------------------------
 */
static void
DragDropSend(srcPtr)
    register DragSource *srcPtr;	/* drag&drop source record */
{
    Registry *registry = srcPtr->registry;
    char *datatype = NULL;

    int status;
    char *sendcmd;
    DD_WinRep *target;
    Tcl_DString buffer;
    DD_PercentSubst subst[3];

    Tcl_DStringInit(&buffer);
    Tcl_DStringAppend(&buffer, srcPtr->pkgcmdResult, -1);

    /*
     *  See if current position is over drop point...
     */
    target = FindTargetWin(srcPtr, srcPtr->token.lastX, srcPtr->token.lastY);

    if (target) {
	char coords[256];

	sprintf(coords, "%d %d", srcPtr->token.lastX, srcPtr->token.lastY);
	status = Tcl_VarEval(registry->interp,
	    "send {", target->ddinterp, "} ", dragDropCmd, " location ", coords,
	    (char *)NULL);

	if (status == TCL_OK) {
	    datatype = DragDropSendHndlr(srcPtr, target->ddinterp, target->ddwin);
	    if (datatype) {
		sendcmd = FindSourceHandler(srcPtr, datatype);
		if (sendcmd && *sendcmd) {
		    Tcl_DString cmdStr;

		    subst[0].letter = 'i';
		    subst[0].value = target->ddinterp;
		    subst[1].letter = 'w';
		    subst[1].value = target->ddwin;
		    subst[2].letter = 'v';
		    subst[2].value = srcPtr->pkgcmdResult;

		    Tcl_DStringInit(&cmdStr);
		    status = Tcl_Eval(registry->interp,
			ExpandPercents(sendcmd, subst, 3, &cmdStr));
		    Tcl_DStringFree(&cmdStr);

		    Tcl_DStringSetLength(&buffer, 0);
		    Tcl_DStringAppend(&buffer, registry->interp->result, -1);
		}
	    } else {
		Tcl_AppendResult(registry->interp, "target \"", target->ddwin,
		    "\" does not recognize handlers for source \"",
		    Tk_PathName(srcPtr->tkwin), "\"", (char *)NULL);
		status = TCL_ERROR;
	    }
	}
	/*
         *  If the "send" command was successful, then tell the
         *  target application to handle the data.
         */
	if (status == TCL_OK && datatype) {
	    status = Tcl_VarEval(registry->interp,
		"send {", target->ddinterp, "} ",
		dragDropCmd, " target ", target->ddwin, " handle ", datatype,
		" {", Tcl_DStringValue(&buffer), "}",
		(char *)NULL);
	}
	/*
         *  Give success/failure feedback to user.
         *  If an error occurred and an error proc is defined,
         *  then use it to handle the error.
         */
	if (status == TCL_OK) {
	    HideToken(&(srcPtr->token));
	} else {
	    RejectToken(&(srcPtr->token));

	    if (registry->errorProc && *registry->errorProc) {
		(void)Tcl_VarEval(registry->interp,
		    registry->errorProc, " {", registry->interp->result, "}",
		    (char *)NULL);
	    }
	}
    }
    Tcl_DStringFree(&buffer);
}

/*
 * ------------------------------------------------------------------------
 *  DragDropSendHndlr()
 *
 *  Queries the drag&drop target under the specified interpreter for a
 *  handler that is compatible with one of the handlers defined for the
 *  source.  Returns a pointer to the appropriate data type, or
 *  NULL if none is found.
 * ------------------------------------------------------------------------
 */
static char *
DragDropSendHndlr(srcPtr, interpName, ddName)
    register DragSource *srcPtr;	/* drag&drop source record */
    char *interpName;		/* interpreter containing drag&drop target */
    char *ddName;		/* drag&drop target pathname */
{
    char *retn = NULL;		/* no handler found yet */
    Tcl_Interp *interp = srcPtr->registry->interp;

    int hndlc, hi, ei;
    char **hndlv, *hlist;
    DragSourceHndl *dsHndl;
    char buffer[1024];

    /*
     *  Query the drag&drop target for its list of known handlers.
     */
    Tcl_ResetResult(interp);	/* for Tcl_AppendResult() below */
    if (Tcl_VarEval(interp,
	    "send {", interpName, "} ", dragDropCmd, " target {", ddName, "} handler",
		    (char *)NULL) != TCL_OK) {
	return NULL;
    }
    hlist = strdup(interp->result);
    if (Tcl_SplitList(interp, hlist, &hndlc, &hndlv) == TCL_OK) {
	/*
         *  If the list of send handlers is specified as "all", then
         *  search through the handlers in order.
         */
	if (strcmp(srcPtr->send, "all") == 0) {
	    for (dsHndl = srcPtr->handlers; dsHndl && !retn; dsHndl = dsHndl->next) {
		for (hi = 0; (hi < hndlc) && !retn; hi++)
		    if (strcmp(dsHndl->dataType, hndlv[hi]) == 0)
			retn = dsHndl->dataType;
	    }
	/*
         *  Otherwise, search through the specified send handlers.
         */
	} else {
	    int elemc;
	    char **elemv;
	    if (Tcl_SplitList(interp, srcPtr->send, &elemc, &elemv) == TCL_OK) {
		for (ei = 0; (ei < elemc) && !retn; ei++) {
		    for (hi = 0; (hi < hndlc) && !retn; hi++) {
			if (strcmp(elemv[ei], hndlv[hi]) == 0) {
			    for (dsHndl = srcPtr->handlers; dsHndl; dsHndl = dsHndl->next) {
				if (strcmp(dsHndl->dataType, elemv[ei]) == 0) {
				    break;
				}
			    }
			    if (dsHndl) {
				retn = dsHndl->dataType;
			    } else {
				sprintf(buffer, "unknown handler \"%.50s\" requested for drag&drop source \"%.200s\"", elemv[ei], Tk_PathName(srcPtr->tkwin));
				Tcl_ResetResult(interp);
				Tcl_AddErrorInfo(interp, buffer);
				Tk_BackgroundError(interp);
			    }
			}
		    }
		}
		free((char *)elemv);
	    } else {
		sprintf(buffer, "drag&drop source has invalid -send: %.200s",
		    srcPtr->send);
		Tcl_ResetResult(interp);
		Tcl_AddErrorInfo(interp, buffer);
		Tk_BackgroundError(interp);
	    }
	}
	free((char *)hndlv);
    }
    free(hlist);

    return retn;
}

/*
 * ------------------------------------------------------------------------
 *  GetWinRepInfo()
 *
 *  Invoked at the start of a "drag" operation to capture the positions
 *  of all windows on the current root.  Queries the entire window
 *  hierarchy and determines the placement of each window.  Queries
 *  "DragDropInfo" property info where appropriate.  This information
 *  is used during the drag operation to determine when the drag&drop
 *  token is over a valid drag&drop target.
 *
 *  Returns the record for the root window, which contains records for
 *  all other windows as children.
 * ------------------------------------------------------------------------
 */
static DD_WinRep *
GetWinRepInfo(srcPtr, registry)
    DragSource *srcPtr;		/* drag&drop source window */
    Registry *registry;		/* drag&drop registration info */
{
    DD_WinRep *wr;

    wr = WinRepAlloc(registry);
    wr->win = DefaultRootWindow(srcPtr->display);
    WinRepInit(wr, srcPtr);

    return wr;
}

/*
 * ------------------------------------------------------------------------
 *  FindTargetWin()
 *
 *  Checks to see if a compatible drag&drop target exists at the given
 *  position.  A target is "compatible" if it is a drag&drop window,
 *  and if it has a handler that is compatible with the current source
 *  window.
 *
 *  Returns a pointer to a structure describing the target, or NULL
 *  if no compatible target is found.
 * ------------------------------------------------------------------------
 */
static DD_WinRep *
FindTargetWin(srcPtr, x, y)
    DragSource *srcPtr;		/* drag&drop source window */
    int x, y;			/* current drag&drop location (virtual coords) */
{
    int vx, vy, vw, vh;
    register char *type;

    DD_WinRep *wr, *wrkid;
    DD_Stack stack;
    DragSourceHndl *shandl;

    /*
     *  If window representations have not yet been built, then
     *  abort this call.  This probably means that the token is being
     *  moved before it has been properly built.
     */
    if (!srcPtr->allwins)
	return NULL;

    /*
     *  Adjust current location for virtual root windows.
     */
    Tk_GetVRootGeometry(srcPtr->tkwin, &vx, &vy, &vw, &vh);
    x += vx;
    y += vy;

    /*
     *  Build a stack of all windows containing the given point,
     *  in order from least to most specific.
     */
    StackInit(&stack);

    wr = srcPtr->allwins;
    if ((x >= wr->x0) && (x <= wr->x1) &&
	(y >= wr->y0) && (y <= wr->y1))
	StackPush((ClientData)wr, &stack);

    while (wr) {
	for (wrkid = wr->kids; wrkid; wrkid = wrkid->next) {
	    if (!wrkid->initialized)
		WinRepInit(wrkid, srcPtr);

	    if ((x >= wrkid->x0) && (x <= wrkid->x1) &&
		(y >= wrkid->y0) && (y <= wrkid->y1)) {
		StackPush((ClientData)wrkid, &stack);
		break;
	    }
	}
	wr = wrkid;		/* continue search */
    }

    /*
     *  Pop windows from the stack until one containing a
     *  "DragDropInfo" property is found.  See if the handlers
     *  listed in this property are compatible with the
     *  given source.
     */
    while ((wr = (DD_WinRep *) StackPop(&stack)) != NULL)
	if (wr->ddprop)
	    break;

    if (wr && wr->ddhandlers) {
	type = wr->ddhandlers;
	while (*type != '\0') {
	    for (shandl = srcPtr->handlers; shandl; shandl = shandl->next)
		if (strcmp(shandl->dataType, type) == 0)
		    break;

	    if (shandl)		/* found a match? */
		break;		/* then stop searching */
	    else {		/* otherwise, move to next handler type */
		while (*type++ != '\0');
	    }
	}
	if (*type == '\0')	/* no handler match? */
	    wr = NULL;		/* then return NULL */
    }
    StackDelete(&stack);

    return wr;
}

/*
 * ------------------------------------------------------------------------
 *  WinRepAlloc()
 *
 *  Returns a new structure for representing X window position info.
 *  Such structures are typically allocated at the start of a drag&drop
 *  operation to capture the placement of all windows on the root
 *  window.  The drag&drop registration list keeps a pool of such
 *  structures so that they can be allocated quickly when needed.
 *  Returns a pointer to an empty structure.
 * ------------------------------------------------------------------------
 */
static DD_WinRep *
WinRepAlloc(registry)
    Registry *registry;		/* drag&drop registration list */
{
    DD_WinRep *wr;

    /*
     *  Return the top-most structure in the pool.
     *  If the pool is empty, add a new structure to it.
     */
    if (registry->pool == NULL) {
	wr = (DD_WinRep *) malloc(sizeof(DD_WinRep));
	wr->next = NULL;
	registry->pool = wr;
    }
    wr = registry->pool;
    registry->pool = wr->next;

    wr->initialized = 0;
    wr->ddprop = NULL;
    wr->ddinterp = wr->ddwin = wr->ddhandlers = NULL;
    wr->parent = wr->kids = wr->next = NULL;
    return wr;
}

/*
 * ------------------------------------------------------------------------
 *  WinRepRelease()
 *
 *  Puts a window representation structure back into the global pool,
 *  making it available for future calls to WinRepAlloc().  Any
 *  associated resources (within the structure) are automatically freed.
 * ------------------------------------------------------------------------
 */
static void
WinRepRelease(wr, registry)
    DD_WinRep *wr;		/* window rep to be freed */
    Registry *registry;		/* drag&drop registration list */
{
    DD_WinRep *wrkid, *wrnext;

    wrnext = NULL;		/* Suppress compiler warning */
    for (wrkid = wr->kids; wrkid; wrkid = wrnext) {
	wrnext = wrkid->next;
	WinRepRelease(wrkid, registry);
    }

    if (wr->ddprop) {
	XFree(wr->ddprop);
    }
    wr->next = registry->pool;	/* put back into pool */
    registry->pool = wr;
}

/*
 * ------------------------------------------------------------------------
 *  WinRepInit()
 *
 *  Invoked during "drag" operations to dig a little deeper into the
 *  root window hierarchy and cache the resulting information.  If a
 *  point coordinate lies within an uninitialized DD_WinRep, this
 *  routine is called to query window coordinates and drag&drop info.
 *  If this particular window has any children, more uninitialized
 *  DD_WinRep structures are allocated.  Further queries will cause
 *  these structures to be initialized in turn.
 * ------------------------------------------------------------------------
 */
static void
WinRepInit(wr, srcPtr)
    DD_WinRep *wr;		/* window rep to be initialized */
    DragSource *srcPtr;		/* drag&drop source managing win rep */
{
    Window ignoreSource = Tk_WindowId(srcPtr->tkwin);
    Window ignoreToken = Tk_WindowId(srcPtr->token.tkwin);

    DD_WinRep *wrkid, *wrtail;

    Window root, parent, *kids;
    unsigned int nkids;
    XWindowAttributes winInfo;

    char *propInfo;
    int i, result, actualFormat;
    Atom actualType;
    unsigned long numItems, bytesAfter;

    /*
     *  If the self-target flag is set, allow the source window to
     *  drop onto itself.  Do not ignore source window during search.
     */
    if (srcPtr->selfTarget)
	ignoreSource = None;

    if (!wr->initialized) {
	/*
         *  Query for the window coordinates.
         */
	if (XGetWindowAttributes(srcPtr->display, wr->win, &winInfo) &&
	    (winInfo.map_state == IsViewable) &&
	    (wr->win != ignoreToken) &&
	    (wr->win != ignoreSource)) {
	    wr->x0 = winInfo.x;
	    wr->y0 = winInfo.y;
	    wr->x1 = winInfo.x + winInfo.width;
	    wr->y1 = winInfo.y + winInfo.height;

	    if (wr->parent) {	/* offset by parent coords */
		wr->x0 += wr->parent->x0;
		wr->y0 += wr->parent->y0;
		wr->x1 += wr->parent->x0;
		wr->y1 += wr->parent->y0;
	    }
	} else {
	    wr->x0 = wr->y0 = -1;
	    wr->x1 = wr->y1 = -1;
	}

	/*
         *  See if this window has a "DragDropInfo" property.
         */
	result = XGetWindowProperty(srcPtr->display, wr->win,
	    srcPtr->ddAtom, 0, MAX_PROP_SIZE, False, XA_STRING,
	    &actualType, &actualFormat,
	    &numItems, &bytesAfter, (unsigned char **)&propInfo);

	if ((result != Success) || 
		(actualFormat != 8) || (actualType != XA_STRING)) {
	    if (propInfo != NULL) {
		XFree((caddr_t) propInfo);
	    }
	    propInfo = NULL;
	}
	wr->ddprop = propInfo;
	if (wr->ddprop) {
	    char *p = wr->ddprop;
	    wr->ddinterp = wr->ddprop;

	    while ((*p != '\0') && (*p != ']'))
		p++;

	    if (*p != '\0') {
		*p++ = '\0';	/* terminate interp name */
		wr->ddwin = p;	/* get start of window name */
	    }
	    while ((*p != '\0') && (*p != ']'))
		p++;

	    if (*p != '\0') {
		*p++ = '\0';	/* terminate window name */
		wr->ddhandlers = p;	/* get start of handler list */

		/*
                 *  Handler strings are of the form:
                 *  "<type> <type> ... <type> "
                 */
		while (*p != '\0') {
		    while ((*p != ' ') && (*p != '\0'))
			p++;

		    *p++ = '\0';/* null terminate handler type */
		}
	    }
	}
	/*
         *  If this window has any children, then create DD_WinReps
         *  for them as well.
         */
	if (XQueryTree(srcPtr->display, wr->win, &root, &parent, &kids, &nkids)) {
	    wrtail = NULL;
	    for (i = nkids - 1; i >= 0; i--) {
		wrkid = WinRepAlloc(srcPtr->registry);
		wrkid->win = kids[i];
		wrkid->parent = wr;

		if (wrtail)
		    wrtail->next = wrkid;
		else
		    wr->kids = wrkid;

		wrtail = wrkid;
	    }
	    if (kids != NULL) {
		XFree((caddr_t) kids);	/* done with list of kids */
	    }
	}
    }
    wr->initialized = ~0;
}

/*
 * ------------------------------------------------------------------------
 *
 *  AddPropertyToTarget --
 *
 *	Attaches a "DragDropInfo" property to the given target window.
 *	This property allows the drag&drop mechanism to recognize the
 *	window as a valid target, and stores important information
 *	including the interpreter managing the target and the pathname
 *	for the target window.  Usually invoked when the target is
 *	first registered or first exposed (so that the X-window really
 *	exists).
 *
 * ------------------------------------------------------------------------ 
 */
static void
AddPropertyToTarget(targetPtr)
    DropTarget *targetPtr;		/* drag&drop target window data */
{
    Tcl_Interp *interp = targetPtr->registry->interp;
    Atom ddProperty;
    char *path, *info;
    DropTargetHndl *thandl;
    Tcl_DString propStr;
    static char command[] = {"winfo name ."};
    unsigned int numBytes;

    if (targetPtr->tkwin == NULL) {
	return;
    }
    Tcl_DStringInit(&propStr);
    path = Tk_PathName(targetPtr->tkwin);
    if (Tcl_Eval(interp, command) == TCL_OK) {
	Tcl_DStringAppend(&propStr, interp->result, -1);
    }
    Tcl_ResetResult(interp);
    Tcl_DStringAppend(&propStr, "]", -1);
    Tcl_DStringAppend(&propStr, path, -1);
    Tcl_DStringAppend(&propStr, "]", -1);
    for (thandl = targetPtr->handlers; thandl; thandl = thandl->next) {
	Tcl_DStringAppend(&propStr, thandl->dataType, -1);
	Tcl_DStringAppend(&propStr, " ", -1);
    }
    ddProperty = XInternAtom(targetPtr->display, propName, False);
    info = Tcl_DStringValue(&propStr);
    numBytes = strlen(info) + 1;
    XChangeProperty(targetPtr->display, Tk_WindowId(targetPtr->tkwin), ddProperty, 
	XA_STRING, 8, PropModeReplace, (unsigned char *)info, numBytes);
    Tcl_DStringFree(&propStr);
}


static void
ActivateToken(tokenPtr, active)
    Token *tokenPtr;
    int active;
{
    int relief;
    Tk_3DBorder *borderPtr;
    int borderWidth;

    Tk_Fill3DRectangle(tokenPtr->tkwin, Tk_WindowId(tokenPtr->tkwin), 
	tokenPtr->outline, 0, 0, Tk_Width(tokenPtr->tkwin), 
	Tk_Height(tokenPtr->tkwin), 0, TK_RELIEF_FLAT);
    if (active) {
	relief = tokenPtr->activeRelief;
	borderPtr = &(tokenPtr->activeBorder);
	borderWidth = tokenPtr->activeBorderWidth;
    } else {
	relief = tokenPtr->relief;
	borderPtr = &(tokenPtr->normalBorder);
	borderWidth = tokenPtr->borderWidth;
    }
    Tk_Fill3DRectangle(tokenPtr->tkwin, Tk_WindowId(tokenPtr->tkwin), 
	*borderPtr, 2, 2, Tk_Width(tokenPtr->tkwin) - 4, 
	Tk_Height(tokenPtr->tkwin) - 4, borderWidth, relief);
}

/*
 * ------------------------------------------------------------------------
 *
 *  TokenEventProc --
 *
 *	Invoked by the Tk dispatcher to handle widget events.
 *	Manages redraws for the drag&drop token window.
 *
 * ------------------------------------------------------------------------
 */
static void
TokenEventProc(clientData, eventPtr)
    ClientData clientData;	/* data associated with widget */
    XEvent *eventPtr;		/* information about event */
{
    Token *tokenPtr = (Token *) clientData;

    if ((eventPtr->type == Expose) && (eventPtr->xexpose.count == 0)) {
	if (tokenPtr->tkwin != NULL) {
	    ActivateToken(tokenPtr, tokenPtr->overTarget);
	}
    } else if (eventPtr->type == DestroyNotify) {
	tokenPtr->tkwin = NULL;
    }
}

/*
 * ------------------------------------------------------------------------
 *
 *  MoveToken --
 *
 *	Invoked during "drag" operations to move a token window to its
 *	current "drag" coordinate.
 *
 * ------------------------------------------------------------------------
 */
static void
MoveToken(srcPtr, tokenPtr)
    DragSource *srcPtr;		/* drag&drop source window data */
    Token *tokenPtr;
{
    Point2D point;
    int x, y;
    int maxWidth, maxHeight;
    int vx, vy, vw, vh;

    /*
     *  Adjust current location for virtual root windows.
     */
    Tk_GetVRootGeometry(srcPtr->tkwin, &vx, &vy, &vw, &vh);
    x = tokenPtr->lastX + vx;
    y = tokenPtr->lastY + vy;

    maxWidth = WidthOfScreen(Tk_Screen(srcPtr->tkwin)) - Tk_Width(tokenPtr->tkwin);
    maxHeight = 
	HeightOfScreen(Tk_Screen(srcPtr->tkwin)) - Tk_Height(tokenPtr->tkwin);
    point = Blt_TranslateBoxCoords(x, y, Tk_Width(tokenPtr->tkwin), 
	Tk_Height(tokenPtr->tkwin), tokenPtr->anchor);

    x = (int)point.x;
    y = (int)point.y;

    if (x > maxWidth) {
	x = maxWidth;
    } else if (x < 0) {
	x = 0;
    }
    if (y > maxHeight) {
	y = maxHeight;
    } else if (y < 0) {
	y = 0;
    }
    if ((x != Tk_X(tokenPtr->tkwin)) || (y != Tk_Y(tokenPtr->tkwin))) {
	Tk_MoveToplevelWindow(tokenPtr->tkwin, x, y);
    }
    RaiseToken(srcPtr, tokenPtr);
}

/*
 * ------------------------------------------------------------------------
 *  UpdateToken --
 *
 *	Invoked when the event loop is idle to determine whether or not
 *	the current drag&drop token position is over another drag&drop
 *	target.
 *
 * ------------------------------------------------------------------------
 */
static void
UpdateToken(clientData)
    ClientData clientData;	/* widget data */
{
    register DragSource *srcPtr = (DragSource *) clientData;
    Token *tokenPtr = &(srcPtr->token);
    int status, result;
    DD_PercentSubst subst[2];

    status = (FindTargetWin(srcPtr, tokenPtr->lastX, tokenPtr->lastY) != NULL);

    if (tokenPtr->overTarget ^ status) {
	ActivateToken(tokenPtr, status);
	/*
         *  If the source has a site command, then invoke it to
         *  modify the appearance of the token window.  Pass any
         *  errors onto the drag&drop error handler.
         */
	if (srcPtr->sitecmd) {
	    char buffer[200];
	    Tcl_DString cmdStr;

	    sprintf(buffer, "%d", status);
	    subst[0].letter = 's';
	    subst[0].value = buffer;
	    subst[1].letter = 't';
	    subst[1].value = Tk_PathName(tokenPtr->tkwin);

	    Tcl_DStringInit(&cmdStr);
	    result = Tcl_Eval(srcPtr->registry->interp,
		ExpandPercents(srcPtr->sitecmd, subst, 2, &cmdStr));
	    Tcl_DStringFree(&cmdStr);

	    if ((result != TCL_OK) && (srcPtr->registry->errorProc != NULL) &&
		(*srcPtr->registry->errorProc)) {

		(void)Tcl_VarEval(srcPtr->registry->interp,
		    srcPtr->registry->errorProc, " {",
		    srcPtr->registry->interp->result, "}",
		    (char *)NULL);
	    }
	}
    }
    tokenPtr->overTarget = status;
}

/*
 * ------------------------------------------------------------------------
 *
 *  HideToken --
 *
 *	Unmaps the drag&drop token.  Invoked directly at the end of a
 *	successful communication, or after a delay if the communication
 *	fails (allowing the user to see a graphical picture of failure).
 *
 * ------------------------------------------------------------------------
 */
static void
HideToken(tokenPtr)
    Token *tokenPtr;
{
    if (tokenPtr->tkwin != NULL) {
	Tk_UnmapWindow(tokenPtr->tkwin);
    }
    tokenPtr->timerToken = NULL;
}

/*
 * ------------------------------------------------------------------------
 *
 *  RejectToken --
 *
 *	Draws a rejection mark on the current drag&drop token, and arranges
 *	for the token to be unmapped after a small delay.
 *
 * ------------------------------------------------------------------------
 */
static void
RejectToken(tokenPtr)
    Token *tokenPtr;
{
    int divisor = 6;		/* controls size of rejection symbol */
    int w, h, lineWidth, x, y, margin;

    margin = 2 * tokenPtr->borderWidth;
    w = Tk_Width(tokenPtr->tkwin) - 2 * margin;
    h = Tk_Height(tokenPtr->tkwin) - 2 * margin;
    lineWidth = (w < h) ? w / divisor : h / divisor;
    lineWidth = (lineWidth < 1) ? 1 : lineWidth;

    w = h = lineWidth * (divisor - 1);
    x = (Tk_Width(tokenPtr->tkwin) - w) / 2;
    y = (Tk_Height(tokenPtr->tkwin) - h) / 2;

    /*
     *  Draw the rejection symbol background (\) on the token window...
     */
    XSetLineAttributes(Tk_Display(tokenPtr->tkwin), tokenPtr->rejectBgGC,
	lineWidth + 4, LineSolid, CapButt, JoinBevel);

    XDrawArc(Tk_Display(tokenPtr->tkwin), Tk_WindowId(tokenPtr->tkwin), 
	tokenPtr->rejectBgGC, x, y, w, h, 0, 23040);

    XDrawLine(Tk_Display(tokenPtr->tkwin), Tk_WindowId(tokenPtr->tkwin),
	tokenPtr->rejectBgGC, x + lineWidth, y + lineWidth, x + w - lineWidth, 
	y + h - lineWidth);

    /*
     *  Draw the rejection symbol foreground (\) on the token window...
     */
    XSetLineAttributes(Tk_Display(tokenPtr->tkwin), tokenPtr->rejectFgGC,
	lineWidth, LineSolid, CapButt, JoinBevel);

    XDrawArc(Tk_Display(tokenPtr->tkwin), Tk_WindowId(tokenPtr->tkwin), 
	tokenPtr->rejectFgGC, x, y, w, h, 0, 23040);

    XDrawLine(Tk_Display(tokenPtr->tkwin), Tk_WindowId(tokenPtr->tkwin), 
	tokenPtr->rejectFgGC, x + lineWidth, y + lineWidth, x + w - lineWidth, 
	y + h - lineWidth);

    /*
     *  Arrange for token window to disappear eventually.
     */
    tokenPtr->timerToken = Tk_CreateTimerHandler(1000, (Tcl_TimerProc *)HideToken, 
	(ClientData)tokenPtr);
}

/*
 * ------------------------------------------------------------------------
 *  StackInit()
 *
 *  Initializes a stack structure, allocating a certain amount of memory
 *  for the stack and setting the stack length to zero.
 * ------------------------------------------------------------------------
 */
static void
StackInit(stack)
    DD_Stack *stack;		/* stack to be initialized */
{
    stack->values = stack->space;
    stack->max = sizeof(stack->space) / sizeof(ClientData);
    stack->len = 0;
}

/*
 * ------------------------------------------------------------------------
 *  StackDelete()
 *
 *  Destroys a stack structure, freeing any memory that may have been
 *  allocated to represent it.
 * ------------------------------------------------------------------------
 */
static void
StackDelete(stack)
    DD_Stack *stack;		/* stack to be deleted */
{
    if (stack->values != stack->space)	/* allocated extra memory? */
	free((char *)stack->values);	/* then free it */

    stack->values = NULL;
    stack->len = stack->max = 0;
}

/*
 * ------------------------------------------------------------------------
 *  StackPush()
 *
 *  Pushes a piece of client data onto the top of the given stack.
 * ------------------------------------------------------------------------
 */
static void
StackPush(cdata, stack)
    ClientData cdata;		/* data to be pushed onto stack */
    DD_Stack *stack;		/* stack */
{
    ClientData *newStack;

    if (stack->len + 1 >= stack->max) {
	stack->max = (stack->max == 0) ? 5 : 2 * stack->max;
	newStack = (ClientData *)
	    malloc((unsigned)(stack->max * sizeof(ClientData)));

	if (stack->values) {
	    memcpy((char *)newStack, (char *)stack->values,
		stack->len * sizeof(ClientData));

	    if (stack->values != stack->space)
		free((char *)stack->values);
	}
	stack->values = newStack;
    }
    stack->values[stack->len++] = cdata;
}

/*
 * ------------------------------------------------------------------------
 *  StackPop()
 *
 *  Pops a bit of client data from the top of the given stack.
 * ------------------------------------------------------------------------
 */
static ClientData
StackPop(stack)
    DD_Stack *stack;		/* stack to be manipulated */
{
    if ((stack->values != NULL) && (stack->len > 0)) {
	return (stack->values[--stack->len]);
    }
    return (ClientData) 0;
}

/*
 * ------------------------------------------------------------------------
 *  ExpandPercents(str,subs,nsubs, dsPtr)
 *
 *  Expands all percent substitutions found in the input "str" that
 *  match specifications in the "subs" list.  Any percent field that
 *  is not found in the "subs" list is left alone.  Returns a string
 *  that remains valid until the next call to this routine.
 * ------------------------------------------------------------------------
 */
static char *
ExpandPercents(string, subs, nsubs, dsPtr)
    char *string;		/* incoming string */
    DD_PercentSubst *subs;	/* array of known substitutions */
    int nsubs;			/* number of elements in subs array */
    Tcl_DString *dsPtr;
{
    register char *chunk, *p;
    char letter;
    char percentSign;
    int i;

    /*
     *  Scan through the copy of the input string, look for
     *  the next '%' character, and try to make the substitution.
     *  Continue doing this to the end of the string.
     */
    chunk = p = string;
    while ((p = strchr(p, '%')) != NULL) {

	/* Copy up to the percent sign.  Repair the string afterwards */
	percentSign = *p;
	*p = '\0';
	Tcl_DStringAppend(dsPtr, chunk, -1);
	*p = percentSign;	

	/* Search for a matching substition rule */
	letter = *(p + 1);
	for (i = 0; i < nsubs; i++) {
	    if (subs[i].letter == letter) {
		break;
	    }
	}
	if (i < nsubs) {
	    /* Make the substitution */
	    Tcl_DStringAppend(dsPtr, subs[i].value, -1);
	} else {
	    /* Copy in the %letter verbatim */
	    static char noSub[3] = "%?";

	    noSub[1] = letter;
	    Tcl_DStringAppend(dsPtr, noSub, -1);
	}
	p += 2;		/* Skip % + letter */
	if (letter == '\0') {
	    p += 1;	/* Premature % substitution (end of string) */
	}
	chunk = p;
    }
    /* Pick up last chunk if a substition wasn't the last thing in the string */
    if (*chunk != '\0') {
	Tcl_DStringAppend(dsPtr, chunk, -1);
    }
    return Tcl_DStringValue(dsPtr);
}


static int
ConfigureToken(interp, srcPtr, argc, argv)
    Tcl_Interp *interp;
    DragSource *srcPtr;
    int argc;
    char **argv;
{
    Token *tokenPtr;

    tokenPtr = &(srcPtr->token);
    if (Tk_ConfigureWidget(interp, srcPtr->tkwin, tokenConfigSpecs, argc, argv, 
		(char *)tokenPtr, TK_CONFIG_ARGV_ONLY) != TCL_OK) {
	return TCL_ERROR;
    }
    return ConfigureSource(interp, srcPtr, 0, (char **)NULL, TK_CONFIG_ARGV_ONLY);
}
