/*
 * bltWinOp.c --
 *
 *	This module implements simple window commands for the BLT toolkit.
 *
 * Copyright 1991-1997 Bell Labs Innovations for Lucent Technologies.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that the
 * copyright notice and warranty disclaimer appear in supporting documentation,
 * and that the names of Lucent Technologies any of their entities not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this software,
 * including all implied warranties of merchantability and fitness.  In no event
 * shall Lucent Technologies be liable for any special, indirect or
 * consequential damages or any damages whatsoever resulting from loss of use,
 * data or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or performance
 * of this software.  
 */

#include "bltInt.h"
#include <X11/Xutil.h>

#define WINOP_VERSION "1.1"

static Tk_Window
NameToWindow(interp, pathName, mainWindow)
    Tcl_Interp *interp;
    char *pathName;
    Tk_Window mainWindow;
{
    Tk_Window tkwin;

    tkwin = Tk_NameToWindow(interp, pathName, mainWindow);
    if ((tkwin != NULL) && (Tk_WindowId(tkwin) == None)) {
	Tk_MakeWindowExist(tkwin);
    }
    return tkwin;
}

#ifdef notdef
static void
RestackToplevel(tkwin, window, how)
    Tk_Window tkwin;
    Window window;
    int how;
{
    XWindowChanges changes;
    unsigned int mask;

    changes.stack_mode = how;
    changes.sibling = None;
    mask = CWStackMode;
    XReconfigureWMWindow(Tk_Display(tkwin), window, Tk_ScreenNumber(tkwin), 
		mask,  &changes);
}
#endif

static int
LowerOp(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    register int i;
    Tk_Window tkwin;

    for (i = 2; i < argc; i++) {
	tkwin = NameToWindow(interp, argv[i], (Tk_Window)clientData);
	if (tkwin == NULL) {
	    return TCL_ERROR;
	}
	XLowerWindow(Tk_Display(tkwin), Blt_XWindowId(tkwin));
    }
    return TCL_OK;
}

static int
RaiseOp(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    register int i;
    Tk_Window tkwin;

    for (i = 2; i < argc; i++) {
	tkwin = NameToWindow(interp, argv[i], (Tk_Window)clientData);
	if (tkwin == NULL) {
	    return TCL_ERROR;
	}
	XRaiseWindow(Tk_Display(tkwin), Blt_XWindowId(tkwin));
    }
    return TCL_OK;
}

static int
MapOp(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    register int i;
    Tk_Window tkwin;

    for (i = 2; i < argc; i++) {
	tkwin = NameToWindow(interp, argv[i], (Tk_Window)clientData);
	if (tkwin == NULL) {
	    return TCL_ERROR;
	}
	XMapWindow(Tk_Display(tkwin), Blt_XWindowId(tkwin));
    }
    return TCL_OK;
}

/*ARGSUSED*/
static int
MoveToOp(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;			/* Unused */
    char **argv;
{
    int x, y;
    Tk_Window tkwin;

    tkwin = NameToWindow(interp, argv[2], (Tk_Window)clientData);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }
    if (Tk_GetPixels(interp, tkwin, argv[3], &x) != TCL_OK) {
	Tcl_AppendResult(interp, ": bad window x-coordinate", (char *)NULL);
	return TCL_ERROR;
    }
    if (Tk_GetPixels(interp, tkwin, argv[4], &y) != TCL_OK) {
	Tcl_AppendResult(interp, ": bad window y-coordinate", (char *)NULL);
	return TCL_ERROR;
    }
    if (Tk_IsTopLevel(tkwin)) {
	XMoveWindow(Tk_Display(tkwin), Blt_XWindowId(tkwin), x, y);
    } else {
	Tk_MoveWindow(tkwin, x, y);
    }
    return TCL_OK;
}

static int
UnmapOp(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    register int i;
    Tk_Window tkwin;

    for (i = 2; i < argc; i++) {
	tkwin = NameToWindow(interp, argv[i], (Tk_Window)clientData);
	if (tkwin == NULL) {
	    return TCL_ERROR;
	}
	XUnmapWindow(Tk_Display(tkwin), Blt_XWindowId(tkwin));
    }
    return TCL_OK;
}

/* ARGSUSED */
static int
QueryOp(interp, tkwin)
    Tcl_Interp *interp;
    Tk_Window tkwin;
{
    int rootX, rootY, childX, childY;
    Window root, child;
    unsigned int mask;

    if (XQueryPointer(Tk_Display(tkwin), Tk_WindowId(tkwin), &root,
	    &child, &rootX, &rootY, &childX, &childY, &mask)) {
	char string[200];

	sprintf(string, "@%d,%d", rootX, rootY);
	Tcl_SetResult(interp, string, TCL_VOLATILE);
    }
    return TCL_OK;
}

/*ARGSUSED*/
static int
WarpToOp(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;			/* Not used */
    char **argv;
{
    Tk_Window tkwin;

    if (argc == 3) {
	if (argv[2][0] == '@') {
	    XPoint point;
	    Window root;
	    
	    tkwin = (Tk_Window)clientData;
	    if (Blt_GetXYPosition(interp, tkwin, argv[2], &point) != TCL_OK) {
		return TCL_ERROR;
	    }
	    root = RootWindow(Tk_Display(tkwin), Tk_ScreenNumber(tkwin));
	    XWarpPointer(Tk_Display(tkwin), None, root, 0, 0, 0, 0,
			 point.x, point.y);
	} else {
	    tkwin = NameToWindow(interp, argv[2], (Tk_Window)clientData);
	    if (tkwin == NULL) {
		return TCL_ERROR;
	    }
	    if (!Tk_IsMapped(tkwin)) {
		Tcl_AppendResult(interp, "can't warp to unmapped window \"",
				 Tk_PathName(tkwin), "\"", (char *)NULL);
		return TCL_ERROR;
	    }
	    XWarpPointer(Tk_Display(tkwin), None, Tk_WindowId(tkwin), 
		0, 0, 0, 0, Tk_Width(tkwin) / 2, Tk_Height(tkwin) / 2);
	}
    }
    return (QueryOp(interp, (Tk_Window)clientData));
}

/*
 * --------------------------------------------------------------------------
 *
 * SnapOp --
 *
 *	Snaps a picture of the graph and stores it in the specified image
 *
 * Results:
 *	Returns a standard Tcl result.  interp->result contains
 *	the list of the graph coordinates. If an error occurred
 *	while parsing the window positions, TCL_ERROR is returned,
 *	then interp->result will contain an error message.
 *
 * -------------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
SnapOp(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;			/* Not used */
    char **argv;
{
    Tk_Window tkwin;
    int result;

    tkwin = NameToWindow(interp, argv[2], (Tk_Window)clientData);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }
    result = Blt_SnapPhoto(interp, tkwin, Tk_WindowId(tkwin), Tk_Width(tkwin), 
	Tk_Height(tkwin), argv[3]);
    return result;
}

static Blt_OpSpec winOps[] =
{
    {"lower", 1, (Blt_Operation)LowerOp, 2, 0, "window ?window?...",},
    {"map", 1, (Blt_Operation)MapOp, 2, 0, "window ?window?...",},
    {"move", 1, (Blt_Operation)MoveToOp, 5, 5, "window x y",},
    {"raise", 1, (Blt_Operation)RaiseOp, 2, 0, "window ?window?...",},
    {"snap", 1, (Blt_Operation)SnapOp, 4, 4, "window photoName",},
    {"unmap", 1, (Blt_Operation)UnmapOp, 2, 0, "window ?window?...",},
    {"warpto", 1, (Blt_Operation)WarpToOp, 2, 3, "?window?",},
};
static int numWinOps = sizeof(winOps) / sizeof(Blt_OpSpec);

/* ARGSUSED */
static int
WindowCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window of interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    Blt_Operation proc;
    int result;

    proc = Blt_LookupOperation(interp, numWinOps, winOps, BLT_OPER_ARG1,
	argc, argv);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (clientData, interp, argc, argv);
    return (result);
}

int
Blt_WinOpInit(interp)
    Tcl_Interp *interp;
{
    static Blt_CmdSpec cmdSpec = { "winop", WindowCmd, };

    return (Blt_InitCmd(interp, "blt", &cmdSpec));
}
