wilcox.test <- function(x, y = NULL, alternative = "two.sided", mu = 0,
			paired = FALSE, exact = FALSE, correct = TRUE) 
{
  CHOICES <- c("two.sided", "less", "greater")
  alternative <- CHOICES[pmatch(alternative, CHOICES)]
  if (length(alternative) > 1 || is.na(alternative)) 
    stop("alternative must be \"two.sided\", \"less\" or \"greater\"")

  if (!missing(mu) && ((length(mu) > 1) || !is.finite(mu)))
    stop("mu must be a single number")

  if (!is.null(y)) {
    DNAME <- paste(deparse(substitute(x)), "and", deparse(substitute(y)))
    if (paired) {
      if (length(x) != length(y))
	stop("x and y must have the same length")
      OK <- complete.cases(x, y)
      x <- x[OK] - y[OK]
      y <- NULL
    }
    else {
      x <- x[is.finite(x)]
      y <- y[is.finite(y)]
    }
  }
  else {
    DNAME <- deparse(substitute(x))
    if (paired)
      stop("y missing for paired test")
    x <- x[is.finite(x)]
  }

  if (length(x) < 1)
    stop("not enough x observations")

  if (exact)
    warning("exact Wilcoxon tests are currently not implemented\n")

  PARAMETER <- NULL
  
  CORRECTION <- 0  

  if (is.null(y)) {
    METHOD <- "Wilcoxon signed rank test"
    x <- x - mu
    r <- rank(abs(x))
    STATISTIC <- sum(r[x > 0])
    names(STATISTIC) <- "V"
    n <- length(x)
    TIES <- table(r)
    z <- STATISTIC - n*(n+1)/4
    SIGMA <- sqrt(n*(n+1)*(2*n+1)/24 - sum(TIES^3-TIES)/48)
  }
  else {
    if (length(y) < 1)
      stop("not enough y observations")
    METHOD <- "Wilcoxon rank sum test"
    r <- rank(c(x - mu, y))
    STATISTIC <- sum(r[seq(along = x)])
    names(STATISTIC) <- "W"
    n.x <- length(x)
    n.y <- length(y)
    TIES <- table(r)
    z <- STATISTIC - n.x*(n.x+n.y+1)/2
    SIGMA <- sqrt((n.x*n.y/12) *
		  ((n.x+n.y+1)
		   - sum(TIES^3-TIES) / ((n.x+n.y)*(n.x+n.y+1))))
  }

  if (correct) {
    CORRECTION <- switch (alternative,
			  "two.sided" = sign(z) * 0.5,
			  "greater" = 0.5,
			  "less" = -0.5)
    METHOD <- paste(METHOD, "with continuity correction")
  }
  
  PVAL <- pnorm((z - CORRECTION) / SIGMA)
  if (alternative == "two.sided")
    PVAL <- 2 * min (PVAL, 1 - PVAL)
  else if (alternative == "greater")
    PVAL <- 1 - PVAL

  NVAL <- mu
  names(NVAL) <- "mu"

  RVAL <- list(statistic = STATISTIC,
	       parameter = PARAMETER,
	       p.value = PVAL,
	       null.value = NVAL,
	       alternative = alternative,
	       method = METHOD,
	       data.name = DNAME)
  class(RVAL) <- "htest"
  return(RVAL)
}
