/***

ifstats.c	- the interface statistics module
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
	
***/

#include <curses.h>
#include <panel.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if_arp.h>
#include <linux/if_packet.h>
#include <net/if.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include "if_ether.h"
#include "ifstats.h"
#include "packet.h"
#include "log.h"
#include "stdwinset.h"
#include "deskman.h"
#include "attrs.h"
#include "ipcsum.h"
#include "error.h"
#include "serv.h"
#include "timer.h"

#define SCROLLUP 0
#define SCROLLDOWN 1

/* from log.c, applicable only to this module */

void writegstatlog(struct iftab *table, unsigned long nsecs, FILE * logfile);
void writedstatlog(char *ifname, float activity, float pps,
			  float peakactivity, float peakpps,
			  struct iftotals *ts, unsigned long nsecs,
			  FILE * logfile);

char *ltrim(char *buf)
{
    char *tmp = buf;

    while (*tmp == ' ')
	tmp++;

    strcpy(buf, tmp);
    return buf;
}

/*
 * Function to check if an interface is already in the interface list.
 * This eliminates duplicate interface entries due to aliases
 */

int ifinlist(struct iflist *list, char *ifname)
{
    struct iflist *ptmp = list;
    int result = 0;

    while ((ptmp != NULL) && (result == 0)) {
	result = (strcmp(ifname, ptmp->ifname) == 0);
	ptmp = ptmp->next_entry;
    }

    return result;
}

void initiflist(struct iflist **list)
{
    FILE *fd;
    char buf[100];
    char ifname[8];
    struct iflist *itmp = NULL;
    struct iflist *tail = NULL;
    unsigned int index = 0;
    int resp;

    *list = NULL;

    fd = fopen("/proc/net/dev", "r");
    if (fd == NULL) {
	errbox("Unable to obtain interface list", ANYKEY_MSG, &resp);
	return;
    }
    fgets(buf, 80, fd);		/* dummy header lines */
    fgets(buf, 80, fd);

    while (!(feof(fd))) {
	strcpy(buf, "");
	fgets(buf, 80, fd);
	if (strcmp(buf, "") != 0) {
	    strcpy(ifname, ltrim(strtok(buf, ":")));

	    if (ifinlist(*list, ifname))	/* ignore entry if already in */
		continue;	/* interface list */

	    itmp = malloc(sizeof(struct iflist));
	    bzero(itmp, sizeof(struct iflist));
	    strcpy(itmp->ifname, ifname);
	    index++;
	    itmp->index = index;

	    if (*list == NULL) {
		*list = itmp;
		itmp->prev_entry = NULL;
	    } else {
		tail->next_entry = itmp;
		itmp->prev_entry = tail;
	    }

	    tail = itmp;
	    itmp->next_entry = NULL;
	}
    }
    fclose(fd);
}

void positionptr(struct iflist *list, struct iflist **ptmp, char *ifname)
{
    struct iflist *plast = NULL;
    int ok = 0;

    *ptmp = list;

    while ((*ptmp != NULL) && (!ok)) {
	ok = (strcmp((*ptmp)->ifname, ifname) == 0);

	if (!ok) {
	    if ((*ptmp)->next_entry == NULL)
		plast = *ptmp;

	    *ptmp = (*ptmp)->next_entry;
	}
    }

    if (*ptmp == NULL) {
	*ptmp = malloc(sizeof(struct iflist));
	bzero(*ptmp, sizeof(struct iflist));
	(*ptmp)->index = plast->index + 1;
	plast->next_entry = *ptmp;
	(*ptmp)->prev_entry = plast;
	(*ptmp)->next_entry = NULL;
	strcpy((*ptmp)->ifname, ifname);
    }
}

void destroyiflist(struct iflist *list)
{
    struct iflist *ctmp;
    struct iflist *ptmp;

    ptmp = list;
    ctmp = ptmp->next_entry;

    do {
	free(ptmp);
	ptmp = ctmp;
	if (ctmp != NULL)
	    ctmp = ctmp->next_entry;
    } while (ptmp != NULL);
}

void updaterates(struct iftab *table, time_t starttime, time_t now,
		 unsigned int idx)
{
    struct iflist *ptmp = table->firstvisible;

    wattrset(table->statwin, HIGHATTR);
    do {
	ptmp->rate = ((float) (ptmp->spanbr * 8 / 1000)) / ((float) (now - starttime));

	if (ptmp->rate > ptmp->peakrate)
	    ptmp->peakrate = ptmp->rate;

	wmove(table->statwin, ptmp->index - idx, 52);
	wprintw(table->statwin, "%8.2f kbits/sec", ptmp->rate);
	ptmp->spanbr = 0;
	ptmp = ptmp->next_entry;
    } while (ptmp != table->lastvisible->next_entry);
}

void printifentry(struct iflist *ptmp, WINDOW * win, unsigned int idx)
{
    unsigned int target_row;

    if ((ptmp->index < idx) || (ptmp->index > idx + 14))
	return;

    target_row = ptmp->index - idx;

    wattrset(win, STDATTR);
    wmove(win, target_row, 1);
    wprintw(win, "%s", ptmp->ifname);
    wattrset(win, HIGHATTR);
    wmove(win, target_row, 12);
    wprintw(win, "%8d", ptmp->total);
    wmove(win, target_row, 22);
    wprintw(win, "%8d", ptmp->iptotal);
    wmove(win, target_row, 32);
    wprintw(win, "%8d", ptmp->noniptotal);
    wmove(win, target_row, 42);
    wprintw(win, "%8d", ptmp->badtotal);
}

void preparescreen(struct iftab *table)
{
    struct iflist *ptmp = table->head;
    unsigned int i = 1;

    table->firstvisible = table->head;

    do {
	printifentry(ptmp, table->statwin, 1);

	if (i <= 15)
	    table->lastvisible = ptmp;

	ptmp = ptmp->next_entry;
	i++;
    } while ((ptmp != NULL) && (i <= 15));
}

void labelstats(WINDOW * win)
{
    wmove(win, 0, 1);
    wprintw(win, " Iface ");
    wmove(win, 0, 15);
    wprintw(win, " Total ");
    wmove(win, 0, 28);
    wprintw(win, " IP ");
    wmove(win, 0, 35);
    wprintw(win, " NonIP ");
    wmove(win, 0, 45);
    wprintw(win, " BadIP ");
    wmove(win, 0, 55);
    wprintw(win, " Activity ");
};

void initiftab(struct iftab *table)
{
    initiflist(&(table->head));

    table->borderwin = newwin(17, 80, 4, 0);
    table->borderpanel = new_panel(table->borderwin);

    move(24, 1);
    stdexitkeyhelp();
    wattrset(table->borderwin, BOXATTR);
    box(table->borderwin, ACS_VLINE, ACS_HLINE);
    labelstats(table->borderwin);
    table->statwin = newwin(15, 78, 5, 1);
    table->statpanel = new_panel(table->statwin);
    stdwinset(table->statwin);
    wtimeout(table->statwin, -1);
    wattrset(table->statwin, STDATTR);
    colorwin(table->statwin);
    wattrset(table->statwin, BOXATTR);
    wmove(table->borderwin, 16, 32);
    wprintw(table->borderwin, " Total, IP, NonIP, and BadIP are packet counts ");
}

/*
 * Scrolling routines for the general interface statistics window
 */

void scrollgstatwin(struct iftab *table, int direction, unsigned int *idx)
{
    wattrset(table->statwin, STDATTR);
    if (direction == SCROLLUP) {
	if (table->lastvisible->next_entry != NULL) {
	    wscrl(table->statwin, 1);
	    table->lastvisible = table->lastvisible->next_entry;
	    table->firstvisible = table->firstvisible->next_entry;
	    (*idx)++;
	    wmove(table->statwin, 14, 0);
	    scrollok(table->statwin, 0);
	    wprintw(table->statwin, "%78c", ' ');
	    scrollok(table->statwin, 1);
	    printifentry(table->lastvisible, table->statwin, *idx);
	}
    } else {
	if (table->firstvisible != table->head) {
	    wscrl(table->statwin, -1);
	    table->firstvisible = table->firstvisible->prev_entry;
	    table->lastvisible = table->lastvisible->prev_entry;
	    (*idx)--;
	    wmove(table->statwin, 0, 0);
	    scrollok(table->statwin, 0);
	    wprintw(table->statwin, "%78c", ' ');
	    scrollok(table->statwin, 1);
	    printifentry(table->firstvisible, table->statwin, *idx);
	}
    }
}

void pagegstatwin(struct iftab *table, int direction, int *idx)
{
    int i = 1;

    if (direction == SCROLLUP) {
	while ((i <= 13) && (table->lastvisible->next_entry != NULL)) {
	    i++;
	    scrollgstatwin(table, direction, idx);
	}
    } else {
	while ((i <= 13) && (table->firstvisible != table->head)) {
	    i++;
	    scrollgstatwin(table, direction, idx);
	}
    }
}

void ifstats(int logging, time_t logspan)
{
    struct iftab table;

    char buf[8192];
    char *packet;
    struct iphdr *ipacket;
    char *tpacket;

    struct sockaddr_pkt fromaddr;

    struct iflist *ptmp = NULL;

    unsigned int idx = 1;

    int fd;
    FILE *logfile = NULL;

    int br;
    char ifname[10];

    int iphlen;
    int ipck, ipcsum;

    int ch;

    int endloop = 0;
    time_t starttime = 0;
    time_t statbegin = 0;
    time_t now = 0;
    time_t startlog = 0;

    initiftab(&table);

    if (logging) {
	logfile = opentlog();

	if (logfile == NULL)
	    logging = 0;
    }
    writelog(logging, logfile, "******** General interface statistics started");

    if (table.head != NULL) {
	open_socket(&fd);

	if (fd < 0)
	    return;

	preparescreen(&table);

	update_panels();
	doupdate();

	starttime = startlog = statbegin = time((time_t *) NULL);

	while (!endloop) {
	    getpacket(fd, buf, &fromaddr, &ch, &br, 5, table.statwin);

	    if (ch != ERR) {
		switch (ch) {
		case KEY_UP:
		    scrollgstatwin(&table, SCROLLDOWN, &idx);
		    break;
		case KEY_DOWN:
		    scrollgstatwin(&table, SCROLLUP, &idx);
		    break;
		case KEY_PPAGE:
		    pagegstatwin(&table, SCROLLDOWN, &idx);
		    break;
		case KEY_NPAGE:
		    pagegstatwin(&table, SCROLLUP, &idx);
		    break;
		case 'Q':
		case 'q':
		case 'X':
		case 'x':
		case 27:
		case 24:
		    endloop = 1;
		    break;
		}
	    }
	    if (br > 0) {
		strcpy(ifname, fromaddr.spkt_device);
		positionptr(table.head, &ptmp, ifname);

		ptmp->total++;

		ptmp->spanbr += br;
		ptmp->br += br;

		if (fromaddr.spkt_protocol == ntohs(ETH_P_IP)) {
		    (ptmp->iptotal)++;
		    adjustpacket(buf, fromaddr.spkt_family,
				 &packet, &br);

		    if (packet == NULL)
			continue;

		    ipacket = (struct iphdr *) packet;
		    iphlen = ipacket->ihl * 4;
		    tpacket = packet + iphlen;

		    /* recalculate checksum */

		    ipcsum = ipacket->check;
		    ipacket->check = 0;
		    ipck = in_cksum((u_short *) ipacket, iphlen);
		    if ((ipcsum != ipck) || (!ipck)) {
			(ptmp->badtotal)++;
			continue;
		    }
		} else {
		    (ptmp->noniptotal)++;
		}
		printifentry(ptmp, table.statwin, idx);
	    }
	    now = time((time_t *) NULL);
	    if ((now - starttime) >= 5) {
		updaterates(&table, starttime, now, idx);
		printelapsedtime(statbegin, now, 16, 1, table.borderwin);
		starttime = now;		
	    }
	    if (((now - startlog) >= logspan) && (logging)) {
		writegstatlog(&table, time((time_t *) NULL) - statbegin, logfile);
		startlog = now;
	    }
	    update_panels();
	    doupdate();
	}

	close(fd);
    }
    del_panel(table.statpanel);
    delwin(table.statwin);
    del_panel(table.borderpanel);
    delwin(table.borderwin);
    update_panels();
    doupdate();

    if (logging) {
	writegstatlog(&table, time((time_t *) NULL) - statbegin, logfile);
	writelog(logging, logfile, "******** General interface statistics stopped");
	fclose(logfile);
    }
    destroyiflist(table.head);
}

void printdetlabels(WINDOW * win)
{
    char *sizelabels = "Pkt Size (bytes)    Count";

    wattrset(win, BOXATTR);
    wmove(win, 2, 13);
    wprintw(win, "Packets     Bytes");
    wmove(win, 12, 2);
    wprintw(win, sizelabels);
    wmove(win, 12, 32);
    wprintw(win, sizelabels);
    wattrset(win, STDATTR);
    wmove(win, 3, 2);
    wprintw(win, "Total:");
    wmove(win, 4, 2);
    wprintw(win, "IP:");
    wmove(win, 5, 2);
    wprintw(win, "TCP:");
    wmove(win, 6, 2);
    wprintw(win, "UDP:");
    wmove(win, 7, 2);
    wprintw(win, "ICMP:");
    wmove(win, 8, 2);
    wprintw(win, "Other IP:");
    wmove(win, 9, 2);
    wprintw(win, "Non-IP:");
    wmove(win, 5, 40);
    wprintw(win, "Activity:");

    wmove(win, 8, 40);
    wprintw(win, "IP Checksum Errors:");

    wmove(win, 13, 2);
    wprintw(win, "1 to 100:");
    wmove(win, 14, 2);
    wprintw(win, "101 to 200:");
    wmove(win, 15, 2);
    wprintw(win, "201 to 300:");
    wmove(win, 16, 2);
    wprintw(win, "301 to 400:");
    wmove(win, 17, 2);
    wprintw(win, "401 to 500:");
    wmove(win, 18, 2);
    wprintw(win, "501 to 600:");
    wmove(win, 19, 2);
    wprintw(win, "601 to 700:");
    wmove(win, 20, 2);
    wprintw(win, "701 to 800:");

    wmove(win, 13, 32);
    wprintw(win, "801 to 900:");
    wmove(win, 14, 32);
    wprintw(win, "901 to 1000:");
    wmove(win, 15, 32);
    wprintw(win, "1001 to 1100:");
    wmove(win, 16, 32);
    wprintw(win, "1101 to 1200:");
    wmove(win, 17, 32);
    wprintw(win, "1201 to 1300:");
    wmove(win, 18, 32);
    wprintw(win, "1301 to 1400:");
    wmove(win, 19, 32);
    wprintw(win, "1401 to 1500:");
    wmove(win, 20, 32);
    wprintw(win, "1500+:");
    update_panels();
    doupdate();
}

void printdetails(struct iftotals *totals, WINDOW * win)
{
    wattrset(win, HIGHATTR);

    /* Print totals on the IP protocols */

    wmove(win, 3, 12);
    wprintw(win, "%8d", totals->total);
    wmove(win, 3, 22);
    wprintw(win, "%8d", totals->bytestotal);
    wmove(win, 4, 12);
    wprintw(win, "%8d", totals->iptotal);
    wmove(win, 4, 22);
    wprintw(win, "%8d", totals->ipbtotal);

    wmove(win, 5, 12);
    wprintw(win, "%8d", totals->tcptotal);
    wmove(win, 5, 22);
    wprintw(win, "%8d", totals->tcpbtotal);

    wmove(win, 6, 12);
    wprintw(win, "%8d", totals->udptotal);
    wmove(win, 6, 22);
    wprintw(win, "%8d", totals->udpbtotal);

    wmove(win, 7, 12);
    wprintw(win, "%8d", totals->icmptotal);
    wmove(win, 7, 22);
    wprintw(win, "%8d", totals->icmpbtotal);

    wmove(win, 8, 12);
    wprintw(win, "%8d", totals->othtotal);
    wmove(win, 8, 22);
    wprintw(win, "%8d", totals->othbtotal);

    /* Print non-IP totals */

    wmove(win, 9, 12);
    wprintw(win, "%8d", totals->noniptotal);
    wmove(win, 9, 22);
    wprintw(win, "%8d", totals->nonipbtotal);

    /* Bad packet count */

    wmove(win, 8, 60);
    wprintw(win, "%8d", totals->badtotal);

    /* Packet size totals */
    wmove(win, 13, 19);
    wprintw(win, "%8d", totals->max100);
    wmove(win, 14, 19);
    wprintw(win, "%8d", totals->max200);
    wmove(win, 15, 19);
    wprintw(win, "%8d", totals->max300);
    wmove(win, 16, 19);
    wprintw(win, "%8d", totals->max400);
    wmove(win, 17, 19);
    wprintw(win, "%8d", totals->max500);
    wmove(win, 18, 19);
    wprintw(win, "%8d", totals->max600);
    wmove(win, 19, 19);
    wprintw(win, "%8d", totals->max700);
    wmove(win, 20, 19);
    wprintw(win, "%8d", totals->max800);

    wmove(win, 13, 49);
    wprintw(win, "%8d", totals->max900);
    wmove(win, 14, 49);
    wprintw(win, "%8d", totals->max1000);
    wmove(win, 15, 49);
    wprintw(win, "%8d", totals->max1100);
    wmove(win, 16, 49);
    wprintw(win, "%8d", totals->max1200);
    wmove(win, 17, 49);
    wprintw(win, "%8d", totals->max1300);
    wmove(win, 18, 49);
    wprintw(win, "%8d", totals->max1400);
    wmove(win, 19, 49);
    wprintw(win, "%8d", totals->max1500);
    wmove(win, 20, 49);
    wprintw(win, "%8d", totals->g1500);
}

void detstats(char *iface, int logging, time_t logspan)
{
    WINDOW *statwin;
    PANEL *statpanel;

    char buf[8192];
    char *packet;
    struct iphdr *ipacket = NULL;
    char *tpacket;

    struct sockaddr_pkt fromaddr;

    int fd;
    FILE *logfile = NULL;
    int br;

    int iphlen;
    int ipck, ipcsum;
    unsigned int iplen;

    struct iftotals totals;

    int ch;

    int endloop = 0;
    time_t starttime, now;
    time_t statbegin, startlog;
    unsigned int spanbr = 0;
    unsigned int spanpkt = 0;
    float activity = 0;
    float peakactivity = 0;
    float pps = 0;
    float peakpps = 0;

    move(24, 1);
    stdexitkeyhelp();
    statwin = newwin(23, 80, 1, 0);
    statpanel = new_panel(statwin);
    stdwinset(statwin);
    wtimeout(statwin, -1);
    wattrset(statwin, BOXATTR);
    colorwin(statwin);
    box(statwin, ACS_VLINE, ACS_HLINE);
    wmove(statwin, 0, 1);
    wprintw(statwin, " Statistics for %s ", iface);
    wattrset(statwin, STDATTR);
    update_panels();
    doupdate();

    bzero(&totals, sizeof(struct iftotals));

    if (logging) {
	logfile = opentlog();

	if (logfile == NULL)
	    logging = 0;
    }
    writelog(logging, logfile, "******** Detailed interface statistics started");
    open_socket(&fd);

    if (fd < 0)
	return;

    printdetlabels(statwin);
    printdetails(&totals, statwin);
    update_panels();
    doupdate();

    spanbr = 0;
    starttime = startlog = statbegin = time((time_t *) NULL);

    while (!endloop) {
	getpacket(fd, buf, &fromaddr, &ch, &br, 5, statwin);

	if (ch != ERR) {
	    switch (ch) {
	    case 'Q':
	    case 'q':
	    case 'X':
	    case 'x':
	    case 24:
	    case 27:
		endloop = 1;
		break;
	    }
	}
	if (br > 0) {
	    if (strcmp(iface, fromaddr.spkt_device) != 0)
		continue;

	    totals.total++;
	    totals.bytestotal += br;

	    spanbr += br;
	    spanpkt++;

	    if (fromaddr.spkt_protocol == ntohs(ETH_P_IP)) {
		totals.iptotal++;
		adjustpacket(buf, fromaddr.spkt_family,
			     &packet, &br);

		if (packet == NULL)
		    continue;

		ipacket = (struct iphdr *) packet;
		iphlen = ipacket->ihl * 4;
		tpacket = packet + iphlen;

		ipcsum = ipacket->check;
		ipacket->check = 0;
		ipck = in_cksum((u_short *) ipacket, iphlen);
		if ((ipcsum != ipck) || (!ipck)) {
		    totals.badtotal++;
		    continue;
		}
		iplen = ntohs(ipacket->tot_len);
		totals.ipbtotal += iplen;

		switch (ipacket->protocol) {
		case IPPROTO_TCP:
		    totals.tcptotal++;
		    totals.tcpbtotal += iplen;
		    break;
		case IPPROTO_UDP:
		    totals.udptotal++;
		    totals.udpbtotal += iplen;
		    break;
		case IPPROTO_ICMP:
		    totals.icmptotal++;
		    totals.icmpbtotal += iplen;
		    break;
		default:
		    totals.othtotal++;
		    totals.othbtotal += iplen;
		    break;
		}

		if (iplen <= 100)
		    totals.max100++;
		else if (iplen <= 200)
		    totals.max200++;
		else if (iplen <= 300)
		    totals.max300++;
		else if (iplen <= 400)
		    totals.max400++;
		else if (iplen <= 500)
		    totals.max500++;
		else if (iplen <= 600)
		    totals.max600++;
		else if (iplen <= 700)
		    totals.max700++;
		else if (iplen <= 800)
		    totals.max800++;
		else if (iplen <= 900)
		    totals.max900++;
		else if (iplen <= 1000)
		    totals.max1000++;
		else if (iplen <= 1100)
		    totals.max1100++;
		else if (iplen <= 1200)
		    totals.max1200++;
		else if (iplen <= 1300)
		    totals.max1300++;
		else if (iplen <= 1400)
		    totals.max1400++;
		else if (iplen <= 1500)
		    totals.max1500++;
		else
		    totals.g1500++;
	    } else {
		totals.noniptotal++;
		totals.nonipbtotal += br;
	    }
	}
	printdetails(&totals, statwin);

	now = time((time_t *) NULL);
	if (now - starttime >= 5) {
	    wattrset(statwin, BOXATTR);
	    printelapsedtime(statbegin, now, 22, 1, statwin);
	    activity = (float) ((spanbr * 8 / 1000) / (float) (now - starttime));
	    pps = (float) (spanpkt) / (float) (now - starttime);
	    spanbr = 0;
	    spanpkt = 0;
	    starttime = now;
	    
	    wattrset(statwin, HIGHATTR);
	    wmove(statwin, 5, 50);
	    wprintw(statwin, "%8.2f kbits/sec", activity);
	    wmove(statwin, 6, 50);
	    wprintw(statwin, "%8.2f packets/sec", pps);

	    if (activity > peakactivity)
		peakactivity = activity;

	    if (pps > peakpps)
		peakpps = pps;
	}
	if (((now - startlog) >= logspan) && (logging)) {
	    writedstatlog(iface, activity, pps, peakactivity, peakpps,
		    &totals, time((time_t *) NULL) - statbegin, logfile);
	    startlog = now;
	}
	update_panels();
	doupdate();
    }

    close(fd);

    if (logging) {
	writedstatlog(iface, activity, pps, peakactivity, peakpps,
		    &totals, time((time_t *) NULL) - statbegin, logfile);
	writelog(logging, logfile, "******** Detailed interface statistics stopped");
	fclose(logfile);
    }
    del_panel(statpanel);
    delwin(statwin);
    update_panels();
    doupdate();
}

void selectiface(char *ifname, int *aborted)
{
    WINDOW *win;
    PANEL *panel;
    WINDOW *bw;
    PANEL *bp;

    int row;
    int ch;
    int exitloop = 0;

    struct iflist *list;
    struct iflist *ptmp;

    initiflist(&list);

    listkeyhelp();
    bw = newwin(14, 24, 6, 19);
    bp = new_panel(bw);
    wattrset(bw, BOXATTR);
    box(bw, ACS_VLINE, ACS_HLINE);

    win = newwin(12, 22, 7, 20);
    panel = new_panel(win);
    stdwinset(win);
    wattrset(win, BOXATTR);
    colorwin(win);
    wmove(bw, 0, 1);
    wprintw(bw, " Select Interface ");

    row = 0;
    ptmp = list;

    wattrset(win, STDATTR);
    do {
	wmove(win, row, 2);
	wprintw(win, "%s", ptmp->ifname);
	ptmp = ptmp->next_entry;
	row++;
    } while ((ptmp != NULL) && (row <= 11));
    update_panels();
    doupdate();

    row = 0;
    ptmp = list;

    do {
	wattrset(win, PTRATTR);
	wmove(win, row, 1);
	waddch(win, ACS_RARROW);
	ch = wgetch(win);
	wmove(win, row, 1);
	wprintw(win, " ");

	wattrset(win, STDATTR);
	switch (ch) {
	case KEY_DOWN:
	    if (ptmp->next_entry != NULL) {
		if (row < 11)
		    row++;
		else {
		    wscrl(win, 1);
		    scrollok(win, 0);
		    wmove(win, 11, 0);
		    wprintw(win, "%22c", ' ');
		    wmove(win, 11, 2);
		    wprintw(win, ptmp->next_entry->ifname);
		    scrollok(win, 1);
		}
		ptmp = ptmp->next_entry;
	    }
	    break;
	case KEY_UP:
	    if (ptmp->prev_entry != NULL) {
		if (row > 0)
		    row--;
		else {
		    wscrl(win, -1);
		    scrollok(win, 0);
		    wmove(win, 0, 0);
		    wprintw(win, "%22c", ' ');
		    wmove(win, 0, 2);
		    wprintw(win, ptmp->prev_entry->ifname);
		    scrollok(win, 1);
		}
		ptmp = ptmp->prev_entry;
	    }
	    break;
	case 13:
	    exitloop = 1;
	    *aborted = 0;
	    break;
	case 27:
	case 24:
	case 'X':
	case 'x':
	case 'Q':
	case 'q':
	    exitloop = 1;
	    *aborted = 1;
	    break;
	}
	update_panels();
	doupdate();
    } while (!exitloop);

    if (!(*aborted))
	strcpy(ifname, ptmp->ifname);

    destroyiflist(list);
    del_panel(panel);
    delwin(win);
    del_panel(bp);
    delwin(bw);
    update_panels();
    doupdate();
}
