/*================================================================
 * load soundfont and virtual bank
 *================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#ifdef linux
#include <linux/soundcard.h>
#include <linux/awe_voice.h>
#else
#include <machine/soundcard.h>
#include <awe_voice.h>
#endif
#include "util.h"
#include "awebank.h"
#include "aweseq.h"
#include "sfopts.h"
#include "config.h"


/*----------------------------------------------------------------
 * prototypes
 *----------------------------------------------------------------*/

/* virtual bank record */
typedef struct _VBank {
	char *name;
	LoadList *list;
	struct _VBank *next;
} VBank;

static int is_virtual_bank(char *path);
static int load_virtual_bank(char *path, LoadList *part_list, int locked);
static int load_patch(char *path, LoadList *lp, LoadList *exlp, int locked, int load_alt);
static int load_map(LoadList *lp, int locked);

static LoadList *make_virtual_list(VBank *v, LoadList *part_list);
static void make_bank_table(FILE *fp);
static void include_bank_table(char *name);
static void free_bank_table(void);

static int do_load_banks(int locked);
static int load_one_bank(VBank *v, int locked);
static int load_mapping_fonts(int locked);
static int load_default_fonts(int locked);
static void mark_loaded_presets(LoadList *pat, LoadList *list);


/*----------------------------------------------------------------*/
/* extensions to be searched */

static char *path_ext[] = {
	".sf2", ".SF2", ".sbk", ".SBK", NULL,
};
static char *path_ext_all[] = {
	".bnk", ".sf2", ".SF2", ".sbk", ".SBK", NULL,
};
static char *path_ext_bank[] = {
	".bnk", NULL,
};

/* search path for font and bank files */
static char *search_path;

/*----------------------------------------------------------------
 * load arbitrary file with specified loading list
 *----------------------------------------------------------------*/

int awe_load_bank(char *name, LoadList *list, int locked)
{
	int rc;
	char sfpath[256];

	if (awe_option.search_path)
		search_path = safe_strdup(awe_option.search_path);
	else {
		char *p = getenv("SFBANKDIR");
		if (p == NULL || *p == 0)
			p = DEFAULT_SF_PATH;
		search_path = safe_strdup(p);
	}

	if (search_file_name(sfpath, name, search_path, path_ext_all)) {
		if (is_virtual_bank(sfpath))
			rc = load_virtual_bank(sfpath, list, locked);
		else
			rc = load_patch(name, list, NULL, locked, TRUE);
	} else
		rc = AWE_RET_NOT_FOUND;

	free(search_path);
	return rc;
}

/*----------------------------------------------------------------
 * virtual bank record handlers
 *----------------------------------------------------------------*/

static VBank *vbanks, *vmap, *vmapkey;
static char *default_font;
static int bank_list_rebuilt;
static LoadList *bank_list, *excl_list;


/* check the extension */
static int is_virtual_bank(char *path)
{
	char *p;
	if ((p = strrchr(path, '.')) == NULL)
		return FALSE;
	if (strcmp(p + 1, "bnk") == 0)
		return TRUE;
	return FALSE;
}

/* read a virtual bank config file and load the specified fonts */
static int load_virtual_bank(char *path, LoadList *part_list, int locked)
{
	int rc;
	FILE *fp;

	if ((fp = fopen(path, "r")) == NULL) {
		fprintf(stderr, "awe: can't open virtual bank %s\n", path);
		return AWE_RET_ERR;
	}
	vbanks = NULL;
	vmap = vmapkey = NULL;
	default_font = NULL;
	bank_list = part_list;
	bank_list_rebuilt = FALSE;
	excl_list = NULL;

	make_bank_table(fp);
	fclose(fp);

	rc = do_load_banks(locked);

	if (default_font)
		free(default_font);
	if (bank_list_rebuilt)
		free_loadlist(bank_list);
	if (excl_list)
		free_loadlist(excl_list);

	free_bank_table();

	return rc;
}
		
/* load banks */
static int do_load_banks(int locked)
{
	int rc;
	VBank *v;

	if (vmap || vmapkey) {
		rc = load_mapping_fonts(locked);
		if (rc != AWE_RET_OK && rc != AWE_RET_SKIP)
			return rc;
	}

	for (v = vbanks; v; v = v->next) {
		rc = load_one_bank(v, locked);
		if (rc != AWE_RET_OK && rc != AWE_RET_SKIP)
			return rc;
	}

	if (default_font) {
		if (bank_list)
			rc = load_default_fonts(locked);
		else
			rc = load_patch(default_font, NULL, excl_list, locked, TRUE);
	}

	return rc;
}

/* load one soundfont file in virtual bank */
static int load_one_bank(VBank *v, int locked)
{
	int rc;
	LoadList *vlist;

	if (bank_list == NULL) {
		if (v->list == NULL)
			return AWE_RET_OK;
		excl_list = merge_loadlist(excl_list, v->list);
		return load_patch(v->name, v->list, NULL, locked, TRUE);
	}

	vlist = make_virtual_list(v, bank_list);
	if (vlist == NULL)
		return AWE_RET_OK;

	rc = load_patch(v->name, vlist, NULL, locked, TRUE);
	mark_loaded_presets(vlist, bank_list);
	free_loadlist(vlist);
	return rc;
}


/* mark the presets as actually loaded */
static void mark_loaded_presets(LoadList *pat, LoadList *list)
{
	LoadList *p, *q;
	for (p = pat; p; p = p->next) {
		if (! p->loaded)
			continue;
		for (q = list; q; q = q->next) {
			if (!q->loaded && awe_match_preset(&p->map, &q->map))
				q->loaded = TRUE;
		}
	}
}


/* load map fonts */
static int load_mapping_fonts(int locked)
{
	LoadList *vmaplist, *p, *q, *t;
	LoadList *new_bank;
	int rc;

	if (vmap) {
		rc = load_map(vmap->list, locked);
		mark_loaded_presets(vmap->list, bank_list);
	}

	if ((rc == AWE_RET_OK || rc == AWE_RET_SKIP) && vmapkey) {
		rc = load_map(vmapkey->list, locked);
		mark_loaded_presets(vmapkey->list, bank_list);
	}
	
	if (bank_list == NULL) {
		if (vmap)
			excl_list = merge_loadlist(excl_list, vmap->list);
		if (vmapkey)
			excl_list = merge_loadlist(excl_list, vmapkey->list);
		return rc;
	}

	/* make preset map list */
	vmaplist = NULL;
	if (vmapkey)
		vmaplist = make_virtual_list(vmapkey, bank_list);

	if (vmap) {
		SFPatchRec pat, map;
		for (p = vmap->list; p; p = p->next) {
			for (q = bank_list; q; q = q->next) {
				if (! awe_match_preset(&q->map, &p->map))
					continue;
				pat = p->pat;
				map = q->map;
				if (pat.keynote == -1)
					pat.keynote = q->pat.keynote;
				for (t = vmaplist; t; t = t->next) {
					if (awe_match_preset(&t->map, &map))
						break;
				}
				if (t == NULL)
					vmaplist = add_loadlist(vmaplist, &pat, &map);
			}
		}
	}

	/* check if the source preset is loaded */
	new_bank = NULL;
	for (p = vmaplist; p; p = p->next) {
		for (q = bank_list; q; q = q->next) {
			if (awe_match_preset(&q->map, &p->pat))
				break;
		}
		if (q == NULL)
			new_bank = add_loadlist(new_bank, &p->pat, &p->pat);
	}
	if (new_bank) {
		bank_list = merge_loadlist(new_bank, bank_list);
		bank_list_rebuilt = TRUE;
	}

	free_loadlist(vmaplist);

	return rc;
}

/* load default font file */
static int load_default_fonts(int locked)
{
	LoadList *vlist, *p, *q;
	int rc;

	/* make the list of fonts not loaded */
	vlist = NULL;
	for (p = bank_list; p; p = p->next) {
		if (!p->loaded)
			vlist = add_loadlist(vlist, &p->pat, &p->map);
	}
			
	if (vlist == NULL)
		return AWE_RET_OK;

	rc = load_patch(default_font, vlist, NULL, locked, TRUE);
	free_loadlist(vlist);

	return rc;
}


/* make part list for each virtual bank */
static LoadList *make_virtual_list(VBank *v, LoadList *part_list)
{
	LoadList *p, *q, *list;
	list = NULL;
	for (p = part_list; p; p = p->next) {
		for (q = v->list; q; q = q->next) {
			if (awe_match_preset(&p->map, &q->map)) {
				list = add_loadlist(list, &q->pat, &q->map);
			}
		}
	}
	return list;
}


/* read a virtual bank config file and build bank table */
static void make_bank_table(FILE *fp)
{
	char line[256], *p, *name;
	SFPatchRec pat, map;
	VBank *v;
	int len;

	while (fgets(line, sizeof(line), fp)) {
		/* discard the linefeed */
		len = strlen(line);
		if (len > 0 && line[len-1] == '\n') line[len-1] = 0; 
		/* skip spaces & comments */
		for (p = line; isspace(*p); p++)
			;
		if (!*p || *p == '#' || *p == '!' || *p == '%')
			continue;

		if (isalpha(*p)) {
			char *arg;

			arg = strschr(p, " \t\r\n");
			if (arg) {
				char *tmp = strschr(arg, " \t\r\n");
				if (tmp) *tmp = 0;
				arg++;
			}
			
			switch (*p) {
			case 'i':
			case 'I':
				include_bank_table(arg);
				break;
			case 'd':
			case 'D':
				if (default_font) free(default_font);
				default_font = safe_strdup(arg);
				break;
			default:
				fprintf(stderr, "awe: illegal bank command %s", line);
				break;
			}
			continue;
		} else if (isdigit(*p)) {
			if (! awe_parse_loadlist(p, &pat, &map, &name))
				continue;
		} else
			continue;

		if (name && *name) {
			/* discard garbage letters */
			if ((p = strschr(name, " \t\n#")) != NULL)
				*p = 0;
		}

		if (name == NULL || !*name) {
			if (map.keynote != -1)
				v = vmapkey;
			else
				v = vmap;
			if (v == NULL) {
				v = (VBank*)safe_malloc(sizeof(VBank));
				v->list = NULL;
				v->name = NULL;
				if (map.keynote != -1)
					vmapkey = v;
				else
					vmap = v;
			}

		} else {
			for (v = vbanks; v; v = v->next) {
				if (strcmp(v->name, name) == 0)
					break;
			}
		}
		if (v == NULL) {
			v = (VBank*)safe_malloc(sizeof(VBank));
			v->list = NULL;
			v->name = safe_strdup(name);
			v->next = vbanks;
			vbanks = v;
		}

		v->list = add_loadlist(v->list, &pat, &map);
	}
}


/* include another bank file */
static void include_bank_table(char *name)
{
	char path[256];
	FILE *fp;
	if (search_file_name(path, name, search_path, path_ext_bank) &&
	    (fp = fopen(path, "r")) != NULL) {
		make_bank_table(fp);
		fclose(fp);
	} else {
		fprintf(stderr, "awe: can't include bank file %s\n", name);
	}
}


/* free bank record table */
static void free_bank_table(void)
{
	VBank *v, *next;
	for (v = vbanks; v; v = next) {
		next = v->next;
		if (v->name)
			safe_free(v->name);
		free_loadlist(v->list);
		safe_free(v);
	}
	vbanks = NULL;
	if (vmap) {
		free_loadlist(vmap->list);
		safe_free(vmap);
	}
	if (vmapkey) {
		free_loadlist(vmapkey->list);
		safe_free(vmapkey);
	}
}


/*----------------------------------------------------------------
 * load sample & info on the sound driver
 *----------------------------------------------------------------*/

static int load_patch(char *name, LoadList *lp, LoadList *exlp, int locked, int load_alt)
{
	FILE *fd;
	char path[256];
	int rc;
	static SFInfo sfinfo;

	if (! search_file_name(path, name, search_path, path_ext)) {
		fprintf(stderr, "awe: can't find font file %s\n", name);
		return AWE_RET_SKIP;
	}
	if ((fd = fopen(path, "r")) == NULL) {
		fprintf(stderr, "awe: can't open SoundFont file %s\n", path);
		return AWE_RET_SKIP;
	}
	if (load_soundfont(&sfinfo, fd, TRUE) < 0) {
		fprintf(stderr, "awe: can't load SoundFont %s\n", path);
		return AWE_RET_SKIP;
	}
	correct_samples(&sfinfo);

	if (! awe_is_ram_fonts(&sfinfo)) {
		fclose(fd);
		fd = NULL;
	}

	awe_open_font(&sfinfo, fd, locked);
	rc = awe_load_font_buffered(&sfinfo, lp, exlp, load_alt);
	awe_close_font(&sfinfo);
	free_soundfont(&sfinfo);
	if (fd) fclose(fd);

	return rc;
}


/*----------------------------------------------------------------
 * load preset mapping
 *----------------------------------------------------------------*/

static int load_map(LoadList *lp, int locked)
{
	int rc;
	if (awe_open_patch("map", AWE_PAT_TYPE_MAP, locked) < 0) {
		fprintf(stderr, "awe: can't access to sequencer\n");
		return AWE_RET_SKIP;
	}

	for (; lp; lp = lp->next) {
		if ((rc = awe_load_map(lp)) != AWE_RET_OK)
			return rc;
		lp->loaded = TRUE;
	}
	awe_close_patch();
	return AWE_RET_OK;
}

