/* 
 * xwave - an interactive audio player, recorder, editor 
 * for the XWindow System
 * 
 * Copyright (C) 1996 Kai Kollmorgen
 * (kkollmor@informatik.uni-rostock.de)
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <signal.h>

#include <X11/Intrinsic.h>

#ifdef linux
#include <linux/soundcard.h>

#elif defined(FreeBSD)
#include <machine/soundcard.h>

#elif defined(sgi)
#include <dmedia/audio.h>

#elif defined(sun)
#include <sys/audioio.h>
#endif

#include "types.h"
#include "audio_file.h"
#include "xwave.h"
#include "xwave_widget.h"
#include "misc.h"
#include "sample_settings.h"
#include "audio.h"

#if defined(linux) || defined (FreeBSD) || defined(sun) 
static int set_dsp(int o_mode,int res,int channels,int freq,int *buf_size);
#endif

#ifdef sgi
static ALport set_dsp(int o_mode,int res,int channels,int freq,int *buf_size);
#endif

extern AppResources app_resources;
static Audio_File af;
static int stop_record;

#if defined (linux) || defined (FreeBSD)
int set_dsp(int o_mode,int res, int channels, int freq, int *buf_size)
{
    int check;
    int audio;
    
    audio=open (app_resources.device, o_mode, 0);
    if (audio == -1) {
	fprintf(stderr,"XWave: Error! Cannot open audio device !\n");
	return(-1);
    }
    
    check=res;
    ioctl(audio, SNDCTL_DSP_SAMPLESIZE, &check);
    if (check != res) {
	fprintf(stderr,"XWave: Device doesn't accept resolution \"%i\"!\n",res);
	return(-1);
    }
    check=channels-1;
    if (ioctl(audio, SNDCTL_DSP_STEREO, &check)==-1) {
	fprintf(stderr,"XWave: Device doesn't accept %i channels !\n",channels);
	return(-1);
    }
    check=freq;
    if (ioctl(audio, SNDCTL_DSP_SPEED, &check)==-1) {
	fprintf(stderr,"XWave: Device doesn't accept frequence \"%i\"!\n",freq);
	return(-1);
    }
    
    ioctl(audio, SNDCTL_DSP_GETBLKSIZE, buf_size);
    if (*buf_size < 1024 || *buf_size > 2*65536) {
	fprintf(stderr,"XWave: Wrong audio buffer size \"%i\"!\n",*buf_size);
	return(-1);
    }
    return(audio); 	
}

void check_audio(Main_Bool *mb)
/* test if there is a play device, and how fast we can play */
{
    int audio,check=0;
    
    audio=open (app_resources.device, O_RDWR, 0);   
    if (audio != -1) {
	mb->canplay=TRUE;
	
	check=16;
	ioctl(audio, SNDCTL_DSP_SAMPLESIZE, &check);
	if (check!=16) mb->canplay16bit=FALSE;
	else mb->canplay16bit=TRUE;
	
	/* the following tests are only for 8bit devices */
	check=8; 
	ioctl(audio, SNDCTL_DSP_SAMPLESIZE, &check);
	check=HIFREQ;
	if (ioctl(audio, SNDCTL_DSP_SPEED, &check)==-1) mb->canplay44khz=FALSE;
	else 
	  if (check<HIFREQ) 
	  mb->canplay44khz=FALSE;
        else mb->canplay44khz=TRUE;
	check=MIDFREQ;
	ioctl(audio, SNDCTL_DSP_SPEED, &check);
	check=1; /* stereo 22050khz 8bit (sbpro and compatible)*/
	if (ioctl(audio, SNDCTL_DSP_STEREO, &check)==-1) mb->canplaystereo=FALSE;
	else {
	    mb->canplaystereo=TRUE;
	    check=HIFREQ; /* stereo 44100khz 8bit (?)*/
	    if (ioctl(audio, SNDCTL_DSP_SPEED, &check)==-1) 
	      mb->canplay44khzstereo=FALSE;
	    else 
	      if (check<HIFREQ) mb->canplay44khzstereo=FALSE;
	    else mb->canplay44khzstereo=TRUE;
	}
	close(audio);
    } else mb->canplay=FALSE;
}

int get_samples(byte* audio_buffer,int mode)
{
    static int audio=-1;
    static int buf_size;
    
    if (mode==AU_READ) return(read(audio, audio_buffer, buf_size));
    
    if (mode==AU_OPEN) {
	int check=4;
	audio=set_dsp(O_RDONLY,LEVRES,LEVMOD,LEVFRQ,&buf_size);
	ioctl(audio,SNDCTL_DSP_SUBDIVIDE,&check);
	ioctl(audio, SNDCTL_DSP_GETBLKSIZE, &buf_size);
	return(buf_size);
    }
    
    if (mode==AU_CLOSE) {
	if (audio!=-1) close(audio);
	audio=-1;
	return(0);
    }
    return(0);
}
#endif

#ifdef sgi
void check_audio(Main_Bool *mb)
{
    ALconfig config=ALnewconfig();
    ALport port=0;
    long pvbuf[2];
    
    switch (AU_RES) {
     case 8: 
	ALsetwidth(config, AL_SAMPLE_8);
	break;
     case 16:
	ALsetwidth(config, AL_SAMPLE_16);
	break;
     case 24:
	ALsetwidth(config, AL_SAMPLE_24);
	break;
    }
    ALsetchannels(config,AU_CHN);
    pvbuf[0] = AL_OUTPUT_RATE;
    pvbuf[1] = AU_RATE;
    ALsetparams(AL_DEFAULT_DEVICE,pvbuf,2);
    
    port = ALopenport("Play","w",config);

    ALfreeconfig(config);
    
    if(port == (ALport)0) {
	mb->canplay=False;
	return;
    }
    
    mb->canplay=True;
    mb->canplay16bit=True;
    mb->canplay44khz=True;
    mb->canplaystereo=True;
    mb->canplay44khzstereo=True;
}


ALport set_dsp(int o_mode,int res, int channels, int freq, int *buf_size)
{
    ALconfig config=ALnewconfig();
    ALport port=0;
    long pvbuf[2];
    
    switch (res) {
     case 8: 
	ALsetwidth(config, AL_SAMPLE_8);
	break;
     case 16:
	ALsetwidth(config, AL_SAMPLE_16);
	break;
     case 24:
	ALsetwidth(config, AL_SAMPLE_24);
	break;
    }
    ALsetchannels(config,channels);
    if (o_mode == O_WRONLY) pvbuf[0] = AL_OUTPUT_RATE;
    if (o_mode == O_RDONLY) pvbuf[0] = AL_INPUT_RATE;
    pvbuf[1] = freq;
    ALsetparams(AL_DEFAULT_DEVICE,pvbuf,2);

    if (o_mode==O_RDONLY) ALsetqueuesize(config,8192);

    *buf_size=(int) ALgetqueuesize(config);

    if (o_mode == O_WRONLY) port = ALopenport("Play","w",config);
    if (o_mode == O_RDONLY) port = ALopenport("Record","r",config);

    ALfreeconfig(config);

    if(port == (ALport)0) {
	fprintf(stderr,"XWave: Error! Cannot open audio device !\n");
	return((ALport) 0);
    }
    return(port);
}

int get_samples(byte* audio_buffer,int mode)
{
    static ALport port;
    static int buf_size;
    int count;
    
    if (mode==AU_READ && buf_size!=-1) {
	ALreadsamps(port, audio_buffer,buf_size/(LEVRES/8));
	return(buf_size);
    }    
    
    if (mode==AU_OPEN) {
	port=set_dsp(O_RDONLY,LEVRES,LEVMOD,LEVFRQ,&buf_size);
	if (port==(ALport)0) buf_size=-1;
	return(buf_size);
    }
    
    if (mode==AU_CLOSE) {
	if (port!=(ALport)0) ALcloseport(port);
	port=0;
	return(0);
    }
    
    return(0);
}
#endif

#ifdef sun
void check_audio(Main_Bool *mb)
/* test if there is a play device, and how fast we can play */
{
    char *dev_name;
    int audio,check=0;
    audio_device_t dev_info;
    audio_info_t au_info;
    
    mb->canplay=False;
    dev_name=malloc(strlen(app_resources.device)+strlen(CTL)+1);
    if (dev_name==NULL) {
	puts(app_resources.err_mem);
	return;
    }
    sprintf(dev_name,"%s%s",app_resources.device,CTL);
    printf("device=%s\n",dev_name);
    audio=open (dev_name, O_WRONLY);
    if (audio != -1) {

	ioctl(audio,AUDIO_GETDEV,&dev_info);
	printf("%s\n%s\n%s\n",dev_info.name,dev_info.version,dev_info.config);
	ioctl(audio,AUDIO_GETINFO,&au_info);
	printf("rate: %i chn: %i res: %i ",au_info.play.sample_rate,
	       au_info.play.channels,au_info.play.precision);
	switch (au_info.play.encoding) {
	 case AUDIO_ENCODING_ULAW:
	    printf("ULAW\n");break;
	 case AUDIO_ENCODING_ALAW:
	    printf("ULAW\n");break;
	 case AUDIO_ENCODING_LINEAR:
	    printf("ULAW\n");break;
	 default:
	    printf("unknown encoding\n");
	}
	close(audio);
    } else  {
	printf("failed.\n");
    }

}

void ini_audio(Main_Bool *mb)
{
    int audio;
    struct audio_info sfd_info;
     
    mb->canplay=False;

    audio=open (app_resources.device, O_WRONLY);
    if (audio != -1) {
	AUDIO_INITINFO(&sfd_info); 
	sfd_info.play.sample_rate = AU_RATE;
	sfd_info.play.channels = AU_CHN;
	sfd_info.play.precision = AU_RES;
	sfd_info.play.encoding = AUDIO_ENCODING_LINEAR;
	if (ioctl(audio, AUDIO_SETINFO, &sfd_info)) 
	  fprintf(stderr,"Can't use desired specs !\n");
	else {
	    mb->canplay=True;
	    mb->canplay16bit=True;
	    mb->canplay44khz=True;
	    mb->canplaystereo=True;
	    mb->canplay44khzstereo=True;
	}
	close(audio);
    }
}


int set_dsp(int o_mode,int res, int channels, int freq, int *buf_size)
{
    int audio;
    struct audio_info sfd_info;
     
    if ((audio=open (app_resources.device, o_mode))==-1) return(-1);

    AUDIO_INITINFO(&sfd_info); 
    sfd_info.play.sample_rate = freq;
    sfd_info.play.channels = channels;
    sfd_info.play.precision = res;
    sfd_info.play.encoding = AUDIO_ENCODING_LINEAR;
    if (ioctl(audio, AUDIO_SETINFO, &sfd_info)) return(-1);
    *buf_size=res/8*freq;
    return(audio);
}

int get_samples(byte* audio_buffer,int mode)
{
    return(-1);
}
#endif


void play_file(char *fname,Main_Bool *mb)
{
#if defined (linux) || defined (FreeBSD) || defined (sun)
    int audio;
#elif defined(sgi)
    ALport port;
#endif
    Audio_File af;
    int buf_size,length;
    char *buffer;
    
    if (af_open(fname,&af,AF_OPEN)==AF_ERROR) {
	kill((pid_t) getppid(),SIGUSR1);
	return;
    }
    
#if defined (linux) || defined (FreeBSD) || defined (sun)
    if ((audio=set_dsp(O_WRONLY,af.bps,af.channels,af.freq,&buf_size))==-1) {
	fprintf(stderr,"XWave: Error ! Cannot set dsp !\n");
	close(af.fd);
	kill((pid_t) getppid(),SIGUSR1);
	return;
    }
#elif defined(sgi)
    if ((port=set_dsp(O_WRONLY,af.bps,af.channels,af.freq,&buf_size))
	==(ALport)0) {
	fprintf(stderr,"XWave: Error ! Cannot set dsp !\n");
	close(af.fd);
	kill((pid_t) getppid(),SIGUSR1);
	return;
    }
#endif
    
    if ((buffer=malloc(buf_size))==NULL) {
	fprintf(stderr,"XWave: Error ! Cannot alloc mem !\n");
	close(af.fd);
#if defined (linux) || defined (FreeBSD) || defined (sun)
	close(audio);
#elif defined(sgi)
	ALcloseport(port);
#endif
	kill((pid_t) getppid(),SIGUSR1);
	return;
    }
    
#if defined (linux) || defined (FreeBSD) || defined (sun)
    while ((length=af_read(af,buffer,buf_size))>0) {
	if (write(audio, buffer, length)==-1) break;
    }
    close(audio);
#elif defined(sgi)
    while ((length=af_read(af,buffer,buf_size))>0) {
	if (ALwritesamps(port, buffer, length/(af.bps/8))==-1) break;
    }
    while (ALgetfilled(port)) sginap(2);
    ALcloseport(port);
#endif
    
    free(buffer);
    close(af.fd);
    kill((pid_t) getppid(),SIGUSR1);
    return;
}

void play_buffer(Main_Data *md)
{
    int buf_size;
    byte *buffer;
    Wave_Data *wd=md->wd;
    int offset=0,playlength,length;
#ifdef sgi
    ALport port;
#elif defined(linux) || defined (FreeBSD) || defined (sun)
    int audio;
    
    if ((audio=set_dsp(O_WRONLY,wd->res,wd->channels,wd->freq,&buf_size))==-1) {
	fprintf(stderr,"XWave: Error ! Cannot set dsp \"%s\"!\n",
		app_resources.device);
	kill((pid_t) getppid(),SIGUSR1);
	return;
    }
#endif    
#ifdef sgi
    if ((port=set_dsp(O_WRONLY,wd->res,wd->channels,wd->freq,&buf_size))
	==(ALport)0) {
	fprintf(stderr,"XWave: Error ! Cannot set dsp !\n");
	kill((pid_t) getppid(),SIGUSR1);
	return;
    }
#endif
    
    playlength=wd->length;
    if (md->wd->ismark) {
	offset=wd->markbeg*wd->bpspl;
	playlength=wd->marklength*wd->bpspl;
    }
    if (md->wd->isplay) {
	offset=wd->markbeg*wd->bpspl;
	playlength-=wd->markbeg*wd->bpspl;
    }
    
    if (md->wd->inmem) {
	buffer=wd->buffer+offset;
	while (playlength>0) {
	    if (playlength<buf_size) buf_size=playlength;
#if defined (linux) || defined (FreeBSD) || defined (sun)
	    if (write(audio, buffer, buf_size)==-1) {
		close(audio);
		kill((pid_t) getppid(),SIGUSR1);
		return;
	    }
#elif defined(sgi)
	    if (ALwritesamps(port,buffer,buf_size/(wd->res/8))==-1) {
		printf("error while writing port (playlength=%i)!\n",playlength);
		ALcloseport(port);
		kill((pid_t) getppid(),SIGUSR1);
		return;
	    }	    
#endif
	
	    playlength-=buf_size;
	    buffer+=buf_size;
	}
    } else {
	Audio_File af;
	
	wd2af(wd,&af);
	af_rewind(af);
	if (af_seek(af,offset,SEEK_CUR)==AF_ERROR) {
#if defined (linux) || defined (FreeBSD) || defined (sun)
	    close(audio);
#elif defined(sgi)
	    ALcloseport(port);
#endif
	    kill((pid_t) getppid(),SIGUSR1);
	    return;
	}
	if (MAXUSHORT<buf_size) buf_size=MAXUSHORT;
	
	while (playlength>0) {
	    if (playlength<buf_size) buf_size=playlength;
	    if ((length=af_read(af,(char*) md->mg->fbuf,buf_size))==-1) {
#if defined (linux) || defined (FreeBSD) || defined (sun)
		close(audio);
#elif defined(sgi)
		ALcloseport(port);
#endif
		kill((pid_t) getppid(),SIGUSR1);
		return;
	    }
#if defined (linux) || defined (FreeBSD) || defined (sun)
	    if (write(audio,(char*) md->mg->fbuf,length)==-1) {
		close(audio);
		kill((pid_t) getppid(),SIGUSR1);
		return;
	    }
#elif defined(sgi)
	    if ((length=ALwritesamps(port,md->mg->fbuf,length/(wd->res/8)))==-1) {
		ALcloseport(port);
		kill((pid_t) getppid(),SIGUSR1);
		return;
	    }
    	    while (ALgetfilled(port)) sginap(2);
#endif
	    playlength-=length;
	}
    }
    
#if defined (linux) || defined (FreeBSD) || defined (sun)
    close(audio);
#elif defined(sgi)
    while (ALgetfilled(port)) sginap(2);
    ALcloseport(port);
#endif
    
    kill((pid_t) getppid(),SIGUSR1);
    return;
}

#ifdef sgi
    ALport port;
#endif

void rec_file(byte res,byte mode,int freq,char *fname)
{
    void write_length();
    byte *data=NULL;
    int abuf_size,count;
#if defined (linux) || defined (FreeBSD) || defined (sun)
    int audio;
    
    if ((audio=set_dsp(O_RDONLY,res,mode,freq,&abuf_size))==-1) {
	fprintf(stderr,"XWave: Error ! Cannot open audio device !\n");
	kill((pid_t) getppid(),SIGUSR2);
	return;
    }
#endif
    
#ifdef sgi
    if ((port=set_dsp(O_RDONLY,res,mode,freq,&abuf_size))==(ALport)0) {
	fprintf(stderr,"XWave: Error ! Cannot open audio device !\n");
	kill((pid_t) getppid(),SIGUSR1);
	return;
    }
#endif
    
    
    if ((data = (byte *) malloc(abuf_size)) == NULL) {
	fprintf(stderr,"XWave: Error while alloc mem for audio_buffer !\n");
#if defined (linux) || defined (FreeBSD) || defined (sun)
	close(audio);
#elif defined(sgi)
	ALcloseport(port);
#endif
	kill((pid_t) getppid(),SIGUSR2);
	return;
    }
    
    af.bps=res;
    af.channels=mode;
    af.freq=freq;
    af.length=0;
    af.comp=app_resources.default_comp;
    af.type=app_resources.default_ftype;
    
    if (af_open(fname,&af,AF_NEW)==AF_ERROR) {
	free(data);
#if defined (linux) || defined (FreeBSD) || defined (sun)
	close(audio);
#elif defined(sgi)
	ALcloseport(port);
#endif
	kill((pid_t) getppid(),SIGUSR2);
	return;
    }
    
    stop_record=FALSE;
    signal(SIGUSR1,write_length);
    
    while(1) {
#if defined (linux) || defined (FreeBSD) || defined (sun)
	if ((count=read(audio, data, abuf_size))==-1) {
	    close(audio);
	    af_close(af);
	    kill((pid_t) getppid(),SIGUSR2);
	    return;
	}
#elif defined(sgi)
	if ((ALreadsamps(port, data,abuf_size/(af.bps/8)))==-1) {
	    ALcloseport(port);
	    af_close(af);
	    kill((pid_t) getppid(),SIGUSR2);
	    return;
	}
	count=abuf_size;
#endif

	if (stop_record) {
#ifdef sgi
    	    ALcloseport(port);
#endif
	    return;
	}
	if ((count=af_write(af,(char*)data,count))==AF_ERROR) {
#if defined (linux) || defined (FreeBSD) || defined (sun)
	    close(audio);
#elif defined(sgi)
	    ALcloseport(port);
#endif
	    af_close(af);
	    kill((pid_t) getppid(),SIGUSR2);
	    return;
	}
	af.length+=count;
    }
}

void write_length()
{
    stop_record=TRUE;
    af_close(af);
    kill((pid_t) getppid(),SIGUSR1);
}

