/*
 * Copyright (c) 1994 Paul Vojta.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * NOTE:
 *	xdvi is based on prior work, as noted in the modification history
 *	in xdvi.c.
 */

#include "xdvi-config.h"
#include <kpathsea/c-ctype.h>
#include <kpathsea/c-fopen.h>
#include <kpathsea/c-vararg.h>

#ifdef VMS
#include <rmsdef.h>
#endif /* VMS */


/*
 *	General utility routines.
 */

/*
 *	Print error message and quit.
 */

#ifdef HAVE_PROTOTYPES
NORETURN void
oops(_Xconst char *message, ...)
#else
/* VARARGS */
NORETURN void
oops(va_alist)
	va_dcl
#endif
{
#ifndef HAVE_PROTOTYPES
	_Xconst char *message;
#endif
	va_list	args;

	Fprintf(stderr, "%s: ", prog);
#ifdef HAVE_PROTOTYPES
	va_start(args, message);
#else
	va_start(args);
	message = va_arg(args, _Xconst char *);
#endif
	(void) vfprintf(stderr, message, args);
	va_end(args);
	Putc('\n', stderr);
#if	PS
	psp.destroy();
#endif
	exit(1);
}

/*
 *	Either allocate storage or fail with explanation.
 */

char *
xmalloc(size, why)
	unsigned	size;
	_Xconst char	*why;
{
	/* Avoid malloc(0), though it's not clear if it ever actually
	   happens any more.  */
	char *mem = malloc(size ? size : 1);

	if (mem == NULL)
	    oops("! Cannot allocate %u bytes for %s.\n", size, why);
	return mem;
}

/*
 *	Allocate bitmap for given font and character
 */

void
alloc_bitmap(bitmap)
	register struct bitmap *bitmap;
{
	register unsigned int	size;

	/* width must be multiple of 16 bits for raster_op */
	bitmap->bytes_wide = ROUNDUP((int) bitmap->w, BITS_PER_BMUNIT) *
	    BYTES_PER_BMUNIT;
	size = bitmap->bytes_wide * bitmap->h;
	bitmap->bits = xmalloc(size != 0 ? size : 1, "character bitmap");
}


/*
 *	Hopefully a self-explanatory name.  This code assumes the second
 *	argument is lower case.
 */

int
memicmp(s1, s2, n)
	_Xconst char	*s1;
	_Xconst char	*s2;
	size_t		n;
{
	while (n > 0) {
	    int i = TOLOWER(*s1) - *s2;
	    if (i != 0) return i;
	    ++s1;
	    ++s2;
	    --n;
	}
	return 0;
}


/*
 *	Close the pixel file for the least recently used font.
 */

static	void
close_a_file()
{
	register struct font *fontp;
	unsigned short oldest = ~0;
	struct font *f = NULL;

	for (fontp = font_head; fontp != NULL; fontp = fontp->next)
	    if (fontp->file != NULL && fontp->timestamp <= oldest) {
		f = fontp;
		oldest = fontp->timestamp;
	    }
	if (f == NULL)
	    oops("Can't find an open pixel file to close");
	Fclose(f->file);
	f->file = NULL;
	++n_files_left;
}

/*
 *	Open a file in the given mode.
 */

FILE *
#ifndef	VMS
xfopen(filename, type)
	_Xconst char	*filename;
	_Xconst char	*type;
#define	TYPE	type
#else
xfopen(filename, type, type2)
	_Xconst char	*filename;
	_Xconst char	*type;
	_Xconst char	*type2;
#define	TYPE	type, type2
#endif	/* VMS */
{
	FILE	*f;

        /* Try not to let the file table fill up completely.  */
	if (n_files_left <= 5)
	  close_a_file();
#ifdef HTEX
	if (URL_aware && ((URLbase != NULL) || (htex_is_url(filename)))) {
		int i;
		i = fetch_relative_url(URLbase, filename, temporary_dir);
		if (i < 0) return NULL;
		wait_for_urls(); /* Don't bother waiting right now... */
		f = fopen(filelist[i].file, TYPE); /* This needs to be set somehow... */
	} else 
#endif /* HTEX */
	f = fopen(filename, TYPE);
	/* Interactive Unix 2.2.1 doesn't set errno to EMFILE
  	   or ENFILE even when it should, but if we do this
  	   unconditionally, then giving a nonexistent file on the
  	   command line gets the bizarre error `Can't find an open pixel
  	   file to close' instead of `No such file or directory'.  */
#ifndef	VMS
	if (f == NULL && (errno == EMFILE || errno == ENFILE))
#else	/* VMS */
	if (f == NULL && errno == EVMSERR && vaxc$errno == RMS$_ACC)
#endif	/* VMS */
	{
	    n_files_left = 0;
	    close_a_file();
	    f = fopen(filename, TYPE);
	}
#ifdef	F_SETFD
	if (f != NULL) (void) fcntl(fileno(f), F_SETFD, 1);
#endif
	return f;
}
#undef	TYPE

/*
 *	Open a file, but temporary disable URL awareness.
 */

#ifdef HTEX
FILE*
xfopen_local(filename, type)
	_Xconst char	*filename;
	_Xconst char	*type;
{
	FILE *f;
	int url_aware_save;
	url_aware_save = URL_aware;
	URL_aware = FALSE;
	f = xfopen(filename, type);
	URL_aware = url_aware_save;
	return f;
}
#endif /* HTEX */

#ifdef	PS_GS
/*
 *	Create a pipe, closing a file if necessary.  This is (so far) used only
 *	in psgs.c.
 */

int
xpipe(fd)
	int	*fd;
{
	int	retval;

	for (;;) {
	    retval = pipe(fd);
	    if (retval == 0 || (errno != EMFILE && errno != ENFILE)) break;
	    n_files_left = 0;
	    close_a_file();
	}
	return retval;
}
#endif	/* PS_GS */


/*
 *
 *      Read size bytes from the FILE fp, constructing them into a
 *      signed/unsigned integer.
 *
 */

unsigned long
num(fp, size)
	register FILE *fp;
	register int size;
{
	register long x = 0;

	while (size--) x = (x << 8) | one(fp);
	return x;
}

long
snum(fp, size)
	register FILE *fp;
	register int size;
{
	register long x;

#if	__STDC__
	x = (signed char) getc(fp);
#else
	x = (unsigned char) getc(fp);
	if (x & 0x80) x -= 0x100;
#endif
	while (--size) x = (x << 8) | one(fp);
	return x;
}
